/*
 *  Program to try and read Tops-20 Dumper format tapes
 *
 *                   Jim Guyton,  Rand Corporation
 *                   Original       10/20/82
 *                   jdg:   -n added 6/11/83
 *                   jdg:    can now extract 8-bit-byte files  2/9/86
 *		     clh:    handle errors and blocked tapes. archive mode
 */

#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mtio.h>
#include "dump.h"
#define MAXBLKS 16
				/* logfile should be changable */
#define LOGFILE "Logfile"

char *ctime(), *index(), *rindex();
char *unixname();

int  fdTape;                    /* File handle for Dumper-20 format tape */
char tapebuffer[MAXBLKS][TAPEBLK];
char *tapeblock;	        /* One logical record from tape */
FILE *fpFile;                   /* Output file handle on extracts */
int debug = 0;
int textflg = 0;                /* Non-zero if retr binary files as text */
int arcmode = 0;		/* non-zero if doing archive retrieves */
int numflg = 0;                 /* Non-zero if using numeric filenames */
int number;                     /* Current output file "number" */

/* data for next archive retrieve */

int arcset, arcfile;		/* desired file */
int arcuid;			/* should be owned by this guy */
char arcname[2000];		/* put it here */

#define TAPE "/dev/rmt12"        /* Default input tape */

int  bytesize;          /* Number of bits/byte in current file */
int  numbytes;          /* Number of bytes in current file */
int  pgcount;           /* Number of twenex pages in file */

int  sequence,newseq;   /* Current sequence number */
int  logrecord;         /* Current logical record in block */
int  eofs;              /* How many EOF's seen */
int  scd;               /* Type of tape (normal, archive, etc) */
int  maxblks;		/* Actual max blocking factor to try */
struct mtget mtgetblk;	/* Data from mtget */
extern int  errno;
int  sawlabels = 0;	/* Nonzero if we have seen labels */

char *pattern = 0;      /* Filename match pattern */

/*
	pgm [-a]  [-f tapefile] [-t] [-n number] pattern

	no pattern == directory only
	no tapefile == /dev/rmt0
	-a == archive retrieves. read specs from stdin
	-t == try to pretend files are 7-bit ascii
	-n == use numeric filenames in extracts, number is 1st name
*/

main(argc, argv)
int argc;
char *argv[];
{
	char *tape = TAPE;              /* Pathname for tape device/file */
	int rc;

	/* Do switch parsing */

	while(argc>1 && argv[1][0] == '-'){
		switch(argv[1][1]){
		case 'f':
			if (argc <= 2) punt("Need filename after -f\n");
			tape = argv[2];
			argc--; argv++;
			break;
		case 't':             /* Force text mode on "binary" files */
			textflg = 1;
			break;
		case 'd':
			debug = atoi(&argv[1][2]);
			printf("Debug value set to %d\n", debug);
			break;
		case 'n':               /* numeric output filenames */
			if (argc <= 2) punt("Need number after -n\n");
			number = atoi(argv[2]);         /* First file name */
			numflg = 1;
			argc--; argv++;
			break;
		case 'a':
			arcmode = 1;
			textflg = 1;
			break;
		default:
			printf("unknown flag %s\n", argv[1]);
			exit(1);
			break;
		}
		argc--;  argv++;
	}

	if (arcmode) {
	  arcuid = geteuid();
	  if (arcuid != 0) {
	    fprintf(stderr, "Only root can use archive mode\n");
	    exit(1);
	  }
	  if (fscanf(stdin, "%d %d %d %s", &arcset, &arcfile, &arcuid, 
		     arcname) != 4)
	    exit(0);
	}	  

	if (argc > 1)
		pattern = argv[1];

	maxblks = MAXBLKS;		/* Normally large blocking factor */
	fdTape = open(tape, 0);         /* Open tape for read */
	if (fdTape == -1)
		punt("Couldn't open 'tape' file %s\n", tape);

#ifdef pyr
/* 
 * On Pyramid's old IOC controller, the maximum blocking factor the
 * hardware can handle is 11.  Since we have some tapes written at
 * 15, it's fairly important to set maxblks down to 11 only on
 * the machine with that old controller.
 */
	
	if (ioctl(fdTape, MTIOCGET, &mtgetblk) >= 0 &&
	    mtgetblk.mt_type == MT_TIOC)  /* IOC/Tapemaster controller */
	  maxblks = 11;
#endif

	sequence = 0;
	rc = 0;
	eofs = 0;
	for ( ; ; )             /* Loop till end of tape */
	{
					 /*** Read a block ***/
	  	
	  	while (1)
		{
		  	if (rc == 0) {
			  rc = read(fdTape, tapebuffer, TAPEBLK * maxblks);
			  logrecord = 0;
			  if (rc < 0) {
			    printf("Error %d on tape after record %d\n",
				   errno, sequence);
			    rc = 0;
			    close(fdTape);
			    fdTape = open(tape, 0); 
			    if (fdTape == -1)
			      punt("Couldn't open 'tape' file %s\n", tape);
			    continue;
			  }
			  if (rc == 80) {
			    if (! sawlabels) {
			      if (strncmp(tapebuffer[0],"VOL1",4) != 0)
				fprintf(stderr,"WARNING: Tape has record of size 80 that is not a legal label\n");
			      tapebuffer[0][10] = 0;
			      printf("[labelled tape VOLID %s]\n",&tapebuffer[0][4]);
			      sawlabels = 1;
			    }
			    rc = 0;
			    eofs = 0;
			    sequence = 0;
			    continue;
			  }
			  if (rc == 0) {
			    if (eofs) {
			      printf("End of Tape\n");
			      exit(0);
			    }
			    printf("End of Saveset\n");
			    eofs++;
			    logrecord = 0;
			    continue;
			  }
 			}
			if (rc > 0 && rc < TAPEBLK) {
			  printf("Bad record size after record %d\n",sequence);
			  exit(1);
			}
			eofs = 0;
			tapeblock = &tapebuffer[logrecord];
			logrecord++;
			rc -= TAPEBLK;
			newseq = getsequence(tapeblock);
			if (newseq <= sequence)
				printf("\nDuplicate record %d ignored\n", 
				       newseq);
			else break;
		}
		if (newseq > sequence + 1)
			printf("\nRecord %d out of sequence, continuing...\n",
				newseq);
		sequence = newseq;

					/*** Do something with it ***/
		switch(getrecordtype(tapeblock))
		{
		  case RectypeData:             /* Data block */
					doDatablock(tapeblock);
					break;

		  case RectypeTphd:             /* Saveset header */
					doSaveset(tapeblock, 0);
					break;

		  case RectypeFlhd:             /* File header */
					doFileHeader(tapeblock);
					break;

		  case RectypeFltr:             /* File trailer */
					doFileTrailer(tapeblock);
					break;

		  case RectypeTptr:             /* Tape trailer */
					doTapeTrailer(tapeblock);
					break;

		  case RectypeUsr:              /* User directory info ? */
					if (debug > 5)
					  printf("User info record skipped\n");
					break;

		  case RectypeCtph:             /* Continued saveset hdr */
					doSaveset(tapeblock, 1);
					break;

		  case RectypeFill:             /* Fill record */
					if (debug > 10)
					  printf("Fill record skipped\n");
					break;

		  default:
					punt("Unknown record type 0x%x\n",
						  getrecordtype(tapeblock));
					break;
		}
	}
}

/* Get the "record type" from the tape block header.  Since it */
/* is stored in 2's complement form, negate it before returning */

getrecordtype(block)
char *block;
{
	long int tl;
	tl = getfield(block, WdoffRectype, BtoffRectype, BtlenRectype);
	return( (int) -tl);
}

getsequence(block)
char *block;
{
	long int tl;
	tl = getfield(block, WdoffRecseq, BtoffRecseq, BtlenRecseq);
	return( (int) tl);
}

int   masks[32] =       /* bitmasks for different length fields */
{              0x00000001, 0x00000003, 0x00000007,
   0x0000000f, 0x0000001f, 0x0000003f, 0x0000007f,
   0x000000ff, 0x000001ff, 0x000003ff, 0x000007ff,
   0x00000fff, 0x00001fff, 0x00003fff, 0x00007fff,
   0x0000ffff, 0x0001ffff, 0x0003ffff, 0x0007ffff,
   0x000fffff, 0x001fffff, 0x003fffff, 0x007fffff,
   0x00ffffff, 0x01ffffff, 0x03ffffff, 0x07ffffff,
   0x0fffffff, 0x1fffffff, 0x3fffffff, 0x7fffffff,
   0xffffffff
};


long
getfield(block, wordoff, bitoff, bitlen)
char *block;            /* Tape block record */
int wordoff;            /* 36-bit word offset */
int bitoff;             /* Bit offset of field (from msb) */
int bitlen;             /* Bit length of field */
{
	char *p;                /* Used to point into record */
	long int w32;           /* First 32 bits of the 36 bit word */
	int   w4;               /* Last 4 bits of the 36 bit word */
	long  w = 0;            /* the word to return */


				/* First, the "illegal" kludge */
	if (bitoff == 0 && bitlen == 36)
	{       bitoff = 4;
		bitlen = 32;

	}
	if (bitlen > 32) punt("I can't get that large a field!\n");

	/* A PDP-10 (or 20) 36-bit word is laid out with the first 32 bits
	   as the first 4 bytes and the last 4 bits are the low order 4 bits
	   of the 5th byte.   The high 4 bits of that byte should be zero */

	p = block + (5*wordoff);        /* Get ptr to word of interest */
	w32 = *p++ & 0377;                      /* First byte */
	w32 = (w32 << 8) | (*p++ & 0377);       /* 2nd */
	w32 = (w32 << 8) | (*p++ & 0377);       /* 3rd */
	w32 = (w32 << 8) | (*p++ & 0377);       /* 4th */
	w4  = *p;                               /* 5th */
	if (w4 > 017) punt("Not a PDP-10 tape!  w4=%o\n", w4);


	/* Get the field right justified in the word "w".
	   There are three cases that I have to handle:
	      [1] field is contained in w32
	      [2] field crosses w32 and w4
	      [3] field is contained in w4
	*/

	if (bitoff+bitlen <= 32)        /* [1] field is contained in w32 */
	{
		w = w32 >> (32 - (bitoff+bitlen));
	}
	else if (bitoff <= 32)          /* [2] field crosses boundary */
	{
		w =  (w32 << (bitoff+bitlen-32))
		   | (w4  >> (36 - (bitoff+bitlen)));
	}
	else                            /* [3] field is contained in w4 */
	{
		w = w4 >> (36 - (bitoff+bitlen));
	}
	w = w & masks[bitlen-1];          /* Trim to proper size */
	return(w);
}


doDatablock(block)
char *block;
{
					    /* max is 5 bytes per word */
	static char buf[(512*5)+1];         /* A page of characters */
	int ct;
	int maxperblock;

	if (debug > 10) printf("*");
	if (fpFile == NULL) return;

					/* only handle 7 and 8 bit bytes */
	switch (bytesize) {
	   case 7:      maxperblock = 512*5; break;
	   case 8:      maxperblock = 512*4; break;
	   default:     return;
	}

	if (numbytes > maxperblock) ct = maxperblock;
	else ct = numbytes;

	if (bytesize == 7) {
		getstring(block, buf, 6, ct);
		buf[ct] = 0;
		fprintf(fpFile, "%s", buf);
	}
	else {          /* if not 7, then 8bit */
		getbytes(block, buf, 6, ct);
		write(fileno(fpFile), buf, ct);
	}
	numbytes -= ct;
}

char *savesettype[4] = {"", "Collection", "Archive", "Migration"};

doSaveset(block, contflag)
char *block;
int  contflag;
{
	static char name[100];
	long t;

	if (debug > 10) printf("\nSaveset header:");
	getstring(block, name, WdoffSSName, sizeof(name));

	t = unixtime(block, WdoffSSDate);
	scd = getfield(block, WdoffSCD, BtoffSCD, BtlenSCD);
	
	printf("%s Saveset '%s', %s\n", savesettype[scd], name, ctime(&t));

}

doFileHeader(block)
char *block;
{
	static char name[100];
	long t;                 /* The time in unix format */
	char *ts;

	if (debug > 5) printf("File Header block:\n");

	getstring(block, name, WdoffFLName, sizeof(name));
	ts = index(name, ';');          /* Chop off ;Pprotection;Aacct */
	*ts = 0;

	t = unixtime(block, WdoffFDB_Wrt);
	ts = ctime(&t) + 4;             /* Skip over day-name field */
	ts[strlen(ts)-1] = 0;             /* Chop off \n at end */

	bytesize = getfield(block, WdoffFDB_BSZ, BtoffFDB_BSZ, BtlenFDB_BSZ);
	numbytes = getfield(block, WdoffFDB_Size, BtoffFDB_Size, BtlenFDB_Size);
	pgcount  = getfield(block, WdoffFDB_PGC, BtoffFDB_PGC, BtlenFDB_PGC);

	if (! arcmode) 
	  printf("%6d %11d(%2d) %s %s\n",
		 pgcount, numbytes, bytesize, ts, name);

	if (pattern && match(name, pattern) ||
	    arcmode && arcset == getfield(block, WdoffSaveSetNum, 
					  BtoffSaveSetNum, BtlenSaveSetNum)
	            && arcfile == getfield(block, WdoffFileNum,
					   BtoffFileNum, BtlenFileNum))
	{
	  	if (arcmode)
		    printf("%6d %11d(%2d) %s %s => %s\n",
			   pgcount, numbytes, bytesize, ts, name, arcname);

					      /* Special hack for bad files */
		if (bytesize != 7 && textflg)
		{
			if (bytesize == 0 || bytesize == 36)     /* Sigh */
			{       bytesize = 7;
				numbytes = numbytes * 5;
			}
		}
		if (bytesize != 7 && bytesize != 8)
			fprintf(stderr, "Skipping -- binary file.\n");
		else
		{
		    if (arcmode) {
		      fpFile = fopen(arcname, "w");
		      if (fpFile == NULL)
			fprintf(stderr, "Skipping -- can't open.\n");
		    }
		    else {
		      fpFile = fopen(unixname(name), "w");
		      if (fpFile == NULL)
			punt("Can't open %s for write!\n", unixname(name));
		      printf("Extracting\n");
		    }
		}
	}
	else
		fpFile = NULL;
}

doFileTrailer(block)
char *block;
{
	if (debug > 10) printf(" File trailer\n");
	if (fpFile != NULL)
	{
		fclose(fpFile);
		fpFile = NULL;
		if (arcmode) {
		  chown(arcname, arcuid, -1);
		  if (fscanf(stdin, "%d %d %d %s", &arcset, &arcfile,
			     &arcuid, arcname) != 4)
		    exit(0);
		}

	}
}

doTapeTrailer(block)
char *block;
{
	if (debug > 10) printf("Tape Trailer");
}

punt(s, arg)
char *s;
int  arg;
{
	fprintf(stderr, s, arg);
	exit(1);
}


getstring(block, s, wordoff, max)
char *block;            /* Tape block */
char *s;                /* Destination string buffer */
int  wordoff;           /* 36-bit offset from start of tape block */
int  max;               /* Max number of characters to xfer into s */
{
	register int i;         /* Counter for five characters per word */
	int ct = 0;             /* Number of characters loaded so far */
	char *orig = s;         /* Save for debugging */

	while (ct < max)
	{
		for (i = 0; i < 5; i++)
		{
			*s = getfield(block, wordoff, i*7, 7);
			if (*s == 0) return;
			s++;
		}
		wordoff++;
		ct += 5;
	}
   /**     punt("String greater than %d characters.\n", max);   **/
}

/* getbytes: like getstring, but ...
   1) uses 8 bit bytes
   2) doesn't stop on a zero
*/
getbytes(block, s, wordoff, max)
char *block;            /* Tape block */
char *s;                /* Destination string buffer */
int  wordoff;           /* 36-bit offset from start of tape block */
int  max;               /* Max number of characters to xfer into s */
{
	register int i;         /* Counter for five characters per word */
	int ct = 0;             /* Number of characters loaded so far */
	char *orig = s;         /* Save for debugging */

	while (ct < max)
	{
		for (i = 0; i < 4; i++)
		{
			*s = getfield(block, wordoff, i*8, 8);
		   /*   if (*s == 0) return;    */
			s++;
		}
		wordoff++;
		ct += 4;
	}
   /**     punt("String greater than %d characters.\n", max);   **/
}


#define SecPerTick  (24.*60.*60.)/0777777
#define DayBaseDelta 0117213            /* Unix day 0 in Tenex format */

long
unixtime(block, wordoff)
char *block;
int  wordoff;
{
	long int t, s;

	t = getfield(block, wordoff, 0, 18);    /* First half is day */
	t -= DayBaseDelta;                      /* Switch to unix base */
						/* Now has # days since */
						/* Jan 1, 1970 */

	s = getfield(block, wordoff, 18, 18);   /* 2nd half is fraction day */
	s = s * SecPerTick;                     /* Turn into seconds */

	s += t*24*60*60;                        /* Add day base */
	return(s);
}


/* See if pattern is in name (very simple) */
match(name, pattern)
char *name, *pattern;
{
	int  plen = strlen(pattern);

	while ((name=index(name, *pattern)))
	{
		if (strncmp(name, pattern, plen)==0) return(1);
		name++;
		if (*name == 0) return(0);         /* May not need */
	}
	return(0);
}

char *
unixname(name)
char *name;
{
	static char newname[200];
	static FILE *log = NULL;
	char *t;

	if (numflg)             /* If numeric filenames */
	{
		if (log == NULL) log = fopen(LOGFILE, "a");
		fprintf(log, "%d is %s\n", number, name);
		sprintf(newname, "%d", number++);
		return(newname);
	}

	name = index(name, '<');        /* Trim off device */
	t = rindex(name, '>');          /* find end of directory */

	/* eventually make subdirectories */
	/* eventually optionally lowify filename */

	strcpy(newname, ++t);   /* Skip over the > */
	t = rindex(newname, '.');       /* find last . */
	*t = 0;                         /* zap it out */
	for (t = newname; *t; t++)	/* lowercase it */
	  if (*t >= 'A' && *t <= 'Z')
	    *t |= 040;
	return(newname);
}
