#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <stdarg.h>

#include <iniparse.h>
#include <rtos.h>

#include "cc11.h"

/* Global variables */

struct sDevice units[MAXUNITS];			/* Device units */
struct sNetport netports[MAXNETPORTS];	/* Network ports */

int use_socket=1;
dictionary *inidict;							/* INI file */

void *th_par, *th_kbd, *th_telnet;		/* Threads */

int BreakKey;

/* Function prorotypes */

int main(int, char**);

/* Code section */

void xprintf(char *str, ...)
{
	va_list ap;
	char *xbuf;

	xbuf = kcalloc(2048, 1);
	va_start(ap, str);
	vsprintf(xbuf, str, ap);
	va_end(ap);

	cprintf(xbuf);
	tprintf(xbuf);

	kfree(xbuf);
}

int ParseINI(dictionary *d)
{
	int i, j, n;
	char tmp[64], *tmp2;

	for(i=0, n=0; i<MAXUNITS; i++)
	{
		units[i].type = TYPE_NONE;
		units[i].access = ACCESS_NONE;
		units[i].cs = cs_alloc();

		sprintf(tmp, "unit%d", i);
		if(iniparser_find_entry(d, tmp))
		{
			/* Section [unitX] found */
			strcat(tmp, ":filename");
			if(tmp2=iniparser_getstr(d, tmp))
			{
				units[i].dev.file.filename = tmp2;
				if(!(units[i].dev.file.file=fopen(tmp2, "r+b")))
				{
					if(errno == ENOENT)
					{
						if(units[i].dev.file.file=fopen(tmp2, "w+b"))
						{
							cprintf("INI: Creating new file %s\r\n", tmp2);
							sprintf(tmp, "unit%d:size", i);
							units[i].size = iniparser_getint(d, tmp, 500);
							tmp2 = malloc(512);
							memset(tmp2, 0, 512);
							if(tmp2)
							{
								for(j=0; j<units[i].size; j++)
									fwrite(tmp2, 512, 1, units[i].dev.file.file);
								fseek(units[i].dev.file.file, 0, SEEK_SET);
								free(tmp2);
							}
						}
						else
						{
							cprintf("INI: Cannot create new file %s\r\n", tmp2);
							continue;
						}
					}
					else
					{
						cprintf("INI: Cannot attach unit %d to file %s, ignoring\r\n",
									 i, tmp2);
						continue;
					}
				}

				units[i].size = GetVolSize(units[i].dev.file.file);
				units[i].buffer = (UBYTE *)kcalloc(512, 1);
				units[i].type = TYPE_FILE;
				cprintf("INI: Unit %d attached to file %s\r\n", i, tmp2);
				n++;
			}
			else
			{
				sprintf(tmp, "unit%d:type", i);
				if(tmp2=iniparser_getstr(d, tmp))
				{
					if(!strcmp(tmp2, "block"))
					{
						/* Block device */

						/* BIOS unit number */
						sprintf(tmp, "unit%d:bios", i);
						units[i].dev.bios.bios = iniparser_getint(d, tmp, 0);

						/* Number of cylinders */
						sprintf(tmp, "unit%d:cyl", i);
						units[i].dev.bios.cyl = iniparser_getint(d, tmp, 40);

						/* Number of heads */
						sprintf(tmp, "unit%d:head", i);
						units[i].dev.bios.hd = iniparser_getint(d, tmp, 1);

						/* Sectors per track */
						sprintf(tmp, "unit%d:sect", i);
						units[i].dev.bios.sec = iniparser_getint(d, tmp, 1);

						/* Bytes per sector */
						sprintf(tmp, "unit%d:secsize", i);
						units[i].dev.bios.secsize = iniparser_getint(d, tmp, 512);

						/* Use LBA */
						sprintf(tmp, "unit%d:lba", i);
						units[i].dev.bios.use_lba = iniparser_getboolean(d, tmp, 0);

						units[i].buffer = (UBYTE *)kcalloc(512, MAXSECS);
						if(!units[i].buffer)
						{
							cprintf("INI: Can't allocate buffer space for unit %d\r\n",
										i);
							continue;
						}

						units[i].size = GetBlkVolSize(&units[i]);
						units[i].type = TYPE_BLOCK;
						cprintf("INI: Unit %d attached as raw device\r\n", i);
						n++;
					}
					else if(!strcmp(tmp2, "network") && use_socket)
					{
						/* Server name */
						sprintf(tmp, "unit%d:server", i);
						tmp2 = iniparser_getstr(d, tmp);
						if(!tmp2)
						{
							cprintf("INI: Missing server name for network unit %d\r\n",
										i);
							continue;
						}
						if(!(units[i].dev.net.host = resolve(tmp2)))
						{
							cprintf("INI: Cannot resolve host %s, ignoring unit %d\r\n",
										tmp2, i);
							continue;
						}

						/* Server port */
						sprintf(tmp, "unit%d:port", i);
						if(!(units[i].dev.net.port = iniparser_getint(d, tmp, 0)))
						{
							cprintf("INI: Missing port number for network unit %d\r\n",
										i);
							continue;
						}

						/* Remote unit number */
						sprintf(tmp, "unit%d:unit", i);
						if((units[i].dev.net.unit = iniparser_getint(d, tmp, -1)) == -1)
						{
							cprintf("INI: Missing remote unit number for network unit %d\r\n",
										i);
							continue;
						}

						units[i].dev.net.sock = kcalloc(sizeof(tcp_Socket),1);
						/* Open connection */
						if(!tcp_open(units[i].dev.net.sock, 0,
											units[i].dev.net.host,
											units[i].dev.net.port, NULL))
						{
							cprintf("INI: Cannot connect network unit %d, ignoring\r\n",
										i);
							continue;
						}

						/* Network block device */
						units[i].buffer = (UBYTE *)kcalloc(NETBUFSIZE, 1);
						if(!units[i].buffer)
						{
							cprintf("INI: Cannot allocate network buffer\r\n");
							continue;
						}

						units[i].type = TYPE_NET;
						units[i].dev.net.hostname = tmp2;

						cprintf("INI: Unit %d attached as network device to host %s\r\n",
									i, tmp2);
						n++;
					}
					else
					{
						cprintf("INI: Unknown device type %s for unit %d, ignoring\r\n",
									tmp2, i);
						continue;
					}
				}
			}
		}
	}

	return n;
}

void Rcv_RW_Param(UWORD *blk, SWORD *wcnt)
{
	*blk	= RcvWord();
	*wcnt	= (WORD)RcvWord();
}

void Sync(void)
{
	while(RcvByte()!=SYNC_CHAR && !BreakKey);
}

int SendGetACK(void)
{
	SndByte(SYNC_CHAR);
	if(RcvByte() != 0x33)
	{
		xprintf("ERROR: NACK received\r\n");
		return 0;
	}
	return 1;
}

void SendGetNACK(void)
{
	SndByte(0xFF);
	RcvByte();
	xprintf("ERROR: NACK sent\r\n");
}

/**************************************/

void KbdThread(DWORD val)
{
	UBYTE ch;

	do
	{
		ch = kgetch();

		switch(toupper(ch))
		{
			case 'D':
				kwritemessage(kmainthread, RTMSG_SHDRV, 0);
				break;

			case 'R':
				Restart();
				break;

			case 27:
				kwritemessage(kmainthread, RTMSG_QUIT, 0);
				break;
		}
	}while(1);
}

void ParThread(DWORD lptport)
{
	UBYTE cmd, unit, b;
	struct sDevice *dev;

	outp(LPT_CTRL, CTRLBYTEINIT);

//	kwindow(1, 10, 80, 25);
//	clrscr();

	do
	{
//		xprintf("Waiting for command... ");

		cmd	= RcvByte();
		if(cmd == SYNC_CHAR)		/* Sync */
		{
			xprintf("SYNC\r\n");
			goto wend;
		}

		unit	= RcvByte() & 7;
		if(BreakKey)
			continue;

		dev	= &units[unit];

		if(dev->type==TYPE_NONE)
		{
			xprintf("I/O error: Unit %d not available\r\n", unit);
			SndByte(ERR_NOUNIT);
			goto wend;
		}
		SndByte(ERR_OK);

		b = RcvByte();
		if(BreakKey)
			continue;

		if(b == SYNC_CHAR)
		{
			xprintf("SYNC received\r\n");
			continue;
		}
		if(b)
		{
			xprintf("ERROR: Remote host sent NACK ($%02X, cmd=%d, unit=%d)\r\n", b, cmd, unit);
			goto wend;
		}

		cs_enter(dev->cs);
		dev->cmd		= cmd;
		dev->unit	= unit;
		DoIO(dev, ACCESS_PARALLEL);
		cs_exit(dev->cs);

wend:
		Sync();

	}while(!BreakKey);
}

void NetPortThread(DWORD param)
{
	tcp_Socket *s;
	UBYTE cmd, unit;
	UWORD w, blk;
	SWORD wcnt;
	struct sDevice *dev;
	struct sNetport *np;
	UBYTE cmdbuf[6];
	UBYTE str[32];
	sockaddr sa;
	int i;
	int netport;

	np = (struct sNetport *)param;
	s = np->s;
	netport = 5001;
//	s = kcalloc(sizeof(tcp_Socket), 1);		/* allocate socket */

	do
	{
		/* wait for incoming connection */
		tcp_listen(s, 5001, 0L, 0, NULL, 0);
		while(!sock_established(s))
			tcp_tick(s);

		i = sizeof(sa);
		if(getpeername(s, &sa, &i))
			sprintf(str, "(unknown)");
		inet_ntoa(str, sa.s_ip);
		xprintf("NET %ld: Connection from %s\r\n", netport, str);

		/* Loop while socket established */
		do
		{
			if(!tcp_tick(s))
			{
				xprintf("NET %ld: Connection closed\r\n", netport);
				break;
			}

			/* get command and parameters */
			xprintf("NET %ld: Waiting for command\r\n", netport);
			if(!SOCK_READ(s, cmdbuf, 6))
			{
				xprintf("NET %ld: Short read while receiving command, aborting I/O\r\n",
							netport);
				continue;	/* Will test if connection still there */
			}

			cmd	= cmdbuf[0];
			unit	= cmdbuf[1] & 7;
			blk	= ((UWORD *)(cmdbuf))[1];
			wcnt	= ((SWORD *)(cmdbuf))[2];

			dev	= &units[unit];

			xprintf("NET %ld: Command received (%d,%d,%u,%d)\r\n",
						netport, cmd, unit, blk, wcnt);

			if(dev->type == TYPE_NONE)
			{
				xprintf("NET %ld: I/O error: Unit %d not available\r\n",
							netport, unit);
				w = ERR_NOUNIT;
				sock_write(s, (byte *)(&w), 2);
				continue;
			}

// xprintf("**NET: Vor SOCK_WRITE.\r\n");
			w = ERR_OK;
			if(!SOCK_WRITE(s, &w, 2))
				continue;

			cs_enter(dev->cs);
			dev->cmd		= cmd;
			dev->unit	= unit;
			dev->blk		= blk;
			dev->wcnt	= wcnt;
			dev->lsock	= s;
// xprintf("**NET: Vor DoIO\r\n");
			DoIO(dev, ACCESS_NETWORK);
			cs_exit(dev->cs);
		}while(1);
	}while(1);
}

void CreateNetports(void)
{
	int i;

	for(i=0; i<MAXNETPORTS; i++)
	{
		netports[i].s = kcalloc(sizeof(tcp_Socket), 1);
		if(!netports[i].s)
			continue;
		netports[i].thread = rt_newthread(NetPortThread, (DWORD)(&netports[i]), 10240, 0, "network port daemon");
	}
}

void DestroyNetports(void)
{
	int i;

	for(i=0; i<MAXNETPORTS; i++)
	{
		if(netports[i].s)
		{
			kdestroythread(netports[i].thread);
			kfree(netports[i].s);
		}
	}
}

void DoIO(struct sDevice *dev, int access)
{
	if(access == ACCESS_NONE)
		return;		/* nothing to do */

	/* enter critical section for this unit */
	dev->access = access;

	switch(dev->type)
	{
		case TYPE_FILE:
			DoFileIO(dev);
			break;

		case TYPE_BLOCK:
			DoBlockIO(dev);
			break;

		case TYPE_NET:
			DoNetIO(dev);
			break;

#if 0
		case TYPE_TAPE:
			break;

		case TYPE_CD:
			break;
#endif
	}

	/* leave critical section */
	dev->access = ACCESS_NONE;
}

void ShowDrives(void)
{
	int i;

	for(i=0; i<MAXUNITS; i++)
	{
		switch(units[i].type)
		{
			case TYPE_FILE:
				xprintf("Unit %d, size %ld, attached to file %s\r\n",
							i, units[i].size, units[i].dev.file.filename);
				break;

			case TYPE_BLOCK:
				xprintf("Unit %d, size %ld, attached as raw device\r\n",
							i, units[i].size);
				break;

			case TYPE_NET:
				xprintf("Unit %d attached to host %s port %d\r\n",
							i, units[i].dev.net.hostname, units[i].dev.net.port);
				break;
		}
	}
}

void Restart(void)
{
	kwritemessage(kmainthread, RTMSG_RESTART, 0);
}

/**************************************/

int main(int argc, char **argv)
{
	int i, errval=0;
	int msg_main;
	DWORD val_main;


	/* RTOS initialization */
	kdebug = 0;
	kpreemptive = 1;
	rt_init(100);
	rt_timerfreq(100);

//	kwindow(1, 1, 80, 10);
//	cputs("\r\n");
	clrscr();

	if(sock_init_noexit())
	{
		cputs("\r\nERROR: IP stack could not be initialized, check configuration\r\n");
		use_socket = 0;
	}

	inidict = iniparser_load("cc11.ini");
	if(!inidict)
	{
		cprintf("ERROR: INI file not found\r\n");
		return 10;
	}

	if(ParseINI(inidict) < 0)
	{
		errval = 10;
		goto err;
	}

	BreakKey = 0;

	cprintf("CC11\r\n");
	cprintf("====\r\n");
	cprintf("(Press ESC to quit)\r\n\n");

	/* Create threads */

	th_par	= rt_newthread(ParThread, 1, 8192, 0, "parallel port thread");
	th_kbd	= rt_newthread(KbdThread, 0, 2048, 0, "keyboard thread");
	if(use_socket)
	{
		th_telnet = rt_newthread(TelnetThread, 0, 4096, 0, "telnet daemon");
		CreateNetports();
	}

	sound(2000);
	rt_sleep(500);
	nosound();

	/* MAIN LOOP */

	do
	{
		kreadmessage(&msg_main, &val_main);
		switch(msg_main)
		{
			case RTMSG_RESTART:
				xprintf("Restarting program...");
				kdestroythread(th_par);
				th_par = rt_newthread(ParThread, 1, 8192, 0, "parallel port thread");
				xprintf("done!\r\n");
				break;

/*
			case RTMSG_NEWTELNET:
				th_telnet = rt_newthread(TelnetThread, 0, 4096, 0, "telnet daemon");
				break;
*/

			case RTMSG_SHDRV:
				ShowDrives();
				break;

			case RTMSG_QUIT:
				/* Quit program */
				BreakKey = 1;
				xprintf("\r\nTerminating program!\r\n");
				break;
		}
	}while(!BreakKey);

	if(use_socket)
	{
		DestroyNetports();
		kdestroythread(th_telnet);
	}
	kdestroythread(th_kbd);
	kdestroythread(th_par);

err:
	for(i=0; i<MAXUNITS; i++)
	{
		switch(units[i].type)
		{
			case TYPE_FILE:
				fclose(units[i].dev.file.file);
				break;

			case TYPE_NET:
				sock_abort(units[i].dev.net.sock);
				break;
		}

		kfree(units[i].buffer);
		cs_free(units[i].cs);
	}

	iniparser_freedict(inidict);

	return errval;
}
