(****************************************************************************)
(*                                                                          *)
(*      ROBOT.INC: Unterroutinen & Definitionen fuer FT Trainingsroboter    *)
(*      ----------------------------------------------------------------    *)
(*                                                                          *)
(*                    (C) Copyright 1986,1987 by AKS                        *)
(*                                                                          *)
(*      Dieses INCLUDE File enthaelt Unterroutinen und Definitionen zur     *)
(*      Ansteuerung des fischertechnik Trainingsroboters aus TURBO PASCAL   *)
(*                                                                          *)
(*      Folgende Anweisung wird im Quellprogramm benoetigt :                *)
(*                                                                          *)
(*                $I ROBOT.INC                                              *)
(*                                                                          *)
(*                                                                          *)
(*      Folgende Objekte werden definiert :                                 *)
(*      -----------------------------------                                 *)
(*                                                                          *)
(*      X_ACHSE,Y_ACHSE         Konstanten fuer Achsen                      *)
(*      Z_ACHSE,ZANGE                                                       *)
(*                                                                          *)
(*      RobotInit               Robotersteuerung initialisieren             *)
(*        procedure RobotInit;                                              *)
(*                                                                          *)
(*      RobotTerm               Robotersteuerung terminieren                *)
(*        procedure RobotTerm;                                              *)
(*                                                                          *)
(*      RobotRaumTestEin        Raumtest einschalten                        *)
(*        procedure RobotRaumTestEin;                                       *)
(*                                                                          *)
(*      RobotRaumTestAus        Raumtest ausschalten                        *)
(*        procedure RobotRaumTestAus;                                       *)
(*                                                                          *)
(*      RobotRaumTestStatus     Status Raumtest holen                       *)
(*        function  RobotRaumTestStatus:boolean;                            *)
(*                                                                          *)
(*      RobotMaxFehler          Maximalen Fehler bei Bewegung setzen        *)
(*        procedure RobotMaxFehler(fehler:integer);                         *)
(*                                                                          *)
(*      RobotMaxFehlerHolen     Maximalen Fehler bei Bewegung abfragen      *)
(*        function  RobotMaxFehlerHolen:integer;                            *)
(*                                                                          *)
(*      RobotDrehzahlSetzen     Drehzahl einer Roboterachse setzen          *)
(*        procedure RobotDrehzahlSetzen(achse,drehzahl:integer);            *)
(*                                                                          *)
(*      RobotDrehzahlHolen      Drehzahl einer Roboterachse abfragen        *)
(*        function  RobotDrehzahlHolen(achse:integer):integer;              *)
(*                                                                          *)
(*      RobotHome               Roboter in HOME Position bewegen            *)
(*        procedure RobotHome;                                              *)
(*                                                                          *)
(*      RobotSchritte           Roboter um Anzahl Schritte um Achsen bewegen*)
(*        procedure RobotSchritte(x,y,z,zange:integer);                     *)
(*                                                                          *)
(*      RobotPos                Roboter auf Absolutposition bewegen         *)
(*        procedure RobotPos(PosX,PosY,PosZ,PosZange:integer);              *)
(*                                                                          *)
(*      RobotPosition           Roboterposition fuer Achse abfragen         *)
(*        function  RobotPosition(achse:integer):integer;                   *)
(*                                                                          *)
(*      RobotHomeA              Roboterachse in HOME Position bewegen       *)
(*        procedure RobotHomeA(achse:integer);                              *)
(*                                                                          *)
(*      RobotSchritteA          Roboter um Anzahl Schritte um Achse bewegen *)
(*        procedure RobotSchritteA(achse,schritte:integer);                 *)
(*                                                                          *)
(*      RobotPosA               Roboter auf Absolutposition (Achse) bewegen *)
(*        procedure RobotPosA(achse,position:integer);                      *)
(*                                                                          *)
(*      RobotZange              Steuerung Roboterzange (AUF=FALSE,ZU=TRUE)  *)
(*        procedure RobotZange(zustand:boolean);                            *)
(*                                                                          *)
(*      RobotZangeAuf           Oeffnen der Roboterzange                    *)
(*        procedure RobotZangeAuf;                                          *)
(*                                                                          *)
(*      RobotZangeZu            Schliessen der Roboterzange                 *)
(*        procedure RobotZangeZu;                                           *)
(*                                                                          *)
(*      RobotSchrittFehler      Fehler nach Roboterbewegung holen           *)
(*        function  RobotSchrittFehler(achse:integer):integer;              *)
(*                                                                          *)
(*      RobotPosFehler          Fehler nach Roboterbewegung holen           *)
(*        function  RobotPosFehler(achse:integer):integer;                  *)
(*                                                                          *)
(****************************************************************************)

(****************************************************************************)
(*      KONSTANTENDEKLARATIONEN                                             *)
(****************************************************************************)

CONST ACHSEN=4; X_ACHSE=1; Y_ACHSE=2; Z_ACHSE=3; ZANGE=4;

(****************************************************************************)
(*      TYPDEKLARATIONEN                                                    *)
(****************************************************************************)

TYPE Robot_Achse = record               (* Definition einer Achse *)
        ZanFlg:boolean;                 (* Flag Achse == Zange *)
        MtrNum:integer;                 (* Motor Nummer *)
        SchGls:integer;                 (* Schalter Gabellichtschranke *)
        SchHom:integer;                 (* Schalter HOME Position *)
        DirPos:integer;                 (* Richtungskonstante positiv *)
        DirNeg:integer;                 (* Richtungskonstante negativ *)
        SchNum:integer;                 (* Schritte Anzahl *)
        SchDir:integer;                 (* Schritte Richtung *)
        MinPos:integer;                 (* Minimale Position *)
        MaxPos:integer;                 (* Maximale Position *)
        end;

     RobotAchsenTabelle   = array[1..ACHSEN] of Robot_Achse;
     RobotPositionTabelle = array[1..ACHSEN] of integer;
     RobotSchrittTabelle  = array[1..ACHSEN] of integer;
     RobotHomeTabelle     = array[1..ACHSEN] of boolean;

(****************************************************************************)
(*      VARIABLENDEKLARATIONEN                                              *)
(****************************************************************************)

VAR RobotAchsenTab:RobotAchsenTabelle;  (* Achsen Tabelle *)
    RobotAktPosTab:RobotPositionTabelle;(* aktuelle Position *)
    RobotFehlerTab:RobotSchrittTabelle; (* Fehlerschritte *)
    RobotFehlerMax:integer;             (* maximal zulssiger Fehler *)
    RobotRaumTest :boolean;             (* Status Raum Test *)

(****************************************************************************)
(*      LOW LEVEL UNTERROUTINEN                                             *)
(****************************************************************************)

(*--------------------------------------------------------------------------*)
(*      _RobotInitInterface: Interface fuer Robotersteuerung initialisieren *)
(*--------------------------------------------------------------------------*)

procedure _RobotInitInterface;
  begin
  r_init;                               (* Interface initialisieren *)
  r_auszeit(AUS);                       (* Auszeit AUS *)
  end;

(*--------------------------------------------------------------------------*)
(*      _RobotTermInterface: Interface terminieren                          *)
(*--------------------------------------------------------------------------*)

procedure _RobotTermInterface;
  begin
  r_term;                               (* Interface terminieren *)
  end;

(*--------------------------------------------------------------------------*)
(*      _RobotInitAchsen: Initialisieren der Roboter Achsentabelle          *)
(*--------------------------------------------------------------------------*)

procedure _RobotInitAchsen;
  var achse:integer;
  begin
  for achse:=1 to ACHSEN do             (* fuer alle Achsen *)
    with RobotAchsenTab[achse] do       (* fuer aktueller Achse *)
      begin
      ZanFlg := (achse = ZANGE);        (* Zangenflag setzen *)
      MtrNum := achse;                  (* Motornummer setzen *)
      SchGls := 2*achse;                (* Schalter Gabellichtschranke *)
      SchHom := SchGls - 1;             (* Schalter HOME Position *)
      DirPos := LINKS;                  (* positive Richtungskonstante *)
      DirNeg := RECHTS;                 (* Richtungskonstante HOME *)
      SchNum := 0;                      (* Schritte Anzahl ruecksetzen *)
      SchDir := AUS;                    (* Schritte Richtung ruecksetzen *)
      case achse of                     (* minimale & maximale Position *)
        X_ACHSE: begin MinPos:= -4000; MaxPos:=  4000; end;
        Y_ACHSE: begin MinPos:=     0; MaxPos:=  1300; end;
        Z_ACHSE: begin MinPos:=     0; MaxPos:=  1300; end
        else     begin MinPos:=  -100; MaxPos:=  1500; end;
        end;
      end;
  end;

(*--------------------------------------------------------------------------*)
(*      _RobotRaumTest: Status Raum Test setzen                             *)
(*--------------------------------------------------------------------------*)

procedure _RobotRaumTest(status:boolean);
  begin
  RobotRaumTest:=status;                (* Status Raumtest setzen *)
  end;

(*--------------------------------------------------------------------------*)
(*      _RobotRaumTestStatus: Status Raum Test holen                        *)
(*--------------------------------------------------------------------------*)

function _RobotRaumTestStatus:boolean;
  begin
  _RobotRaumTestStatus:=RobotRaumTest;  (* Status Raumtest holen *)
  end;

(*--------------------------------------------------------------------------*)
(*      _RobotMaxFehler: Maximalen Fehler fuer Bewegung/Position setzen     *)
(*--------------------------------------------------------------------------*)

procedure _RobotMaxFehler(fehler:integer);
  begin
  RobotFehlerMax:=fehler;               (* Maximalen Fehler setzen *)
  end;

(*--------------------------------------------------------------------------*)
(*      _RobotMaxFehlerHolen: maximalen Fehler holen                        *)
(*--------------------------------------------------------------------------*)

function _RobotMaxFehlerHolen:integer;
  begin
  _RobotMaxFehlerHolen:=RobotFehlerMax; (* Maximalen Fehler setzen *)
  end;

(*--------------------------------------------------------------------------*)
(*      _RobotDrehzahlSetzen: Setzen der Drehzahl einer Roboterachse        *)
(*--------------------------------------------------------------------------*)

procedure _RobotDrehzahlSetzen(achse,drehzahl:integer);
  begin
  with RobotAchsenTab[achse] do         (* fuer aktuelle Achse *)
    r_drehzahl(MtrNum,drehzahl);        (* Drehzahl setzen *)
  end;

(*--------------------------------------------------------------------------*)
(*      _RobotDrehzahlHolen: Holen der Drehzahl einer Roboterachse          *)
(*--------------------------------------------------------------------------*)

function _RobotDrehzahlHolen(achse:integer):integer;
  var drehzahl,dummy:integer;
  begin
  with RobotAchsenTab[achse] do         (* fuer aktuelle Achse *)
    r_status(MtrNum,dummy,dummy,dummy,drehzahl);(* Drehzahl holen *)
  _RobotDrehzahlHolen:=drehzahl;        (* Ergebnis setzen *)
  end;

(*--------------------------------------------------------------------------*)
(*      _RobotSchritteSetzen: Schritte fuer Bewegung setzen                 *)
(*--------------------------------------------------------------------------*)

procedure _RobotSchritteSetzen(achse,schritte:integer);
  var dir:integer;
  begin
  with RobotAchsenTab[achse] do         (* fuer aktuelle Achse *)
    begin
    if schritte > 0 then dir:=DirPos    (* positive Richtung falls > 0 *)
                    else dir:=DirNeg;   (* sonst Richtung HOME Position *)
    schritte:=abs(schritte);            (* Anzahl Schritte in Richtung *)
    SchNum:=schritte; SchDir:=dir;      (* Schritte & Richtung merken *)
    if not ZanFlg then                  (* wenn nicht Zange *)
      r_glsset(MtrNum,SchGls,dir,schritte);     (* Schritte setzen *)
    end;
  end;

(*--------------------------------------------------------------------------*)
(*      _RobotBewegungAusfuehren: Bewegung des Roboters ausfuehren          *)
(*--------------------------------------------------------------------------*)

procedure _RobotBewegungAusfuehren;
  var achse,drz,wtz:integer;
  begin
  r_glsexe;                             (* Bewegung durchfuehren *)
  for achse:=1 to ACHSEN do             (* fuer alle Achsen *)
    with RobotAchsenTab[achse] do       (* mit aktuellen Werten *)
      if ZanFlg and (SchNum > 0) then   (* wenn Zange *)
        begin
        drz:=_RobotDrehzahlHolen(achse);(* Drehzahl holen *)
        wtz:=SchNum div (10 * drz);     (* Wartezeit berechnen *)
        if wtz < 1 then wtz:=1;         (* Minimum testen *)
        r_motori(MtrNum,SchDir,wtz);    (* Motor einschalten *)
        SchNum:=0; SchDir:=AUS;         (* Zangenwerte rcksetzen *)
        end;
  end;

(*--------------------------------------------------------------------------*)
(*      _RobotFehlerHolen: Fehler nach Bewegen des Roboters holen           *)
(*--------------------------------------------------------------------------*)

function _RobotFehlerHolen(achse:integer):integer;
  var fehler:integer;
  begin
  with RobotAchsenTab[achse] do         (* fuer aktuelle Achse *)
    begin
    if ZanFlg then fehler:=0            (* wenn Zange dann kein Fehler *)
    else
      begin
      fehler:=r_glsget(MtrNum);         (* Echte Schritte holen *)
      if fehler=-1 then                 (* wenn berlauf *)
        begin
        writeln;                        (* Meldung auf Bildschirm *)
        writeln('*** IMPULSUEBERLAUF [MOTOR=',MtrNum,',SCHALTER=',SchGls,'] ***');
        writeln;
        fehler:=0;                      (* Fehler nicht zu bestimmen *)
        end
      else fehler:=fehler-SchNum;       (* sonst Fehlerschritte berechnen *)
      end;
    _RobotFehlerHolen:=fehler;          (* Fehlerschritte setzen *)
    end;
  end;

(*--------------------------------------------------------------------------*)
(*      _RobotVorFehler: Voraussichtlichen Fehler bei Bewegung berechnen    *)
(*--------------------------------------------------------------------------*)

function _RobotVorFehler(achse,schritte,drehzahl:integer):integer;
  var fehler,vorzeichen:integer;
  begin
  with RobotAchsenTab[achse] do         (* mit aktueller Achse *)
    if ZanFlg then fehler:=0            (* kein Fehler falls Zange *)
    else
      begin
      if RobotFehlerMax = -1 then       (* wenn Fehlerkorrektur nicht aktiv *)
         fehler:=0                      (* kein Fehler *)
      else
        begin
        if schritte > 0 then vorzeichen:=1      (* Vorzeichen setzen *)
                        else vorzeichen:=-1;
        schritte:=abs(schritte);        (* Absolutbetrag Schritte *)
        case drehzahl of
          1: fehler:=0;                 (* Fehler bei Drehzahl 1 *)
          2: fehler:=3;                 (* Fehler bei Drehzahl 2 *)
          3: fehler:=6;                 (* Fehler bei Drehzahl 3 *)
          4: fehler:=8;                 (* Fehler bei Drehzahl 4 *)
          5: fehler:=9;                 (* Fehler bei Drehzahl 5 *)
          6: fehler:=10;                (* Fehler bei Drehzahl 6 *)
          7: fehler:=12                 (* Fehler bei Drehzahl 7 *)
          else fehler:=14;              (* Fehler bei Drehzahl 8 *)
          end;
        if schritte < fehler then fehler:=0
        else if schritte < (2*fehler) then fehler:=fehler div 2;
        fehler:=vorzeichen*fehler;      (* Vorzeichen beruecksichtigen *)
        end;
      end;
  _RobotVorFehler:=fehler;              (* Ergebnis setzen *)
  end;

(*--------------------------------------------------------------------------*)
(*      _RobotAchsenHome: Roboterachsen in HOME Position bringen            *)
(*--------------------------------------------------------------------------*)

procedure _RobotAchsenHome(home:RobotHomeTabelle);
  var StaTab:RobotHomeTabelle;            (* Zangen Statustabelle *)
      achse:integer; flag:boolean;
  begin
  for achse:=1 to ACHSEN do             (* fr alle Achsen *)
    if home[achse] then                 (* wenn HOME fuer Achse gewuenscht *)
      with RobotAchsenTab[achse] do     (* fuer aktuelle Achse *)
        begin
        r_motor(MtrNum,DirNeg);         (* Motor Richtung HOME *)
        StaTab[achse]:=TRUE;            (* Statusflag Zange setzen *)
        end;
  repeat
    begin
    flag:=TRUE;                         (* Flag setzen *)
    for achse:=1 to ACHSEN do           (* fr alle Achsen *)
      if home[achse] then               (* wenn HOME fuer Achse gewuenscht *)
        with RobotAchsenTab[achse] do   (* fuer aktuelle Achse *)
          if ZanFlg and StaTab[achse] then (* wenn Zange & Status gesetzt *)
            begin
            if r_schalter(SchHom) then  (* wenn Schalter zu *)
              begin
              r_motor(MtrNum,DirPos);   (* Motor Richtung HOME *)
              StaTab[achse]:=FALSE;     (* Statusflag Zange setzen *)
              end;
            flag:=FALSE;                (* weitermachen *)
            end
          else
            if not r_schalter(SchHom) then (* wenn HOME Position erreicht *)
              begin
              r_motor(MtrNum,AUS);      (* Motor ausschalten *)
              home[achse]:=FALSE;       (* keine weitere Ueberpruefung *)
              RobotAktPosTab[achse]:=0; (* Position initialisieren *)
              end
            else flag:=FALSE;           (* weitermachen *)
    end;
  until flag;
  end;

(*--------------------------------------------------------------------------*)
(*      _RobotBewegen: Roboter um Achse um Anzahl Einheiten bewegen         *)
(*--------------------------------------------------------------------------*)

procedure _RobotBewegen(schritte:RobotSchrittTabelle);
  var achse,fehler,pos:integer;
  begin
  for achse:=1 to ACHSEN do             (* fuer alle Achsen *)
    begin
    if RobotRaumTest then               (* wenn Raumtest ein *)
      with RobotAchsenTab[achse] do
        begin
        pos := RobotAktPosTab[achse]+schritte[achse];
        if pos < MinPos then pos := MinPos;
        if pos > MaxPos then pos := MaxPos;
        schritte[achse] := pos - RobotAktPosTab[achse];
        end;
    _RobotSchritteSetzen(achse,schritte[achse]);(* Schritte setzen *)
    end;
  _RobotBewegungAusfuehren;             (* Ausfuehren der Bewegung *)
  for achse:=1 to ACHSEN do             (* fuer alle Achsen *)
    begin
    fehler:=_RobotFehlerHolen(achse);   (* Fehler holen *)
    if schritte[achse] < 0 then         (* wenn negative Bewegung *)
      fehler:=-fehler;                  (* Vorzeichen umdrehen *)
    RobotFehlerTab[achse]:=fehler;      (* Fehler setzen *)
    RobotAktPosTab[achse]:=             (* Neue Position setzen *)
      RobotAktPosTab[achse] + schritte[achse] + fehler;
    end;
  end;

(*--------------------------------------------------------------------------*)
(*      _RobotSchritte: Roboter um Schritten um Achsen bewegen              *)
(*                      Fehler werden korrigiert                            *)
(*--------------------------------------------------------------------------*)

procedure _RobotSchritte(schritte:RobotSchrittTabelle);
  var AktSch:RobotSchrittTabelle;
      AltPos:RobotPositionTabelle;
      AltDrz:array[1..ACHSEN] of integer;
      StaTab:array[1..ACHSEN] of boolean;
      achse,drz:integer; status:boolean;
  begin
  for achse:=1 to ACHSEN do             (* fuer alle Achsen *)
    begin
    AltPos[achse]:=RobotAktPosTab[achse];(* Aktuelle Position retten *)
    AltDrz[achse]:=                     (* Aktuelle Drehzahl retten *)
      _RobotDrehzahlHolen(achse);
    StaTab[achse]:=TRUE;                (* Status auf "Fehler zu grosz" *)
    end;
  repeat
    status:=TRUE;                       (* Annahme Fehler kleiner Max. *)
    for achse:=1 to ACHSEN do           (* fuer alle Achsen *)
      if StaTab[achse] then             (* wenn Fehler noch zu grosz *)
        begin
        AktSch[achse]:=                 (* notwendige Schritte berechnen *)
          schritte[achse]-(RobotAktPosTab[achse]-AltPos[achse]);
        drz:=_RobotDrehzahlHolen(achse);(* Drehzahl holen *)
        AktSch[achse]:=AktSch[achse]-   (* Voraus. Fehler beruecksichtigen *)
                       _RobotVorFehler(achse,AktSch[achse],drz);
        end
    else AktSch[achse]:=0;              (* sonst keine Bewegung *)
    _RobotBewegen(AktSch);              (* Bewegung um aktuelle Schritte *)
    for achse:=1 to ACHSEN do           (* fuer alle Achsen *)
      if StaTab[achse] then             (* wenn Fehler noch zu grosz *)
        if (RobotFehlerMax <> -1) and   (* wenn Fehlertest EIN *)
          (abs(RobotFehlerTab[achse]) > RobotFehlerMax) then
          begin
          drz:=_RobotDrehzahlHolen(achse); (* Drehzahl holen *)
          if drz = 1 then               (* Ende falls minimale Drehzahl *)
             StaTab[achse]:=FALSE
          else                          (* sonst *)
            begin
            if drz > 2 then drz:=2      (* Drehzahl verkleineren *)
                       else drz:=1;
            _RobotDrehzahlSetzen(achse,drz); (* neue Drehzahl setzen *)
            status:=FALSE;              (* Fehler zu grosz *)
            end;
          end
        else StaTab[achse]:=FALSE;      (* sonst Ende anzeigen *)
  until status;
  for achse:=1 to ACHSEN do             (* fuer alle Achsen *)
    _RobotDrehzahlSetzen(achse,AltDrz[achse]);  (* Drehzahl setzen *)
  end;

(*--------------------------------------------------------------------------*)
(*      _RobotPositionieren: Roboter auf absolute Position bewegen          *)
(*--------------------------------------------------------------------------*)

procedure _RobotPositionieren(position:RobotPositionTabelle);
  var SchrittTab:RobotSchrittTabelle;
  var achse:integer;
  begin
  for achse:=1 to ACHSEN do             (* fuer alle Achsen *)
    SchrittTab[achse]:=                 (* Schritte fuer Achse setzen *)
                   position[achse] - RobotAktPosTab[achse];
  _RobotSchritte(SchrittTab);           (* Roboterschritte ausfuehren *)
  end;

(****************************************************************************)
(*      HIGH LEVEL UNTERROUTINEN                                            *)
(****************************************************************************)

(*--------------------------------------------------------------------------*)
(*      RobotInit: Robotersteuerung Initialisieren                          *)
(*--------------------------------------------------------------------------*)

procedure RobotInit;
  begin
  _RobotInitInterface;                  (* Interface initialisieren *)
  _RobotInitAchsen;                     (* Achsentabelle initialisieren *)
  _RobotRaumTest(TRUE);                 (* Raumtest EIN *)
  _RobotMaxFehler(1);                   (* maximalen Fehler setzen *)
  end;

(*--------------------------------------------------------------------------*)
(*      RobotTerm: Robotersteuerung Terminieren                             *)
(*--------------------------------------------------------------------------*)

procedure RobotTerm;
  begin
  _RobotTermInterface;                  (* Interface terminieren *)
  end;

(*--------------------------------------------------------------------------*)
(*      RobotRaumTestEin: Raumtest einschalten                              *)
(*--------------------------------------------------------------------------*)

procedure RobotRaumTestEin;
  begin
  _RobotRaumTest(TRUE);                 (* Raumtest einschalten *)
  end;

(*--------------------------------------------------------------------------*)
(*      RobotRaumTestAus: Raumtest ausschalten                              *)
(*--------------------------------------------------------------------------*)

procedure RobotRaumTestAus;
  begin
  _RobotRaumTest(FALSE);                (* Raumtest ausschalten *)
  end;

(*--------------------------------------------------------------------------*)
(*      RobotRaumTestStatus: Status Raumtest holen                          *)
(*--------------------------------------------------------------------------*)

function RobotRaumTestStatus:boolean;
  begin
  RobotRaumTestStatus:=                 (* Status Raumtest holen *)
    _RobotRaumTestStatus;
  end;

(*--------------------------------------------------------------------------*)
(*      RobotMaxFehler: Maximalen Fehler fuer Bewegung/Positionieren setzen *)
(*--------------------------------------------------------------------------*)

procedure RobotMaxFehler(schritte:integer);
  begin
  if schritte < 0 then schritte:=-1;    (* Ausschalten falls negativ *)
  _RobotMaxFehler(schritte);            (* Maximalen Fehler setzen *)
  end;

(*--------------------------------------------------------------------------*)
(*      RobotMaxFehlerHolen: Aktuellen maximalen Fehler holen               *)
(*--------------------------------------------------------------------------*)

function RobotMaxFehlerHolen:integer;
  begin
  RobotMaxFehlerHolen:=                (* Maximalen Fehler holen *)
    _RobotMaxFehlerHolen;
  end;

(*--------------------------------------------------------------------------*)
(*      RobotHome: Roboter in HOME Position bringen                         *)
(*--------------------------------------------------------------------------*)

procedure RobotHome;
  var HomeTab:RobotHomeTabelle; achse:integer;
  begin
  for achse:=1 to ACHSEN do             (* fuer alle Achsen *)
    HomeTab[achse]:=TRUE;               (* HOME Position anfahren *)
  _RobotAchsenHome(HomeTab);            (* Roboterachsen in HOME Position *)
  end;

(*--------------------------------------------------------------------------*)
(*      RobotDrehzahlSetzen: Setzen der Drehzahl einer Roboterachse         *)
(*--------------------------------------------------------------------------*)

procedure RobotDrehzahlSetzen(achse,drehzahl:integer);
  begin
  _RobotDrehzahlSetzen(achse,drehzahl); (* Drehzahl setzen *)
  end;

(*--------------------------------------------------------------------------*)
(*      RobotDrehzahlHolen: Holen der Drehzahl einer Roboterachse           *)
(*--------------------------------------------------------------------------*)

function RobotDrehzahlHolen(achse:integer):integer;
  begin
  RobotDrehzahlHolen:=                  (* Drehzahl holen *)
    _RobotDrehzahlHolen(achse);
  end;

(*--------------------------------------------------------------------------*)
(*      RobotPosition: Position einer Achse holen                           *)
(*--------------------------------------------------------------------------*)

function RobotPosition(achse:integer):integer;
  begin
  RobotPosition:=RobotAktPosTab[achse]; (* Position setzen *)
  end;

(*--------------------------------------------------------------------------*)
(*      RobotSchritte: Roboter um Anzahl Schritte um Achse bewegen          *)
(*--------------------------------------------------------------------------*)

procedure RobotSchritte(SchritteX,SchritteY,SchritteZ,SchrZange:integer);
  var SchrittTabelle:RobotSchrittTabelle;
  begin
  SchrittTabelle[X_ACHSE]:=SchritteX;   (* X Schritte setzen *)
  SchrittTabelle[Y_ACHSE]:=SchritteY;   (* Y Schritte setzen *)
  SchrittTabelle[Z_ACHSE]:=SchritteZ;   (* Z Schritte setzen *)
  SchrittTabelle[ZANGE]  :=SchrZange;   (* Schritte fuer Zange setzen *)
  _RobotSchritte(SchrittTabelle);       (* Roboterbewegung ausfuehren *)
  end;

(*--------------------------------------------------------------------------*)
(*      RobotSchrittFehler: Fehler nach Schrittbewegung holen               *)
(*--------------------------------------------------------------------------*)

function RobotSchrittFehler(achse:integer):integer;
  begin
  RobotSchrittFehler:=RobotFehlerTab[achse];    (* Fehler holen *)
  end;

(*--------------------------------------------------------------------------*)
(*      RobotPos: Roboter auf absolute Position bewegen                     *)
(*--------------------------------------------------------------------------*)

procedure RobotPos(PositionX,PositionY,PositionZ,PosZange:integer);
  var PosTabelle:RobotPositionTabelle;
  begin
  PosTabelle[X_ACHSE]:=PositionX;       (* X Position setzen *)
  PosTabelle[Y_ACHSE]:=PositionY;       (* Y Position setzen *)
  PosTabelle[Z_ACHSE]:=PositionZ;       (* Z Position setzen *)
  PosTabelle[ZANGE]  :=PosZange;        (* Zangenposition setzen *)
  _RobotPositionieren(PosTabelle);      (* Roboter positionieren *)
  end;

(*--------------------------------------------------------------------------*)
(*      RobotPosFehler: Fehler nach Positionierung holen                    *)
(*--------------------------------------------------------------------------*)

function RobotPosFehler(achse:integer):integer;
  begin
  RobotPosFehler:=                      (* Fehler holen *)
                  RobotSchrittFehler(achse);
  end;

(*--------------------------------------------------------------------------*)
(*      RobotHomeA: Roboterachse in HOME Position bringen                   *)
(*--------------------------------------------------------------------------*)

procedure RobotHomeA(achse:integer);
  var HomeTab:RobotHomeTabelle; i:integer;
  begin
  for i:=1 to ACHSEN do                 (* fuer alle Achsen *)
    HomeTab[i]:=FALSE;                  (* HOME Position nicht anfahren *)
  HomeTab[achse]:=TRUE;                 (* Wert fuer aktuelle Achse *)
  _RobotAchsenHome(HomeTab);            (* HOME Position anfahren *)
  end;

(*--------------------------------------------------------------------------*)
(*      RobotSchritteA: Roboter um Anzahl Schritte um Achse bewegen         *)
(*--------------------------------------------------------------------------*)

function RobotSchritteA(achse,schritte:integer):integer;
  var SchrittTabelle:RobotSchrittTabelle;
  var i:integer;
  begin
  for i:=1 to ACHSEN do                 (* fuer alle Achsen *)
    SchrittTabelle[i]:=0;               (* Schritte auf 0 setzen *)
  SchrittTabelle[achse]:=schritte;      (* Schritte fuer Achse setzen *)
  _RobotSchritte(SchrittTabelle);       (* Roboterschritte ausfuehren *)
  RobotSchritteA:=RobotSchrittFehler(achse);  (* Fehler holen *)
  end;

(*--------------------------------------------------------------------------*)
(*      RobotPosA: Roboter auf absolute Position fuer Achse bewegen         *)
(*--------------------------------------------------------------------------*)

function RobotPosA(achse,position:integer):integer;
  var PosTabelle:RobotPositionTabelle;
      i:integer;
  begin
  for i:=1 to ACHSEN do                 (* fuer alle Achsen *)
    PosTabelle[i]:=RobotPosition(i);    (* keine Bewegung *)
  PosTabelle[achse]:=position;          (* Wert fuer aktuelle Achse *)
  _RobotPositionieren(PosTabelle);      (* Roboter positionieren *)
  RobotPosA:=RobotPosFehler(achse);     (* Fehler holen *)
  end;

(*--------------------------------------------------------------------------*)
(*      RobotZange: Steuerung der Roboterzange                              *)
(*--------------------------------------------------------------------------*)

procedure RobotZange(zustand:boolean);
  var pos,min,max:integer;              (* Zangenpositionen *)
  begin
  with RobotAchsenTab[ZANGE] do         (* fuer Zange *)
    begin min:=MinPos; max:=MaxPos; end;(* minimale & maximale Position *)
  pos := RobotPosition(ZANGE);          (* aktuelle Position holen *)
  if (((pos >= 0) and     zustand) or   (* Test ob HOME sinnvoll *)
      ((pos <= 0) and not zustand)) then
    RobotHomeA(ZANGE);                  (* Zange in HOME Position *)
  if zustand then pos := min            (* wenn Zange schliessen *)
             else pos := max;           (* wenn Zange oeffnen *)
  pos := RobotPosA(ZANGE,pos);          (* Zange Positionieren *)
  end;

(*--------------------------------------------------------------------------*)
(*      RobotZangeAuf: Oeffnen der Roboterzange                             *)
(*--------------------------------------------------------------------------*)

procedure RobotZangeAuf;
  begin
  RobotZange(FALSE);                    (* Zange oeffnen *)
  end;

(*--------------------------------------------------------------------------*)
(*      RobotZangeZu: Schliessen der Roboterzange                           *)
(*--------------------------------------------------------------------------*)

procedure RobotZangeZu;
  begin
  RobotZange(TRUE);                     (* Zange oeffnen *)
  end;
