(*-------------------------------------------------------------------------*)
(*               CDBeta  -- Cumulative Beta Distribution                   *)
(*-------------------------------------------------------------------------*)

FUNCTION CDBeta(     X,     Alpha, Beta: REAL;
                     Dprec, MaxIter    : INTEGER;
                 VAR Cprec             : REAL;
                 VAR Iter              : INTEGER;
                 VAR Ifault            : INTEGER  ) : REAL;

(*-------------------------------------------------------------------------*)
(*                                                                         *)
(*       Function:  CDBeta                                                 *)
(*                                                                         *)
(*       Purpose:   Evaluates CPDF of Incomplete Beta Function             *)
(*                                                                         *)
(*       Calling Sequence:                                                 *)
(*                                                                         *)
(*            P     := CDBeta(     X, Alpha, Beta: REAL;                   *)
(*                                 Dprec, Maxitr : INTEGER;                *)
(*                             VAR Cprec         : REAL;                   *)
(*                             VAR Iter          : INTEGER;                *)
(*                             VAR Ifault        : INTEGER  ) : REAL;      *)
(*                                                                         *)
(*                 X      --- Upper percentage point of PDF                *)
(*                 Alpha  --- First shape parameter                        *)
(*                 Beta   --- Second shape parameter                       *)
(*                 Dprec  --- Number of digits of precision required       *)
(*                 Maxitr --- Maximum number of iterations                 *)
(*                 Cprec  --- Actual resulting precision                   *)
(*                 Iter   --- Iterations actually used                     *)
(*                 Ifault --- error indicator                              *)
(*                            = 0:  no error                               *)
(*                            = 1:  argument error                         *)
(*                                                                         *)
(*                 P      --- Resultant probability                        *)
(*                                                                         *)
(*       Calls:                                                            *)
(*                                                                         *)
(*            ALGama                                                       *)
(*                                                                         *)
(*       Method:                                                           *)
(*                                                                         *)
(*            The continued fraction expansion as given by                 *)
(*            Abramowitz and Stegun (1964) is used.  This                  *)
(*            method works well unless the minimum of (Alpha, Beta)        *)
(*            exceeds about 70000.                                         *)
(*                                                                         *)
(*            An error in the input arguments results in a returned        *)
(*            probability of -1.                                           *)
(*                                                                         *)
(*-------------------------------------------------------------------------*)

VAR
   Epsz : REAL;
   A    : REAL;
   B    : REAL;
   C    : REAL;
   F    : REAL;
   Fx   : REAL;
   Apb  : REAL;
   Zm   : REAL;
   Alo  : REAL;
   Ahi  : REAL;
   Blo  : REAL;
   Bhi  : REAL;
   Bod  : REAL;
   Bev  : REAL;
   Zm1  : REAL;
   D1   : REAL;
   Aev  : REAL;
   Aod  : REAL;

   Ntries : INTEGER;

   Qswap  : BOOLEAN;
   Qdoit  : BOOLEAN8ppφ  ϜpppGpppϜ	Opppϒ  OppϒϙOϐOpϙOOppϘOϙOOρpϙOO@pppϜOO@pppϘ O  O@pppσOppϞOO@ppϜO@ppφ O@ppϜOOF@ppφ	O@ppL;
   PP     : REAL;
   H      : REAL;
   A1     : REAL;
   B1     : REAL;
   Eprec  : REAL;

   Done   : BOOLEAN;

   Jter   : INTEGER;

LABEL 10, 30, 9000;

BEGIN (* BetaInv *)

   Ierr    := 1;
   BetaInv := P;
                                   (* Check validity of arguments *)

   IF( ( Alpha <= 0.0 ) OR ( Beta <= 0.0 ) ) THEN GOTO 9000;
   IF( ( P > 1.0 ) OR ( P < 0.0 ) )          THEN GOTO 9000;

                                   (* Check for P = 0 or 1        *)

   IF( ( P = 0.0 ) OR ( P = 1.0 ) )          THEN
      BEGIN
         Iter   := 0;
         Cprec  := MaxPrec;
         GOTO 9000;
      END;

                                  (* Set precision *)
   IF Dprec > MaxPrec THEN
      Dprec := MaxPrec
   ELSE IF Dprec <= 0 THEN
      Dprec := 1;

   Cprec  := Dprec;

   Eps    := PowTen( -2 * Dprec );

                                   (* Flip params if needed so that *)
                                   (* P for evaluation is <= .5     *)
   IF( P > 0.5 ) THEN
      BEGIN
         A      := Beta;
         B      := Alpha;
         PP     := 1.0 - P;
      END
   ELSE
      BEGIN
         A      := Alpha;
         B      := Beta;
         PP     := P;
      END;
                                   (* Generate initial approximation.  *)
                                   (* Several different ones used,     *)
                                   (* depending upon parameter values. *)
   Ierr   := 0;

   Cmplbt := ALGama( A ) + ALGama( B ) - ALGama( A + B );
   Fi     := Ninv( 1.0 - PP );

   IF( ( A > 1.0 ) AND ( B > 1.0 ) ) THEN
      BEGIN
         R      := ( Fi * Fi - 3.0 ) / 6.0;
         S      := 1.0 / ( A + A - 1.0 );
         T      := 1.0 / ( B + B - 1.0 );
         H      := 2.0 / ( S + T );
         W      := Fi * SQRT( H + R ) / H - ( T - S ) *
                   ( R + 5.0 / 6.0 - 2.0 / ( 3.0 * H ) );
         Xi     := A / ( A + B * EXP( W + W ) );
      END
   ELSE
      BEGIN

         R      := B + B;
         T      := 1.0 / ( 9.0 * B );
         T      := R * PowerI( ( 1.0 - T + Fi * SQRT( T ) ) , 3 );

         IF( T <= 0.0 ) THEN
            Xi     := 1.0 - EXP( ( LN( ( 1.0 - PP ) * B ) + Cmplbt ) / B )
         ELSE
            BEGIN

               T      := ( 4.0 * A + R - 2.0 ) / T;

               IF( T <= 1.0 ) THEN
                  Xi := EXP( (LN( PP * A ) + Cmplbt) / PP )
               ELSE
                  Xi := 1.0 - 2.0 / ( T + 1.0 );

            END;

      END;
                                   (* Force initial estimate to *)
                                   (* reasonable range.         *)

   IF ( Xi < 0.0001 ) THEN Xi := 0.0001;
   IF ( Xi > 0.9999 ) THEN Xi := 0.9999;

                                   (* Set up Newton-Raphson loop *)

   A1     := 1.0 - A;
   B1     := 1.0 - B;
   Fim1   := 0.0;
   Sq     := 1.0;
   Xim1   := 1.0;
   Iter   := 0;
   Done   := FALSE;

                                   (* Begin Newton-Raphson loop  *)
   REPEAT

                                      (* achieved, or maximum iterations *)
                                   (* exceeded.                       *)

   IF ( Qswap ) THEN
      CDBeta := 1.0 - F
   ELSE
      CDBeta := F;

                                   (* Calculate precision of result *)

   IF ABS( F - Fx ) <> 0.0 THEN
      Cprec := -LogTen( ABS( F - Fx ) )
   ELSE
      Cprec := MaxPrec;

9000:  (* Error exit *)

END   (* CDBeta *);
