/*===========================================================================*/
/*                                                                           */
/* File    : WINWHERE.C                                                      */
/*                                                                           */
/* Purpose : File finder for Microsoft Windows 3.0                           */
/*                                                                           */
/* History : 9/14/90 Created by Marc Adler @ Magma Systems                   */
/*                              15 Bodwell Terrace, Millburn NJ, 07041       */
/*                              BBS (201) 912-0668, 2400/1200 baud N-8-1     */
/*                                                                           */
/* WINWHERE 1.0 (c) Copyright 1990 Ziff Communications Co.                   */
/*===========================================================================*/

#include <stdlib.h>
#include <ctype.h>
#include <dos.h>
#include <fcntl.h>
#include <io.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <string.h>

#ifdef MEWEL
#include "window.h"
#undef  NULL
#define NULL  0
#else
#include <windows.h>
#define _DialogBox    DialogBox
#endif

#include "winwhere.h"


/*
  Structure which contains a file-type/file-spec pairing
*/
typedef struct tagFileTypeInfo
{
  char szType[65];
  char szSpec[16];
} FILETYPEINFO;
#define MAXFILETYPES  8
FILETYPEINFO FileTypeInfo[MAXFILETYPES];
int nFileTypes = 0;

/*
  Structure which contains the current search parameters
*/
typedef struct tagSearchInfo
{
  WORD fFlags;
#define SEARCH_DATE  0x0001
#define DATE_EQ      0x0002
#define DATE_LT      0x0004
#define DATE_GT      0x0008
#define SEARCH_SIZE  0x0010
#define SIZE_EQ      0x0020
#define SIZE_LT      0x0040
#define SIZE_GT      0x0080

  DWORD dwSize;
  WORD  wDate;
} SEARCHINFO;
SEARCHINFO SearchInfo;

#define ISLEAP(y)  ((y)%4 == 0 && (y)%100 != 0 || (y) % 400 == 0)

/*
  List of device names used for validation
*/
PSTR pszDevices[] =
{
  "aux",  "com1",  "com2",  "com3",  "com4", "con",
  "lpt1", "lpt2",  "lpt3",  "lpt4",  "nul",  "prn",
  NULL
};



HANDLE hInst = 0;          /* instance handle */
HWND   hFoundListBox;      /* handle to the listbox which saves found files */
WORD   iCurrFileType = 0;  /* index of current file type */


/*
  Function declarations
*/
BOOL FAR PASCAL WinwhereDlgProc(HWND, unsigned, WORD, LONG);
BOOL FAR PASCAL AddTypeDlgProc(HWND, unsigned, WORD, LONG);
BOOL FAR PASCAL GenCommandDlgProc(HWND, unsigned, WORD, LONG);
BOOL FAR PASCAL About(HWND, unsigned, WORD, LONG);
BOOL PASCAL WinwhereDriver(PSTR, HWND, WORD);
BOOL FAR PASCAL FindFilesInDirectory(PSTR, WORD, HWND, HWND);
void UpdateFileTypeCombo(HWND);


/****************************************************************************/
/*                                                                          */
/* Function : WinMain()                                                     */
/*                                                                          */
/* Purpose  : Main initialization function for a WinApp. All that this one  */
/*            does is to invoke the main dialog box.                        */
/*                                                                          */
/* Returns  : Zero.                                                         */
/*                                                                          */
/****************************************************************************/
int PASCAL WinMain(hInstance, hPrevInstance, lpCmdLine, nCmdShow)
  HANDLE hInstance;
  HANDLE hPrevInstance;
  LPSTR  lpCmdLine;
  int    nCmdShow;
{
  FARPROC lpProcWinwhere;

  hInst = hInstance;
  lpProcWhere = MakeProcInstance(WhereDlgProc, hInstance);
  _DialogBox(hInst, "Where", NULL, lpProcWhere);
  return 0;
}


/****************************************************************************/
/*                                                                          */
/* Function : WinwhereDlgProc()                                             */
/*                                                                          */
/* Purpose  : The dialog box procedure for the main "winwhere" dialog box.  */
/*                                                                          */
/* Returns  : TRUE if we processed the message, FALSE if not.               */
/*                                                                          */
/****************************************************************************/
BOOL FAR PASCAL WinwhereDlgProc(hDlg, message, wParam, lParam)
  HWND hDlg;
  unsigned message;
  WORD wParam;
  LONG lParam;
{
  FARPROC lpProc;
  int     rc, i;
  char    szBuf[80];
  char    *s;

  static int iDaysInMonth[12] =
  {
    31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31
  };


  switch (message)
  {
    case WM_INITDIALOG :
      /*
        We want to have an icon associated with this dialog box if the
        user chooses to minimize the dialog box. We load the "winwhere"
        icon from the resource file, and stick it into the hIcon
        field in the dialog box's class structure.
      */
#ifndef MEWEL
      SetClassWord(hDlg, GCW_HICON, LoadIcon(hInst, "Winwhere"));
#endif

      hFoundListBox = GetDlgItem(hDlg, ID_FOUNDLIST);
      UpdateFileTypeCombo(GetDlgItem(hDlg, ID_FILETYPE));
      SendDlgItemMessage(hDlg, ID_FILETYPE, CB_SETCURSEL,
                         iCurrFileType = 0, 0L);

      EnableWindow(GetDlgItem(hDlg, ID_FINDOPTS), FALSE);

      EnableWindow(GetDlgItem(hDlg, ID_DATE), FALSE);
      EnableWindow(GetDlgItem(hDlg, ID_EARLYDATE), FALSE);
      EnableWindow(GetDlgItem(hDlg, ID_LATERDATE), FALSE);
      EnableWindow(GetDlgItem(hDlg, ID_EQUALDATE), FALSE);

      EnableWindow(GetDlgItem(hDlg, ID_SIZE), FALSE);
      EnableWindow(GetDlgItem(hDlg, ID_LARGERSIZE), FALSE);
      EnableWindow(GetDlgItem(hDlg, ID_SMALLERSIZE), FALSE);
      EnableWindow(GetDlgItem(hDlg, ID_EQUALSIZE), FALSE);

      /*
        Determine which drives are present in the system
      */
      SendDlgItemMessage(hDlg, ID_DRIVELIST, LB_DIR,
                         0xC000, (LONG) (LPSTR) "~~~.~~~");

      SetFocus(GetDlgItem(hDlg, ID_FILENAME));
      SetDlgItemText(hDlg, ID_FILENAME, FileTypeInfo[0].szSpec);
      SendDlgItemMessage(hDlg, ID_FILENAME, EM_SETSEL, 
                               NULL, MAKELONG(0, 0x7FFF));

      return FALSE;   /* Indicates the focus is set to a control */


    case WM_COMMAND:
      switch (wParam)
      {
        case ID_BYSIZE  :
          if (IsDlgButtonChecked(hDlg, ID_BYSIZE))
          {
            EnableWindow(GetDlgItem(hDlg, ID_SIZE), TRUE);
            EnableWindow(GetDlgItem(hDlg, ID_LARGERSIZE), TRUE);
            EnableWindow(GetDlgItem(hDlg, ID_SMALLERSIZE), TRUE);
            EnableWindow(GetDlgItem(hDlg, ID_EQUALSIZE), TRUE);
            CheckRadioButton(hDlg, ID_SMALLERSIZE, ID_EQUALSIZE, ID_EQUALSIZE);
            SetFocus(GetDlgItem(hDlg, ID_SIZE));
          }
          else
          {
            EnableWindow(GetDlgItem(hDlg, ID_SIZE), FALSE);
            EnableWindow(GetDlgItem(hDlg, ID_LARGERSIZE), FALSE);
            EnableWindow(GetDlgItem(hDlg, ID_SMALLERSIZE), FALSE);
            EnableWindow(GetDlgItem(hDlg, ID_EQUALSIZE), FALSE);
          }
          break;

        case ID_BYDATE  :
          if (IsDlgButtonChecked(hDlg, ID_BYDATE))
          {
            EnableWindow(GetDlgItem(hDlg, ID_DATE), TRUE);
            EnableWindow(GetDlgItem(hDlg, ID_EARLYDATE), TRUE);
            EnableWindow(GetDlgItem(hDlg, ID_LATERDATE), TRUE);
            EnableWindow(GetDlgItem(hDlg, ID_EQUALDATE), TRUE);
            CheckRadioButton(hDlg, ID_EARLYDATE, ID_EQUALDATE, ID_EQUALDATE);
            SetFocus(GetDlgItem(hDlg, ID_DATE));
          }
          else
          {
            EnableWindow(GetDlgItem(hDlg, ID_DATE), FALSE);
            EnableWindow(GetDlgItem(hDlg, ID_EARLYDATE), FALSE);
            EnableWindow(GetDlgItem(hDlg, ID_LATERDATE), FALSE);
            EnableWindow(GetDlgItem(hDlg, ID_EQUALDATE), FALSE);
          }
          break;


        case ID_FILETYPE:
           switch (HIWORD(lParam))
           {
             case CBN_SELCHANGE :
               /*
                 When the current selection in the file-type combo box
                 changes, we extract the file spec and put it into
                 the edit field.
               */
               iCurrFileType = (int) 
                 SendDlgItemMessage(hDlg, ID_FILETYPE, CB_GETCURSEL, 0, 0L);
               SetDlgItemText(hDlg, ID_FILENAME,
                              FileTypeInfo[iCurrFileType].szSpec);
               SendDlgItemMessage(hDlg, ID_FILENAME, EM_SETSEL,
                                  NULL, MAKELONG(0, 0x7FFF));
               break;
           }
           return TRUE;


        case ID_ADDFILE :
          lpProc = MakeProcInstance(AddTypeDlgProc, hInst);
          rc = _DialogBox(hInst, "AddType", hDlg, lpProc);
          FreeProcInstance(lpProc);

          if (rc == TRUE)
          {
            UpdateFileTypeCombo(GetDlgItem(hDlg, ID_FILETYPE));
            SendDlgItemMessage(hDlg, ID_FILETYPE, CB_SETCURSEL,
                              iCurrFileType = 0, 0L);
            SetFocus(GetDlgItem(hDlg, ID_FILENAME));
            SetDlgItemText(hDlg, ID_FILENAME, FileTypeInfo[0].szSpec);
            SendDlgItemMessage(hDlg, ID_FILENAME, EM_SETSEL,
                               NULL, MAKELONG(0, 0x7FFF));
          }

          break;

        case ID_FINDOPTS :
          lpProc = MakeProcInstance(GenCommandDlgProc, hInst);
          rc = _DialogBox(hInst, "GenCommand", hDlg, lpProc);
          FreeProcInstance(lpProc);
#ifdef MEWEL
          if (rc == ID_EXEC)
          {
            WinDraw(hDlg);
          }
#endif
          break;


        case IDOK     :
          /*
            The user pressed the OK button, and wants to start a search.
            Gather up all of the values which the user input, and call
            the search driver.
          */
          GetDlgItemText(hDlg, ID_FILENAME, szBuf, sizeof(szBuf));
          if (!szBuf[0])
          {
            MessageBox(hDlg, "No filename specified.", "Error",
                       MB_OK | MB_ICONHAND);
            return TRUE;
          }

          /*
            Do a little validation on the file name
          */
          for (s = szBuf;  *s;  s++)
          {
            if (strchr("|<>[]:+=;,\"/", *s))
            {
              MessageBox(hDlg, "Illegal character in file name.", "Error",
                         MB_OK | MB_ICONHAND);
              return TRUE;
            }
          }

          /*
            There is a small anomaly in the findfirst() function. If we
            specify the name of a device (ie- PRN, AUX), then it will be
            found in *every* subdirectory. So, weed out device names.
            Note - we just check against the reserved DOS device names,
            and not against devices which are installed in config.sys.
          */
          for (i = 0;  pszDevices[i] != NULL;  i++)
            if ((s = strstr(szBuf, pszDevices[i])) == szBuf)
            {
              /*
                The initial part of the filename matches the device.
                We must weed out filenames like "AUX" or "AUX.FOO",
                but not filenames like "AUXIL.C".
              */
              s = szBuf + strlen(pszDevices[i]);
              if (*s == '\0' || *s == '.')
              {
                MessageBox(hDlg, "You cannot specify the name of a device",
                           "Error", MB_OK | MB_ICONHAND);
                return TRUE;
              }
            }


          memset((char *) &SearchInfo, 0, sizeof(SearchInfo));

          if (IsDlgButtonChecked(hDlg, ID_BYDATE))
          {
            char szDate[32], *pTok;
            int  month = 0, day = 0, year = 0;

            GetDlgItemText(hDlg, ID_DATE, szDate, sizeof(szDate));
            if (!szDate[0])
            {
              MessageBox(hDlg, "No date specified.", "Error",
                         MB_OK | MB_ICONHAND);
              return TRUE;
            }

            if ((pTok = strtok(szDate, "/-")) != NULL)
              month = atoi(pTok);
            if ((pTok = strtok(NULL, "/-")) != NULL)
              day = atoi(pTok);
            if ((pTok = strtok(NULL, "/-")) != NULL)
              year = atoi(pTok);

            if (!month || !day || !year)
            {
date_err:
              MessageBox(hDlg, "Incorrect date specified. Must be mm/dd/yy.",
                               "Error", MB_OK | MB_ICONHAND);
              return TRUE;
            }

            /*
              Do a little date validation here
            */
            if (year >= 1900)
              year -= 1900;
            if (year >= 80)
              year -= 80;
            if (month <= 0 || month > 12   ||
                day < 1    || day > iDaysInMonth[month-1] ||
                year < 0   || year >= 2000 ||
                (month == 2 && day == 29 && !ISLEAP(year + 80)))
              goto date_err;


            /*
              Turn the date into a word-length value which is the same
              format as the date field in the structure returned by
              the _dos_findfirst and _dos_findnext functions.
            */
            SearchInfo.wDate = (day & 0x1F) |
                               ((month & 0x0F) << 5) |
                               ((year & 0x7F) << 9);

            SearchInfo.fFlags |= SEARCH_DATE;
            if (IsDlgButtonChecked(hDlg, ID_EARLYDATE))
              SearchInfo.fFlags |= DATE_LT;
            else
            if (IsDlgButtonChecked(hDlg, ID_LATERDATE))
              SearchInfo.fFlags |= DATE_GT;
            else
              SearchInfo.fFlags |= DATE_EQ;
          }

          if (IsDlgButtonChecked(hDlg, ID_BYSIZE))
          {
            char szSize[32];
            GetDlgItemText(hDlg, ID_SIZE, szSize, sizeof(szSize));
            if (!szSize[0])
            {
              MessageBox(hDlg, "No size specified.",
                         "Error", MB_OK | MB_ICONHAND);
              return TRUE;
            }
            SearchInfo.dwSize = atol(szSize);

            /*
              Do a little size validation. Check for negative numbers,
              and for illegal characters in the size string.
            */
            if (SearchInfo.dwSize < 0)
            {
size_err:
              MessageBox(hDlg, "Invalid size specified.",
                         "Error", MB_OK | MB_ICONHAND);
              return TRUE;
            }
            for (s = szSize;  *s;  s++)
              if (!isdigit(*s))
                goto size_err;


            SearchInfo.fFlags |= SEARCH_SIZE;
            if (IsDlgButtonChecked(hDlg, ID_SMALLERSIZE))
              SearchInfo.fFlags |= SIZE_LT;
            else
            if (IsDlgButtonChecked(hDlg, ID_LARGERSIZE))
              SearchInfo.fFlags |= SIZE_GT;
            else
              SearchInfo.fFlags |= SIZE_EQ;
          }

          WinwhereDriver(szBuf, hDlg, ID_DRIVELIST);

          break;

        case IDCANCEL :
          /*
            The user wants to leave the program...
          */
          EndDialog(hDlg, 0);
          break;

        case ID_HELP :
          /*
            Display the "About" box
          */
          lpProc = MakeProcInstance(About, hInst);
          _DialogBox(hInst, "About", hDlg, lpProc);
          FreeProcInstance(lpProc);
          break;
      }
      return TRUE;

    default:
      return FALSE;
  }
}


/****************************************************************************/
/*                                                                          */
/* Function : WinwhereDriver                                                */
/*                                                                          */
/* Purpose  : Main driver for the search engine.                            */
/*                                                                          */
/* Returns  : TRUE if files were located, FALSE if not.                     */
/*                                                                          */
/****************************************************************************/
BOOL PASCAL WinwhereDriver(PSTR szSearchSpec, HWND hDlg, WORD idDriveListBox)
{
  HWND hDriveListBox;
  WORD wItem;
  HCURSOR hOldCursor;
  char szFileSpec[128];
  char chDrive;
  int  afDriveMask[26];
  int  nSelected;
  BOOL bContinue = TRUE;

  /*
    Get the handle to the directory listbox, and the number of
    entries in the listbox.
  */
  hDriveListBox = GetDlgItem(hDlg, idDriveListBox);


  /* 
    Disable run button & change cursor to hourglass
  */
  EnableWindow(GetDlgItem(hDlg, IDOK), FALSE);
  hOldCursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
            
  /*
    Erase the contents of the listbox so that we may put the found
    file names in there.
  */
  SendMessage(hFoundListBox, WM_SETREDRAW, FALSE, 0L);
  SendMessage(hFoundListBox, LB_RESETCONTENT, 0, 0L);

  /*
    If no drives were selected, then search the current directory
  */
  if (SendMessage(hDriveListBox, LB_GETSELCOUNT, 0, 0L) == 0)
  {
    FindFilesInDirectory(szSearchSpec, _A_NORMAL, hFoundListBox,
                         GetDlgItem(hDlg, ID_PATH));
    goto bye;
  }


  /*
    Search all of the drives for the desired file
  */
  /*
    Determine which drives are present in the system
  */
  nSelected = SendMessage(hDriveListBox, LB_GETSELITEMS, 26, 
                          (DWORD) (LPSTR) afDriveMask);
  for (wItem = 0;  wItem < nSelected;  wItem++)
  {
    SendMessage(hDriveListBox, LB_GETTEXT, afDriveMask[wItem],
                                      (LONG)(LPSTR) szFileSpec);
    chDrive = szFileSpec[2];
    sprintf(szFileSpec, "%c:\\%s", chDrive, szSearchSpec);
    bContinue = FindFilesInDirectory(szFileSpec, _A_NORMAL, 
                            hFoundListBox, GetDlgItem(hDlg, ID_PATH));

  }

  /*
    Check for null search operation
  */
bye:
  if (SendMessage(hFoundListBox, LB_GETCOUNT, 0, 0L) == 0L)
    MessageBox(hDlg, "No files found matching criteria!", 
                     "Winwhere", MB_OK);
  else
  {
    EnableWindow(hFoundListBox, TRUE);
    EnableWindow(GetDlgItem(hDlg, ID_FINDOPTS), TRUE);
  }
               
  /* 
    Activate redrawing
    Restore cursor & repaint display
  */
  SendMessage(hFoundListBox, WM_SETREDRAW, TRUE, 0L);
  SetCursor(hOldCursor);
  EnableWindow(GetDlgItem(hDlg, IDOK), TRUE);
  InvalidateRect(hFoundListBox, NULL, TRUE);
  UpdateWindow(hFoundListBox);
  InvalidateRect(GetDlgItem(hDlg, ID_PATH), NULL, TRUE);
  UpdateWindow(GetDlgItem(hDlg, ID_PATH));
}


/****************************************************************************/
/*                                                                          */
/* Function : FindFilesInDirectory()                                        */
/*                                                                          */
/* Purpose  : Searches a single directory for files who match the search    */
/*            criteria. Recurses on subdirectories.                         */
/*                                                                          */
/* Returns  : TRUE if we sould continue the search.                         */
/*                                                                          */
/****************************************************************************/
BOOL FAR PASCAL FindFilesInDirectory(PSTR szFileSpec, WORD wAttributes, 
                                     HWND hListBox, HWND hPath)
{
   BOOL bContinue;           
   BOOL bDirOutput;          
   BOOL bFound;
   char szPath[64];          
   char szSpec[64];          
   char szEntry[64];         
   PSTR pSlash;
   HDC  hDC;
   RECT r;
   struct find_t ff;   

   /* initialization */
   bContinue = TRUE;
   bDirOutput = FALSE;

   /*
     Print a little message in the static path field
   */
   hDC = GetDC(hPath);
   GetClientRect(hPath, &r);
   FillRect(hDC, &r, GetStockObject(WHITE_BRUSH));
   if ((pSlash = strrchr(szFileSpec, '\\')) != NULL)
   {
     *pSlash = '\0';
     strcpy(szPath, szFileSpec);
     strcpy(szSpec, pSlash+1);
   }
   else
   {
     strcpy(szPath, ".");
     strcpy(szSpec, szFileSpec);
   }
   TextOut(hDC, 0, 0, strlwr(szFileSpec), strlen(szFileSpec));
   if (pSlash)
     *pSlash = '\\';
   ReleaseDC(hPath, hDC);

      
   /*
     Look at all of the normal files which match the file spec.
   */
   if (_dos_findfirst(szFileSpec, wAttributes, &ff) == 0)
   {
     do
     {
       bFound = TRUE;

       if (SearchInfo.fFlags & SEARCH_SIZE)
       {
         if ((SearchInfo.fFlags & SIZE_LT) && ff.size >= SearchInfo.dwSize ||
             (SearchInfo.fFlags & SIZE_GT) && ff.size <= SearchInfo.dwSize ||
             (SearchInfo.fFlags & SIZE_EQ) && ff.size !=SearchInfo.dwSize)
           bFound = FALSE;
       }
       if (SearchInfo.fFlags & SEARCH_DATE)
       {
         if ((SearchInfo.fFlags & DATE_LT) && ff.wr_date >= SearchInfo.wDate ||
             (SearchInfo.fFlags & DATE_GT) && ff.wr_date <= SearchInfo.wDate ||
             (SearchInfo.fFlags & DATE_EQ) && ff.wr_date !=SearchInfo.wDate)
           bFound = FALSE;
       }

       if (bFound)
       {
         /*
           Add the found file to the listbox. Use the complete path name.
         */
         sprintf(szEntry, "%s\\%s", szPath, ff.name);
         strlwr(szEntry);
         if (SendMessage(hListBox, LB_INSERTSTRING, -1,
                                    (LONG)(LPSTR) szEntry) < 0)
         {
           MessageBox(GetParent(hListBox), "Insufficient Memory!", 
                                 "File Finder", MB_OK | MB_ICONHAND);
           return FALSE;
         }
       }
     } while (_dos_findnext(&ff) == 0);
   } 
   
   /*
     We need to recurse on any subdirectories which were found.
   */
   sprintf(szEntry, "%s\\*.*", szPath);
   if (_dos_findfirst(szEntry, _A_SUBDIR, &ff) == 0)
     do
     {
       if (ff.attrib == _A_SUBDIR && ff.name[0] != '.')
       {
         sprintf(szEntry, "%s\\%s\\%s", szPath, ff.name, szSpec);
         bContinue = FindFilesInDirectory(szEntry, wAttributes,
                                          hListBox, hPath);
       }
     } while (bContinue && _dos_findnext(&ff) == 0);
   

   return bContinue;
}


/****************************************************************************/
/*                                                                          */
/* Function : GenCommandDlgProc()                                           */
/*                                                                          */
/* Purpose  : Dialog box proc for the find-options dialog box.              */
/*                                                                          */
/* Returns  : TRUE if we processed the message, FALSE if not.               */
/*                                                                          */
/****************************************************************************/
BOOL FAR PASCAL GenCommandDlgProc(hDlg, message, wParam, lParam)
  HWND hDlg;
  unsigned message;
  WORD wParam;
  LONG lParam;
{
  HWND hListBox;
  char szBuf[128];
  int  nSelected, i;


  switch (message)
  {
    case WM_INITDIALOG :
      /*
        Transfer the filenames from the main dialog box's listbox
        to this listbox. Initially select all strings in the listbox.
      */
      hListBox = GetDlgItem(hDlg, ID_FOUNDLIST);
      nSelected = SendMessage(hFoundListBox, LB_GETCOUNT, 0, 0L);
      for (i = 0;  i < nSelected;  i++)
      {
        SendMessage(hFoundListBox, LB_GETTEXT,
                    i, (LONG) (LPSTR) szBuf);
        SendMessage(hListBox, LB_INSERTSTRING, -1, (LONG) (LPSTR) szBuf);
        SendMessage(hListBox, LB_SETSEL, TRUE, MAKELONG(i, 0));
      }

      return TRUE;

    case WM_COMMAND:
      switch (wParam)
      {
        case IDOK     :
        {
          int  aiIndex[128];
          char szName[65];
          char szCmd[128];
          char *pCmd, *pBuf;
          char szBatchFile[65];
          int  fd;
          OFSTRUCT OF;

          GetDlgItemText(hDlg, ID_BATCHFILE, szBatchFile, sizeof(szBatchFile));
          if (szBatchFile[0] == '\0')
          {
            MessageBox(hDlg, "You must input the name of a batch file",
                             "Error", MB_OK | MB_ICONHAND);
            return TRUE;
          }

          GetDlgItemText(hDlg, ID_COMMAND, szCmd, sizeof(szCmd));
          if (szCmd[0] == '\0')
          {
            MessageBox(hDlg, "You must input a command",
                             "Error", MB_OK | MB_ICONHAND);
            return TRUE;
          }

          hListBox = GetDlgItem(hDlg, ID_FOUNDLIST);
          nSelected = SendMessage(hListBox,
                                  LB_GETSELITEMS,
                                  sizeof(aiIndex)/sizeof(int),
                                  (DWORD) (LPSTR) aiIndex);
          if (nSelected <= 0)
          {
            MessageBox(hDlg, "You must select at least one file",
                             "Error", MB_OK);
            return TRUE;
          }

          /*
            Create the batch file.
          */
          fd = OpenFile(szBatchFile, &OF, OF_CREATE | OF_WRITE);
          if (fd < 0)
          {
            sprintf(szBuf,
                    "Couldn't open the batch file %s [Code %d]", 
                    szBatchFile, OF.nErrCode);
            MessageBox(hDlg, szBuf, "Open Error", MB_OK);
            return TRUE;
          }

          /*
            For all of the selected files, generate a command string and
            write it to the specified batch file.
          */
          for (i = 0;  i < nSelected;  i++)
          {
            SendMessage(hListBox, LB_GETTEXT,
                        aiIndex[i], (LONG)(LPSTR) szName);
            /*
              Substitute the '!' with the name of the file
            */
            for (pCmd = szCmd, pBuf = szBuf;  *pCmd;  )
              if (*pCmd == '!')
              {
                strcpy(pBuf, szName);
                pBuf = szBuf + strlen(szBuf);
                pCmd++;
              }
              else
                *pBuf++ = *pCmd++;
            *pBuf++ = '\r';
            *pBuf++ = '\n';
            *pBuf = '\0';

            _lwrite(fd, szBuf, strlen(szBuf));
          }

          _lclose(fd);

          /*
            Perhaps the user wants to execute the batch file?
          */
          if (MessageBox(hDlg, "Do you want to execute the batch file now?",
                               "Batch file written", MB_YESNO) == IDYES)
          {
            if ((pCmd = strchr(szBatchFile, '.')) != NULL)
              *pCmd = '\0';       /* get rid of the .BAT extension */
            WinExec(szBatchFile, 1);
            EndDialog(hDlg, ID_EXEC);
          }
          else
            EndDialog(hDlg, TRUE);
          break;
        } 

        case IDCANCEL :
          EndDialog(hDlg, 0);
          break;
      }
      return TRUE;

    default:
      return FALSE;
  }
}


/****************************************************************************/
/*                                                                          */
/* Function : AddTypeDlgProc                                                */
/*                                                                          */
/* Purpose  : Dialog box proc for the dialog box which lets the user add    */
/*            a filetype/filespec pairing to the current list.              */
/*                                                                          */
/* Returns  : TRUE if we processed the message, FALSE if not.               */
/*                                                                          */
/****************************************************************************/
BOOL FAR PASCAL AddTypeDlgProc(hDlg, message, wParam, lParam)
  HWND hDlg;
  unsigned message;
  WORD wParam;
  LONG lParam;
{
  char  szType[65];
  char  szSpec[65];

  switch (message)
  {
    case WM_INITDIALOG :
      return TRUE;

    case WM_COMMAND:
      switch (wParam)
      {
        case IDOK     :
          /*
            Get the file type string
          */
          GetDlgItemText(hDlg, ID_FILETYPE, szType, sizeof(szType));
          if (szType[0] == '\0')
          {
            MessageBox(hDlg, "You must input a file type", "Error", MB_OK);
            return TRUE;
          }

          /*
            Get the file spec string
          */
          GetDlgItemText(hDlg, ID_FILENAME, szSpec, sizeof(szSpec));
          if (szSpec[0] == '\0')
          {
            MessageBox(hDlg, "You must input a file specification",
                             "Error", MB_OK | MB_ICONHAND);
            return TRUE;
          }
          
          /*
            Add the pair to the filetype list
          */
          AddFileType(szType, szSpec);

          EndDialog(hDlg, TRUE);
          break;

        case IDCANCEL :
          EndDialog(hDlg, FALSE);
          break;
      }
      return TRUE;

    default:
      return FALSE;
  }
}


/****************************************************************************/
/*                                                                          */
/* Function : AddFileType()                                                 */
/*                                                                          */
/* Purpose  : Adds the two passed strings to the list of filetypes          */
/*                                                                          */
/* Returns  : TRUE if added, FALSE if not.                                  */
/*                                                                          */
/****************************************************************************/
AddFileType(PSTR szType, PSTR szSpec)
{
  int   i;
  char  szBuf[256];

  if (nFileTypes < MAXFILETYPES)
  {
    /*
      Add it to the array as the last element.
    */
    strcpy(FileTypeInfo[nFileTypes].szType, szType);
    strcpy(FileTypeInfo[nFileTypes].szSpec, szSpec);
    nFileTypes++;

    /*
      Create one long string which consists of all the filetypes and
      filespecs. Each string is separated by a blank. We write
      this string to the WIN.INI file.
    */
    szBuf[0] = '\0';
    for (i = 0;  i < nFileTypes;  i++)
    {
      strcat(szBuf, FileTypeInfo[i].szType);
      strcat(szBuf, " ");
      strcat(szBuf, FileTypeInfo[i].szSpec);
      strcat(szBuf, " ");
    }

    WriteProfileString("Winwhere", "FileTypes", szBuf);
    return TRUE;
  }
  else
    return FALSE;
}


/****************************************************************************/
/*                                                                          */
/* Function : UpdateFileTypeCombo()                                         */
/*                                                                          */
/* Purpose  : Reads the filetype/filespec pairs from the WIN.INI file,      */
/*            and adds them to the filetype combo box.                      */
/*                                                                          */
/* Returns  : Nothing.                                                      */
/*                                                                          */
/****************************************************************************/
void UpdateFileTypeCombo(HWND hCombo)
{
  int  i;
  char szBuf[256], *pTok;

  /*
    Read the long filetype/filespec string from the WIN.INI file.
  */
  GetProfileString("Winwhere", "FileTypes", "Anything *.*", szBuf,           
                    sizeof(szBuf));

  /*
    Extract the pairs and add them to the filetype list
  */
  nFileTypes = 0;
  if ((pTok = strtok(szBuf, " ")) != NULL)
    strcpy(FileTypeInfo[nFileTypes].szType, pTok);
  while (pTok)
  {
    if ((pTok = strtok(NULL, " ")) != NULL)
    {
      strcpy(FileTypeInfo[nFileTypes++].szSpec, pTok);
      if ((pTok = strtok(NULL, " ")) != NULL)
        strcpy(FileTypeInfo[nFileTypes].szType, pTok);
    }
  }

  /*
    Refresh the filetype combobox in the main dialog box.
  */
  SendMessage(hCombo, CB_RESETCONTENT, 0, 0L);
  for (i = 0;  i < nFileTypes;  i++)
    SendMessage(hCombo, CB_ADDSTRING, 0,
               (LONG) (LPSTR) FileTypeInfo[i].szType);
}


/****************************************************************************/
/*                                                                          */
/* Function :  About                                                        */
/*                                                                          */
/* Purpose  :  Dialog box proc for the about box.                           */
/*                                                                          */
/* Returns  : TRUE if we processed the message, FALSE if not.               */
/*                                                                          */
/****************************************************************************/
BOOL FAR PASCAL About(HWND hDlg, unsigned message, WORD wParam, LONG lParam)
{
  switch (message)
  {
    case WM_INITDIALOG:
      return TRUE;

    case WM_COMMAND:
      if (wParam == IDOK)
      {
        EndDialog(hDlg, TRUE);
        return TRUE;
      }
      break;
  }
  return FALSE;
}


#ifdef MEWEL
main()
{
  WinInit();
  WinUseSysColors(NULLHWND, TRUE);
  return WinMain(NULL, NULL, (LPSTR) NULL, TRUE);
}
#endif


