{*****************************************************************************
*  OOPDEMO : demonstriert die objektorientierten Features von QuickPascal    *
*            anhand mehrerer Objekte, die sich ber den Bildschirm bewegen   *
*            und dabei vom Bildschirmrand reflektiert werden.                *
**--------------------------------------------------------------------------**
*  Autor            : MICHAEL TISCHER                                        *
*  entwickelt am    : 22.07.1989                                             *
*  letztes Update am: 22.07.1989                                             *
*****************************************************************************}

{$M-}                         { Objekt-Initialisierung nicht mehr berprfen }

program oopdemo;

uses Crt;                                     { Unit mit GotoXY, ClrEol etc. }

const ANZ_OBJEKTE = 10;                      { Anzahl zu erzeugender Objekte }

{== Objekt- und Typdeklarationen ============================================}

type RDaten = record                { Daten fr die Reflektion an einer Wand }
                NeueRichtung : array [1..3] of byte;        { Richtungscodes }
                MoveX, MoveY : integer;                   { Richtungsoffsets }
              end;

type punkt = object                          { das Basisobjekt ist der Punkt }
       X, Y     : integer;                     { Position auf dem Bildschirm }
       Richtung : byte;                                  { Bewegungsrichtung }
       procedure pinit( XPos, YPos : integer);
       procedure draw;                                        { Objekt malen }
       procedure move;                                  { Objekt verschieben }
       function  GetX : integer;                               { X-Pos holen }
       function  GetY : integer;                               { Y-Pos holen }
       function  check( NeuX, NeuY : integer ) : boolean;
     end;

type rechteck = object( punkt )               { Rechteck folgt aus dem Punkt }
       XLen, YLen : integer;                                  { Seitenlngen }
       procedure rinit( XPos, YPos, dX, dY : integer );
       function check( NeuX, NeuY : integer ) : boolean; override;
       procedure draw; override;
     end;

type dreieck = object( punkt )               { ein gleichschenkliges Dreieck }
       SLen : integer;                                        { Seitenlngen }
       procedure dinit( XPos, YPos, dS : integer );
       function check( NeuX, NeuY : integer ) : boolean; override;
       procedure draw; override;
     end;

{== initialisierte globale Variablen (typisierte Konstanten) ================}

const BewDa : array [1..4] of RDaten =       { Bewegungsdaten fr Reflektion }
 (
  ( NeueRichtung :( 4, 2, 3 ) ; MoveX :  1; MoveY : -1 ),
  ( NeueRichtung :( 3, 1, 4 ) ; MoveX : -1; MoveY : -1 ),
  ( NeueRichtung :( 2, 4, 1 ) ; MoveX : -1; MoveY :  1 ),
  ( NeueRichtung :( 3, 1, 2 ) ; MoveX :  1; MoveY :  1 )
 );

{== Methoden des Objekts Punkt ==============================================}

{*****************************************************************************
*  PUNKT.PINIT: ldt die Koordinate eines Punktes (bzw. eines nachfolgenden  *
*               Objekts) in die Objekt-Variablen X und Y und whlt eine zu-  *
*               fllige Bewegungsrichtung aus.                               *
**--------------------------------------------------------------------------**
*  Eingabe : X, Y = Koordinate des Objekts                                   *
*  Ausgabe : keine                                                           *
*****************************************************************************}

procedure punkt.pinit( XPos, YPos : integer);
begin
  self.X := XPos; self.Y := YPos;                          { Position merken }
  self.Richtung := Random( 3 ) + 1;             { Bewegungsrichtung auslosen }
end;

{*****************************************************************************
*  PUNKT.CHECK : stellt fest, ob die Koordinate des bergebenen Bildschirm-  *
*                punktes noch innerhalb des Bewegungsbereichs der Objekte    *
*                auf dem Bildschirm liegt.                                   *
**--------------------------------------------------------------------------**
*  Eingabe : X, Y = Bildschrimposition                                       *
*  Ausgabe : TRUE, wenn der Punkt noch innerhalb des Bewegungsbereichs liegt,*
*            sonst FALSE.                                                    *
*****************************************************************************}

function punkt.check( NeuX, NeuY : integer ) : boolean;

begin
  check := (NeuX >= 1) and (NeuX <= 80) and (NeuY >= 2) and (NeuY <=24);
end;

{*****************************************************************************
*  PUNKT.GETX : liefert die Spalte, in der sich ein Objekt vom Typ PUNKT     *
*               oder eines der nachfolgenden Objekte befindet.               *
**--------------------------------------------------------------------------**
*  Eingabe : keine                                                           *
*  Ausgabe : Spalte des Objekts                                              *
*****************************************************************************}

function punkt.GetX : integer;

begin
  GetX := self.X;                     { Spaltenposition aus dem Objekt holen }
end;

{*****************************************************************************
*  PUNKT.GETY : liefert die Zeile, in der sich ein Objekt vom Typ PUNKT oder *
*               eines der nachfolgenden Objekte befindet.                    *
**--------------------------------------------------------------------------**
*  Eingabe : keine                                                           *
*  Ausgabe : Zeile des Objekts                                               *
*****************************************************************************}

function punkt.GetY : integer;

begin
  GetY := self.Y;                      { Zeilenposition aus dem Objekt holen }
end;

{*****************************************************************************
*  PUNKT.DRAW : Malt ein Objekt vom Typ PUNKT in der aktuelle eingestellten  *
*               Ausgbefarbe                                                  *
**--------------------------------------------------------------------------**
*  Eingabe : keine                                                           *
*  Ausgabe : keine                                                           *
*  Info    : Diese Methode wird von der Methode MOVE aufgerufen, um das      *
*            Objekt auf dem Bildschirm zu zeichnen oder vom Bildschirm zu    *
*            entfernen.                                                      *
*****************************************************************************}

procedure punkt.draw;
begin
  GotoXY( self.X, self.Y );              { Cursor auf Objekt-Position setzen }
  write( '' );                                             { Punkt zeichnen }
end;

{*****************************************************************************
*  PUNKT.MOVE : verschiebt ein OBJEKT vom Typ PUNKT oder eines der nachfol-  *
*               genden Typen in der aktuellen Bewegungsrichtung ber den     *
*               Bildschirm. Stt das Objekt dabei an den Rand des Bewegungs-*
*               bereichs, wird die Bewegungsrichtung so verndert, da das   *
*               Objekt sich weiter bewegen kann.                             *
*---------------------------------------------------------------------------**
*  Eingabe : keine                                                           *
*  Ausgabe : keine                                                           *
*****************************************************************************}

procedure punkt.move;

var i, r : byte;      { dienen der Berechnung einer neuen Bewegungsrichtung  }

begin
  with self do                        { auf die Felder des Objekts zugreifen }
    begin
      TextColor( 0 );                            { Zeichen unsichtbar machen }
      draw;                                              { Objekt ausblenden }
      if ( not check( X + BewDa[self.Richtung].MoveX,               { Crash? }
                      Y + BewDa[Richtung].MoveY ) ) then
        begin                                     { Ja, neue Richtung suchen }
          i := 0;
          repeat                          { nach einer neuen Richtung suchen }
            inc( i );             { Index fr Richtungszhler inkrementieren }
            r := BewDa[Richtung].NeueRichtung[i];            { neue Richtung }
          until check( X + BewDa[r].MoveX, Y + BewDa[r].MoveY );
          Richtung := r;                    { neue Richtung gefunden, merken }
        end;
      inc( X, BewDa[Richtung].MoveX );           { Bewegung in neue Richtung }
      inc( Y, BewDa[Richtung].MoveY );
      TextColor( 7 );                       { Zeichen wieder sichtbar machen }
      draw;                                  { Objekt an neuer Pos. zeichnen }
    end;
end;

{== Methoden des Objekts Rechteck ===========================================}

{*****************************************************************************
*  RECHTECK.RINIT : initialisiert ein Objekt vom Typ RECHTECK, indem es die  *
*                   Position des Objekts sowie die Lnge der beiden Seiten   *
*                   speichert.                                               *
**--------------------------------------------------------------------------**
*  Eingabe : X, Y = Koordinate des Objekts                                   *
*            dX   = Seitenlnge horizontal                                   *
*            dY   = Seitenlnge vertikal                                     *
*  Ausgabe : keine                                                           *
*****************************************************************************}

procedure rechteck.rinit( XPos, YPos, dX, dY : integer);
begin
  self.XLen := dX;                                     { Seitenlngen merken }
  self.YLen := dY;
  self.pinit( XPos, YPos );                          { Objekt initialisieren }
end;

{*****************************************************************************
*  RECHTECK.CHECK : stellt fest, ob sich das bergebene Objekt vom Typ RECHT-*
*                   ECK noch innerhalb des Bewegungsbereichs der Objekte auf *
*                   dem Bildschirm befindet.                                 *
**--------------------------------------------------------------------------**
*  Eingabe : NeuX, NeuY = Ausgangsposition fr das zu berprfende Recht-    *
*                         eck                                                *
*  Ausgabe : TRUE, wenn das Objekt noch innerhalb des Bewegungsbereichs      *
*            liegt, sonst FALSE.                                             *
*  Info    : Der Test wird durchgefhrt, indem die Gltigkeit der Koor-      *
*            dinaten der vier Ecken des Rechtecks mit Hilfe der Methode      *
*            PUNKT.CHECK berprft wird.                                     *
*****************************************************************************}

function rechteck.check( NeuX, NeuY : integer ) : boolean;
begin
  check := inherited self.check( NeuX, NeuY )                         and
           inherited self.check( NeuX+self.XLen-1, NeuY )             and
           inherited self.check( NeuX+self.XLen-1, NeuY+self.YLen-1 ) and
           inherited self.check( NeuX, NeuY+self.YLen-1 );
end;

{*****************************************************************************
*  RECHTECK.DRAW : mal ein Objekt vom Typ RECHTECK an seiner augenblick-     *
*                  lichen Position auf dem Bildschirm                        *
**--------------------------------------------------------------------------**
*  Eingabe : keine                                                           *
*  Ausgabe : keine                                                           *
*  Info    : Diese Methode wird von der Methode MOVE aufgerufen, um das      *
*            Objekt auf dem Bildschirm zu zeichnen oder vom Bildschirm zu    *
*            entfernen.                                                      *
*****************************************************************************}

procedure rechteck.draw;

var i : byte;                                              { Schleifenzhler }

begin
  with self do                        { auf die Felder des Objekts zugreifen }
    begin
      {-- obere Linie zeichnen ----------------------------------------------}

      GotoXY( self.GetX, self.GetY ); Write('');
      for i := 1 to XLen-2 do write( '' ); Write('');

      {-- senkrechte Linien zeichnen ----------------------------------------}

      for i := 1 to YLen - 1 do
        begin
          GotoXY( self.GetX, self.GetY + i ); write( '' );
          GotoXY( self.GetX + XLen - 1, self.GetY + i ); write( '' );
        end;

      {-- untere Linie ziehen -----------------------------------------------}

      GotoXY( self.GetX, self.GetY + YLen - 1 );  Write('');
      for i := 1 to XLen-2 do write( '' ); Write('');
    end;
end;

{== Methoden des Objekts Dreieck ============================================}

{*****************************************************************************
*  DREIECK.DINIT : initialisiert ein Objekt vom Typ DREIECK, indem es die    *
*                  Position des Objekts sowie die Lnge der Grundseite       *
*                  speichert.                                                *
**--------------------------------------------------------------------------**
*  Eingabe : X, Y = Koordinate des Objekts                                   *
*            dS   = Lnge der Grundseiten                                    *
*            dY   = Seitenlnge vertikal                                     *
*  Ausgabe : keine                                                           *
*  Info    : die Lnge der beiden Schenkel entspricht der halben Lnge der   *
*            Grundseite.                                                     *
*****************************************************************************}

procedure dreieck.dinit( XPos, YPos, dS : integer);
begin
  self.SLen := dS or 1;                      { Seitenlnge mu ungerade sein }
  self.pinit( XPos, YPos );                          { Objekt initialisieren }
end;

{*****************************************************************************
*  DREIECK.CHECK : stellt fest, ob sich das bergebene Objekt vom Typ DREI-  *
*                  ECK noch innerhalb des Bewegungsbereichs der Objekte auf  *
*                  dem Bildschirm befindet.                                  *
**--------------------------------------------------------------------------**
*  Eingabe : NeuX, NeuY = die zu berprfende Ausgangspositon des Dreiecks   *
*  Ausgabe : TRUE, wenn das Objekt noch innerhalb des Bewegungsbereichs      *
*            liegt, sonst FALSE.                                             *
*  Info    : Der Test wird durchgefhrt, indem die Gltigkeit der Koor-      *
*            dinaten der drei Ecken des Dreiecks mit Hilfe der Methode       *
*            PUNKT.CHECK berprft wird.                                     *
*****************************************************************************}

function dreieck.check( NeuX, NeuY : integer ) : boolean;

begin
  check := inherited self.check( NeuX, NeuY )                      and
           inherited self.check( NeuX + self.SLen - 1, NeuY )      and
           inherited self.check( NeuX + ( self.SLen shr 1 ),
                                 NeuY - ( self.SLen shr 1 ) );
end;

{*****************************************************************************
*  DREIECK.DRAW : mal ein Objekt vom Typ DREIECK an seiner augenblicklichen  *
*                 Bildschirmosition                                          *
**--------------------------------------------------------------------------**
*  Eingabe : keine                                                           *
*  Ausgabe : keine                                                           *
*  Info    : Diese Methode wird von der Methode MOVE aufgerufen, um das      *
*            Objekt auf dem Bildschirm zu zeichnen oder vom Bildschirm zu    *
*            entfernen.                                                      *
*****************************************************************************}

procedure dreieck.draw;

var i : byte;                                              { Schleifenzhler }

begin
  with self do              { erlaubt den Zugriff auf die Felder des Objekts }
    begin
      {-- Grundseite malen --------------------------------------------------}

      GotoXY( self.GetX, self.GetY );
      for i := 1 to SLen do write( '' );

      {-- die beiden Schenkel zeichnen --------------------------------------}

      for i := 1 to ( SLen shr 1 ) do
        begin
          GotoXY( self.GetX + i, self.GetY - i ); write( '*' );
          GotoXY( self.GetX + SLen - i - 1, self.GetY - i ); write( '*' );
        end;
    end;
end;

{============================================================================}
{==    Hauptprogramm    =====================================================}
{============================================================================}

var r     : array [1..ANZ_OBJEKTE] of rechteck;         { Arrays mit jeweils }
    d     : array [1..ANZ_OBJEKTE] of dreieck;        { einem Typ von Objekt }
    p     : array [1..ANZ_OBJEKTE] of punkt;
    anzr,                        { Anzahl der Dreiecke, Punkte und Rechtecke }
    anzd,
    anzp ,
    i     : integer;                                       { Schleifenzhler }
    ch    : char;                                   { zum Abfragen der Taste }

begin
  Randomize;                               { Zufallsgenerator initialisieren }
  repeat
    {-- Bildschirm aufbauen -------------------------------------------------}

    TextColor( 7 );  TextBackground( 0 );                  { normale Schrift }
    ClrScr;                                             { Bildschirm lschen }
    TextColor( 0 );  TextBackground( 7 );                  { inverse Schrift }
    ClrEol;                              { erste Bildschirmzeile invertieren }
    write( '     OOPDEMO - demonstriert die objektorientierten Features',
           ' von QuickPascal' );
    GotoXY( 1, 25 );                      { Cursor in letzte Bildschirmzeile }
    ClrEol;                                       { und auch die invertieren }
    write( ' SPACE = neuer Durchlauf  RETURN = Ende           ',
           '(c) 1989 by Michael Tischer' );
    TextBackground( 0 );               { Hintergrundfarbe auf schwarz setzen }
    anzr := 0;                                { es gibt bisher weder Punkte, }
    anzd := 0;                                { noch Kreise, noch Dreiecke   }
    anzp := 0;

    {-- Objekte zufllig erzeugen -------------------------------------------}

    for i := 1 to ANZ_OBJEKTE do                      { ANZ_OBJEKTE erzeugen }
      case Random(3) of                           { Art des Objekts auslosen }
        0 : begin                                           { Punkt erzeugen }
              inc( anzp );                    { Anzahl Punkte inkrementieren }
              new( p[anzp ] );                             { Objekt erzeugen }
              p[ anzp ].pinit( Random(79)+1, Random(24)+2 );
          end;
        1 : begin                                        { Rechteck erzeugen }
              inc( anzr );                 { Anzahl Rechtecke inkrementieren }
              new( r[ anzr ] );                            { Objekt erzeugen }
              r[ anzr ].rinit( Random(60)+1, Random(15)+2,
                               Random(10)+3, Random(5)+2 );
            end;
        2 : begin                                         { Dreieck erzeugen }
              inc( anzd );                  { Anzahl Dreiecke inkrementieren }
              new( d[ anzd ] );                            { Objekt erzeugen }
              d[ anzd ].dinit( Random(60)+1, Random(10)+14, Random(7)+3 );
            end;
      end;

    {-- Objekte bewegen, bis eine Taste bettigt wird -----------------------}

    ch := #0;                                    { noch keine Taste bettigt }
    repeat
      for i := 1 to ANZ_OBJEKTE do                 { die Objekte durchlaufen }
        begin
          if ( i <= anzp ) then                  { gibt es noch einen Punkt? }
            p[ i ].move;                                 { Ja, Punkt bewegen }
          if ( i <= anzr ) then                 { gibt es noch ein Rechteck? }
            r[ i ].move;                              { Ja, Rechteck bewegen }
          if ( i <= anzd ) then                  { gibt es noch ein Dreieck? }
            d[ i ].move;                               { Ja, Dreieck bewegen }
        end;
      if KeyPressed then                        { wurde eine Taste bettigt? }
        begin
          ch := ReadKey;                             { bettigte Taste holen }
          if ( ch = #0 ) then                    { erweiterter Tastaturcode? }
            begin                     { Ja, Taste holen aber nicht verwerten }
              ch := ReadKey;
              ch := #0;                                    { Taste wegwerfen }
            end;
        end;
    until ( ch = ' ' ) or ( ch = #13 );

    {-- die erzeugten Objekte wieder lschen --------------------------------}

    for i := 1 to ANZ_OBJEKTE do                   { die Objekte durchlaufen }
      begin
        if ( i <= anzp ) then                    { gibt es noch einen Punkt? }
          dispose( p[ i ] );                                   { Ja, lschen }
        if ( i <= anzr ) then                   { gibt es noch ein Rechteck? }
          dispose( r[ i ] );                                   { Ja, lschen }
        if ( i <= anzd ) then                    { gibt es noch ein Dreieck? }
          dispose( d[ i ] );                                   { Ja, lschen }
      end;
  until ( ch = #13 );               { wiederholen, bis Return bettigt wurde }
  ClrScr;                                               { Bildschirm lschen }
end.
