/*********************************************************************

 ----  H E L P M G R  ------ Help-Manager for MS-Windows applications 

======================================================================

This module controls and organizes the help management for a
MS-Windows application which has linked it.

----------------------------------------------------------------------

Copyright 1989 by
          Marcellus Buchheit, Buchheit software research
          Zaehringerstrasse 47, D-7500 Karlsruhe 1
          Phone (0) 721/37 67 76    (West Germany)

Release 1.00 of 89-Jul-29 --- All rights reserved.

*********************************************************************/

#define  NOMINMAX
#include <WINDOWS.H>
#include "HELPMGR.H"

/* further C standard headers */
#include <STDLIB.H>
#include <STRING.H>

/* window function parameter macros */
#define P2LO LOWORD(ulP2)
#define P2HI HIWORD(ulP2)
#define LADDR(r) (LONG)(LPSTR)(r)

/* local module data values */
static WORD sxChar; /* width of a SYSTEM character */
static WORD syChar; /* height of a standard SYSTEM character */

static HWND hwMain; /* handle of application's main menu */
static HANDLE hiMain; /* handle of application's instance */

/* description line definitions */
#define S_CMD_DESCR 45 /* number of characters in description line */
static HWND hwLine;    /* window handle of command description line */

/* current help mode */
enum {HPM_NONE,HPM_INDEX,HPM_CONTEXT};
static int iHelpMode=HPM_NONE;

static HWND hwDbxHelp; /* window handle for the help dialog boxes */
FARPROC rfdHelp;       /* instance of help dialog box function */
FARPROC rfwSubEdit;    /* subclassing function for "edit" class */
FARPROC rfwOrgEdit;    /* window function of original "edit" class */
static int iTextPos;   /* text position in help text */
static int iTextLen;   /* number of lines in help text */

static int iMenuCmd;    /* last selected menu command */

/* current help text environment */
static int iCurHelpText; /* current set help-text, 0 if no text set */
static int iTextFirst;   /* minimum general help-text index */
static int iTextLast;    /* maximum general help-text index */
static HANDLE hIndex;    /* handle of index table resource */

/* current message/dialog box environment */
static int iCurDlgBox;    /* ident of selected dialog box ident */
static int iCurDlgItem;   /* ident of selected dialog box item */
static int iCurMsgString; /* ident of message string */

/* pointers to message filter hook functions */
FARPROC rfhMsgFilter,rfhPrevMsgFilter;


/*********************************************************************
 H p m S e t C m d L i n e
======================================================================

This function sets and displayes the command description line after
the application window was resized.

Parameters:
 syClient  is the new vertical size of the client area of
           the application window.

Return:
 none

*********************************************************************/

VOID HpmSetCmdLine(WORD syClient)

 {SetWindowPos
   (hwLine,NULL,-1,syClient-(syChar+4-1),0,0,
    SWP_NOACTIVATE|SWP_NOZORDER|SWP_NOSIZE
   );
  ShowWindow(hwLine,SW_SHOWNA);
 } /* HpmSetCmdLine() */


/*********************************************************************
 G e t T e x t I n d e x
======================================================================

This function determines the index number of the help text for the
entry <i> in the help keyword index and returns it as a WORD. The
first WORD in the list (at 0) contains the maximum available help text
of the index.

Parameters:
 hwBox  is the handle of the dialog box window.

Return:
 The index of the text is returned. It is 0 if it was not determined.

*********************************************************************/

static int GetTextIndex(WORD i)

 {WORD FAR *ru;
  int iVal;
 
  if (hIndex==NULL) return 0; /* memory full => not found */
  /* get index of help text via resource table (type RCDATA) */
  ru=(WORD FAR*)LockResource(hIndex);
  iVal=ru[i]; UnlockResource(hIndex);
  return iVal;
 } /* GetTextIndex() */


/*********************************************************************
 G e t I n d e x P o s
======================================================================

This function gets the entry in the index list for the text
<iCurHelpText> if this is not 0. The position of this or the previous
position in the index is returned. Extremely the lowest index value is
returned.

Parameters:
 none

Return:
 The position of the current help text is returned. If <iCurHelpText>
 is 0 the first entry in the index is returned.

*********************************************************************/

static int GetIndexPos(VOID)

 {WORD FAR *rmuTable,FAR *ru;
  int iEntry,nEntry;
  WORD iFound=0;
 
  if (!hIndex||iCurHelpText==0) return 0; /* memory too small */
  /* get index of help text via resource table (type RCDATA) */
  rmuTable=(WORD FAR*)LockResource(hIndex); nEntry=rmuTable[0];
  /* search in index list */
  for (iEntry=0,ru=rmuTable+1;iEntry<nEntry;iEntry++,ru++)
   {if (*ru<=iCurHelpText && *ru>rmuTable[iFound+1])
     {/* set new "previous" element */
      iFound=iEntry;
     } /* if */
   } /* for */
  UnlockResource(hIndex);
  return iFound;
 } /* GetIndexPos() */


/*********************************************************************
 L o a d H e l p I n d e x
======================================================================

This function reads the help index word list from resource HLS_INDEX
and sets it into the listbox IDLIST of dialog box <hwBox>.

Parameters:
 hwBox  is the handle of the dialog box window.

Return:
 none

*********************************************************************/

static VOID LoadHelpIndex(HWND hwBox)

 {HANDLE hIndex;
  LPSTR rzText,rzEnd;
  BYTE c;

  hIndex=FindResource
   (hiMain,MAKEINTRESOURCE(HLS_INDEX),MAKEINTRESOURCE(HELPLIST));
  if (hIndex) hIndex=LoadResource(hiMain,hIndex);
  if (!hIndex) return; /* memory too small */
  rzText=LockResource(hIndex);
  while (*rzText!=26)
   {/* read next string until end of line */
    rzEnd=rzText+1;
    while ((c=*rzEnd)!=0 && c!='\r' && c!='\n' && c!=26) rzEnd++;
    *rzEnd=0; /* temporary end of string */
    SendDlgItemMessage(hwBox,IDLIST,LB_ADDSTRING,-1,LADDR(rzText));
    while ((c=*rzEnd)<' ' && c!=26) rzEnd++; /* search next string */
    rzText=rzEnd; /* new start of string */
   } /* while */
  UnlockResource(hIndex); /* resource can be moved/discarded */
 } /* LoadHelpIndex() */


/* needed forward references */
BOOL FAR PASCAL fdHelpIndex(HWND hw,WORD iMsg,WORD uP1,DWORD ulP2);
LONG FAR PASCAL fwSubEdit(HWND hw,WORD iMsg,WORD uP1,DWORD ulP2);
BOOL FAR PASCAL fdHelpText(HWND hw,WORD iMsg,WORD uP1,DWORD ulP2);


/*********************************************************************
 S e t I n d e x H e l p
======================================================================

This function creates the index help modeless dialog box and sets the
new help mode.

Parameters:
 none

Return:
 none

*********************************************************************/

static VOID SetIndexHelp(VOID)

 {if (!hIndex)
   {/* load index for the first time */
    hIndex=FindResource
     (hiMain,MAKEINTRESOURCE(HLS_INDEX),MAKEINTRESOURCE(RT_RCDATA));
    if (hIndex) hIndex=LoadResource(hiMain,hIndex);
    if (!hIndex) return; /* cannot load */
   } /* if */
  /* create modeless dialog box */
  rfdHelp=MakeProcInstance(fdHelpIndex,hiMain);
  if (CreateDialog
       (hiMain,MAKEINTRESOURCE(DBX_HPMINDEX),hwMain,rfdHelp))
   {/* help box created */
    iHelpMode=HPM_INDEX;
   } /* if */
 } /* SetIndexHelp() */


/*********************************************************************
 S e t C o n t e x t H e l p
======================================================================

This function creates the context help modeless dialog box and sets
the new help mode.

Parameters:
 iNewText  is the index of the new help text.

Return:
 none

*********************************************************************/

static VOID SetContextHelp(int iNewText)

 {iCurHelpText=iNewText; /* set new text */
  rfdHelp=MakeProcInstance(fdHelpText,hiMain); /* dialog function */
  if (CreateDialog
       (hiMain,MAKEINTRESOURCE(DBX_HPMTEXT),hwMain,rfdHelp))
   {/* help box created */
    iHelpMode=HPM_CONTEXT;
   } /* if */
 } /* SetContextHelp() */


/*********************************************************************
 R e s e t H e l p
======================================================================

This function resets the help mode and destroyes the current help
modeless dialog box.

Parameters:
 none

Return:
 none

*********************************************************************/

static VOID ResetHelp(VOID)

 {DestroyWindow(hwDbxHelp); hwDbxHelp=NULL; iHelpMode=HPM_NONE;
  FreeProcInstance(rfdHelp);
 } /* ResetHelp() */


/*********************************************************************
 f d H e l p I n d e x
======================================================================

### Dialog Box function ###

This function processes any messages received by the "Help index"
dialog box.

Parameters:
standard message data

Return:
standard dialog box function value.

*********************************************************************/

BOOL FAR PASCAL fdHelpIndex(HWND hw,WORD iMsg,WORD uP1,DWORD ulP2)
   
 {int i;
 
  switch (iMsg)
   {case WM_INITDIALOG:
      hwDbxHelp=hw; /* set global variable */
      LoadHelpIndex(hw); /* load resource with help-index keywords */
      /* select current set index entry */
      SendDlgItemMessage(hw,IDLIST,LB_SETCURSEL,GetIndexPos(),0L);
      return TRUE; /* no focus set */
    case WM_COMMAND:
      switch (uP1)
       {case IDLIST:
          if (P2HI!=LBN_DBLCLK) return 0L; /* consumed */
          /* double-click causes IDOK: continue */
        case IDOK:
          /* get selected index */
          i=(int)SendDlgItemMessage(hw,IDLIST,LB_GETCURSEL,0,0L);
          if (i==LB_ERR||(i=GetTextIndex(i+1))==0) i=0; /* no text */
        case IDCANCEL:
          /* close modeless index dialog box */
          ResetHelp();
          if (uP1!=IDCANCEL && i!=0)
           {/* switch into context help */
            iTextFirst=HTX_FIRST;
            iTextLast=GetTextIndex(0); /* set range */
            SetContextHelp(i);
           } /* if */
          return TRUE; /* consumed */
       } /* switch */
      break;
   } /* switch */
  return FALSE;
 } /* fdHelpIndex() */


/*********************************************************************
 L o a d H e l p T e x t
======================================================================

This function reads the help text from resource with value <iHelpText>
and sets it into the "edit" field IDTEXT of dialog box <hwBox>. The
extra scroll bar for the "edit" field is initialized and set to first
line. The previous and next buttons are disabled if no general help
text is available in the corresponding direction.

Parameters:
 hwBox .... is the handle of the dialog box window.
 iHelpText  is the resource number of the help text.

Return:
 TRUE if the text is loaded, FALSE if not

*********************************************************************/

static BOOL LoadHelpText(HWND hwBox,int iHelpText)

 {HANDLE hIndex;
  LPSTR rzText,rzEnd;
  BYTE c;

  hIndex=FindResource
   (hiMain,MAKEINTRESOURCE(iHelpText),MAKEINTRESOURCE(HELPTEXT));
  if (hIndex) hIndex=LoadResource(hiMain,hIndex);
  if (!hIndex) return FALSE; /* text not loaded */
  rzText=LockResource(hIndex);
  /* first line is title */
  rzEnd=rzText; while ((c=*rzEnd)!=0 && c!='\r' && c!='\n') rzEnd++;
  *rzEnd=0; /* temporary end of title */
  SetDlgItemText(hwBox,IDTITLE,rzText);
  while ((c=*rzEnd)<' ' && c!=26) rzEnd++; /* search start of text */
  /* search end of help text */
  rzText=rzEnd; while ((c=*rzEnd)!=0 && c!=26) rzEnd++;
  *rzEnd=0; /* temporary end of text */
  SetDlgItemText(hwBox,IDTEXT,rzText);
  UnlockResource(hIndex); /* resource can be moved/discarded */
  /* initialize scroll bar */
  iTextLen=(int)SendDlgItemMessage(hwBox,IDTEXT,EM_GETLINECOUNT,0,0L);
  iTextLen=max(iTextLen-10+1,1); iTextPos=1;
  SetScrollRange(GetDlgItem(hwBox,IDTXSRL),SB_CTL,1,iTextLen,FALSE);
  SetScrollPos(GetDlgItem(hwBox,IDTXSRL),SB_CTL,iTextPos,TRUE);
  return TRUE; /* text is loaded */
 } /* LoadHelpText() */


/*********************************************************************
 f w S u b E d i t
======================================================================

### Subclassing window function ###

This function avoids any text change in a window of the "Edit" class.
The text can only be scrolled by using the EM_LINESCROLL message.

Parameters:
standard message data

Return:
standard window function return value.

*********************************************************************/

LONG FAR PASCAL fwSubEdit(HWND hw,WORD iMsg,WORD uP1,DWORD ulP2)

 {LONG vRetValue;
 
  vRetValue=0L;
  switch (iMsg)
   {case WM_SETFOCUS:
    case WM_KEYDOWN:
    case WM_CHAR:
      /* ignore input focus/character input: caret cannot be set */
      return 0L;
   } /* switch */
  /* call original window function */
  return CallWindowProc(rfwOrgEdit,hw,iMsg,uP1,ulP2);
 } /* fwSubEdit() */


/*********************************************************************
 f d H e l p T e x t
======================================================================

### Dialog Box function ###

This function processes any messages received by the DBX_SIZE dialog
box.

Parameters:
standard message data.

Return:
standard dialog box function value.

*********************************************************************/

BOOL FAR PASCAL fdHelpText(HWND hw,WORD iMsg,WORD uP1,DWORD ulP2)
   
 {int i;
 
  switch (iMsg)
   {case WM_INITDIALOG:
      hwDbxHelp=hw; /* set global variable */
      /* subwindowing of IDTEXT "edit" window: no changes possible */
      rfwOrgEdit=(FARPROC)SetWindowLong
       (GetDlgItem(hw,IDTEXT),GWL_WNDPROC,(LONG)rfwSubEdit);
      /* load resource with specified help text */
      LoadHelpText(hw,iCurHelpText);
      return TRUE; /* no focus set */
    case WM_VSCROLL:
      switch (uP1)
       {case SB_TOP:
          i=1; break;
        case SB_BOTTOM:
          i=iTextLen; break;
        case SB_LINEDOWN:
          i=min(iTextPos+1,iTextLen); break;
        case SB_LINEUP:
          i=max(iTextPos-1,1); break;
        case SB_PAGEDOWN:
          i=min(iTextPos+10,iTextLen); break;
        case SB_PAGEUP:
          i=max(iTextPos-10,1); break;
        case SB_THUMBPOSITION:
          i=P2LO; break;
        default:
          /* ignore */
          return FALSE;
       } /* switch */
      /* send scrolling command to edit field */
      if (i!=iTextPos)
       {SendDlgItemMessage
         (hw,IDTEXT,EM_LINESCROLL,0,MAKELONG(i-iTextPos,0));
        iTextPos=i;
        SetScrollPos(GetDlgItem(hw,IDTXSRL),SB_CTL,iTextPos,TRUE);
       } /* if */
      return FALSE;
    case WM_COMMAND:
      switch (uP1)
       {case IDPREV:
          if (iCurHelpText<=iTextFirst)
           {/* at start of table: invalid selection */
            MessageBeep(0); 
           }
          else
           {/* load predecessor help-text if possible */
            i=iCurHelpText-1;
            if (LoadHelpText(hw,i)) iCurHelpText=i;
           } /* if */
          break;
        case IDNEXT:
          if (iCurHelpText>=iTextLast)
           {/* at end of table: invalid selection */
            MessageBeep(0);
           }
          else
           {/* load successor help-text if possible */
            i=iCurHelpText+1;
            if (LoadHelpText(hw,i)) iCurHelpText=i;
           } /* if */
          break;
        case IDOK:
        case IDCANCEL:
          /* close modeless index dialog box */
          DestroyWindow(hw); hwDbxHelp=NULL; iHelpMode=HPM_NONE;
          FreeProcInstance(rfdHelp);          
          if (uP1==IDOK) SetIndexHelp();
          return TRUE; /* consumed */
       } /* switch */
      break;
   } /* switch */
  return FALSE;
 } /* fdHelpText() */


/*********************************************************************
 S w i t c h I n d e x H e l p
======================================================================

This function activates the index help.

Parameters:
 none

Return:
 none

*********************************************************************/

static VOID SwitchIndexHelp(VOID)

 {switch (iHelpMode)
   {case HPM_NONE:
      iCurHelpText=0; SetIndexHelp(); iHelpMode=HPM_INDEX; break;
    case HPM_INDEX:
      /* activate context help window */
      SetFocus(hwDbxHelp); break;
    case HPM_CONTEXT:
      ResetHelp(); SetIndexHelp(); break;
   } /* switch */
 } /* SwitchIndexHelp() */


/*********************************************************************
 S w i t c h C o n t e x t H e l p
======================================================================

This function activates the context help with the help text
<iHelpText>.

Parameters:
 iNewText .. is the index of the new help text.
 iAreaFirst  is the first value of the current help text area. If this value
             is 0, for both bound values a default area is used.
 iAreaLast   is the last value of the current help text area. If the
             value is 0, a default value is used.

Return:
 none

*********************************************************************/

static VOID SwitchContextHelp
 (int iNewText,int iAreaFirst,int iAreaLast)

 {/* set first and last bound of current help text area */
  if (iAreaFirst==0)
   {/* use default values */
    if (iNewText>=HTX_FIRST && iNewText<HTX_SPECIAL)
     {/* standard help text area (general, menu etc) */
      iTextFirst=HTX_FIRST; iTextLast=GetTextIndex(0);
     }
    else
     {/* no range */
      iTextFirst=iNewText; iTextLast=iNewText;
     } /* if */
   }
  else
   {/* set both values */
    iTextFirst=iAreaFirst; iTextLast=iAreaLast;
   } /* if */
  switch (iHelpMode)
   {case HPM_NONE:
      SetContextHelp(iNewText); break;
    case HPM_INDEX:
      ResetHelp(); SetContextHelp(iNewText); break;
    case HPM_CONTEXT:
      /* change text if needed, activate context help window */
      if (iCurHelpText)
       {/* change help text */
        LoadHelpText(hwDbxHelp,iNewText); iCurHelpText=iNewText;
       } /* if */
      SetFocus(hwDbxHelp); break;
   } /* switch */
 } /* SwitchContextHelp() */


/*********************************************************************
 H p m P r o c e s s H e l p M s g
======================================================================

This function processes all messages which are received by the
application window function if they are of interest for the help
manager.

Parameters:
 hw .. is the window handle of the application window.
 iMsg  is the message which was received by the application window
       function.
 uP1   is the word parameter of the message.
 ulP2  is the long word parameter of the message.

Return:
 If a non-zero value is returned, the function has consumed the
 message. Otherwise the standard function DefWndProc must be called.

*********************************************************************/

LONG HpmProcessHelpMsg(HWND hw,WORD iMsg,WORD uP1,DWORD ulP2)

 {BYTE z[S_CMD_DESCR+1];
  WORD uHcs=0;

  switch (iMsg)
   {case WM_MENUSELECT:
      if (P2HI!=NULL)
       {if (P2LO&MF_POPUP)
         {/* popup-window header: get code of first command in menu */
          uHcs=GetMenuItemID(uP1,0);
         }
        else if (P2HI!=NULL)
         {/* menu command selected: determine description string */
          uHcs=uP1;
         } /* if */
        iMenuCmd=uHcs; /* command code of menu */
        if (P2LO&MF_POPUP) iMenuCmd--; /* popup-code */
       } /* if */
      if (uHcs>=0xF000 && P2LO&MF_SYSMENU)
       {/* determine predefined entries in system menu */
        iMenuCmd=uHcs; /* command code of menu */
        switch (uHcs)
         {case SC_RESTORE:
            uHcs=HCS_SYSREST; break;
          case SC_MOVE:
            uHcs=HCS_SYSMOVE; break;
          case SC_SIZE:
            uHcs=HCS_SYSSIZE; break;
          case SC_MINIMIZE:
            uHcs=HCS_SYSMIN; break;
          case SC_MAXIMIZE:
            uHcs=HCS_SYSMAX; break;
          case SC_CLOSE:
            uHcs=HCS_SYSCLOSE; break;
          default:
            uHcs=0; break;
         } /* switch */
       }
      else
       {/* other command: convert to help command string */
        uHcs+=2000;
       } /* if */
      if (uHcs!=0)
       {/* command exists: load description string */
        if (P2LO&MF_POPUP) uHcs--; /* use predecessor for popup */
        LoadString(hiMain,uHcs,z,sizeof(z));
       }
      else
       {/* string is empty */
        z[0]=0;
       } /* if */
      /* send string to command description line */
      SetWindowText(hwLine,z);
      break;
    case WM_COMMAND:
      if (uP1==CMD_HELP_INDEX)
       {/* switch index help (on/off) */
        SwitchIndexHelp();
       }
      else if (uP1==CMD_HELP_CONTEXT)
       {/* activate context help: currently not implemented */
       } /* if */
      break; /* not consumed */
   } /* switch */
  return 0L; /* not consumed */
 } /* HpmProcessHelpMsg() */


/*********************************************************************
 S w i t c h D i a l o g H e l p
======================================================================

This function finds the help text for the current selected dialog box
item. If it is found the help text is displayed and TRUE is returned.
Otherwise FALSE is returned.

Parameters:
 none

Used variables:
 iCurDlgBox,iCurDlgItem 

Return:
 none

*********************************************************************/

static int SwitchDialogHelp(VOID)

 {HANDLE hTab;
  WORD FAR *ru;
  int iFirst,iLast,iText;
  BOOL bLastDlg;
  HWND hw;

  hTab=FindResource
   (hiMain,MAKEINTRESOURCE(HLS_DIALOG),MAKEINTRESOURCE(RT_RCDATA));
  if (hTab) hTab=LoadResource(hiMain,hTab);
  if (!hTab) return 0; /* not found or memory too small */
  /* get index of help text via resource table (type RCDATA) */
  ru=(WORD FAR*)LockResource(hTab);
  hw=GetFocus(); /* assumed that selected item in dialog box */
  /* window must be child window otherwise no dialog item */
  if (!(GetWindowLong(hw,GWL_STYLE)&WS_CHILD)) return FALSE;
  /* read ident of child window */
  iCurDlgItem=GetWindowWord(hw,GWW_ID);
  /* search item in table */
  bLastDlg=FALSE; iText=0;
  do
   {if (*ru!=iCurDlgBox && *ru)
     {/* skip dialog box area */
      ru++; /* skip dialog box ident */
      while (*ru) ru+=2; /* skip items of dialog box */
      ru++; /* skip end of dialog box area */   
     }
    else
     {/* dialog box matched: find item */
      if (!*ru) bLastDlg=TRUE; /* default box => last chance */
      ru++; /* skip dialog box ident */
      iFirst=*(ru+1); /* first element in dialog box text area */
      /* check if dialog item matches */
      while (*ru)
       {/* search item */
        if (*ru==iCurDlgItem)
         {/* item found => get ident of help text, exit loop */
          iText=*(ru+1);
          /* search last element */
          while (*ru) ru+=2; /* skip items of dialog box */
          iLast=*(ru-1); /* get last element of dialog box text area */
          goto Found; /* entry is found */
         } /* if */
        ru+=2; /* next item */
       } /* while */
      ru++; /* skip end of dialog box area */
     } /* if */
   }
  while (!bLastDlg);
  Found:
  UnlockResource(hTab); FreeResource(hTab);
  if (iText!=0)
   {/* help text found: display it */
    if (bLastDlg||iText<HTX_SPECIAL)
     {/* no range possible or general text */
      iFirst=0;
     } /* if */
    SwitchContextHelp(iText,iFirst,iLast); return TRUE; /* found */
   } /* if */
  return FALSE;
 } /* SwitchDialogHelp() */


/*********************************************************************
 S w i t c h M e n u H e l p
======================================================================

This function finds the help text for the current selected menu. If it
is found the help text is displayed and TRUE is returned. Otherwise
FALSE is returned.

Parameters:
 none

Return:
 none

*********************************************************************/

static int SwitchMenuHelp(VOID)

 {HANDLE hTab;
  WORD FAR *ru;

  hTab=FindResource
   (hiMain,MAKEINTRESOURCE(HLS_MENU),MAKEINTRESOURCE(RT_RCDATA));
  if (hTab) hTab=LoadResource(hiMain,hTab);
  if (!hTab) return 0; /* not found or memory too small */
  /* get index of help text via resource table (type RCDATA) */
  ru=(WORD FAR*)LockResource(hTab);
  while (*ru)
   {if (*ru==iMenuCmd) break; /* command found */
    ru+=2; /* next pair */
   } /* while */
  UnlockResource(hTab); FreeResource(hTab);
  if (*ru==iMenuCmd)
   {/* help text found: display it */
    SwitchContextHelp(*(ru+1),0,0); return TRUE; /* found */
   } /* if */
  return FALSE;
 } /* SwitchMenuHelp() */


/*********************************************************************
 S w i t c h M e s s a g e H e l p
======================================================================

This function finds the help text for the current displayed message
box. If it is found the help text is displayed and TRUE is returned.
Otherwise FALSE is returned.

Parameters:
 none

Used variables:
 iCurDlgBox,iCurDlgItem 

Return:
 none

*********************************************************************/

static int SwitchMessageHelp(VOID)

 {HANDLE hTab;
  WORD FAR *ru;
  int iText;
  BOOL bLastDlg;

  if (!iCurMsgString) return FALSE; /* no string */
  hTab=FindResource
   (hiMain,MAKEINTRESOURCE(HLS_MESSAGE),MAKEINTRESOURCE(RT_RCDATA));
  if (hTab) hTab=LoadResource(hiMain,hTab);
  if (!hTab) return 0; /* not found or memory too small */
  /* get index of help text via resource table (type RCDATA) */
  ru=(WORD FAR*)LockResource(hTab); iText=0;
  while (*ru)
   {/* outer loop: search selected message string */
    if (*ru++!=iCurMsgString)
     {/* skip message string area */
      while (!*ru)
       {/* skip dialog box area */
        ru++; /* skip dialog box ident */
        while (*ru) ru+=2; /* skip items of dialog box */
        ru++; /* skip end of dialog box area */
       } /* while */
      ru++; /* skip end of message string area */
     }
    else
     {/* message string matched: find dialog box & item */
      bLastDlg=FALSE;
      do
       {if (*ru!=iCurDlgBox && *ru)
         {/* skip dialog box area */
          ru++; /* skip dialog box ident */
          while (*ru) ru+=2; /* skip items of dialog box */
          ru++; /* skip end of dialog box area */   
         }
        else
         {/* dialog box matched: find item */
          if (!*ru) bLastDlg=TRUE; /* default box => last chance */
          ru++; /* skip dialog box ident */
          /* check if dialog item matches */
          while (*ru)
           {/* search item */
            if (*ru==iCurDlgItem)
             {/* item found => get ident of help text, exit loop */
              iText=*(ru+1); goto Found;
             } /* if */
            ru+=2; /* next item */
           } /* while */
          ru++; /* skip end of dialog area */
         } /* if */
       }
      while (!bLastDlg);
     } /* if */
   } /* while */
  Found:
  UnlockResource(hTab); FreeResource(hTab);
  if (iText!=0)
   {/* help text found: display it */
    SwitchContextHelp(iText,0,0); return TRUE; /* found */
   } /* if */
  return FALSE;
 } /* SwitchMessageHelp() */


/*********************************************************************
 f h M s g F i l t e r
======================================================================

### Hook filter procedure ###

This function is the filter function for all dialog-box/message-box/
menu messages to check if a help-manager button is pressed.

If the WM_KEYDOWN message is received and the key VK_F1 is pressed,
the context help is displayed or actualized.
All others messages are not consumed. 

*********************************************************************/

DWORD FAR PASCAL fhMsgFilter(int iCode,WORD uP,MSG FAR *rwm)
  
 {int iMsg;
  
  if (iCode>=0)
   {iMsg=rwm->message; /* current message */
    if (iMsg==WM_KEYDOWN && rwm->wParam==VK_F1 && rwm->hwnd!=hwDbxHelp)
     {/* F1-button pressed: activate/change help */
      switch (iCode)
       {case MSGF_DIALOGBOX:
          /* determine help text/dialog box entry and display it */
          if (SwitchDialogHelp())
           {/* message consumed */
            rwm->message=WM_NULL; return 0L;
           } /* if */
          break; /* not consumed */
        case MSGF_MENU:
          /* determine help text/current menu entry and display it */
          if (SwitchMenuHelp())
           {/* message consumed: close menu by replacing F1 by ESC */
            rwm->message=WM_KEYDOWN; rwm->wParam=VK_ESCAPE; return 0L;
           } /* if */
          break; /* not consumed */
        case MSGF_MESSAGEBOX:
          /* determine help text/current message box and display it */
          if (SwitchMessageHelp())
           {/* message consumed */
            rwm->message=WM_NULL; return 0L;
           } /* if */
          break; /* not consumed */
       } /* switch */
     } /* if */
   } /* if */
  /* not processed => use default */
  return DefHookProc
          (iCode,uP,LADDR(rwm),(FARPROC FAR*)&rfhPrevMsgFilter);
 } /* fhMsgFilter() */


/*********************************************************************
 H p m C h e c k M e s s a g e
======================================================================

This function checks if the message [*rwm] is for the help manager.
If so the message is processed and TRUE is returned.
Otherwise FALSE is returned.

Parameters:
 rwm  is the address of the window message.
 
Return:
 TRUE if the message was processed by the help manager
 and FALSE if not.

*********************************************************************/

BOOL HpmCheckMessage(LPMSG rwm)

 {BOOL b=FALSE;
  if (hwDbxHelp)
   {/* check if help dialog box message */
    b=IsDialogMessage(hwDbxHelp,rwm);
   } /* if */
  return b;
 } /* HpmCheckMessage() */


/*********************************************************************
 H p m S e t M s g B o x E n v
======================================================================

This function sets the environment constants for the message box which is
displayed in near future. The values are used if the user presses the
[SHIFT][F1] keyboard combination to get help about the message.

Parameters:
 iItem    is the ident of the item in the dialog box or 0 if no
          special item is selected.
 iString  is the ident of the string which is displayed in the
          message box. If this value is 0, no help can be chosen about
          the displayed message (for example if memory to small).

Return:
 none

*********************************************************************/

VOID HpmSetMsgBoxEnv(WORD iItem,WORD iString)

 {/* set values to module variables */
  iCurDlgItem=iItem; iCurMsgString=iString;
 } /* HpmSetMsgBoxEnv() */


/*********************************************************************
 H p m S e t D l g B o x E n v
======================================================================

This function sets the identification value of the current used dialog box.
This is the value of the resource file description of the dialog box.
The value is used if the user presses the [SHIFT][F1] keyboard
combination to get help about the current dialog item of a message
string.

Parameters:
 iDlgBox  is the ident of the dialog box or 0 if no special
          dialog box is selected.

Return:
 none

*********************************************************************/

VOID HpmSetDlgBoxEnv(WORD iDlgBox)

 {/* set value to module variable */
  iCurDlgBox=iDlgBox;
 } /* HpmSetDlgBoxEnv() */


/*********************************************************************
 H p m I n i t
======================================================================

This function initializes the help manager. If creates a small child
window at the left bottom corner of the client area of the window
<hwAppl>.

Parameters:
 hwAppl  is the window handle of the application in which client area
         the help manager's command description line is displayed.
 hiAppl  is the module instance handle of the application.

Return:
 TRUE if help manager is initialized FALSE if not.

*********************************************************************/

BOOL HpmInit(HWND hwAppl,HWND hiAppl)
  
 {TEXTMETRIC wtm;
  HDC hdc;

  hwMain=hwAppl; /* store window handle of application's menu */
  hiMain=hiAppl; /* set instance of application */
  /* determine size of standard display characters */
  hdc=CreateIC("Display",NULL,NULL,NULL);
  GetTextMetrics(hdc,&wtm);
  sxChar=wtm.tmAveCharWidth; syChar=wtm.tmHeight;
  DeleteDC(hdc);
  /* create command descriptor line window */
  hwLine=CreateWindow
   ("static","",WS_BORDER|SS_LEFT|WS_CLIPSIBLINGS|WS_CHILD,
    0,0,S_CMD_DESCR*sxChar,syChar+4,hwAppl,NULL,hiAppl,NULL
   );
  if (!hwLine) return FALSE; /* not created => error */
  /* set subclass for text browsing */
  rfwSubEdit=MakeProcInstance((FARPROC)fwSubEdit,hiMain);
  /* set hook function for dialog-box/menu/message-box analysing */
  rfhMsgFilter=MakeProcInstance((FARPROC)fhMsgFilter,hiMain);
  rfhPrevMsgFilter=SetWindowsHook(WH_MSGFILTER,rfhMsgFilter);
  return TRUE;
 } /* HpmInit() */

/* =============================
   End of WINDOWS module HELPMGR
   =============================
*/
