/*********************************************************************

 ----  S H A P E  ---------- MS-Windows Application -----------------

======================================================================

This MS-Windows application is a small program for the demonstration
and study of the Help manager. The program permits the drawing of
simple geometric objects in different colors and shapes.

----------------------------------------------------------------------

Copyright 1989 by
          Marcellus Buchheit, Buchheit software research
          Zaehringerstrasse 47, D-7500 Karlsruhe 1
          Phone (0) 721/37 67 76    (West Germany)

Release 1.10 of 89-Jul-13 --- All rights reserved.

*********************************************************************/

/* define/undefine non-debugging option name */
#undef  NDEBUG

#define NOMINMAX
#include <WINDOWS.H>
#include "HELPMGR.H"
#include "DEFS.H"

/* further C standard headers */
#include <STDLIB.H>
#include <STRING.H>
#include <STDIO.H>

/* window function parameter macros */
#define P2LO LOWORD(ulP2)
#define P2HI HIWORD(ulP2)
#define LADDR(r) (LONG)(LPSTR)(r)

/* geometric objects */
enum
 {OBJ_NONE,OBJ_ELLIPSE,OBJ_RHOMBUS,OBJ_SECTOR
 };

/* color index values */
enum
 {COLOR_RED,COLOR_GREEN,COLOR_BLUE
 };

WORD iObject=OBJ_NONE; /* object to paint */

/* size of object in percent parts of client area */
WORD sxObject=50; /* horizontal */
WORD syObject=50; /* vertical */

/* drawing color of object */
DWORD wrgbObject=RGB(0,0,0); /* default is black */

/* brushing for filling geometric object */
LOGBRUSH wlbrObject={BS_HATCHED,RGB(0,0,0),HS_BDIAGONAL};


/*********************************************************************

 -------------------- Application Variables -------------------------

*********************************************************************/

BYTE *zAppName="SHAPE"; /* application module name */
BYTE *rzAppTitle;       /* pointer to application title name */
BYTE *rzNoMemory;       /* pointer to error string "no memory" */
HANDLE hiMain;          /* handle to application instance */
HANDLE hAccel;          /* handle to standard accelerator table */


/* --------------
   window handles
   --------------
*/

HWND hwMain; /* handle to main window */


/*********************************************************************
 E r r o r N o M e m
======================================================================

The error message "Not enough memory" is displayed in a message box.

Parameters:
 hw  is the window handle of the next active window and NULL if no
     window exists.

Used variables:
 rzAppTitle  is the local handle to the application title name.
 rzNoMemory  is the local handle to the no-memory error string.

Return:
 None

*********************************************************************/

VOID ErrorNoMem(HWND hw)
  
 {HpmSetMsgBoxEnv(0,0); /* no help activation is possible */
  MessageBox(hw,rzNoMemory,rzAppTitle,MB_ICONHAND|MB_OK);
 } /* ErrorNoMem() */


/*********************************************************************
 P r i n t E r r o r
======================================================================

The error string with resource value <iString> is displayed at screen
in a message box. The environment for the help manager is set.

Parameters:
 hw ...... is the window handle of the next active window and
           NULL if no such window exists.
 iString   is the identification of the error string.
 iDlgItem  is the identification of the message box item which causes
           the error.
 
Used variables:
 rzAppTitle  is the local handle to the application title name.

Return:
 None

*********************************************************************/

VOID PrintError(HWND hw,WORD iString,WORD iDlgItem)
  
 {BYTE z[255];

  HpmSetMsgBoxEnv(iDlgItem,iString);
  LoadString(hiMain,iString,z,sizeof(z));
  MessageBox(hw,z,rzAppTitle,MB_ICONASTERISK|MB_OK);
 } /* PrintError() */


/*********************************************************************
 R e a d D l g I t e m W o r d
======================================================================

This function reads a numerical unsigned integer value in range
0..uMax (including) from the "edit" item <iItem> and stores it at
address <ruDest> if the specification was correct. If so, TRUE is
returned.
In case of any error (bad characters specified or overflow), the error
string STE_NUMVAL is printed in a message box, the input focus is set
to <iItem>, all characters of the field are selected and FALSE is
returned.

Parameters:
 hwBox .. is the window handle of the dialog box.
 iItem .. is the item identifier of the edit field.
 ruDest   is the address of a WORD variable where the
          read value is stored.
 uMax ... is the maximum value which is permitted.

Return:
 If the value is specified correctly, it is stored at address [ruDest] and
 TRUE is returned. Otherwise FALSE is returned.

*********************************************************************/

BOOL ReadDlgItemWord(HWND hwBox,WORD iItem,WORD *ruDest,WORD uMax)
  
 {BOOL bTrns;
  WORD u;
  
  u=GetDlgItemInt(hwBox,iItem,&bTrns,FALSE);
  if (bTrns && u<=uMax)
   {/* no error: store value, return */
    *ruDest=u; return TRUE; /* no error */
   } /* if */
  /* invalid specification or overflow: error message, set focus */
  PrintError(NULL,STE_NUMVAL,iItem);
  /* select all characters in item */
  SendDlgItemMessage(hwBox,iItem,EM_SETSEL,0,MAKELONG(0,32767));
  SetFocus(GetDlgItem(hwBox,iItem));
  return FALSE; /* error */
 } /* ReadDlgItemWord() */


/*********************************************************************
 P a i n t D l g I t e m
======================================================================

This function paints the contents of the dialog item <iItem>
of dialog box <hwBox>.

Parameters:
 hwBox  is the window handle of the dialog box.
 iItem  is the item identifier of the field to paint.

Used variables:
 rzAppTitle  is the local handle to the application title name.

Return:
 None

*********************************************************************/

VOID PaintDlgItem(HWND hwBox,HWND iItem)
  
 {HWND hwItem;
  HDC  hdc;
  RECT wrcItem;
  HBRUSH hbrItem;
 
  hwItem=GetDlgItem(hwBox,iItem);
  if (!hwItem) return; /* item not available */
  /* repaint contents of item */
  GetClientRect(hwItem,&wrcItem); hdc=GetDC(hwItem);
  InvalidateRect(hwItem,NULL,TRUE);
  /* specify painting brush */
  switch (iItem)
   {case IDTHATCH:
      hbrItem=CreateHatchBrush(HS_BDIAGONAL,wrgbObject);
      break;
    case IDTSOLID:
      hbrItem=CreateSolidBrush(wrgbObject);
      break;
   } /* switch */
  SelectObject(hdc,hbrItem);
  /* draw full contents of window with specified brush */
  Rectangle(hdc,-1,-1,wrcItem.right+1,wrcItem.bottom+1);
  ReleaseDC(hwItem,hdc); DeleteObject(hbrItem);
 } /* PaintDlgItem() */


/*********************************************************************

 -------------------- Dialog-Box functions --------------------------

*********************************************************************/

/*********************************************************************
 f d S i z e
======================================================================

### Dialog Box function ###

This function processes any messages received
by the DBX_SIZE dialog box.

Parameters:
standard message data (see fwMain)

Return:
standard dialog box function value. DialogBox() returns TRUE.

*********************************************************************/

BOOL FAR PASCAL fdSize(HWND hw,WORD iMsg,WORD uP1,DWORD ulP2)
   
 {WORD sx,sy;
 
  switch (iMsg)
   {case WM_INITDIALOG:
      SetDlgItemInt(hw,IDHORZ,sxObject,FALSE);
      SetDlgItemInt(hw,IDVERT,syObject,FALSE);
      return TRUE; /* no focus set */
    case WM_COMMAND:
      switch (uP1)
       {case IDOK:
          /* read values in edit fields */
          if (!ReadDlgItemWord(hw,IDHORZ,&sx,100)) break;
          if (!ReadDlgItemWord(hw,IDVERT,&sy,100)) break;
          /* store set values */
          sxObject=sx; syObject=sy;
        case IDCANCEL:
          /* close dialog box */
          EndDialog(hw,uP1);
          return TRUE;
       } /* switch */
      break;
   } /* switch */
  return FALSE;
 } /* fdSize() */


/*********************************************************************
 f d C o l o r
======================================================================

### Dialog Box function ###

This function processes any messages received by
the DBX_COLOR dialog box.

Parameters:
standard message data (see fwMain)

Return:
standard dialog box function value. DialogBox() returns TRUE.

*********************************************************************/

BOOL FAR PASCAL fdColor(HWND hw,WORD iMsg,WORD uP1,DWORD ulP2)
   
 {int i;
  static struct
   {int  vColor;
    HWND hw;
   }
  Entry[3];
  BYTE z[5];
  DWORD uL;
  int vColor;
 
  switch (iMsg)
   {case WM_INITDIALOG:
      uL=wrgbObject; /* current set color */
      for (i=COLOR_RED;i<=COLOR_BLUE;i++)
       {/* set scroll range, scroll value and percent values */
        Entry[i].vColor=(100*(BYTE)uL+128)/255;
        uL>>=8; /* move to next color */
        Entry[i].hw=GetDlgItem(hw,IDRED+i);
        sprintf(z,"%d%%",Entry[i].vColor);
        SetDlgItemText(hw,IDVRED+i,z);
        SetScrollRange(Entry[i].hw,SB_CTL,0,100,FALSE);
        SetScrollPos(Entry[i].hw,SB_CTL,Entry[i].vColor,TRUE);
       } /* for */
      return TRUE; /* no focus set */
    case WM_CTLCOLOR:
      switch (GetWindowWord(P2LO,GWW_ID))
       {case IDTRED:
          SetTextColor(uP1,RGB(255,0,0)); break;
        case IDTGREEN:
          SetTextColor(uP1,RGB(0,255,0)); break;
        case IDTBLUE:
          SetTextColor(uP1,RGB(0,0,255)); break;
        default:
          return NULL;
       } /* switch */
      return (BOOL)GetStockObject(NULL_BRUSH);
    case WM_HSCROLL:
      for (i=COLOR_RED;i<=COLOR_BLUE;i++)
       {if (P2HI!=Entry[i].hw) continue;
        vColor=Entry[i].vColor;
        switch (uP1)
         {case SB_TOP:
            vColor=0; break;
          case SB_BOTTOM:
            vColor=100; break;
          case SB_LINEUP:
            if (vColor!=0) vColor--;
            break;
          case SB_LINEDOWN:
            if (vColor<100) vColor++;
            break;
          case SB_PAGEUP:
            if (vColor>10) vColor-=10; else vColor=0;
            break;
          case SB_PAGEDOWN:
            if (vColor<90) vColor+=10; else vColor=100;
            break;
          case SB_THUMBPOSITION:
          case SB_THUMBTRACK:
            vColor=P2LO; break;
         } /* switch */
        if (vColor!=Entry[i].vColor)
         {/* set new values */
          Entry[i].vColor=vColor;
          SetScrollPos(Entry[i].hw,SB_CTL,vColor,TRUE);
          sprintf(z,"%d%%",Entry[i].vColor);
          SetDlgItemText(hw,IDVRED+i,z);
         } /* if */
       } /* for */
      break;
    case WM_COMMAND:
      switch (uP1)
       {case IDOK:
          /* store set color values */
          wrgbObject=RGB
           ((Entry[COLOR_RED].vColor*255+50)/100,
            (Entry[COLOR_GREEN].vColor*255+50)/100,
            (Entry[COLOR_BLUE].vColor*255+50)/100
           );
          wlbrObject.lbColor=wrgbObject;
        case IDCANCEL:
          /* close dialog box */
          EndDialog(hw,uP1); return TRUE;
       } /* switch */
      break;
   } /* switch */
  return FALSE;
 } /* fdColor() */


/*********************************************************************
 f d B r u s h
======================================================================

### Dialog Box function ###

This function processes any messages received by
the DBX_BRUSH dialog box.

Parameters:
standard message data (see fwMain)

Return:
standard dialog box function value. DialogBox() returns TRUE.

*********************************************************************/

BOOL FAR PASCAL fdBrush(HWND hw,WORD iMsg,WORD uP1,DWORD ulP2)
   
 {static WORD iStyle; /* current selected style */
 
  switch (iMsg)
   {case WM_INITDIALOG:
      switch (wlbrObject.lbStyle)
       {case BS_HATCHED:
          iStyle=IDHATCH; break;
        case BS_SOLID:
          iStyle=IDSOLID; break;
        default:
          iStyle=IDHOLLOW; break;
       } /* switch */
      CheckRadioButton(hw,IDHOLLOW,IDSOLID,iStyle);
      return TRUE; /* no focus set */
    case WM_PAINT:
      PaintDlgItem(hw,IDTHATCH); PaintDlgItem(hw,IDTSOLID);
      break;
    case WM_COMMAND:
      switch (uP1)
       {case IDHOLLOW:
        case IDHATCH:
        case IDSOLID:
          if (P2HI==BN_CLICKED)
           {iStyle=uP1; CheckRadioButton(hw,IDHOLLOW,IDSOLID,iStyle);
           } /* if */
          break;
        case IDOK:
          /* store set value */
          switch (iStyle)
           {case IDHOLLOW:
              wlbrObject.lbStyle=BS_HOLLOW; break;
            case IDHATCH:
              wlbrObject.lbStyle=BS_HATCHED; break;
            case IDSOLID:
              wlbrObject.lbStyle=BS_SOLID; break;
           } /* switch */
        case IDCANCEL:
          /* close dialog box */
          EndDialog(hw,uP1); return TRUE;
       } /* switch */
      break;
   } /* switch */
  return FALSE;
 } /* fdBrush() */


/*********************************************************************
 R e a d D i a l o g
======================================================================

This function creates a modeless dialog box, displayes it at screen
and returns if the user closes the box.
The function returns TRUE if the function DialogBox() returns IDOK and
FALSE otherwise. If the dialog box is not created because the memory
is too small, a message box with an error is created and FALSE is
returned.

Parameters:
 fd .. is the dialog box function.
 uBox  is the number of the dialog box (DBX_...-key).

Return:
 TRUE if the dialog box was created and the OK-button was pressed to
 close it or FALSE if the CANCEL-button was pressed or no box was
 created.

*********************************************************************/

BOOL ReadDialog(WORD uBox,FARPROC rfd)

 {FARPROC rfdInst;
  int v;
 
  rfdInst=MakeProcInstance(rfd,hiMain);
  HpmSetDlgBoxEnv(uBox); /* set identification of current box */
  if (rfdInst!=NULL)
   {v=DialogBox(hiMain,MAKEINTRESOURCE(uBox),hwMain,rfdInst);
    FreeProcInstance(rfdInst);
    if (v!=-1)
     {/* box was created: return TRUE if OK-button pressed */
      return v==IDOK;
     } /* if */
   } /* if */
  /* procedure-instance or box not created: print error message */
  ErrorNoMem(hwMain); return FALSE;
 } /* ReadDialog() */


/*********************************************************************
 C m d M a i n 
======================================================================

The command code <iCmd> of the main window is executed. Here only
command codes from the system menu (CMD_...) are executed. Commands
from child windows or similiar sources are not analysed here.

Parameters:
 iCmd  is the command code.

Return:
 TRUE if the command was consumed, FALSE if not.

*********************************************************************/

BOOL CmdMain(int iCmd)

 {switch(iCmd)
   {case CMD_ELLIPSE:
      iObject=OBJ_ELLIPSE; break;
    case CMD_SECTOR:
      iObject=OBJ_SECTOR; break;
    case CMD_RHOMBUS:
      iObject=OBJ_RHOMBUS; break;
    case CMD_SIZE:
      if (!ReadDialog(DBX_SIZE,fdSize)) return TRUE;
      break;
    case CMD_COLOR:
      if (!ReadDialog(DBX_COLOR,fdColor)) return TRUE;
      break;
    case CMD_BRUSH:
      if (!ReadDialog(DBX_BRUSH,fdBrush)) return TRUE;
      break;
    case CMD_CLEAR:
      iObject=OBJ_NONE; break;
    default:
      /* other commands: not consumed */
      return FALSE;
   } /* switch */
  /* redraw object */
  InvalidateRect(hwMain,NULL,TRUE); return TRUE;
 } /* CmdMain() */


/*********************************************************************

 ----------------------- Main Window function -----------------------

*********************************************************************/

LONG FAR PASCAL fwMain(HWND hw,WORD iMsg,WORD uP1,DWORD ulP2)

 {PAINTSTRUCT wps;
  HDC hdc;
  HPEN hpnOld;
  HBRUSH hbrOld,hbrFrame;
  POINT pCenter;
  int sxHalf,syHalf;
  RECT wrc;
  LONG ulRet;
  HRGN hrgnE,hrgnS;
  HCURSOR hcrSave;

  switch (iMsg)
   {case WM_CREATE:
      /* set application window handle */
      hwMain=hw; return 0L;
    case WM_SIZE:
      if (uP1!=SIZEICONIC) HpmSetCmdLine(P2HI);
      break;
    case WM_PAINT:
      hdc=BeginPaint(hw,&wps);
      hpnOld=SelectObject(hdc,CreatePen(0,1,wrgbObject));
      hbrOld=SelectObject(hdc,CreateBrushIndirect(&wlbrObject));
      GetClientRect(hw,&wrc);
      pCenter.x=wrc.right/2; pCenter.y=wrc.bottom/2;
      sxHalf=(pCenter.x*sxObject)/100;
      syHalf=(pCenter.y*syObject)/100;
      switch (iObject)
       {case OBJ_ELLIPSE:
          Ellipse
           (hdc,pCenter.x-sxHalf,pCenter.y-syHalf,
            pCenter.x+sxHalf,pCenter.y+syHalf
           );
          break;
        case OBJ_RHOMBUS:
         {POINT mpRhombus[4];
          mpRhombus[0].x=pCenter.x-sxHalf; mpRhombus[0].y=pCenter.y;
          mpRhombus[1].x=pCenter.x; mpRhombus[1].y=pCenter.y-syHalf;
          mpRhombus[2].x=pCenter.x+sxHalf; mpRhombus[2].y=pCenter.y;
          mpRhombus[3].x=pCenter.x; mpRhombus[3].y=pCenter.y+syHalf;
          Polygon(hdc,mpRhombus,4); break;
         } /* case */
        case OBJ_SECTOR:
          /* longer delay: set hour glass cursor */
          hcrSave=SetCursor(LoadCursor(NULL,IDC_WAIT));
          /* create region of elliptic part (very slow!) */
          hrgnE=CreateEllipticRgn
           (pCenter.x-sxHalf,pCenter.y-syHalf,
            pCenter.x+sxHalf,pCenter.y+3*syHalf
           );
          /* create region of rectangle part */
          hrgnS=CreateRectRgn
           (pCenter.x-sxHalf,pCenter.y-syHalf,
            pCenter.x+sxHalf,pCenter.y+syHalf
           );
          /* combine both regions and draw cotents of regions */
          CombineRgn(hrgnS,hrgnE,hrgnS,RGN_AND); PaintRgn(hdc,hrgnS);
          /* draw frame */
          hbrFrame=CreateSolidBrush(wrgbObject);
          FrameRgn(hdc,hrgnS,hbrFrame,1,1); DeleteObject(hbrFrame);
          SetCursor(hcrSave); /* set old cursor */
          break;
       } /* switch */
      /* select default painting objects, delete created objects */
      DeleteObject(SelectObject(hdc,hpnOld));
      DeleteObject(SelectObject(hdc,hbrOld));
      EndPaint(hw,&wps); return 0L;
    case WM_COMMAND:
      if (uP1!=CMD_EXIT)
       {/* user control command */
        if (CmdMain(uP1))
         {/* command executed, but help manager needs it too */
          return HpmProcessHelpMsg(hw,iMsg,uP1,ulP2);
         } /* if */
        break; /* not consumed */
       } /* if */
      /* continue */
    case WM_CLOSE:
      DestroyWindow(hw); return 0L;
    case WM_DESTROY:
      PostQuitMessage(0); return 0L;
   } /* switch */
  ulRet=HpmProcessHelpMsg(hw,iMsg,uP1,ulP2);
  if (ulRet!=0L) return ulRet;
  return DefWindowProc(hw,iMsg,uP1,ulP2);
 } /* fwMain() */


/*********************************************************************

 ------ Window dependent initialization (one for all instances) -----

*********************************************************************/

/*********************************************************************
 C r e a t e C l a s s
======================================================================

The classes of all application specific windows are registered.

Parameters:
 none
 
Return:
 TRUE   if all classes created.
 FALSE  if any error during creation (memory error).

*********************************************************************/

BOOL CreateClass(VOID)

 {WNDCLASS wwc;

  /* --- Create window class of Application --- */
  wwc.lpszClassName=zAppName; wwc.hInstance=hiMain;
  wwc.lpfnWndProc=fwMain;
  wwc.hCursor=LoadCursor(NULL,IDC_ARROW);
  wwc.hbrBackground=GetStockObject(WHITE_BRUSH);
  wwc.style=CS_HREDRAW|CS_VREDRAW;
  wwc.hIcon=NULL;
  wwc.lpszMenuName=MAKEINTRESOURCE(MNU_MAIN);
  wwc.cbClsExtra=0; wwc.cbWndExtra=0;
  /* register window, return if error */
  if (!RegisterClass(&wwc)) return FALSE;
 } /* CreateClass() */


/*********************************************************************

 ------ Instance dependent initialization (for every instance) ------

*********************************************************************/

/*********************************************************************
 C r e a t e A p p W i n d o w
======================================================================

All globally existing windows are created by this function. 
         
Parameters:
 none
 
Return:
TRUE is returned if all data read, otherwise FALSE
(memory too small).

*********************************************************************/

BOOL CreateAppWindow(VOID)
  
 {/* application's window, return if error */
  if (!CreateWindow
   (zAppName,rzAppTitle,WS_OVERLAPPEDWINDOW|WS_CLIPCHILDREN,
    CW_USEDEFAULT,0,CW_USEDEFAULT,0,NULL,NULL,hiMain,NULL)
   )
    return FALSE;
  return TRUE;
 } /* CreateAppWindow() */


/*********************************************************************
 I n i t I n s t a n c e
======================================================================

### The main init module entry point ###

The complete initialization of a new instance of the application
window. If no previous instance exists, the application window is
initialized and the common data for all instances are created.
Otherwise, common data are copied from the previous instance to the
new instance. All individual data in the instance data segment are
initialized. If memory is too small, a error message box is displayed
and FALSE is returned.

Parameters:
 hiNew      is the handle to the new instance of window.
 hiPrev     is the handle to the first instance of window (or NULL).
 rzCmdLine  points to the command line buffer.
 vCmdShow   is the entry style of window for ShowWindow().

Return:
 TRUE is returned if initialization complete,
 otherwise FALSE (memory too small).

*********************************************************************/

BOOL InitInstance
      (HANDLE hiNew,HANDLE hiPrev,LPSTR rzCmdLine,int vCmdShow)

 {BYTE z[80];
  WORD sz;

  hiMain=hiNew; /* set instance global */
  /* ---- load the two strings for the no-memory error message ---- */
  sz=LoadString(hiMain,STAPPTITLE,z,sizeof(z));
  rzAppTitle=(BYTE*)LocalAlloc(LMEM_FIXED,sz+1); 
  if (!rzAppTitle) return FALSE;
  strcpy(rzAppTitle,z);
  sz=LoadString(hiMain,STNOMEM,z,sizeof(z));
  rzNoMemory=(BYTE*)LocalAlloc(LMEM_FIXED,sz+1);
  if (!rzNoMemory) return FALSE;
  strcpy(rzNoMemory,z);
  /* ----- initialize first/further instance ----- */
  if (!hiPrev)
   {/* first instance: create window class, exit if error */
    if (!CreateClass()) goto MemError;
   } /* if */
  /* create global application window */
  if (!CreateAppWindow()) goto MemError;
  /* load accelerator table */
  hAccel=LoadAccelerators(hiMain,MAKEINTRESOURCE(ACC_MAIN));
  if (hAccel==NULL) return FALSE;
  /* initialize help manager */
  HpmInit(hwMain,hiMain);
  /* show created main window */
  ShowWindow(hwMain,vCmdShow); UpdateWindow(hwMain);
  return TRUE;
  MemError:
  /* error: memory too small */
  ErrorNoMem(NULL);
  return FALSE; /* return with error */
 } /* InitInstance() */


/*********************************************************************

 ------------------------ Main function -----------------------------

*********************************************************************/

WORD PASCAL WinMain
      (HANDLE hiNew,HANDLE hiPrev,LPSTR rzCmdLine,int vCmdShow)

 {MSG wm;

  /* Initialize (window and) instance, return if error */
  if (!InitInstance(hiNew,hiPrev,rzCmdLine,vCmdShow))
    return 255;
  /* --- application execution loop --- */
  while (GetMessage(&wm,NULL,0,0))
   {if (!HpmCheckMessage(&wm))
     {/* message for application */
      if (!TranslateAccelerator(hwMain,hAccel,&wm))
       {TranslateMessage(&wm); DispatchMessage(&wm);
       } /* if */
     } /* if */
   } /* while */
  return wm.wParam;
 } /* WinMain() */

/* ================================
   End of WINDOWS Application SHAPE
   ================================
*/
