/*********************************************************************

 ----  D U M P  ------------ MS-Windows Application -----------------
                             Module DUMP.C

======================================================================

This is the kernel module of the DUMP application.

----------------------------------------------------------------------

Copyright 1989 by
          Marcellus Buchheit, Buchheit software research
          Zaehringerstrasse 47, D-7500 Karlsruhe 1
          Phone (0) 721/37 67 76    (West Germany)

Release 1.00 of 89-Sep-13 --- All rights reserved.

*********************************************************************/

/* read common header files */
#include "DUMP.H"


/*********************************************************************

 ----------------------- Application Variables ----------------------

*********************************************************************/

BYTE zAppName[]="DUMP"; /* application module name */
BYTE *rzAppTitle; /* pointer to application title name */
BYTE *rzNoMemory; /* pointer to error string "no memory" */
HANDLE hiMain;    /* handle to application instance */
HWND hwMain;     /* handle to main window */

FILESPEC FileSpec; /* file specification of dump-file */
int sxChar,syChar; /* size of system characters (fixed font size!) */
int nAddrChars=0;  /* number of used address field characters */
BYTE mcHex[]="0123456789ABCDEF";

PANESPEC PaneSpec;    /* application's pane specification */
PANESPEC *rCPS=NULL; /* pointer to current pane specification */

/* pointer to resources */
HCURSOR hcrWait;   /* hourglass */


/*********************************************************************

 ----------------------- Dialog-Box functions -----------------------

*********************************************************************/

/*********************************************************************
 f d A b o u t
======================================================================

### Dialog Box function ###

This function processes any messages received by the "About"
dialog box.

Parameters:
 standard message data (see fwMain)

Return:
 standard dialog box function value
 DialogBox() returns TRUE.

*********************************************************************/

BOOL FAR PASCAL fdAbout(HWND hw,WORD iMsg,WORD uP1,DWORD ulP2)
   
 {if (iMsg==WM_COMMAND)
   {/* OK pressed */
    EndDialog(hw,TRUE); return TRUE;
   }
  else if (iMsg==WM_INITDIALOG)
   {/* no focus is set */
    return TRUE;
   }
  else return FALSE;
 } /* fdAbout() */


/*********************************************************************
 S e t P a n e s
======================================================================

This function sets the paning for the main window.
It determines and sets values which are used by the pane manager
during resizing, scrolling and drawing. The values are stored into the
structure <PaneSpec.H> for horizontal values and <PaneSpec.V> for
vertical values. If this function is explicitely called by the
application, the function SetPaneWindow() must be called to resize and
redraw the main window contents.

Parameters:
 none

Used Variables:
 PaneSpec

Return:
 none

*********************************************************************/

VOID SetPanes(VOID)

 {LONG s;
  RECT wrc;
  HDC hdc;
  TEXTMETRIC wtm;

  rCPS=&PaneSpec; /* set pointer to pane specification */
  /* ----- create system-dependent constants ----- */
  hdc=CreateIC("Display",NULL,NULL,NULL);
  SelectObject(hdc,GetStockObject(OEM_FIXED_FONT));
  GetTextMetrics(hdc,&wtm);
  sxChar=wtm.tmAveCharWidth; syChar=wtm.tmHeight;
  /* set fixed values of drawing specifications */
  PaneSpec.V.sUnit=wtm.tmHeight+2;
  PaneSpec.V.sMedian=wtm.tmDescent;
  PaneSpec.V.sHead=wtm.tmHeight+2;
  DeleteDC(hdc);
  PaneSpec.V.iMin=0;
  PaneSpec.V.iMax=max(FileSpec.vSize-1,0)/16; /* per line 16 bytes */
  PaneSpec.V.iPos=0; PaneSpec.V.iBase[0]=0;
  /* set horizontal values */
  PaneSpec.H.iMin=0; PaneSpec.H.iMax=22;
  PaneSpec.H.iPos=0; PaneSpec.H.iBase[0]=0;
  PaneSpec.H.sUnit=(5*sxChar)/2;
  PaneSpec.H.sMedian=0;
  /* determine width of header from file size value */
  nAddrChars=4; /* at least 4 digits for address */
  s=FileSpec.vSize;
  while (s>=65536L)
   {/* increase size by one digit */
    nAddrChars++; s/=16;
   } /* while */
  PaneSpec.H.sHead=(nAddrChars+1)*sxChar; /* convert into units */
  CreatePaneWindows(); /* create pane windows for <rCPS> */
  /* set new application window sizes, redraw window contents */
  GetClientRect(hwMain,&wrc); SetPaneWindow(wrc.right,wrc.bottom);
 } /* SetPanes() */


/*********************************************************************
 C m d M a i n 
======================================================================

The command code <iCmd> of the main window is executed. Here only
command codes from the system menu (CMD_...) are executed.
Commands from child windows or similiar sources are not analysed here.

Parameters:
 iCmd  is the command code.

Return:
 TRUE if the command was consumed, FALSE if not.

*********************************************************************/

BOOL CmdMain(int iCmd)

 {HCURSOR hcrSave;
  BYTE z1[80],z2[80];

  switch(iCmd)
   {case CMD_ABOUT:
      /* ----- display about dialog box ----- */
      ReadDialog(DB_ABOUT,(FARPROC)fdAbout); return TRUE;
    case CMD_OPEN:
      /* ----- read name of file  ----- */
      if (!ReadDialog(DB_OPEN_FILE,fdOpenFile))
        return TRUE; /* cancelled */
      hcrSave=SetCursor(hcrWait); /* set hour glass */
      if (rCPS!=NULL)
       {/* ----- close old file, free memory ----- */
        CloseFile(&FileSpec);
        EnableMenuItem
         (GetMenu(hwMain),CMD_SPLIT,MF_BYCOMMAND|MF_GRAYED);
        ClosePanes(); /* pane manager not further valid */
       } /* if */
      /* ----- load specified file ----- */
      strcpy(zCurrentFile,zNewFile); /* set current file */
      FileSpec.hf=hfNewFile; /* set new handle */
      if (LoadFile(&FileSpec))
       {/* file header is loaded: change window text, set file spec */
        LoadString(hiMain,STFILETITLE,z1,sizeof(z1));
        sprintf(z2,z1,zCurrentFile); SetWindowText(hwMain,z2);
        SetPanes(); /* set new size of window-contents, redraw it */
        EnableMenuItem
         (GetMenu(hwMain),CMD_SPLIT,MF_BYCOMMAND|MF_ENABLED);
       }
      else
       {/* cannot load file: close it again */
        CloseFile(&FileSpec);
        SetWindowText(hwMain,rzAppTitle); /* set standard text */
        InvalidateRect(hwMain,NULL,TRUE); /* clear window) */
       } /* if */
      SetCursor(hcrSave); /* set old cursor */
      return TRUE;
    case CMD_SPLIT:
      /* change size of panes */
      ChangePaneSize();
   } /* switch */
  return FALSE; /* not consumed */
 } /* CmdMain() */


/*********************************************************************

 ------------------------- Drawing Functions ------------------------

*********************************************************************/

/* application specific drawing tools */
HPEN hpnDot;
HPEN hpnLine;
HBRUSH hbrBkgr;


/*********************************************************************
 D r a w T r a c k P o s
======================================================================

This function draws the preliminary value of the address which is
determined by moving the holding-down scroll bar thumb. The value is
written into the first line of the horizontal header in pane 0 or
pane 2. If the value is different from the current set position, it is
displayed inverted. If it is the same value it is not changed. The
function draws the value immediately.

Parameters:
 iTrackBase  is the current tracking base value.

Used Globals:
 rCPS

Return:
 none

*********************************************************************/

VOID DrawTrackPos(int iPane,LONG iTrackBase)

 {HDC hdc;
  BOOL bOrg; /* TRUE if original location */
  RECT wrcBkgr;
  BYTE z[10+1];
  int i;

  /* determine DC, determine drawing point */
  hdc=GetDC(hwMain); SelectObject(hdc,GetStockObject(OEM_FIXED_FONT));
  bOrg=iTrackBase==rCPS->V.iBase[iPane>1];
  SelectObject(hdc,GetStockObject(OEM_FIXED_FONT));
  SetTextColor(hdc,GetSysColor(bOrg? COLOR_WINDOWTEXT:COLOR_WINDOW));
  SetBkColor(hdc,GetSysColor(bOrg? COLOR_WINDOW:COLOR_WINDOWTEXT));
  wrcBkgr.left=rCPS->Pane[0].wrc.left;
  wrcBkgr.right=wrcBkgr.left+rCPS->H.sHead-1;
  wrcBkgr.top=rCPS->Pane[iPane].wrc.top+(iPane>1? 0:rCPS->V.sHead);
  wrcBkgr.bottom=wrcBkgr.top+rCPS->V.sUnit-1;
   {/* --- draw hex number of address --- */
    LONG v=iTrackBase;
    z[nAddrChars-1]='0'; /* last digit */
    for (i=nAddrChars-2;i>=0;i--) {z[i]=mcHex[v&0xF]; v>>=4;}
    ExtTextOut
     (hdc,wrcBkgr.left+sxChar/2,wrcBkgr.top,
      ETO_OPAQUE,&wrcBkgr,z,nAddrChars,NULL
     );
   } /* block */
  ReleaseDC(hwMain,hdc);
  if (bOrg)
   {/* redrawing not needed: invalidate rectangle */
    rCPS->wrcTrackRedraw.left=rCPS->wrcTrackRedraw.right;
   }
  else
   {/* tracking field must be redrawn: set rectangle */
    rCPS->wrcTrackRedraw=wrcBkgr;
   } /* if */
 } /* DrawTrackPos() */


/*********************************************************************
 C r e a t e D r a w i n g T o o l s
======================================================================

This function creates GDI tools which are needed for the repainting of
the main window contents by the function DrawPane. The default values
are set.

Parameters:
 hdc  is the device context for the drawing.

Return:
 none

*********************************************************************/

VOID CreateDrawingTools(HDC hdc)

 {/* create needed drawing tools */
  hpnDot=CreatePen(PS_DOT,1,GetSysColor(COLOR_WINDOWTEXT));
  hpnLine=CreatePen(PS_SOLID,1,GetSysColor(COLOR_WINDOWTEXT));
  hbrBkgr=CreateSolidBrush(GetSysColor(COLOR_WINDOW));
  /* set drawing tools */
  SelectObject(hdc,GetStockObject(OEM_FIXED_FONT));
  SetTextColor(hdc,GetSysColor(COLOR_WINDOWTEXT));
  SelectObject(hdc,hpnLine); SelectObject(hdc,hbrBkgr);
 } /* CreateDrawingTools() */


/*********************************************************************
 D e s t r o y D r a w i n g T o o l s
======================================================================

This function destroyes all GDI tools which are created by the
preceded call of CreateDrawingTools().

Parameters:
 none

Return:
 none

*********************************************************************/

VOID DestroyDrawingTools(VOID)

 {DeleteObject(hpnDot); DeleteObject(hpnLine); DeleteObject(hbrBkgr);
 } /* DestroyDrawingTools() */


/*********************************************************************
 S e t P a n e F o c u s
======================================================================

The focus is switched to pane <iPane> or destroyed if <iPane> is -1.
This function should set to caret to the specified pane or destroy the
caret.

Parameters:
 iPane .. is the index of a valid pane which contains the focus. This
          can be a value of 0..3. If the value is -1, the application
          window loses the input focus.
 bChange  is TRUE if the focus is changed between the panes without
          changing the focus of the application and FALSE if the
          application has got or lost the focus.

Used Globals:
 Pane

Return:
 none

*********************************************************************/

VOID SetPaneFocus(int iPane,BOOL bChange)

 {POINT pCaret;

  if (iPane<0)
   {/* hide caret (focus lost), return */
    DestroyCaret(); return;
   } /* if */
  /* calculate caret position in pane (after 1st char. in 1st line) */
  pCaret.x=rCPS->Pane[iPane].wrc.left+sxChar/4;
  if (!(iPane&1)) pCaret.x+=rCPS->H.sHead; /* add vertical header */
  pCaret.y=rCPS->Pane[iPane].wrc.top+syChar;
  if (iPane<2) pCaret.y+=rCPS->V.sHead; /* add horizontal header */
  if (!bChange)
   {/* create caret */
    CreateCaret(rCPS->hwPane,NULL,sxChar,rCPS->V.sMedian);
   } /* if */
  /* set caret position */
  SetCaretPos(pCaret.x,pCaret.y);
  if (!bChange) ShowCaret(rCPS->hwPane); /* display caret */
 } /* SetPaneFocus() */


/*********************************************************************
 D r a w P a n e
======================================================================

This function redraws one of the available panes of the main window.
At most 4 panes can be exist in the following order:

 +-----+-----+-+
 |     |     |^|   pane 0 exists always.
 |  0  |  1  |0|   pane 1 exists only if rCPS->pxMullion is non-0.
 |     |     |v|
 +-----+-----+-+   pane 2 exists only if rCPS->pyTransom is non-0.
 |     |     |^|
 |  2  |  3  |2|
 |     |     |v|   pane 3 exists only if rCPS->pxMullion and
 +-----+-----+-+   rCPS->pyTransom are both non-0.
 |<-1->|<-3->|
 +-----+-----+

Parameters:
 hdc ... is the display context for drawing the client area of the
         main window.
 iPane   is the index of the pane in range 0..3.
 sxPane  is the width of the pane.
 syPane  is the height of the pane.

Used Globals:
 rCPS

Return:
 none

*********************************************************************/

VOID DrawPane(HDC hdc,int iPane,WORD sxPane,WORD syPane)

 {int vV,vH;
  int iV,iH,i,s,s1,iWrt;
  int iVStart,nLines;
  int iHStart,iHEnd,iHBase;
  HANDLE hmgData; /* handle of global memory block */
  BYTE FAR *rd; /* pointer to global memory data */
  RECT wrcClip,wrcLines;
  POINT pBase,pStart,pDraw;
  int pyTerm=0; /* terminator line location */
  LONG vAddr,vA;
  BYTE z[60+1];
  int mv[60+1];
  
  GetClipBox(hdc,&wrcClip);
  /* set pane base values */
  pBase.x=iPane&1? 0:rCPS->H.sHead; pBase.y=iPane>1? 0:rCPS->V.sHead;
  /* nothing to draw => return */
  if (wrcClip.left>pBase.x+23*rCPS->H.sUnit) return;
  iHBase=(int)rCPS->H.iBase[iPane&1]; /* index for first column */
  /* ----- determine values for valid vertical drawing area ----- */
  iVStart=(max(wrcClip.top,pBase.y)-pBase.y)/rCPS->V.sUnit; /* line */
  vAddr=(rCPS->V.iBase[iPane>1]+iVStart); /* address of first line */
  pStart.y=pBase.y+iVStart*rCPS->V.sUnit; /* drawing start point */
  /* ----- determine number of lines to draw ----- */
  nLines=(wrcClip.bottom-pBase.y+rCPS->V.sUnit-1)/
         rCPS->V.sUnit-iVStart;
  if (nLines<=0) nLines=1; /* at least one line */
  if (vAddr+(nLines-1)>rCPS->V.iMax)
     nLines=(int)(rCPS->V.iMax-vAddr)+1;
  if (!(iPane&1) && wrcClip.left<rCPS->H.sHead)
   {/* ===== draw vertical header ===== */
    LONG vA=vAddr;
    if (iPane<2 && wrcClip.top<rCPS->V.sHead)
     {/* draw headline for vertical header */
      int s;
      s=sprintf
       (z,nAddrChars<5? "Addr":nAddrChars<8? "Addr.":"Address");
      ExtTextOut(hdc,((nAddrChars-s+1)*sxChar)/2,0,0,NULL,z,s,NULL);
     } /* if */
    pDraw.y=pStart.y; pDraw.x=sxChar/2;
    for (iV=0;iV<nLines;iV++)
     {/* --- draw hex number of address --- */
      LONG v=vA;
      z[nAddrChars-1]='0'; /* last digit */
      for (i=nAddrChars-2;i>=0;i--) {z[i]=mcHex[v&0xF]; v>>=4;}
      ExtTextOut(hdc,pDraw.x,pDraw.y,0,NULL,z,nAddrChars,NULL);
      pDraw.y+=rCPS->V.sUnit; vA++;
     } /* for */
    /* --- draw vertical header separator line --- */
    pDraw.x=rCPS->H.sHead-1;
    pDraw.y=min(pStart.y+nLines*rCPS->V.sUnit,wrcClip.bottom);
    MoveTo(hdc,pDraw.x,wrcClip.top); LineTo(hdc,pDraw.x,pDraw.y);
   } /* if */
  /* ----- exclude header from line clipping area ----- */
  wrcLines=wrcClip; /* set line drawing rectangle */
  if (wrcLines.left<pBase.x) wrcLines.left=pBase.x;
  /* ===== define space separation for ExtTextOut() ===== */
  for (i=32;i<40;i++) mv[i]=sxChar;
  memcpy(mv+40,mv+32,8*sizeof(int));
  mv[39]=3*(sxChar/2); /* separation between 8 characters */
  if (iPane<2 && wrcClip.top<rCPS->V.sHead)
   {/* ===== draw horizontal headline ===== */
    /* --- change ExtTextOut()-spacing --- */
    for (i=0;i<16;i++) mv[i]=rCPS->H.sUnit;
    mv[15]=rCPS->H.sUnit-(sxChar+2)/4; /* separation */
    memcpy(mv+16,mv+32,16*sizeof(int));
    /* --- create hex code byte for headline --- */
    for (i=0;i<16;i++) z[i]=mcHex[i];
    memcpy(z+16,mcHex,16); /* set 0..9A..F */
    /* draw complete line */
    pDraw.x=pBase.x-iHBase*rCPS->H.sUnit+3*(sxChar+2)/4+1;
    ExtTextOut(hdc,pDraw.x,0,ETO_CLIPPED,&wrcLines,z,32,mv);
   } /* if */
  /* ===== draw contents of lines: hex values characters ===== */
   {int *ri=mv;
    int vd=rCPS->H.sUnit-sxChar;
    /* --- change ExtTextOut()-spacing --- */
    for (i=0;i<16;i++) {*ri++=sxChar; *ri++=vd;}
    *(ri-1)=rCPS->H.sUnit-sxChar/2-(sxChar+2)/4; /* separation */
   } /* block */
  /* determine drawing location */
  pDraw.x=pBase.x-iHBase*rCPS->H.sUnit+(sxChar+2)/4+1;
  pDraw.y=pStart.y; vA=16*vAddr;
  hmgData=NULL; /* no buffer selected */
  for (iV=0;iV<nLines;iV++)
   {/* ----- draw one hex code line ----- */
    int iHex=0;
    int iChar=iHex+32;
    if (!hmgData && vA<FileSpec.vSize)
     {/* new data block must be read from file */
      hmgData=ReadDataBlock(&FileSpec,vA);
      if (!hmgData) break; /* memory error => exit loop */
      rd=GlobalLock(hmgData);
      rd+=(WORD)vA&0xFFF; /* to current read location */
     } /* if */
    for (i=0;i<16;i++)
     {/* draw single hex code byte */
      BYTE d;
      /* end of file reached => break */
      if (vA>=FileSpec.vSize)
       {/* end of file: set spaces instead characters */
        z[iHex++]=' '; z[iHex++]=' '; z[iChar++]=' ';
       }
      else
       {/* set hex- and character-value */
        d=*rd++; vA++;
        z[iHex++]=mcHex[d>>4]; z[iHex++]=mcHex[d&0xF];
        z[iChar++]=d; /* character code */
       } /* if */
     } /* for */
    if (((WORD)vA&0xFFF)==0)
     {/* new data block must be read */
      GlobalUnlock(hmgData); hmgData=NULL;
     } /* if */
    /* draw complete line */
    ExtTextOut(hdc,pDraw.x,pDraw.y,ETO_CLIPPED,&wrcLines,z,iChar,mv);
    pDraw.y+=rCPS->V.sUnit; /* next line */
   } /* for */
  if (hmgData) GlobalUnlock(hmgData);
  if (vA>=FileSpec.vSize)
   {/* terminating separator line must be drawn: set location */
    pyTerm=pDraw.y;
   } /* if */
  /* ===== draw vertical separator lines ===== */
  iHStart=(max(wrcClip.left,pBase.x)-pBase.x)/rCPS->H.sUnit+iHBase;
  iHEnd=(wrcClip.right-pBase.x+rCPS->H.sUnit-1)/rCPS->H.sUnit+iHBase;
  SelectObject(hdc,hpnDot);
  pDraw.y=min(pStart.y+nLines*rCPS->V.sUnit,wrcClip.bottom);
  if (iHStart<=8 && iHEnd>8)
   {pDraw.x=pBase.x+(8-iHBase)*rCPS->H.sUnit;
    MoveTo(hdc,pDraw.x,wrcClip.top); LineTo(hdc,pDraw.x,pDraw.y);
   } /* if */
  if (iHStart<=16 && iHEnd>16)
   {pDraw.x=pBase.x+(16-iHBase)*rCPS->H.sUnit;
    MoveTo(hdc,pDraw.x,wrcClip.top); LineTo(hdc,pDraw.x,pDraw.y);
   } /* if */
  if (iHStart<20 && iHEnd>19)
   {pDraw.x=pBase.x+(19-iHBase)*rCPS->H.sUnit+(5*sxChar+1)/4;
    MoveTo(hdc,pDraw.x,wrcClip.top); LineTo(hdc,pDraw.x,pDraw.y);
   } /* if */
  SelectObject(hdc,hpnLine);
  if (iHStart<=23 && iHEnd>23)
   {pDraw.x=pBase.x+(23-iHBase)*rCPS->H.sUnit;
    MoveTo(hdc,pDraw.x,wrcClip.top); LineTo(hdc,pDraw.x,pDraw.y);
   } /* if */
  pDraw.x=min(pBase.x+(23-iHBase)*rCPS->H.sUnit,wrcClip.right);
  if (iPane<2 && wrcClip.top<rCPS->V.sHead)
   {/* ===== draw horizontal header separator line ===== */
    MoveTo(hdc,wrcClip.left,rCPS->V.sUnit-2);
    LineTo(hdc,pDraw.x,rCPS->V.sUnit-2);
   } /* if */
  if (pyTerm)
   {/* ===== draw terminator line ===== */
    MoveTo(hdc,wrcClip.left,pyTerm);
    LineTo(hdc,pDraw.x+1,pyTerm);
   } /* if */
  /* ===== draw dotted horizontal separator lines ===== */
  iV=8-((int)vAddr&7); if (iV==0) iV=8;
  pStart.y+=iV*rCPS->V.sUnit-1; /* vertical point of first line */
  SelectObject(hdc,hpnDot);
  for (;iV<=nLines;iV+=8) 
   {MoveTo(hdc,wrcClip.left,pStart.y); LineTo(hdc,pDraw.x,pStart.y);
    pStart.y+=8*rCPS->V.sUnit; /* next line position */
   } /* for */
 } /* DrawPane() */


/*********************************************************************

 ----------------------- Main Window function -----------------------

*********************************************************************/

LONG FAR PASCAL fwMain(HWND hw,WORD iMsg,WORD uP1,DWORD ulP2)

 {LONG vRet;

  if (rCPS)
   {/* call pane-manager's message processor */
    vRet=ProcessPaneMsg(hw,iMsg,uP1,ulP2);
    if (vRet) return vRet; /* consumed */
   } /* if */
  switch (iMsg)
   {case WM_CREATE:
      /* set application window handle */
      hwMain=hw; PaneSpec.hwPane=hw;
      return 0L;
    case WM_KEYDOWN:
     {BOOL bCtrl=GetKeyState(VK_CONTROL)>>15;
      BOOL bVert=!bCtrl;
      int  uScrollCmd;
      static BYTE miVScroll[]={0,0,2,2};
      static BYTE miHScroll[]={1,3,1,3};
      switch (uP1)
       {case VK_PRIOR:
          /* page up/left */
          uScrollCmd=SB_PAGEUP; break;
        case VK_NEXT:
          /* page down/right */
          uScrollCmd=SB_PAGEDOWN; break;
        case VK_HOME:
          /* start of line/data */
          uScrollCmd=SB_TOP; bVert=!bVert;
          break;
        case VK_END:
          /* end of line/data */
          uScrollCmd=SB_BOTTOM; bVert=!bVert;
          break;
        case VK_UP:
          /* line up */
          uScrollCmd=SB_LINEUP; bVert=TRUE;
          break;
        case VK_DOWN:
          /* line down */
          uScrollCmd=SB_LINEDOWN; bVert=TRUE;
          break;
        case VK_LEFT:
          /* line left */
          uScrollCmd=SB_LINEUP; bVert=FALSE;
          break;
        case VK_RIGHT:
          /* line right */
          uScrollCmd=SB_LINEDOWN; bVert=FALSE;
          break;
        default:
          goto DEFAULT; /* not consumed */
       } /* switch */
      /* execute scrolling */
      ScrollPane
       (bVert? miVScroll[rCPS->iFocusPane]:
               miHScroll[rCPS->iFocusPane],uScrollCmd,0);
      return 0L; /* consumed */
     } /* case */
    case WM_COMMAND:
      if (uP1!=CMD_EXIT)
       {/* user control command */
        CmdMain(uP1);
        break; /* not consumed */
       } /* if */
      /* continue */
    case WM_CLOSE:
      DestroyWindow(hw); return 0L;
    case WM_DESTROY:
      PostQuitMessage(0); return 0L;
   } /* switch */
  DEFAULT:
  return DefWindowProc(hw,iMsg,uP1,ulP2);
 } /* fwMain() */


/*********************************************************************

 ----- Window Dependent Initialization (One for All Instances) ------

*********************************************************************/

/*********************************************************************
 C r e a t e C l a s s
======================================================================

The classes of all application specified windows are registered.
Following classes exists:

Parameters:
 none
 
Return:
 TRUE   if all classes created.
 FALSE  if any error during creation (memory error).

*********************************************************************/

BOOL CreateClass(VOID)

 {WNDCLASS wwc;

  /* --- create window class of application --- */
  wwc.lpszClassName=zAppName; wwc.hInstance=hiMain;
  wwc.lpfnWndProc=fwMain;
  wwc.hCursor=LoadCursor(NULL,IDC_ARROW);
  wwc.hbrBackground=GetStockObject(WHITE_BRUSH);
  wwc.style=0; /* CS_HREDRAW or CS_VREDRAW not needed! */
  wwc.hIcon=LoadIcon(hiMain,MAKEINTRESOURCE(ICO_MAIN));
  wwc.lpszMenuName=MAKEINTRESOURCE(MNU_MAIN);
  wwc.cbClsExtra=0; wwc.cbWndExtra=0;
  /* register window, return if error */
  if (!RegisterClass(&wwc)) return FALSE;
  return TRUE;
 } /* CreateClass() */


/*********************************************************************

 ------ Instance Dependent Initialization (For Every Instance) ------

*********************************************************************/


/*********************************************************************
 C r e a t e A p p W i n d o w
======================================================================

All global existing windows are created by this function. 
         
Parameters:
 none
 
Return:
TRUE is returned if all data read, otherwise FALSE (memory too small).

*********************************************************************/

BOOL CreateAppWindow(VOID)
  
 {int i;
  BYTE z[50];

  /* create application window */
  if (!CreateWindow
   (zAppName,rzAppTitle,WS_OVERLAPPEDWINDOW|WS_CLIPCHILDREN,
    CW_USEDEFAULT,0,CW_USEDEFAULT,0,NULL,NULL,hiMain,NULL)
   )
    return FALSE;
  return TRUE;
 } /* CreateAppWindow() */

 
/*********************************************************************
 I n i t I n s t a n c e
======================================================================

### The main init module entry point ###

The complete initialization of a new instance of the application
window. If no previous instance exists, the application window is
initialized and the common data for all instances are created.
Otherwise, common data are copied from the previous instance to the
new instance. All individual data in the instance data segment are
initialized. If memory is too small, a error message box is displayed
and FALSE is returned.

Parameters:
 hiNew .... is the handle to the new instance of the application.
 hiPrev ... is the handle to the first instance of the application
            (or NULL).
 rzCmdLine  points to the command line buffer.
 vCmdShow   is the entry style of window for ShowWindow().

Return:
 TRUE is returned if initialization complete,
 otherwise FALSE (memory too small).

*********************************************************************/

BOOL InitInstance
      (HANDLE hiNew,HANDLE hiPrev,LPSTR rzCmdLine,int vCmdShow)

 {hiMain=hiNew; /* set instance global */
  /* ---- load the two strings for the no-memory error message ---- */
   {BYTE z[80];
    WORD sz;
    sz=LoadString(hiMain,STAPPTITLE,z,sizeof(z));
    rzAppTitle=(BYTE*)LocalAlloc(LMEM_FIXED,sz+1); 
    if (!rzAppTitle) return FALSE;
    strcpy(rzAppTitle,z);
    sz=LoadString(hiMain,STNOMEM,z,sizeof(z));
    rzNoMemory=(BYTE*)LocalAlloc(LMEM_FIXED,sz+1);
    if (!rzNoMemory) return FALSE;
    strcpy(rzNoMemory,z);
   } /* block */
  /* ----- initialize first/further instance ----- */
  if (!hiPrev)
   {/* first instance: create window class, exit if error */
    if (!CreateClass()) goto MEM_ERROR;
   } /* if */
  InitPaneManager(hiPrev);
  /* create global application window */
  if (!CreateAppWindow()) goto MEM_ERROR;
  /* show created main window */
  ShowWindow(hwMain,vCmdShow); UpdateWindow(hwMain);
  return TRUE;
  /* ===== error: memory too small ===== */
  MEM_ERROR:
  ErrorNoMem(NULL);
  return FALSE; /* return with error */
 } /* InitInstance() */


/*********************************************************************

 --------------------------- Main function --------------------------

*********************************************************************/

WORD PASCAL WinMain
      (HANDLE hiNew,HANDLE hiPrev,LPSTR rzCmdLine,int vCmdShow)

 {MSG wm;

  /* Initialize (window and) instance, return if error */
  if (!InitInstance(hiNew,hiPrev,rzCmdLine,vCmdShow)) return 255;
  /* --- application execution loop --- */
  while (GetMessage(&wm,NULL,0,0))
   {TranslateMessage(&wm); DispatchMessage(&wm);
   } /* while */
  return wm.wParam;
 } /* WinMain() */


/* ============================================
   Windows application DUMP: end of module DUMP
   ============================================
*/
