/*********************************************************************

 ----  D U M P  ------------ MS-Windows Application -----------------
                             Module FILE.C

======================================================================

This Module contains the file management and utility functions of the
DUMP application.

----------------------------------------------------------------------

Copyright 1989 by
          Marcellus Buchheit, Buchheit software research
          Zaehringerstrasse 47, D-7500 Karlsruhe 1
          Phone (0) 721/37 67 76    (West Germany)

Release 1.00 of 89-Sep-22 --- All rights reserved.

*********************************************************************/

/* read common header files */
#include "DUMP.H"

/* used low-level Windows disk-I/O functions */
int FAR PASCAL _lopen(LPSTR lpPathName,int iReadWrite);
int FAR PASCAL _lclose(int hFile);
LONG FAR PASCAL _llseek(int hFile,LONG lOffset,int iOrigin);
WORD FAR PASCAL _lread(int hFile,VOID FAR* lpBuffer,WORD wBytes);


/*********************************************************************

 -------------------- File-Management-Variables ---------------------

*********************************************************************/

BYTE zCurrentFile[12+1]; /* name of current file (without path) */
BYTE zNewFile[12+1]; /* name of new file (specified by open dialog) */
int hfNewFile;       /* handle to new file, <=0 if file not opened */

/* --- file I/O dialog variables --- */
BYTE zCurrentPath[128];          /* current path specification */
BYTE zDefaultFileSpec[13]="*.*"; /* default file specification */
BYTE zDefaultExtend[5]=".*";   /* default extend string (all files) */


/*********************************************************************
 E r r o r N o M e m
======================================================================

The error message "Not enough memory" is displayed in a message box.

Parameters:
 hw  is the window handle of the next active window and NULL if no
     window exists.

Used variables:
 rzAppTitle  is the local handle to the application title name.
 rzNoMemory  is the local handle to the no-memory error string.

Return:
 None

*********************************************************************/

VOID ErrorNoMem(HWND hw)
  
 {MessageBox(hw,rzNoMemory,rzAppTitle,MB_ICONHAND|MB_OK);
 } /* ErrorNoMem() */


/*********************************************************************
 P r i n t M e s s a g e
======================================================================

The string with resource index <iString> is displayed in a message box
with an asterisk. The string can contain placeholder for a printf()-
call. Each of this placeholder is replaced by the arguments which
follows after <iString> in the variable length argument of
PrintMessage().

Parameters:
 iCtrl  is the index of the control string.
Further parameters can follow and specify the replacing 

Used variables:
 rzAppTitle  is the pointer to the application title name.

Return:
 None

*********************************************************************/

VOID PrintMessage(WORD iString,...)
  
 {BYTE zMsg[256],zString[128];
  va_list rVarArg;
  
  /* load string from resource */
  LoadString(hiMain,iString,zString,sizeof(zString));
  va_start(rVarArg,iString); /* set rVarArg to first option param. */
  vsprintf(zMsg,zString,rVarArg); /* create message string */
  va_end(rVarArg);
  /* display message */
  MessageBox(NULL,zMsg,rzAppTitle,MB_ICONASTERISK|MB_OK);
 } /* PrintMessage() */


/*********************************************************************
 R e a d D i a l o g
======================================================================

This function creates a modeless dialog box, displayes it at screen
and returns if the user closes the box.
The function returns TRUE if the function DialogBox() returns IDOK and
FALSE otherwise. If the dialog box is not created because the memory
is too small, a message box with an error is created and FALSE is
returned.

Parameters:
 uBox  is the number of the dialog box (DBX_...).
 rfd   is the address of the dialog box function.

Return:
 TRUE if the dialog box was created and the OK-button was pressed to
 close it or FALSE if the CANCEL-button was pressed or no box was
 created.

*********************************************************************/

BOOL ReadDialog(WORD uBox,FARPROC rfd)

 {FARPROC rfdInst;
  int v;
 
  rfdInst=MakeProcInstance(rfd,hiMain);
  if (rfdInst!=NULL)
   {v=DialogBox(hiMain,MAKEINTRESOURCE(uBox),hwMain,rfdInst);
    FreeProcInstance(rfdInst);
    if (v!=-1)
     {/* box was created: return TRUE if OK-button pressed */
      return v==IDOK;
     } /* if */
   } /* if */
  /* procedure-instance or box not created: print error message */
  ErrorNoMem(hwMain); return FALSE;
 } /* ReadDialog() */


/*********************************************************************
 L o a d F i l e
======================================================================

This function analyses the length of the opened file <hf> and
allocates the memory for the virtual reading of the file data.

Parameters:
 rSpec  is the address of the file specification of the current file.

Return:
 TRUE is returned if the file was read correctly and the memory was
 allocated or FALSE otherwise.

*********************************************************************/

BOOL LoadFile(FILESPEC *rSpec)

 {LONG vSize; /* size of file */
  HANDLE hmg;

  vSize=_llseek(rSpec->hf,0L,2); /* read last position = length */
  if (vSize>0x4000000L) /* 64 MBytes */
   {/* file too long: error message, return FALSE */
    PrintMessage(ST_TOO_LONG,zCurrentFile);
    return FALSE;
   } /* if */
  rSpec->vSize=vSize; /* set into specification */
  /* calc. size of data block header (one handle per 4 KB file size */
  rSpec->seHead=(WORD)((vSize+4095L)/4096);
  /* allocate memory block (init. with 0 => no data area loaded) */
  hmg=GlobalAlloc
   (GMEM_MOVEABLE|GMEM_ZEROINIT,(LONG)(sizeof(HANDLE)*rSpec->seHead));
  if (!hmg)
   {/* memory too small: error message, return FALSE */
    ErrorNoMem(hwMain); return FALSE;
   } /* if */
  rSpec->hmgHead=hmg; /* set handle */
  return TRUE; /* no error */
 } /* LoadFile() */


/*********************************************************************
 R e a d D a t a B l o c k
======================================================================

This function reads the 4KB data block of address <uaFile> from the
file with specification <rSpec> and returns the handle of this data
block. This block must be locked immediately to avoid that the system
discards the block. The function analyses if the 4KB-block is already
loaded. If so, no new reload operation is started. Otherwise the
contents of the block is read from disk.

Parameters:
 rSpec   is the address of the file specification of the current file.
 uaFile  is the address value of the needed data.

Return:
 The valid handle of the global block is returned or NULL if the data
 are not read. In the last case a message box was displayed which has
 informated the user.

*********************************************************************/

HANDLE ReadDataBlock(FILESPEC *rSpec,DWORD uaFile)

 {WORD iBlock;
  HANDLE FAR *rhBlock;
  HANDLE FAR *rhHead;
  HANDLE FAR *rh;
  HANDLE hmgBlock;
  BYTE FAR *rdBlock;
  int v;

  /* calculate block number from address */
  iBlock=(WORD)(uaFile/4096);
  rhHead=(HANDLE FAR*)GlobalLock(rSpec->hmgHead);
  rh=rhHead+iBlock;
  hmgBlock=*rh;
  if (hmgBlock==NULL || (GlobalFlags(hmgBlock)&GMEM_DISCARDED))
   {/* ----- (re-)load data block from disk ----- */
    if (hmgBlock) GlobalFree(hmgBlock); /* free discarded block */
    hmgBlock=GlobalAlloc
     (GMEM_MOVEABLE|GMEM_DISCARDABLE|GMEM_NOTIFY,4096L);
    if (!hmgBlock)
     {/* memory too small: error message, return FALSE */
      ErrorNoMem(hwMain); goto RETURN;
     } /* if */
    /* set read location to start of 4-KB-block */
    _llseek(rSpec->hf,uaFile & 0xFFFFF000L,0);
    /* read data from file into data block (4 KB or less) */
    rdBlock=(BYTE FAR*)GlobalLock(hmgBlock);
    v=_lread(rSpec->hf,rdBlock,4096);
    GlobalUnlock(hmgBlock);
    if (v==-1)
     {/* error during read */
      PrintMessage(ST_BAD_READ,zCurrentFile); goto RETURN;
     } /* if */
    *rh=hmgBlock; /* set handle of allocated block */
   } /* if */
  RETURN:
  GlobalUnlock(rSpec->hmgHead);
  return hmgBlock;
 } /* ReadDataBlock() */


/*********************************************************************
 C l o s e F i l e
======================================================================

This function closes the opened file with specification <rSpec>. All
allocated memory is freed.

Parameters:
 rSpec  is the address of the file specification.

Return:
 none

*********************************************************************/

VOID CloseFile(FILESPEC *rSpec)

 {int i;

  _lclose(rSpec->hf); /* close file */
  if (rSpec->hmgHead)
   {/* ----- free all allocated data ----- */
    HANDLE FAR *rhmg;
    HANDLE FAR *rhHead;
    rhHead=(HANDLE FAR*)GlobalLock(rSpec->hmgHead);
    rhmg=rhHead;
    for (i=0;i<rSpec->seHead;i++,rhmg++)
     {if (*rhmg!=NULL)
       {/* free allocated data */
        GlobalFree(*rhmg);
       } /* if */
     } /* for */
    /* free data of header */
    GlobalUnlock(rSpec->hmgHead); GlobalFree(rSpec->hmgHead);
    rSpec->hmgHead=NULL; /* no header */
   } /* if */
 } /* CloseFile() */


/*********************************************************************

 -------------------- Open-File dialog box functions ----------------
 
*********************************************************************/

/*********************************************************************
 C h a n g e D e f a u l t E x t e n d
======================================================================

The extension of the file specification <rzFile> is used as new
default file extend if it exists and if it contains no wildcards.
This default extention is stored in <zDefaultExtend>.

Parameters:
 rzFile  is the address of string which contains a valid file
         specification.

Used Variables:
 zDefaultExtend

Return:
 none

*********************************************************************/

VOID ChangeDefaultExtend(PSTR rzFile)

 {PSTR rz;
 
  rz=strchr(rzFile,'.');
  if (rz!=NULL && !strpbrk(rz,"*?")) strcpy(zDefaultExtend,rz);
 } /* ChangeDefaultExtend() */


/*********************************************************************
 A d d D e f a u l t E x t e n d
======================================================================

The extension of the file specification <zFile> is set by
<zDefaultExtend> if <rzFile> contains no extend (no '.' in the
specification).

Parameters:
 rzFile  is the address of string which contains a valid file
         specification.

Used Variables:
 zDefaultExtend

Return:
 none

*********************************************************************/

VOID AddDefaultExtend(PSTR rzFile)

 {PSTR rz;

  /* determine if <rzFile> contains extend */
  rz=rzFile+strlen(rzFile)-1;
  while (*rz!=':' && *rz!='\\' && rz>rzFile && *rz!='.')
   {rz=(PSTR)AnsiPrev(rzFile,rz);
   } /* while */
  if (*rz!='.')
   {/* add extension */
    strcat(rzFile,zDefaultExtend);
   } /* if */
 } /* AddDefaultExtend() */


/*********************************************************************
 C h e c k F i l e E d i t
======================================================================

It is checked if the edit field of the dialog box with windows <hw>
contains characters. If so the IDOK item of the dialog box is enabled.
If the field is empty it is disabled. A disabling/enabling process is
only executed if the state changes. This is controlled by the flag at
address <rbEnabled>.

Parameters:
 hw ....... is the window handle of the dialog box.
 rbEnabled  is the address of the enable flag.

Return:
 none

*********************************************************************/

VOID CheckFileEdit(HWND hw,BOOL *rbEnabled)

 {if ((SendMessage(GetDlgItem(hw,IDFILENAME),
      WM_GETTEXTLENGTH,0,0L)!=0)!=*rbEnabled)
   {/* change enable state */
    *rbEnabled=!*rbEnabled;
    EnableWindow(GetDlgItem(hw,IDOK),*rbEnabled);
   } /* if */
 } /* CheckFileEdit() */


/*********************************************************************
 S e p a r a t e F i l e N a m e
======================================================================

The file name location of file specification <rzFile> is determined
and this address is returned.

Parameters:
 rzFile  is the address of string which contains a valid file
         specification.

Return:
 The address of the file name location in the file specification is
 returned.

*********************************************************************/

PSTR SeparateFileName(PSTR rzFile)

 {PSTR rz;

  /* search from end of string */
  rz=rzFile+strlen(rzFile)-1;
  while (*rz!=':' && *rz!='\\' && rz>rzFile)
         rz=(PSTR)AnsiPrev(rzFile,rz);
  if (rz!=rzFile) rz=(PSTR)AnsiNext(rz);
  return rz; /* address of first character */
 } /* SeparateFileName() */


/*********************************************************************
 f d O p e n F i l e
======================================================================

### Dialog Box function ###

This function processes any messages received by the "OpenFile" dialog
box.

Parameters:
standard message data (see fwMain)

Further input:
The desired directory is set and after return the last displayed
directory is set (also if CANCEL).

Return:
standard dialog box function value
DialogBox() returns the code of the pressed terminate button 
(IDOK or IDCANCEL).

*********************************************************************/

BOOL FAR PASCAL fdOpenFile(HWND hw,WORD iMsg,WORD uP1,DWORD ulP2)
   
 {static BOOL bEnabled;
  char z[128+5];
  char zSlct[24+1];
  PSTR rz;
  WORD u;
 
  switch (iMsg)
   {case WM_INITDIALOG:
      /* set contents of file list box (at current directory) */
      DlgDirList(hw,zDefaultFileSpec,IDFILELIST,IDPATH,0);
      DlgDirList(hw,zDefaultFileSpec,IDDIRLIST,NULL,0xC010);
      SetDlgItemText(hw,IDFILENAME,zDefaultFileSpec);
      /* select file entry complete */
      SendDlgItemMessage(hw,IDFILENAME,EM_SETSEL,0,MAKELONG(0,32767));
      /* enable/disable "open" item depending filename exists */
      bEnabled=TRUE; CheckFileEdit(hw,&bEnabled);
      SetFocus(GetDlgItem(hw,IDFILENAME));
      return FALSE; /* focus is set */
    case WM_COMMAND:
      switch (uP1)
       {/* analyse box item */
        case IDFILENAME:
          if (HIWORD(ulP2)==EN_CHANGE)
           {/* activate/deactive "open" item */
            CheckFileEdit(hw,&bEnabled);
           } /* if */
          return TRUE;
        case IDFILELIST:
        case IDDIRLIST:
          switch (HIWORD(ulP2))
           {case LBN_SELCHANGE:
              /* get selected entry (file, directory, drive) */
              DlgDirSelect(hw,zSlct,uP1);
              if (uP1==IDDIRLIST)
               {/* specify new directory/drive */
                GetDlgItemText(hw,IDFILENAME,z,sizeof(z));
                rz=SeparateFileName(z);
                if (*rz==0||!strpbrk(rz,"*?"))
                 {/* no file select entry: set default */
                  rz=zDefaultFileSpec;
                 } /* if */
                strcat(zSlct,rz); /* add selection mask */
               } /* if */
              SetDlgItemText(hw,IDFILENAME,zSlct);
              /* select file entry complete */
              SendDlgItemMessage
               (hw,IDFILENAME,EM_SETSEL,0,MAKELONG(0,32767));
              /* unselect other table */
              SendDlgItemMessage
               (hw,uP1==IDFILELIST? IDDIRLIST:IDFILELIST,
                LB_SETCURSEL,-1,0L);
              break;
            case LBN_DBLCLK:
              /* first click has transf. selection, now like IDOK */
              uP1=IDOK; goto OPEN_FILE;
           } /* switch */
          return TRUE;
        case IDOK:
          /* return if "open" field invalid */
          if (!bEnabled) return TRUE;
        OPEN_FILE:
          /* get contents of file edit line */
          GetDlgItemText(hw,IDFILENAME,z,128); AnsiUpper(z);
          if (DlgDirList(hw,z,IDDIRLIST,IDPATH,0xC010))
           {/* specification is directory: set new file list */
            AddDefaultExtend(z);
            DlgDirList(hw,z,IDFILELIST,NULL,0);
            SetDlgItemText(hw,IDFILENAME,z);
            /* set new default specification */
            strcpy(zDefaultFileSpec,z);
            return TRUE;
           } /* if */
          /* ----- Single specified file: open it ----- */
          AddDefaultExtend(z); /* add extend if not specified */
          strcpy(zNewFile,SeparateFileName(z)); /* det. file spec. */
          hfNewFile=_lopen(zNewFile,OF_READ);
          if (hfNewFile<0)
           {/* not opened: print error, continue */
            if (strpbrk(zNewFile,"*?"))
             {/* bad directory: generate beep */
              MessageBeep(0);
             }
            else
             {/* file not found: error message */
              PrintMessage(ST_NO_FILE,zNewFile);
             } /* if */
            /* set focus to file line, select full line */
            SetFocus(GetDlgItem(hw,IDFILENAME));
            SendDlgItemMessage
             (hw,IDFILENAME,EM_SETSEL,0,MAKELONG(0,32767));
            /* don't abort box */
            break;
           } /* if */
          /* terminate dialog */
        case IDCANCEL:
          EndDialog(hw,uP1);
          break;
       } /* switch */
      return TRUE;
   } /* switch */
  return FALSE;
 } /* fdOpenFile() */


/* ============================================
   Windows application DUMP: end of module FILE
   ============================================
*/
