/*********************************************************************

 ----  D U M P  ------------ MS-Windows Application -----------------
                             Module PANE.C

======================================================================

This module contains the complete pane management for the DUMP
application.

----------------------------------------------------------------------

Copyright 1989 by
          Marcellus Buchheit, Buchheit software research
          Zaehringerstrasse 47, D-7500 Karlsruhe 1
          Phone (0) 721/37 67 76    (West Germany)

Release 1.00 of 89-Sep-13 --- All rights reserved.

*********************************************************************/

/* read common header files */
#include "DUMP.H"


/*********************************************************************

 -------------------- Pane-Management-Variables ---------------------

*********************************************************************/

BOOL bKeySplit; /* the pane size is changing by using the keyboard */

/* split box window style */
#define SPS_HORZ 0 /* horizontal split box */
#define SPS_VERT 1 /* vertical split box */

/* message for split box */
#define SPM_SPLIT  WM_USER /* split command */

/* dialog element sizes */
int sxMullion; /* width of horizontal split area */
int syTransom; /* height of vertical split area */
int syHScroll; /* height of horizontal scroll bar */
int sxVScroll; /* width of vertical scroll bar */

/* pointer to resources */
HCURSOR hcrSplitH; /* horizontal split for panes */
HCURSOR hcrSplitV; /* vertical split for panes */ 
HCURSOR hcrSplitX; /* full split for panes */


/*********************************************************************
 S e t S c r o l l V a l u e s
======================================================================

This function calculates and sets the values for all visible scroll
bars. The values are determined by the document size, the current
location and size of the panes. It is assumed that the bottom or right
location of the scroll bar sets the last line or column of the
document to the bottom or right position of the pane.

This function is called by the pane manager if any of the panes
changes its size. It must be called by the application if any of
values in the [rCPS]-structure is changed, except the <.iPos>-values.
The parameter <bRedraw> must be TRUE if this function is called after
changing the size of the document by the application, except the panes
in the pane window are resized afterwards.

Parameters:
 bRedraw  is TRUE if the scroll bars must redrawn and FALSE if not.

Used Globals:
 rCPS

Return:
 none

*********************************************************************/

VOID SetScrollValues(BOOL bRedraw)

 {int i;
  int iMin,iMax,iPos,iExt;
  int nUnits;
  DATASPEC *rSpec;
  SCROLL *rScroll;
  LONG iBase,iNewBase;

  for (i=0,rScroll=rCPS->Scroll;i<4;i++,rScroll++)
   {if (!rScroll->bVisible) continue; /* only visible scroll bars */
    rSpec=i&1? &rCPS->H:&rCPS->V; iBase=rSpec->iBase[i>=2];
    /* determine number of vertical/horizontal device units */
    switch (i)
     {case 0:
        nUnits=rCPS->Pane[0].wrc.bottom-
               rCPS->Pane[0].wrc.top-rSpec->sHead;
        break;
      case 1:
        nUnits=rCPS->Pane[0].wrc.right-
               rCPS->Pane[0].wrc.left-rSpec->sHead;
        break;
      case 2:
        nUnits=rCPS->Pane[2].wrc.bottom-rCPS->Pane[2].wrc.top; break;
      case 3:
        nUnits=rCPS->Pane[1].wrc.right-rCPS->Pane[1].wrc.left; break;
     } /* switch */
    /* ignore median, divide by unit's height/width */ 
    nUnits=(nUnits+rSpec->sMedian)/rSpec->sUnit;
    if (nUnits<0) nUnits=0; /* minimum must be 0 */
    /* enlarge extend if needed */
    if (rSpec->uExt==0) rSpec->uExt=1;
    iExt=(int)((rSpec->iMax-rSpec->iMin+32767)/32768);
    if (iExt>rSpec->uExt) rSpec->uExt=iExt;
    /* calculate minimum and maximum bar value (round down and up) */
    iMin=(int)(rSpec->iMin/rSpec->uExt);
    iMax=(int)((rSpec->iMax+rSpec->uExt-nUnits)/rSpec->uExt);
    if (iMax<iMin) iMax=iMin; /* set minimum, invalid. scroll bar */ 
    /* calculate pos. value in bar (round up/down), reduce to range */
    iPos=(int)((iBase+rSpec->uExt/2)/rSpec->uExt);
    if (iPos<iMin) iPos=iMin; else if (iPos>iMax) iPos=iMax;
    if (iBase<(iNewBase=(LONG)iMin*rSpec->uExt)||
        iBase>(iNewBase=(LONG)iMax*rSpec->uExt)
       )
     {/* reduce base to valid area */
      rSpec->iBase[i>=2]=iNewBase;
     } /* if */
    /* ----- set scroll bar values ----- */
    rScroll->iMin=iMin; rScroll->iMax=iMax; rScroll->iPos=iPos;
    rScroll->nUnits=nUnits; /* base elements per height/width */
    SetScrollRange(rScroll->hw,SB_CTL,iMin,iMax,FALSE);
    SetScrollPos(rScroll->hw,SB_CTL,iPos,bRedraw);
   } /* for */
 } /* SetScrollValues() */


/*********************************************************************
 S e t P a n e W i n d o w
======================================================================

This function sets the interior of the pane window. This function is
called if the size of the pane window is changed or the splitting of
the window context is changed.

Parameters:
 sxWindow  is the (new) width of the window.
 syWindow  is the (new) height of the window.
 
Used variables:
 rCPS

Return:
 none

*********************************************************************/

VOID SetPaneWindow(int sxWindow,int syWindow)

 {RECT wrcClient;
  struct
   {POINT p;
    int sx,sy;
   }
  ScrollBar[4];
  int i;
  int nCont;

  if (GetFocus()==rCPS->hwPane && rCPS->Pane[0].wrc.left!=-1)
   {/* turn off current focus */
    SetPaneFocus(-1,FALSE);
   } /* if */
  /* ----- determine new size of pane client rectangles ----- */
  rCPS->Pane[0].wrc.left=0; rCPS->Pane[0].wrc.top=0;
  rCPS->Pane[0].wrc.right=sxWindow-sxVScroll+2;
  rCPS->Pane[0].wrc.bottom=syWindow-syHScroll+2;
  rCPS->Pane[1]=rCPS->Pane[0];
  rCPS->Pane[2]=rCPS->Pane[0]; rCPS->Pane[3]=rCPS->Pane[0];
  rCPS->Pane[0].bValid=TRUE;
  rCPS->Pane[1].bValid=rCPS->Pane[2].bValid=
                       rCPS->Pane[3].bValid=FALSE;
  if (rCPS->pxMullion!=0)
   {/* divide pane 0 and 1 */
    rCPS->Pane[0].wrc.right=rCPS->Pane[0].wrc.left+rCPS->pxMullion;
    rCPS->Pane[1].wrc.left+=rCPS->pxMullion+sxMullion;
    rCPS->Pane[1].bValid=TRUE;
   } /* if */
  if (rCPS->pyTransom!=0)
   {/* divide pane 0 and 2 */
    rCPS->Pane[0].wrc.bottom=rCPS->Pane[0].wrc.top+rCPS->pyTransom;
    rCPS->Pane[2].wrc.top+=rCPS->pyTransom+syTransom;
    rCPS->Pane[2].bValid=TRUE;
    if (rCPS->pxMullion!=0)
     {/* determine part 3 */
      rCPS->Pane[2].wrc.right=rCPS->Pane[0].wrc.right;
      rCPS->Pane[1].wrc.bottom=rCPS->Pane[0].wrc.bottom;
      rCPS->Pane[3].wrc.left=rCPS->Pane[1].wrc.left;
      rCPS->Pane[3].wrc.top=rCPS->Pane[2].wrc.top;
      rCPS->Pane[3].bValid=TRUE;
     } /* if */
   } /* if */
  /* check current selected pane */
  if (!rCPS->Pane[rCPS->iFocusPane].bValid)
   {/* pane with focus doesn't exists: set to pane 0 */
    rCPS->iFocusPane=0;
   } /* if */
  /* ----- determine scroll bar locations, sizes and values ----- */
  /* vertical (top) scroll bar */
  ScrollBar[0].p.x=rCPS->Pane[1].wrc.right;
  ScrollBar[0].p.y=rCPS->Pane[0].wrc.top+(rCPS->pyTransom?
                   -1:syTransom-1);
  ScrollBar[0].sx=sxVScroll-1;
  ScrollBar[0].sy=rCPS->Pane[0].wrc.bottom-rCPS->Pane[0].wrc.top-
                  (rCPS->pyTransom? -2:syTransom-2);
  /* horizontal (left) scroll bar */
  ScrollBar[1].p.x=rCPS->Pane[0].wrc.left+(rCPS->pxMullion?
                   -1:sxMullion-1);
  ScrollBar[1].p.y=rCPS->Pane[2].wrc.bottom;
  ScrollBar[1].sx=rCPS->Pane[0].wrc.right-rCPS->Pane[0].wrc.left-
                  (rCPS->pxMullion? -2:sxMullion-2);
  ScrollBar[1].sy=syHScroll-1;
  rCPS->Scroll[0].bVisible=rCPS->Scroll[1].bVisible=TRUE;
  if (rCPS->pyTransom)
   {/* set vertical bottom scroll bar */
    rCPS->Scroll[2].bVisible=TRUE;
    ScrollBar[2]=ScrollBar[0];
    ScrollBar[2].p.y=rCPS->Pane[2].wrc.top-1;
    ScrollBar[2].sy=rCPS->Pane[2].wrc.bottom-rCPS->Pane[2].wrc.top+2;
   }
  else
   {/* vertical bottom scroll bar not visible */
    rCPS->Scroll[2].bVisible=FALSE;
   } /* if */
  if (rCPS->pxMullion)
   {/* set horizontal right scroll bar */
    rCPS->Scroll[3].bVisible=TRUE;
    ScrollBar[3]=ScrollBar[1];
    ScrollBar[3].p.x=rCPS->Pane[1].wrc.left-1;
    ScrollBar[3].sx=rCPS->Pane[1].wrc.right-rCPS->Pane[1].wrc.left+2;
   }
  else
   {/* horizontal right scroll bar not visible */
    rCPS->Scroll[3].bVisible=FALSE;
   } /* if */
  SetScrollValues(FALSE); /* set scroll bar ranges & positions */
  /* ----- set split fields ----- */
  SetWindowPos
   (rCPS->hwHSplit,NULL,rCPS->Pane[0].wrc.left+rCPS->pxMullion,
    rCPS->pxMullion? rCPS->Pane[0].wrc.top:rCPS->Pane[2].wrc.bottom,
    sxMullion,
    rCPS->pxMullion? syWindow-rCPS->Pane[0].wrc.top+1:syHScroll-1,
    SWP_NOACTIVATE|SWP_NOZORDER|SWP_NOREDRAW
   );
  InvalidateRect(rCPS->hwHSplit,NULL,TRUE);
  if (!IsWindowVisible(rCPS->hwHSplit))
    ShowWindow(rCPS->hwHSplit,SW_SHOWNA);
  SetWindowPos
   (rCPS->hwVSplit,NULL,
    rCPS->pyTransom? rCPS->Pane[0].wrc.left:rCPS->Pane[1].wrc.right,
    rCPS->Pane[0].wrc.top+rCPS->pyTransom,
    rCPS->pyTransom? sxWindow-rCPS->Pane[0].wrc.left+1:sxVScroll-1,
    syTransom,
    SWP_NOACTIVATE|SWP_NOZORDER|SWP_NOREDRAW
   );
  InvalidateRect(rCPS->hwVSplit,NULL,TRUE);
  if (!IsWindowVisible(rCPS->hwVSplit))
     ShowWindow(rCPS->hwVSplit,SW_SHOWNA);
  /* ----- repositionate scroll bars ----- */
  for (i=0;i<4;i++)
   {/* draw scroll bar ScrollBar[i] if exists */
    if (rCPS->Scroll[i].bVisible)
     {/* draw scroll bar */
      SetWindowPos
       (rCPS->Scroll[i].hw,NULL,ScrollBar[i].p.x,ScrollBar[i].p.y,
        ScrollBar[i].sx,ScrollBar[i].sy,
        SWP_NOACTIVATE|SWP_NOZORDER|SWP_NOREDRAW
       );
      InvalidateRect(rCPS->Scroll[i].hw,NULL,TRUE);
      /* show scroll bar if it is not visible */
      if (!IsWindowVisible(rCPS->Scroll[i].hw))
        ShowWindow(rCPS->Scroll[i].hw,SW_SHOWNA);
     }
    else
     {/* scroll bar is not visible: hide it if needed */
      if (IsWindowVisible(rCPS->Scroll[i].hw))
        ShowWindow(rCPS->Scroll[i].hw,SW_HIDE);
     } /* if */
   } /* for */
  /* ----- redraw pane window ----- */
  InvalidateRect(rCPS->hwPane,NULL,TRUE); UpdateWindow(rCPS->hwPane);
  if (GetFocus()==rCPS->hwPane)
   {/* turn on new/current focus */
    SetPaneFocus(rCPS->iFocusPane,FALSE);
   } /* if */
 } /* SetPaneWindow() */


/*********************************************************************
 S c r o l l P a n e
======================================================================

This function analyses the scroll command and scrolls the relevant
panes. The scroll command can be entered via the keyboard or a scroll
bar.

Parameters:
 iScroll  is the index of the scroll bar (0..3) or -1 if the scroll
          command was entered via keyboard. In the last case the
          activate pane and its connected pane is scrolled.
 uCmd ... is the scroll command code. Following values are possible:
            SB_TOP ........... to first line or first column
            SB_BOTTOM ........ to last line or last column
            SB_LINEUP ........ one line up or one column left
            SB_LINEDOWN ...... one line down or one column right
            SB_PAGEUP ........ one page up or one page left
            SB_PAGEDOWN ...... one page down or one page right
            SB_THUMBPOSITION   set to position <iNewPos>
            SB_THUMBTRACK .... drag to position <iNewPos>
 iNewPos  is only active if <uCmd> is SB_THUMBPOSITION or
          SB_THUMBTRACK. It contains the set value of the scroll bar.

Return:
 none

*********************************************************************/

VOID ScrollPane(int iScroll,WORD uCmd,int iNewPos)

 {LONG iBase;
  LONG *riBase;
  int  iSbPos;
  SCROLL *rScroll;
  DATASPEC *rSpec;
  int nUnitsPerPage;

  /* set pointers and index to data structures */
  rScroll=rCPS->Scroll+iScroll; rSpec=iScroll&1? &rCPS->H:&rCPS->V;
  riBase=rSpec->iBase+(iScroll>=2); iBase=*riBase;
  /* reduce units per page if page contains more than one unit */
  nUnitsPerPage=rScroll->nUnits-(rScroll->nUnits>1);
  switch (uCmd)
   {case SB_TOP:
      iBase=rSpec->iMin; break;
    case SB_BOTTOM:
      iBase=max(rSpec->iMin,rSpec->iMax-nUnitsPerPage); break;
    case SB_LINEUP:
      if (iBase>rSpec->iMin) iBase--;
      break;
    case SB_LINEDOWN:
      if (iBase<rSpec->iMax-nUnitsPerPage) iBase++;
      break;
    case SB_PAGEUP:
      iBase=max(iBase-nUnitsPerPage,rSpec->iMin); break;
    case SB_PAGEDOWN:
      if (rSpec->iMax>nUnitsPerPage)
       {iBase=min(iBase+nUnitsPerPage,rSpec->iMax-nUnitsPerPage);
       } /* if */
      break;
    case SB_THUMBPOSITION:
      if (rCPS->wrcTrackRedraw.left<rCPS->wrcTrackRedraw.right)
       {/* erase track redraw position */
        InvalidateRect(rCPS->hwPane,&rCPS->wrcTrackRedraw,TRUE);
        /* set rectangle to "empty" (invalid) */
        rCPS->wrcTrackRedraw.left=rCPS->wrcTrackRedraw.right;
        UpdateWindow(rCPS->hwPane); /* redraw before scrolling */
       } /* if */
    case SB_THUMBTRACK:
      iBase=(LONG)iNewPos*rSpec->uExt; /* multiply pos. with extent */
      if (iBase<rSpec->iMin) iBase=rSpec->iMin;
        else if (iBase>rSpec->iMax) iBase=rSpec->iMax;
      if (uCmd==SB_THUMBTRACK && !(iScroll&1))
        DrawTrackPos(iScroll,iBase);    
   } /* switch */
  if (iBase==*riBase) return; /* no location change */
  /* ----- execute scrolling: update contents of panes ----- */
  SetPaneFocus(-1,FALSE); /* switch off caret */
  if (uCmd!=SB_THUMBTRACK)
   {/* ----- set new scroll bar value ----- */
    RECT wrcScroll;
    iSbPos=(int)((iBase+rSpec->uExt/2)/rSpec->uExt);
    if (iSbPos<rScroll->iMin) iSbPos=rScroll->iMin;
      else if (iSbPos>rScroll->iMax) iSbPos=rScroll->iMax;
    if (iSbPos!=rScroll->iPos)
     {/* set new scroll bar location */
      rScroll->iPos=iSbPos;
      SetScrollPos(rScroll->hw,SB_CTL,iSbPos,TRUE);
     } /* if */
    /* --- determine rectangle to scroll/redraw in pane window --- */
    wrcScroll.left=iScroll==3? rCPS->Pane[1].wrc.left:
      iScroll==1? rCPS->Pane[0].wrc.left+rCPS->H.sHead:
                  rCPS->Pane[0].wrc.left;
    wrcScroll.top=iScroll==2? rCPS->Pane[2].wrc.top:
      iScroll==0? rCPS->Pane[0].wrc.top+rCPS->V.sHead:
                  rCPS->Pane[0].wrc.top;
    wrcScroll.right=iScroll==1? rCPS->Pane[0].wrc.right:
                                rCPS->Pane[1].wrc.right;
    wrcScroll.bottom=iScroll==0? rCPS->Pane[0].wrc.bottom:
                                 rCPS->Pane[2].wrc.bottom;
    if (labs(*riBase-iBase)<(LONG)rScroll->nUnits)
     {/* scroll pane(s) into specified direction, redraw new parts */
      HDC hdc;
      RECT wrcUpd;
      int sxScroll,syScroll;
      int vDelta;
      /* --- calculate scrolling rectangle --- */
      /* determine scrolling offset (negative if scroll down/right */
      vDelta=(int)(*riBase-iBase)*rSpec->sUnit;
      /* --- calculate scrolling range and offsets --- */
      if (iScroll&1)
       {/* horizontal scrolling */
        sxScroll=vDelta; syScroll=0;
        if (vDelta<0)
         {/* move remainder to left */
          wrcScroll.left-=vDelta;
         }
        else
         {/* scroll remainder to right */
          wrcScroll.right-=vDelta;
         } /* if */
       }
      else
       {/* vertical scrolling */
        sxScroll=0; syScroll=vDelta;
        if (vDelta<0)
         {/* move remainder to top */
          wrcScroll.top-=vDelta;
         }
        else
         {/* move remainder to bottom */
          wrcScroll.bottom-=vDelta;
         } /* if */
       } /* if */
      hdc=GetDC(rCPS->hwPane);
      /* execute scrolling, enlarge update rectangle if needed */
      ScrollDC(hdc,sxScroll,syScroll,&wrcScroll,NULL,NULL,&wrcUpd);
      if (sxScroll<0 && wrcUpd.left>wrcUpd.right+sxScroll)
       {wrcUpd.left=wrcUpd.right+sxScroll;
       }
      else if (sxScroll>0 && wrcUpd.right<wrcUpd.left+sxScroll)
       {wrcUpd.right=wrcUpd.left+sxScroll;
       } /* if */
      if (syScroll<0 && wrcUpd.top>wrcUpd.bottom+syScroll)
       {wrcUpd.top=wrcUpd.bottom+syScroll;
       }
      else if (syScroll>0 && wrcUpd.bottom<wrcUpd.top+syScroll)
       {wrcUpd.bottom=wrcUpd.top+syScroll;
       } /* if */
      ReleaseDC(rCPS->hwPane,hdc);
      InvalidateRect(rCPS->hwPane,&wrcUpd,TRUE);
     }
    else
     {/* ----- redraw concerned panes completely new ----- */
      InvalidateRect(rCPS->hwPane,&wrcScroll,TRUE);
     } /* if */
    *riBase=iBase; /* set new location */
    UpdateWindow(rCPS->hwPane); /* draw immediately */
    SetPaneFocus(rCPS->iFocusPane,FALSE); /* turn on active caret */
   } /* if */
 } /* ScrollPane() */


/*********************************************************************

 ---------------------- Pane-Split-Management -----------------------

*********************************************************************/

/*********************************************************************
 G e t C r o s s S p l i t R e c t
======================================================================

This function calculates the rectangle of the intersection of the two
current set split bars. The coordinates of the rectangle are relative
in the window <hw> and stored into <wrcDest>. The function returns
TRUE if the rectangle exists and FALSE if not.

Parameters:
 hw  is the handle of one of split windows.
 
Used variables:
 rCPS

Return:
 TRUE if the intersection rectangle exists, FALSE if not.

*********************************************************************/

BOOL GetCrossSplitRect(HWND hw,RECT *rwrcDest)

 {RECT wrc1,wrc2;
  POINT p1,p2;
  /* no intersection => return with FALSE */
  if (rCPS->pxMullion==0||rCPS->pyTransom==0) return FALSE;
  GetWindowRect(rCPS->hwHSplit,&wrc1);
  GetWindowRect(rCPS->hwVSplit,&wrc2);
  if (!IntersectRect(&wrc1,&wrc1,&wrc2)) return FALSE;
  p1.x=wrc1.left; p1.y=wrc1.top; p2.x=wrc1.right; p2.y=wrc1.bottom;
  ScreenToClient(hw,&p1); ScreenToClient(hw,&p2);
  SetRect(rwrcDest,p1.x,p1.y,p2.x,p2.y); return TRUE;
 } /* GetCrossSplitRect() */


/*********************************************************************
 G e t C e n t e r e d M u l l i o n
======================================================================

This function returns a centered mullion location of the pane window
with dimensions <wrcPane>.

Parameters:
 wrcPane  is client rectangle of the pane window.
  
Used variables:
 rwrc  is a pointer to the client rectangle of the pane window.

Return:
 The vertical mullion value is returned.

*********************************************************************/

int GetCenteredMullion(RECT *rwrc)

 {return ((rwrc->right-sxMullion-sxVScroll-rCPS->H.sHead)/
   (2*rCPS->H.sUnit))*rCPS->H.sUnit+rCPS->H.sHead-rCPS->H.sMedian;
 } /* GetCenteredMullion() */


/*********************************************************************
 G e t C e n t e r e d T r a n s o m
======================================================================

This function returns a centered transom location of the pane window
with dimensions <wrcPane>.

Parameters:
 rwrc  is a pointer to the client rectangle of the pane window.
  
Used variables:
 rCPS

Return:
 The horizontal transom value is returned.

*********************************************************************/

int GetCenteredTransom(RECT *rwrc)

 {return ((rwrc->bottom-syTransom-syHScroll-rCPS->V.sHead)/
    (2*rCPS->V.sUnit))*rCPS->V.sUnit+rCPS->V.sHead-rCPS->V.sMedian;
 } /* GetCenteredTransom() */


/*********************************************************************
 L i m i t S p l i t P o s
======================================================================

This function checks if the point [rp] is within the pane window
client area which is specified in <wrc>. If so the position is not
changed. Otherwise the location is reduced to the limit values.

Parameters:
 rwrc  is a pointer to the client rectangle of the pane window.
 rp .. is a pointer to the position value.
  
Used variables:
 rCPS

Return:
 The horizontal transom value is returned.

*********************************************************************/

VOID LimitSplitPos(RECT *rwrc,POINT *rp)

 {int sMax;
  if (rp->x<0) rp->x=0;
    else if (rp->x>(sMax=rwrc->right-sxVScroll-sxMullion+2))
      rp->x=sMax;
  if (rp->y<0) rp->y=0;
    else if (rp->y>(sMax=rwrc->bottom-syHScroll-syTransom+2))
      rp->y=sMax;
 } /* LimitSplitPos() */


/*********************************************************************

 --------------------- "Split" Window function ----------------------

*********************************************************************/

LONG FAR PASCAL fwSplit(HWND hw,WORD iMsg,WORD uP1,DWORD ulP2)

 {BOOL bVert; /* TRUE if vertical splitting */
  static BOOL bSetSplit; /* TRUE if split setting terminated */
  static POINT pSplit; /* current split point (cross of corner) */
  static int sxMouseOffs; /* x-offset between bar and mouse loc. */
  static int syMouseOffs; /* y-offset between bar and mouse loc. */

  bVert=(BOOL)GetWindowLong(hw,GWL_STYLE)&1;
  switch (iMsg)
   {case WM_PAINT:
     {PAINTSTRUCT wps;
      HDC hdc;
      HBRUSH hbrDraw;
      HPEN hpnDraw;
      RECT wrc;
      hdc=BeginPaint(hw,&wps);
      hpnDraw=CreatePen(PS_SOLID,1,GetSysColor(COLOR_WINDOWFRAME));
      hbrDraw=CreateSolidBrush(GetSysColor(COLOR_WINDOWFRAME));
      SelectObject(hdc,hpnDraw);
      if (GetCrossSplitRect(hw,&wrc))
       {/* exclude cross rectangle from clipping area */
        InflateRect(&wrc,bVert? -1:0,bVert? 0:-1);
        ExcludeClipRect(hdc,wrc.left,wrc.top,wrc.right,wrc.bottom);
       } /* if */
      GetClientRect(hw,&wrc);
      if (bVert)
       {if (rCPS->pyTransom!=0)
         {/* draw rectangle of vertical split bar */
          Rectangle(hdc,-1,0,wrc.right-sxVScroll+2,wrc.bottom);
         } /* if */
        /* draw vertical split box */
        SelectObject(hdc,hbrDraw);
        Rectangle(hdc,wrc.right-sxVScroll+1,0,wrc.right-1,wrc.bottom);
       }
      else
       {if (rCPS->pxMullion!=0)
         {/* draw rectangle of horizontal split bar */
          Rectangle(hdc,0,-1,wrc.right,wrc.bottom-syHScroll+2);
         } /* if */
        /* draw horizontal split box */
        SelectObject(hdc,hbrDraw);
        Rectangle
         (hdc,0,wrc.bottom-syHScroll+1,wrc.right,wrc.bottom-1);
       } /* if */
      EndPaint(hw,&wps);
      DeleteObject(hpnDraw); DeleteObject(hbrDraw); return 0L;
     } /* case */
    case WM_LBUTTONDOWN:
      /* during keyboard-using: convert to "release-button" */
      if (bKeySplit) iMsg=WM_LBUTTONUP;
    case WM_MOUSEMOVE:
    case WM_LBUTTONDBLCLK:
    case WM_LBUTTONUP:
    case SPM_SPLIT:
     {RECT wrcCross; /* intersection rect. of the both split bars */
      POINT pMouse; /* current mouse location */
      BOOL bCross; /* TRUE if mouse location within split bar cross */
      static BOOL bSplitting; /* during splitting operation */
      static BOOL bHSet; /* horizontal setting of mullion */
      static BOOL bVSet; /* vertical setting of transom */
      HWND hwPane=GetParent(hw); /* handle of pane-window */
      if (iMsg==SPM_SPLIT)
       {/* ---- start splitting with keyboard-/mouse-interface ---- */
        RECT wrc;
        POINT pMouseSet;
        GetClientRect(hwPane,&wrc); /* client of parent window */
        bCross=TRUE; bHSet=TRUE; bVSet=TRUE;
        /* set offset between split bars and mouse point */
        sxMouseOffs=sxMullion/2; syMouseOffs=syTransom/2;
        if (rCPS->pxMullion==0 && rCPS->pyTransom==0)
         {/* set start location to centering split */
          pMouse.x=GetCenteredMullion(&wrc);
          pMouse.y=GetCenteredTransom(&wrc);
         }
        else
         {/* start at current setting */
          pMouse.x=0; pMouse.y=0;
          if (rCPS->pxMullion!=0) pMouse.x=rCPS->pxMullion;
          if (rCPS->pyTransom!=0) pMouse.y=rCPS->pyTransom;
         } /* if */
        pMouse.x+=sxMouseOffs; pMouse.y+=syMouseOffs;
        /* set cursor to split location */
        pMouseSet=pMouse;
        ClientToScreen(hwPane,&pMouseSet);
        SetCursor(hcrSplitX); SetCursorPos(pMouseSet.x,pMouseSet.y);
        SetCapture(hw); bSplitting=TRUE; bSetSplit=FALSE;
       }
      else
       {/* ----- determine if mouse is within split bar cross ----- */
        pMouse.x=P2LO; pMouse.y=P2HI;
        bCross=GetCrossSplitRect(hw,&wrcCross)&&
               PtInRect(&wrcCross,pMouse);
        /* get current mouse position as parent window coordinates */
        ClientToScreen(hw,&pMouse); ScreenToClient(hwPane,&pMouse);
        if (iMsg==WM_MOUSEMOVE||iMsg==WM_LBUTTONDOWN)
         {/* set cursor of field */
          SetCursor(bHSet&&bVSet||
                    bCross? hcrSplitX:bVert? hcrSplitV:hcrSplitH);
         } /* if */
       } /* if */
      if (iMsg==WM_LBUTTONDOWN||iMsg==WM_LBUTTONDBLCLK)
       {/* ----- enter splitting mode ----- */
        SetCapture(hw); bSplitting=TRUE; bSetSplit=FALSE;
        /* set splitting mode */
        if (bCross)
         {/* simultanous change of mullion and transom */
          bHSet=TRUE; bVSet=TRUE;
         }
        else
         {/* change only one direction */
          bHSet=!bVert; bVSet=bVert;
         } /* if */
        /* set mouse position deltas */
        sxMouseOffs=pMouse.x-rCPS->pxMullion;
        syMouseOffs=pMouse.y-rCPS->pyTransom;
       } /* if */
      if (bSplitting)
       {/* ====== redraw splitting line if needed ====== */
        HDC hdc;
        RECT wrc;
        BOOL bVChange=TRUE;
        BOOL bHChange=TRUE;
        GetClientRect(hwPane,&wrc); /* client of parent window */
        /* get and analyse mouse location */
        if (iMsg==WM_LBUTTONDBLCLK)
         {/* ----- center split line or destroy it ----- */
          if (bHSet)
           {/* analyse mullion: valid => destroy, center otherwise */
            pMouse.x=rCPS->pxMullion? 0:GetCenteredMullion(&wrc);
           } /* if */
          if (bVSet)
           {/* analyse transom: valid => destroy, center otherwise */
            pMouse.y=rCPS->pyTransom? 0:GetCenteredTransom(&wrc);
           } /* if */
          bSetSplit=TRUE; /* end of repositioning */
         }
        else
         {/* determine new mouse location */
          pMouse.x-=sxMouseOffs; pMouse.y-=syMouseOffs;
         } /* if */
        LimitSplitPos(&wrc,&pMouse);
        if (iMsg==WM_LBUTTONUP) bSetSplit=TRUE; /* end of repos. */
        hdc=GetDC(hwPane);
        SelectObject(hdc,GetStockObject(GRAY_BRUSH));
        /* repaint mullion or transom line */
        if (iMsg!=WM_LBUTTONDOWN && iMsg!=WM_LBUTTONDBLCLK &&
            iMsg!=SPM_SPLIT)
         {/* remove old mullion or transom line if needed */
          bHChange=bHSet&&pMouse.x!=pSplit.x;
          bVChange=bVSet&&pMouse.y!=pSplit.y;
          if (bHSet && (bHChange||bSetSplit))
           {/* remove old mullion line marker */
            PatBlt(hdc,pSplit.x,0,sxMullion,wrc.bottom,PATINVERT);
           } /* if */
          if (bVSet && (bVChange||bSetSplit)) 
           {/* remove old transom line marker */
            PatBlt(hdc,0,pSplit.y,wrc.right,syTransom,PATINVERT);
           } /* if */
         } /* if */
        if (bSetSplit)
         {/* ====== terminate splitting mode ====== */
          if (bHSet&&pMouse.x!=rCPS->pxMullion||
              bVSet&&pMouse.y!=rCPS->pyTransom)
           {/* reorder interior of main application window */
            if (bHSet)
             {/* set new mullion: check width of panes */
              if (pMouse.x<rCPS->H.sHead+
                  rCPS->H.sUnit-rCPS->H.sMedian)
               {/* left pane too small: destroy mullion, use right */
                rCPS->H.iBase[0]=rCPS->H.iBase[1];
                pMouse.x=0;
               }
              else if 
               (pMouse.x>=
                wrc.right-sxVScroll-rCPS->H.sUnit+rCPS->H.sMedian)
               {/* right pane too small: destroy mullion, use left */
                pMouse.x=0;
               }
              else if (rCPS->pxMullion==0)
               {/* new pane created: set index of base unit in pane */
                rCPS->H.iBase[1]=rCPS->H.iBase[0]+(pMouse.x-
                      rCPS->H.sHead+rCPS->H.sMedian)/rCPS->H.sUnit;
               } /* if */
              rCPS->pxMullion=pMouse.x;
              if (rCPS->pxMullion==0 && rCPS->iFocusPane&1)
               {/* focus-pane is destroyed: reduce to 0 or 2 */
                rCPS->iFocusPane&=2;
               } /* if */
             } /* if */
            if (bVSet)
             {/* set new transom: check height of panes */
              if (pMouse.y<rCPS->V.sHead+
                  rCPS->V.sUnit-rCPS->V.sMedian)
               {/* top pane too small: destroy transom, use bottom */
                rCPS->V.iBase[0]=rCPS->V.iBase[1];
                pMouse.y=0;
               }
              else if (pMouse.y>=wrc.bottom-syHScroll-rCPS->V.sUnit+
                       rCPS->V.sMedian)
               {/* bottom pane too small: destroy transom, use left */
                pMouse.y=0;
               }
              else if (rCPS->pyTransom==0)
               {/* new pane: determine index of base unit in pane */
                rCPS->V.iBase[1]=rCPS->V.iBase[0]+(pMouse.y-
                      rCPS->V.sHead+rCPS->V.sMedian)/rCPS->V.sUnit;
               } /* if */
              rCPS->pyTransom=pMouse.y;
              if (rCPS->pyTransom==0 && rCPS->iFocusPane>=21)
               {/* focus-pane is destroyed: reduce to 0 or 1 */
                rCPS->iFocusPane-=2;
               } /* if */
             } /* if */
            SetPaneWindow(wrc.right,wrc.bottom);
           } /* if */
          bHSet=FALSE; bVSet=FALSE; bVChange=FALSE; bHChange=FALSE;
          ReleaseCapture(); bSplitting=FALSE; bKeySplit=FALSE;
         } /* if */
        if (bHSet && bHChange)
         {/* draw new mullion */
          pSplit.x=pMouse.x; /* set new point */
          PatBlt(hdc,pSplit.x,0,sxMullion,wrc.bottom,PATINVERT);
         } /* if */
        if (bVSet && bVChange)
         {/* draw transom selection rectangle */
          pSplit.y=pMouse.y; /* set new point */
          PatBlt(hdc,0,pSplit.y,wrc.right,syTransom,PATINVERT);
         } /* if */
        ReleaseDC(hwPane,hdc);
       } /* if */
      return 0L;
     } /* case */
    case WM_KEYDOWN:
     {RECT wrc;
      HWND hwPane=GetParent(hw);
      POINT pMouse,pMouseSet;
      BOOL bCtrl=GetKeyState(VK_CONTROL)>>15;
      /* abort if keyboard interface not active */
      if (!bKeySplit) return 0L;
      GetClientRect(hwPane,&wrc);
      /* ----- analyse pressed key ----- */
      pMouse=pSplit; /* get last location */
      switch (uP1)
       {case VK_LEFT:
          if (bCtrl) pMouse.x-=1;
          else
           {pMouse.x=
              max((pMouse.x-rCPS->H.sHead+rCPS->H.sMedian)/
                  rCPS->H.sUnit-1,0)
              *rCPS->H.sUnit+rCPS->H.sHead-rCPS->H.sMedian;
           } /* if */
          break;
        case VK_RIGHT:
          if (bCtrl) pMouse.x+=1;
          else
           {pMouse.x=((pMouse.x-rCPS->H.sHead+rCPS->H.sMedian)/
                      rCPS->H.sUnit+1)
                     *rCPS->H.sUnit+rCPS->H.sHead-rCPS->H.sMedian;
           } /* if */
          break;
        case VK_UP:
          if (bCtrl) pMouse.y-=1;
          else
           {pMouse.y=
              max((pMouse.y-rCPS->V.sHead+rCPS->V.sMedian)/
                  rCPS->V.sUnit-1,0)
              *rCPS->V.sUnit+rCPS->V.sHead-rCPS->V.sMedian;
           } /* if */
          break;
        case VK_DOWN:
          if (bCtrl) pMouse.y+=1;
          else
           {pMouse.y=((pMouse.y-rCPS->V.sHead+rCPS->V.sMedian)/
                      rCPS->V.sUnit+1)
                     *rCPS->V.sUnit+rCPS->V.sHead-rCPS->V.sMedian;
           } /* if */
          break;
        case VK_HOME:
          pMouse.x=rCPS->H.sHead; pMouse.y=rCPS->V.sHead; /* origin */
          break;
        case VK_END:
          pMouse.x=32767; pMouse.y=32767; /* end location */
          break;
        case VK_PRIOR:
          /* set left column or top line */
          if (bCtrl) pMouse.x=rCPS->H.sHead; else
            pMouse.y=rCPS->V.sHead;
        case VK_NEXT:
          /* set right column or bottom line */
          if (bCtrl) pMouse.x=32767; else pMouse.y=32767;
          break;
        case VK_ESCAPE:
          /* ignore new setting (set cross bars to old values */
          bSetSplit=TRUE;
          pMouse.x=rCPS->pxMullion; pMouse.y=rCPS->pyTransom;
          break;
        case VK_RETURN:
          /* use new setting */
          bSetSplit=TRUE; break;
       } /* switch */
      LimitSplitPos(&wrc,&pMouse);
      /* set new mouse position */
      pMouseSet.x=pMouse.x+sxMouseOffs;
      pMouseSet.y=pMouse.y+syMouseOffs;
      ClientToScreen(hwPane,&pMouseSet);
      /* set new position (and send WM_MOUSEMOVE command) */
      SetCursorPos(pMouseSet.x,pMouseSet.y);
     } /* case */
   } /* switch */
  return DefWindowProc(hw,iMsg,uP1,ulP2);
 } /* fwSplit() */


/*********************************************************************
C h a n g e P a n e S i z e
======================================================================

This function change the size of the panes as a result of calling the
"split" command in the application menu. The user interface is full
CUA-compatible. The size of the panes can be changed by keyboard or
mouse.

Parameters:
 none
 
Return:
 none

*********************************************************************/

VOID ChangePaneSize(VOID)

 {SetPaneFocus(-1,FALSE); /* switch off caret */
  SetFocus(rCPS->hwVSplit); /* change focus */
  bKeySplit=TRUE; /* activate keyboard input */
  /* send to split window for setting default-panes (middled) */
  SendMessage(rCPS->hwVSplit,SPM_SPLIT,0,0L); 
 } /* ChangePaneSize() */


/*********************************************************************
I n i t P a n e M a n a g e r
======================================================================

This function initializes the complete pane manager.

Parameters:
 hiPrev  is the handle of the previous instance of NULL.
 
Return:
 TRUE if pane manager completely created otherwise FALSE.

*********************************************************************/

BOOL InitPaneManager(HANDLE hiPrev)

 {WNDCLASS wwc;
 
  if (!hiPrev)
   {/* --- create window class of split control area --- */
    wwc.lpszClassName="Split"; wwc.hInstance=hiMain;
    wwc.lpfnWndProc=fwSplit;
    wwc.hCursor=NULL;
    wwc.hbrBackground=COLOR_WINDOW+1;
    wwc.style=CS_DBLCLKS; wwc.hIcon=NULL; wwc.lpszMenuName=NULL;
    wwc.cbClsExtra=0; wwc.cbWndExtra=0;
    /* register window, return if error */
    if (!RegisterClass(&wwc)) return FALSE;
   } /* if */
  /* determine system-specific constants */
  sxMullion=(GetSystemMetrics(SM_CXHTHUMB)+2)/4;
  syTransom=(GetSystemMetrics(SM_CYVTHUMB)+2)/4;
  syHScroll=GetSystemMetrics(SM_CYHSCROLL);
  sxVScroll=GetSystemMetrics(SM_CXVSCROLL);
  hcrSplitH=LoadCursor(hiMain,MAKEINTRESOURCE(CRS_SPLITH));
  hcrSplitV=LoadCursor(hiMain,MAKEINTRESOURCE(CRS_SPLITV));
  hcrSplitX=LoadCursor(hiMain,MAKEINTRESOURCE(CRS_SPLITX));
  return TRUE;
 } /* InitPaneManager() */


/*********************************************************************
 C r e a t e P a n e W i n d o w s
======================================================================

This function creates the pane child windows for the pane
specification <rCPS>.

Parameters:
 none

Used Globals:
 rCPS

Return:
 TRUE if the pane windows are crated, otherwise FALSE.

*********************************************************************/

BOOL CreatePaneWindows(VOID)

 {int i;

  /* reset paning, invalidate pane sizes */
  rCPS->pxMullion=0; rCPS->pyTransom=0; rCPS->Pane[0].wrc.left!=-1;
  rCPS->iFocusPane=0;
  /* create window of horizontal split field */
  rCPS->hwHSplit=CreateWindow
   ("Split","",WS_CHILD|SPS_HORZ,0,0,0,0,
    rCPS->hwPane,IDHSPLIT,hiMain,NULL);
  if (!rCPS->hwHSplit) return FALSE;
  /* create window of vertical split field */
  rCPS->hwVSplit=CreateWindow
   ("Split","",WS_CHILD|SPS_VERT,0,0,0,0,
    rCPS->hwPane,IDVSPLIT,hiMain,NULL);
  if (!rCPS->hwVSplit) return FALSE;
  /* create scroll bar childs for pane window */
  for (i=0;i<4;i++)
   {rCPS->Scroll[i].hw=CreateWindow
     ("ScrollBar","",WS_CHILD|(i&1? SBS_HORZ:SBS_VERT),
      0,0,0,0,rCPS->hwPane,IDPSCROLL+i,hiMain,NULL);
    if (!rCPS->Scroll[i].hw) return FALSE;
   } /* for */
  return TRUE;
 } /* CreatePaneWindows() */


/*********************************************************************
C l o s e P a n e s
======================================================================

This function closes the panes of the pane specification <rCPS> and
destroys all child windows of the pane window. The pane manager is no
longer valid for this window and <rCPS> is set to NULL.

Parameters:
 none
 
Used Globals:
 rCPS

Return:
 none

*********************************************************************/

VOID ClosePanes(VOID)

 {int i;
  if (GetFocus()==rCPS->hwPane && rCPS->Pane[0].wrc.left!=-1)
   {/* turn off pane focus */
    SetPaneFocus(-1,FALSE);
   } /* if */
  DestroyWindow(rCPS->hwHSplit); DestroyWindow(rCPS->hwVSplit);
  for (i=0;i<4;i++)
   {/* destroy scroll bar windows */
    DestroyWindow(rCPS->Scroll[i].hw);
   } /* for */
  rCPS=NULL;
 } /* ClosePanes() */


/*********************************************************************
 P r o c e s s P a n e M s g
======================================================================

This function is called if the pane window has received a message. All
usable messages are analysed if processed if possible. The function
returns 1L if the message was processed completely and 0L if not.
This function may be called only if <rCPS> is valid for <hw>.

Parameters:
 see standard window function

Return:
 none

*********************************************************************/

LONG ProcessPaneMsg(HWND hw,WORD iMsg,WORD uP1,DWORD ulP2)

 {int iPane;
  PANE *rPane;
  RECT wrc;
  short iSavedDC;

  switch (iMsg)
   {case WM_SIZE:
      /* resize interior of pane window */
      SetPaneWindow(P2LO,P2HI);
      return 0L;
    case WM_PAINT:
     {HDC hdc;
      PAINTSTRUCT wps;
      int vRegion;
      SetPaneFocus(-1,FALSE);
      hdc=BeginPaint(hw,&wps);
      CreateDrawingTools(hdc);
      for (iPane=0,rPane=rCPS->Pane;iPane<4;iPane++,rPane++)
       {/* determine if <iPane> is valid */
        if (!rPane->bValid) continue; /* invalid pane */
        /* save display context, reduce clipping rectangle to pane */
        iSavedDC=SaveDC(hdc);
        vRegion=IntersectClipRect
         (hdc,rPane->wrc.left,rPane->wrc.top,
          rPane->wrc.right,rPane->wrc.bottom
         );
        if (vRegion!=NULLREGION)
         {/* set drawing viewport to pane */
          SetViewportOrg
           (hdc,rCPS->Pane[iPane].wrc.left,rCPS->Pane[iPane].wrc.top);
          /* call application specific drawing function */
          DrawPane
           (hdc,iPane,rPane->wrc.right-rPane->wrc.left,
            rPane->wrc.bottom-rPane->wrc.top
           );
         } /* if */
        /* restore prev. device context (with full clipping range) */
        RestoreDC(hdc,iSavedDC);
       } /* for */
      EndPaint(hw,&wps); DestroyDrawingTools();
      SetPaneFocus(rCPS->iFocusPane,FALSE);
      return 1L;
     } /* case */
    case WM_SETFOCUS:
      /* window gets the focus: set caret to active pane */
      SetPaneFocus(rCPS->iFocusPane,FALSE);
      break;
    case WM_KILLFOCUS:
      /* window loses the focus: hide caret */
       if (rCPS->Pane[0].wrc.left!=-1) SetPaneFocus(-1,FALSE);
      break;
    case WM_VSCROLL:
    case WM_HSCROLL:
      /* execute scrolling of panes */
      ScrollPane(GetWindowWord(P2HI,GWW_ID)-IDPSCROLL,uP1,P2LO);
      return 1L; /* consumed */
    case WM_KEYDOWN:
      /* ------ analyse if pane control character ------ */
      if (uP1==VK_F6)
       {/* ---- switch from pane to pane ---- */
        static BYTE miNextPane[]={1,3,0,2};
        static BYTE miPrevPane[]={2,0,3,1};
        int iNewPane=rCPS->iFocusPane;
        BOOL bPrev=GetKeyState(VK_SHIFT)>>15;
        do
         {iNewPane=bPrev? miPrevPane[iNewPane]:miNextPane[iNewPane];
          iNewPane&=3; /* reduce to 0..3 */
          if (rCPS->Pane[iNewPane].bValid) break;
         }
        while (iNewPane!=rCPS->iFocusPane);
        if (iNewPane!=rCPS->iFocusPane)
         {/* set new focus pane */
          rCPS->iFocusPane=iNewPane;
          SetPaneFocus(rCPS->iFocusPane,TRUE);
         } /* if */
        return 1L; /* consumed */
       } /* if */
      return 0L; /* not consumed */
    case WM_LBUTTONDOWN:
      /* set focus to clicked pane */
      if (GetFocus()!=rCPS->hwPane)
       {/* set focus to application, no caret moving */
        SetFocus(rCPS->hwPane); return 1L; /* consumed */
       }
      else
       {/* determine new pane */
        int i;
        for (i=0;i<4;i++)
         {if (rCPS->Pane[i].bValid &&
              PtInRect(&rCPS->Pane[i].wrc,MAKEPOINT(ulP2)))
           {/* set new pane */
            if (i!=rCPS->iFocusPane)
             {/* set <i> as new focus */
              rCPS->iFocusPane=i; SetPaneFocus(rCPS->iFocusPane,TRUE);
             } /* if */
            break;
           } /* if */
         } /* for */
       } /* if */
      break; /* not completely consumed */
   } /* switch */
  return 0L; /* not consumed */
 } /* ProcessPaneMsg() */


/* ============================================
   Windows application DUMP: end of module PANE
   ============================================
*/
