//     (C) Copyright Microsoft Corp. 1991.  All rights reserved.
//
//     You have a royalty-free right to use, modify, reproduce and 
//     distribute the Sample Files (and/or any modified version) in 
//     any way you find useful, provided that you agree that 
//     Microsoft has no warranty obligations or liability for any 
//     Sample Application Files which are modified. 

/****************************************************************************

    MODULE    : SNDPLY.C

    PURPOSE   : DLL that exports functions helpfull in playing sounds
		using MCI.

    FUNCTIONS : LibMain
		WEP
		OpenWaveDevice
		PlayAndCloseWave
		CloseWaveDevice
		OpenPlayCloseWave
		SndPlaySnd
		ErrorProc

    COMMENTS  : This DLL implements playing wavefiles through the MCI command
		interface.

    HISTORY   :

****************************************************************************/

#include <windows.h>
#include <stdio.h>
#include "plysnd.h"
#include <mmsystem.h>


HANDLE hInst;

// forward declarations

WORD FAR PASCAL OpenWaveDevice(HWND);
void FAR PASCAL PlayAndCloseWave(HWND,WORD,LPSTR);
BOOL FAR PASCAL CloseWaveDevice(HWND,WORD);
void FAR PASCAL OpenPlayCloseWave(HWND,LPSTR);
BOOL FAR PASCAL SndPlaySnd(HWND,LPSTR);
void FAR PASCAL ErrorProc(DWORD);


/****************************************************************************

    FUNCTION  :  LibMain(HANDLE,WORD,WORD,LPSTR)

    PURPOSE   : Initializes the heap of the DLL.

    COMMENTS  : This DLL implements playing wavefiles through the MCI command
		interface.

    HISTORY   :

****************************************************************************/

BOOL FAR PASCAL LibMain(hInstance,wDataSeg,cbHeap,lpszCmdLine)

HANDLE hInstance;
WORD   wDataSeg;
WORD   cbHeap;
LPSTR	lpszCmdLine;

{
     if (cbHeap == 0)
	return 0;

     hInst=hInstance;

     LocalInit(wDataSeg, 0 , (WORD)cbHeap);
}


/****************************************************************************

    FUNCTION  :  WEP (int)

    PURPOSE   : Windows Exit Procedure.  This function is called when the
		DLL exits.

    COMMENTS  :  This function must be included in all DLL's.  In low memory
		 situations the DLL could crash in the WEP so it is necessary
		 to place the WEP in a fixed (non-discardable) segment.

    HISTORY   :

****************************************************************************/



VOID FAR PASCAL WEP (bSystemExit)
int  bSystemExit;
{
    return;
}


/****************************************************************************

    FUNCTION  :  OpenWaveDevice(HWND)

    PURPOSE   :  This function opens the device waveaudio and leaves it open.
		 The device could also be opened by assigning the string
		 "waveaudio" as follows.

		 mciopen.lpstrDeviceType="waveaudio"

		 Using this format you would only have to specify

		 MCI_OPEN_TYPE for dwflags.

    COMMENTS  :  Opening the driver once and leaving it open saves time.
		 You can then use this driver ID in the following calls
		 to open elements (wavefiles).	This saves time since if
		 you open an element without specifying an open device ID
		 mci must open the driver and close the driver each time.
		 When playing multiple elements (wavefiles) this excess time
		 can be costly.  Therefore to conserve time you should open
		 the waveaudio device, play the waves, close the waveaudio
		 device.


    HISTORY   :

****************************************************************************/


WORD FAR PASCAL OpenWaveDevice(hWnd)

HWND hWnd;
{

     MCI_OPEN_PARMS mciopen;
     DWORD dwRes;
     WORD wGlobalDeviceID;
     DWORD dwFlags;


     mciopen.wDeviceID=0;
     mciopen.lpstrDeviceType=(LPSTR)MCI_DEVTYPE_WAVEFORM_AUDIO;


     // when specifying the device ID you must also include the
     //MCI_OPEN_TYPE flag.

     MessageBox(hWnd,"About to open the Device","YES",MB_OK);

     dwFlags= MCI_OPEN_TYPE_ID | MCI_OPEN_TYPE;

     dwRes=mciSendCommand(0,MCI_OPEN,dwFlags,(DWORD)(LPSTR)&mciopen);

     if (dwRes)
	{
	ErrorProc(dwRes);
	return(FALSE);
	}

     wGlobalDeviceID=mciopen.wDeviceID;

     return(wGlobalDeviceID);
     }


/****************************************************************************

    FUNCTION  :  PlayAndCloseWave(HWND,WORD,LPSTR)

    PURPOSE   :  This function takes a currently open device ID and a
		 string representing a wave file and plays that wavefile.
		 When the wavefile is finished playing the function closes
		 the wavefile.

    COMMENTS  :  dwFlags is set to MCI_WAIT so that the function will not
		 return until the wavefile is done playing.  If you want
		 the function to return before the wavefile is finished
		 playing omit the MCI_WAIT flag.

    HISTORY   :

****************************************************************************/


void FAR PASCAL PlayAndCloseWave(hWnd,wGlobalDeviceID,lpWaveName)

HWND hWnd;
WORD wGlobalDeviceID;
LPSTR lpWaveName;
{

     MCI_OPEN_PARMS mciopen;
     MCI_PLAY_PARMS mciplay;
     DWORD dwRes;
     WORD wDeviceID;
     DWORD dwFlags;
     char szWaveMessage[80];



     // Open the wave file independent of the device.	This will speed up
     // the playing of the wavefile.


     MessageBox(hWnd,(LPSTR)"Opening the Wave file","YES",MB_OK);

     dwFlags= MCI_OPEN_ELEMENT;

     mciopen.lpstrElementName=(LPSTR)lpWaveName;

     // All strings must be initialized otherwise this will crash under
     // the debug version of mmsystem.

     mciopen.lpstrDeviceType="\0";
     mciopen.lpstrAlias="\0";

     dwRes=mciSendCommand(wGlobalDeviceID,MCI_OPEN,dwFlags,(DWORD)(LPSTR)&mciopen);


     if (dwRes)
	{
	ErrorProc(dwRes);
	return;
	}

     // play the wave file.

     wDeviceID=mciopen.wDeviceID;

     mciplay.dwFrom=0;
     dwFlags=MCI_WAIT | MCI_PLAY | MCI_FROM;


     dwRes=mciSendCommand(wDeviceID,MCI_PLAY,dwFlags,(DWORD)(LPSTR)&mciplay);

     if (dwRes)
	{
	  ErrorProc(dwRes);
	  return;
	}

     dwRes=mciSendCommand(wDeviceID,MCI_CLOSE,0,NULL);

     if (dwRes)
	 {
	 ErrorProc(dwRes);
	 return;
	 }

     return;
     }


/****************************************************************************

    FUNCTION  :  CloseWaveDevice(HWND,WORD)

    PURPOSE   :  This function closes a currently open waveaudio device.


    COMMENTS  :	 To conserve system resources you should always close a
		 device when you are not using it.

    HISTORY   :

****************************************************************************/


BOOL FAR PASCAL CloseWaveDevice(hWnd,wGlobalDeviceID)

HWND hWnd;
WORD wGlobalDeviceID;

{
     DWORD dwRes;

     // Close the global device

      dwRes=mciSendCommand(wGlobalDeviceID,MCI_CLOSE,0,NULL);

     if (dwRes)
	{
	ErrorProc(dwRes);
	return(FALSE);
	}



     return(TRUE);
     }



/****************************************************************************

    FUNCTION  :  OpenPlayCloseWave(HWND,LPSTR)

    PURPOSE   :  This function demonstrates how to open an element (wavefile)
		 play the wavefile and close that element in one fell swoop.

    COMMENTS  :  Rather than opening the global device "waveaudio" before
		 playing the wave we let MCI do this for us.  After playing
		 the element we immediately close the element and MCI closes
		 the global device "waveaudio" for you.  Unfortunately doing
		 this is slow and is not advisable if you will be playing several
		 wave files at a time.	If you plan on playing several wave files
		 at a time it is advisable for speed to use the previously
		 demonstrated method.  Which is opening the global waveaudio device.
		 Leaving that device open and passing it to the open for the element
		 (wavefile).  Playing all the wavefiles and then closing the global
		 waveaudio device when you are finished.

    HISTORY   :

****************************************************************************/


void FAR PASCAL OpenPlayCloseWave(hWnd,lpWavefile)

HWND hWnd;
LPSTR lpWavefile;


{
     MCI_OPEN_PARMS mciopen;
     MCI_PLAY_PARMS mciplay;
     DWORD dwRes;
     WORD wDeviceID;
     WORD wGlobalDeviceID;
     DWORD dwFlags;



     MessageBox(NULL,lpWavefile,(LPSTR)"Playing Wave File",MB_OK);

     mciopen.lpstrDeviceType=(LPSTR)MCI_DEVTYPE_WAVEFORM_AUDIO;
     mciopen.lpstrElementName=(LPSTR)lpWavefile;


     dwFlags=MCI_OPEN_TYPE_ID | MCI_OPEN_TYPE | MCI_OPEN_ELEMENT;



     dwRes=mciSendCommand(0,MCI_OPEN,dwFlags,(DWORD)(LPSTR)&mciopen);

     if (dwRes)
	   {
	   ErrorProc(dwRes);
	   return;
	   }

     wDeviceID=mciopen.wDeviceID;
     mciplay.dwFrom=0;
     dwFlags=MCI_WAIT | MCI_FROM;
     dwRes=mciSendCommand(wDeviceID,MCI_PLAY,dwFlags,(DWORD)(LPSTR)&mciplay);

     if (dwRes)
	   {
	   ErrorProc(dwRes);
	   return;
	   }

     mciSendCommand(wDeviceID,MCI_CLOSE,0L,NULL);

     return;
}


/****************************************************************************

    FUNCTION  :  SndPlaySnd(HWND,LPSTR)

    PURPOSE   :  This function demonstrates how to use the function
		 sndPlaySound.	This is the easiest way to play a wave file.

    COMMENTS  :  The one setback to using sndPlaySound is that the entire
		 sound is loaded into memory at once.  If you have a very
		 large sound sndPlaySound will fail if the sound will not
		 load into physical memory.

    HISTORY   :

****************************************************************************/

BOOL FAR PASCAL SndPlaySnd(hWnd,lpWavefile)

HWND hWnd;
LPSTR lpWavefile;


{
     DWORD dwFlags;



     MessageBox(NULL,lpWavefile,(LPSTR)"Playing Wave File",MB_OK);


     dwFlags=SND_ASYNC;

     if (sndPlaySound(lpWavefile,dwFlags))
	   return(TRUE);
     else
	   return(FALSE);

}


/****************************************************************************

    FUNCTION  :  ErrorProc(WORD)

    PURPOSE   :  ErrorProc calls mciGetErrorString to display an error
		 message returned by MCI.

    COMMENTS  :

    HISTORY   :

****************************************************************************/

void FAR PASCAL ErrorProc(dwResult)

DWORD dwResult;

{
     HANDLE hMem;
     LPSTR lpStringBuff;


     hMem=GlobalAlloc(GHND,80);

     if (hMem)
	  {
	   lpStringBuff=GlobalLock(hMem);
	   if (lpStringBuff)
		{
		 if(mciGetErrorString(dwResult,lpStringBuff,80))
		       MessageBox(NULL,lpStringBuff,"ERROR",MB_OK);
		 else
		       MessageBox(NULL,"Generic Error","ERROR",MB_OK);

		 GlobalUnlock(hMem);
		}
	    else
		MessageBox(NULL,"Lock Failed","ERROR",MB_OK);

	    GlobalFree(hMem);
	    }
	else
	    MessageBox(NULL,"Alloc Failed","ERROR",MB_OK);

	return;
}
