/* tar.c - Tape ARchive utility program (main function)
 * Author: T.V.Shaporev
 * Creation date: 14 Dec 1990
 *
 * The program works in a same fashion under UNIX (most clones) and MS-DOS.
 * The main idear was to develop a mean for file transferring via diskette
 * between different operating systems - such as all UNIX clones, MS-DOS,
 * RSX, VAX/VMS - and all others which support tar format on a diskette.
 *
 * First step on this way (made in 1989) lies in adapting common UNIX tar
 * program to MS-DOS.
 *
 * On the second step
 *  - some bugs were fixed (especially in DOS-applied codes) and some
 *    optimization were done;
 *  - nonstandard (under DOS) diskette formats were added (i.e.
 *    DEC Rainbow and 80 tracks & 9 sectors format)
 *  - the possibility for compress encoding were included
 *    (this compressor has the best ratio among all others which
 *    I know, but don't ask me about its speed). Compressed-file
 *    format is compatible with the common versions of tar, so You
 *    can extract the compressed image from the archive by the
 *    common program, but I doubt You could uncompress them at last.
 *
 * On the fird step the program was totally (newly) overwritten to bypass
 * any copyright exclamations. In fact, it must be considered new program
 * but I prefer to continue version enumeration (I hope, nobody cares).
 *
 * I think, this program must be called Tar (with capital first letter)
 * to distinguish it from regular UNIX tar.
 *
 * The program's behaviour is analogous to usual tar, and I hope, its
 * internal help will be enough to understand the differences.
 *
 * The program doesn't perform any text file conversion - it passes
 * strict binary image of each file. If You has a problems with reading
 * DOS text files under UNIX (or UNIX files under DOS) please use my
 * dostext program.
 *
 * The program must be compiled by Turbo C 2.0 compiler (in a compact
 * model) under MS-DOS. Please don't replace dynamic arrays back to
 * static and automatic - MS-DOS compilers dislike them.
 *
 *			tim	tim@ecsc.mipt.su	14 Dec 1990
 */
/* Version 3.01
 * Handling of the 'l' option corrected
 */
/* Version 3.02                                       31 Dec 1990
 *  - great deal of minor corrections
 *  - u<pdate> option expanded to extracting
 *  - j<ournal> option added (comment storying files)
 *  - wildcards * and ? are now processed in archive; this may be
 *    suppressed by s<trict> option
 *  - d<elete> option added to replace previous occurencies of files
 *    in archive on storying, or deleting files from archive when
 *    whithout a<dd> parameter - this is for file archives only!
 */
/* Version 3.03                                       22 Feb 1991
 *  - an error corrected in mismatch() (file extract.c)
 *  - decreased stack requirements for tree processing
 *    (see store() in store.c and putfiles() in tar.c)
 *  - added codes to prevent archive file self-storying
 *    (not quite reliable for MS-DOS)
 *  - bincall() invocations changed by calls to rmdir() and mkdir()
 *    this is done automatically for 386/ix and may be switched by
 *    RMKDIR macro
 */
/* Version 3.04                                       29 Jul 1991
 *  - a direct intialization of static global variables inserted into
 *    lzencode() and lzdecode() - see file lzpack.c
 */
/* Version 3.04b                                      03 Oct 1991
 *  - a minor correction of bincall()
 *  - added default block number = sectors on track (MS-DOS)
 */
/* Version 3.05                                       11 Nov 1991
 *  - block factor for diskette writing is set to 1 for most BIOS
 *    compatibility
 *  - scantape() is slightly optimized
 */
/* Version 3.06                                       17 Dec 1991
 *  - n<onest> option applied to all actions (see inarg() in extract.c)
 *  - command-line-extension-file option (responce file) added
 *    for vak (vak@kiae.su) request (see append() in tar.c)
 *  - p<ermission> option added to save directories permissions (UNIX)
 */
/* Version 3.06b                                      22 Dec 1991
 *   - UNIX to DOS renaming algorithm (dot elimination) slightly
 *     changed (see extract.c)
 *   - most of output redirected to stdout (vs stderr)
 */
/* Version 3.07                                       28 Dec 1992
 *   - all unary apostrofs in string constants preserved by backslashes
 *   - reading file list from stdin allowed (see append() in tar.c)
 *   - input redirected to /dev/tty under UNIX
 *   - support for traditional UNIX compression algorithm
 *   - few changes in converting UNIX names for DOS
 */
/* Version 3.07b                                      20 Jan 1993
 *   - gethead() does not return FALSE while errors,
 *     scantape() looks for tape reading errors
 */
/* Version 3.08                                       22 Feb 1993
 *   - method-dependent comression indicator masks (see percent.c)
 *   - 'z' option applied to catalog printing (see catalog())
 *   - compatibility corrections in directory structure checking
 *   - st.st_size == codesize - means file unpacked! (see extract.c)
 */
/* Version 3.09                                       14 Mar 1993
 *   - a bug fixed wich prevents archiving unpacked files at all
 *     (ha-ha!) - see store.c
 *   - changed header description to support new features
 *     see define.h
 *   - support for P1003 and GNU file types - 't' option only! -
 *     see extract.c
 *   - small changes in #ifdef-s to distinguish SCO UNIXes from
 *     XENIXes - see store.c
 *   - regular file processing extracted into separate source
 *     files (see savefile.c & restore.c)
 *   - support for devices and FIFOs added
 *   - 'l' option added for DOS (copy linked files)
 *   - support for System V extents - see extract.c and restore.c
 *     to read archives only, extents may not be unpacked on the
 *     fly - alas, that's all for now
 *   - an error corrected in roll.c
 */
/* Version 3.10                                       28 Jun 1993
 *   - a bug fixed in old compression code (see lzpack.c)
 *   - added possibility to run through compress (',' comma option)
 *     see tar.c, tape.c and compress.c
 *   - comments will not be printed unless 'j' is given (extract.c)
 *   - separated compress-related codes
 */
/* Version 3.11                                       14 Jul 1993
 *   - support for QIC-02 streamers (first version!)
 *     devices supported: fastape, everex
 */
/* Version 3.12                                       29 Sen 1993
 *   - slack area in archive is filled by nulls to improve compression
 *   - added support for Wangtek (QIC-02) device
 *   - a bug fixed in memory release ('pk_out' variable - see _done())
 *   - program support for QIC-02 drive number and tape format
 *     selection
 *   - experimental (!) support for appending QIC-02 tapes
 *     (see qback() in qicface.c)
 *   - LZW support splitted into compressor and extractor and
 *     the letter included in official release (see unlzw.c etc.)
 *   - get default file name from TAPE environment variable
 *   - 'o' flag for DOS means prevent file overwriting
 */
/* Version 3.12b                                      10 Nov 1993
 *   - an error corrected in QIC device selection (see qicface.c)
 *   - eliminated idle rewindings around QIC tape initialisation
 */
/* Version 3.13                                       26 Dec 1993
 *   - online inflatter (unzip) and corresponding '.' (point) option
 */
#include "sysup.h"
#include "modern.h"
#include "diszip.h"
#include "lzwbits.h"
#include "lzwhead.h"
#include "compress.h"

static char note[] = "\n\
   Tape ARchive utility        v3.13%c        (C) 1990-93 Tim Shaporev\n";

#ifdef USE_COMPRESS
#	define SIGN_VERSION '+'
#else
#	define SIGN_VERSION '-'
#endif

#ifdef UNIX
static char help[] = "\n\
   Usage: tar -<options> [tapefile] [blocksize] [disksize] file ...\n\n\
   Options are:\n\
     c - put files to new archive     s - no wildcards for archive\n\
     a,r - add files to archive       i - ignore read errors\n\
     y - replace files to archive     m - forget files origin date/time\n\
     x - extract files from archive   o - forget files owner\n\
     t - show archive catalog         l - type missed links\n\
     d - delete files in archive      p - save directories & permissions\n\
     u - update files                 n - no proceed with dir nesting\n\
     v - verbose                      / - omit left \'/\' in file names\n\
     w - acknowledge operations       0...7 - number of tape device\n\
     e - compress encode files        j - comment storying files\n\
     z - old-fashion compression      f - next arg is an archive file\n\
     , - run through compressor       b - next arg is a block number\n\
     . - run through (g)unzip         @ - next arg is a responce file\n\
";
#endif

#ifdef MSDOS
static char help[] = "\n\
   Usage: tar -<options> [tapefile] [blocksize] [disksize] file ...\n\n\
   Options are:                       s - no wildcards for archive\n\
     c - put files to new archive     i - ignore read errors\n\
     a,r - add files to archive       m - forget files date/time\n\
     y - replace files to archive     n - no proceed with dir nesting\n\
     x - extract files from archive   l - copy linked files\n\
     t - show archive catalog         o - prevent files overwriting\n\
     d - delete files in archive      \\ - omit left \'\\\' in file names\n\
     u - update files                 : - omit DOS drive name\n\
     v - verbose                      0...3 - number of diskette drive\n\
     w - acknowledge operations       j - comment storying files\n\
     e - compress encode files        f - next arg is an archive file\n\
     z - old-fashion compression      b - next arg is a block number\n\
     , - run through compressor       k - next arg is K diskette size\n\
     . - run through (g)unzip         @ - next arg is a responce file\n\
\n\
   Most of options may be combined. Wildcards * and ? are o\'k\n\
";
static char devlist[] = "\n\
   The following \"file names\" will be treated as diskette format/size:\n\
\tfd048ss8  - 160K\tfd048ds8  - 320K\tfd135ds9  - 720K\n\
\tfd048ss9  - 180K\tfd048ds9  - 360K\tfd135ds18 - 1.4M\n\
\tfd096ds9  - 720K\tfd096ds15 - 1.2M\trainbow\n\
\n\
   Streamer \"file name\" syntax (full form) is:\n\
\t<device>:base:=<base address>h,dma:=<DRQ>[,irq:=<IRQ>][,norewind]\n\
   Streamer device clones supported are:\n\
\tarchive,\teverex,\t\twangtek\n\
";
#endif

#include <signal.h>
#include <stdio.h>
#ifdef MSDOS
#	include <string.h>
#	include <stdlib.h>
#	include <dos.h>
#	include <dir.h>
#	include <io.h>
#else
	char *strcpy(), *strcat(), *strncpy();
        char *getenv(), *malloc(), *realloc();
	int  open(), read(), close(), link(), unlink(), isatty();
	int  strlen(), strncmp(), atoi();
        void exit(), free();
	long lseek();
#endif

#define __ALLOCEXT__
#include "define.h"

#include "lzpack.h"
#include "roll.h"

#ifdef UNIX
#	ifdef MAXNAMSIZ
#		define MAXPATH MAXNAMSIZ+1
#	else
#		define MAXPATH 512+1
#	endif
#endif

#ifdef UNIX
#	ifndef RMKDIR
int bincall(name, args)
char *name, *args;
{
   extern int fork(), execl(), wait();
   int i; register k;
   char b[24];

   if (fork() == 0) {
      (void)execl(strcat(strcpy(b, "/bin/"),     name), name, args, 0);
      (void)execl(strcat(strcpy(b, "/usr/bin/"), name), name, args, 0);
      k = -1;
   } else {
      (void)wait(&i); k = i>>8;
   }
   return k;
}
#	endif
#endif

#ifdef MSDOS
void takename(dst, src)
register char *dst, *src;
{
   do {
      *(dst++) = *src >= 'A' && *src <= 'Z' ? *src + ('z'-'Z') :
                 *src == '\\' ? '/' : *src;
   } while (*(src++));
}
#endif

short headsum(h)
struct header *h;
{
   register short i, j;

   for (i=0, j=0; i<BLKSIZE; i++) {
      j += i >= MAXTNAME+3*8+2*12 && i < MAXTNAME+3*8+2*12+8 ?
           ' ' : *((unsigned char *)h + i);
   }
#if ~0 != 0177777
   return j & 0177777;
#else
   return j;
#endif
}

struct node *finditem(name, prev, head)
char *name; struct node **prev, *head;
{
   register struct node *this;
   register i;

   *prev = this = head; i = 1;
   while (this && i>0) {
      if ((i = strncmp(name, this->name, MAXTNAME)) > 0) {
         *prev = this; this = this->next;
      }
   }
   return i ? NONE : this;
}

struct node *additem(name, prev, head)
char *name; struct node *prev, **head;
{
   register struct node *this;

   if ((this = (struct node *)malloc(sizeof(struct node))) == NULL)
      return NONE;
   (void)strncpy(this->name, name, MAXTNAME);
   this->prev = prev;
   if (prev != NONE) {
      if ((this->next = prev->next) != NONE) this->next->prev = this;
      prev->next = this;
   } else {/* initialise the list */
      this->next = NONE;
      (*head) = this;
   }
   return this;
}

void delitem(this, head)
struct node *this, **head;
{
   if (this == *head) {/* head of the list */
      if (this->next != NONE) {
         this->next->prev = this->prev != this ? this->prev : this->next;
      }
      this = this->next;
      free(*head);
      *head = this;
   } else {
      if (this->next != NONE) this->next->prev = this->prev;
      this->prev->next = this->next;
      free(this);
   }
}

static char *responce = NULL, **argvector = NULL;

static void _done __ARGS__(( void ))
{
   register struct node *p, *q;
#ifdef MSDOS
   if (devtype == DEV_QIC2) qend();
   if (archname) free(archname);
#endif
   if (responce) {
      free(responce); free((char*)argvector);
   }
   p = timehead; while (p) { q = p->next; free(p); p = q; }
#ifdef UNIX
   p = linkhead; while (p) { q = p->next; free(p); p = q; }
#endif
   if (hwrite >= 0 && hwrite != handle) {
      (void)close(hwrite); (void)unlink(scratch);
   }
   if (io_2nd && io_2nd!=io_buf) free(io_2nd);
   if (io_buf) free(io_buf);
   if (pk_out && pk_out!=pk_inp) free(pk_out);
   if (pk_inp) free(pk_inp);
   unzfree();
#ifdef USE_COMPRESS
   z_reltab();
#endif
   z_relmem();
   lzrelmem();
   delroll();
}

void done(n)
int n;
{
   _done(); exit(n);
}

void outmem(f)
FILE *f;
{
   (void)fprintf(f, "Tar: not enough memory\n");
   done(ESMALL);
}

char *salloc(n)
int n;
{
   register char *p;

   if ((p = malloc(n)) == NULL) outmem(stderr);
   return p;
}

int yes_no(d)
char d;
{
   register int c;
#ifdef MSDOS
   extern int getkey __ARGS__((void));

   do {
      c = getkey();
   } while (c!='y' && c!='Y' && c!='n' && c!='N' && c!='q' && c!='Q' &&
            c!='\r' && c!='\n' && c!=27 && c!=3);
   if (c == 3) {
      cbreak = TRUE;
   } else if (c >= ' ') {
      (void)fprintf(stderr, "%c", c);
   } else if (d) {
      (void)fprintf(stderr, "%c", d);
   }
   (void)fprintf(stderr, "\n");
#else
   if ((c = getc(myinp)) == '\n') {
      c = d;
   } else {
      while (getc(myinp) != '\n') ;
   }
#endif
   if (c == 'q' || c == 'Q') done(EXIT);
   return c == 'y' || c == 'Y';
}

void prmode(c, m)
char c; int m;
{
   (void)fprintf(myout, "%c%c%c%c%c%c%c%c%c%c", c,
      (m & S_IREAD ? 'r' : '-'), (m & S_IWRITE? 'w' : '-'),
      (m & S_ISUID ? 's' : m & S_IEXEC ? 'x' : '-'),
      (m & 00040   ? 'r' : '-'), (m & 00020   ? 'w' : '-'),
      (m & S_ISGID ? 's' : m & 00010   ? 'x' : '-'),
      (m & 00004   ? 'r' : '-'), (m & 00002   ? 'w' : '-'),
      (m & S_ISVTX ? 't' : m & 00001   ? 'x' : '-'));
}

int okwork(c, p, s, n)
char c, p, *n;
struct stat *s;
{
   register m;

   (void)fprintf(stderr, "%c ", c);
   if (v_flag) {
      if (p == ' ' && (m = s->st_mode & S_IFMT) != 0) {
         switch (m) {
            case S_IFREG: break;
            case S_IFDIR: p = 'd'; break;
            case S_IFIFO: p = 'p'; break;
            case S_IFCHR: p = 'c'; break;
            case S_IFBLK: p = 'b'; break;
#ifdef S_IFLNK
            case S_IFLNK: p = 'l'; break;
#endif
            default:      p = '?';
         }
      }
      prmode(p, (int)(s->st_mode));
      (void)fprintf(stderr," %3d/%1d %7ld ",s->st_uid,s->st_gid,s->st_size);
   }
   (void)fprintf(stderr, "%s : ", n);
   return YES_NO();
}

void onintr() { (void)signal(SIGINT,  SIG_IGN); cbreak = TRUE; }
#ifdef UNIX
void onquit() { (void)signal(SIGQUIT, SIG_IGN); cbreak = TRUE; }
void onhup()  { (void)signal(SIGHUP,  SIG_IGN); cbreak = TRUE; }
#endif

static void set_sig __ARGS__(( void ))
{
   if (signal(SIGINT, SIG_IGN) != SIG_IGN) (void)signal(SIGINT, onintr);
#ifdef UNIX
   if (signal(SIGHUP, SIG_IGN) != SIG_IGN) (void)signal(SIGHUP, onhup );
   if (signal(SIGQUIT,SIG_IGN) != SIG_IGN) (void)signal(SIGQUIT,onquit);
#endif
}

static void delfile __ARGS__(( void ))
{
   if (v_flag)
      (void)fprintf(myout, "d %s, %ld bytes, %ld tape blocks\n",
         hblock->name, st.st_size, (st.st_size + (BLKSIZE-1))/BLKSIZE);
   if (usize()) skipfile();
}

static void putfiles __ARGS__(( int, char ** ));

static void putfiles(argc, argv)
int argc; char *argv[];
{
   register i;
   char fnmbuf[MAXPATH];

   for (i=0; i<argc; i++) {
      if (strlen(argv[i]) > MAXTNAME) {
         (void)fprintf(myout, "Tar: \'%s\' name too long\n", argv[i]);
         continue;
      }
      takename(fnmbuf, argv[i]);
      store(fnmbuf);
   }
}

static void append __ARGS__(( int*, char***, char* ));
#define INDMAX 32760
#define INDMIN 1024
#define INDLOW 1023

static void append(argc, argv, fname)
int *argc; char ***argv, *fname;
{
   register i;
   register l = 0;
   register n;

   if (fname[0] == '-' && fname[1] == '\0') {
      /* standard input */ i = 0;
   } else if ((i = open(fname, O_RDONLY)) < 0 || fstat(i, &st) < 0) {
      (void)fprintf(stderr, "Tar: can\'t process \'%s\'\n", fname);
      done(ERRARG);
   }
   if ((st.st_mode & S_IFMT) == S_IFREG) {
      if (st.st_size >= INDMAX) {
         (void)fprintf(stderr, "Tar: \'%s\' exceeds limit in size\n", fname);
         done(ERRARG);
      }
      responce = salloc((l = (int)st.st_size) + 1);
/* Note: effective text file size differs from the real one under MS-DOS */
      if ((l = read(i, responce, l)) < 1) {
         (void)fprintf(stderr, "Tar: can\'t read \'%s\'\n", fname);
         done(ERRARG);
      }
      (void)close(i);
   } else if ((st.st_mode & S_IFMT) == S_IFCHR || /* character device */
              (st.st_mode & S_IFMT) == S_IFIFO) { /* pipe */
      int j, m; long s;

      for (responce=0, m=INDMAX+1; m>=INDMIN; ) {
         if ((responce = malloc(m)) != NULL) goto ok;
         m = ((m+INDLOW) & ~INDLOW) / 2;
      }
      outmem(stderr);
ok:
      s = 0;
      do {
         l = (int)s;
         j = read(i, responce+l, m-l);
         s += j;
      } while (j>0 && s<m);
      if (j < 0) {
         (void)fprintf(stderr, "Tar: error reading \'%s\'\n", fname);
         done(ERRARG);
      }
      if (s >= m) {
         if (m < INDMAX) {
            outmem(stderr);
         } else {
            (void)fprintf(stderr,"Tar: \'%s\' exceeds limit in size\n",fname);
            done(ERRARG);
         }
      }
#ifndef NO_REALLOC
      if (l+1 < m) {
         if ((responce = realloc(responce, l+1)) == NULL) {
            (void)fprintf(stderr, "Tar: error shrinking memory block\n");
            done(ERRARG);
         }
      }
#endif
   } else {
      (void)fprintf(stderr, "Tar: \'%s\' not a file\n", fname);
      done(ERRARG);
   }
   responce[l] = '\0';

   i = 0; n = 0;
   while (i<l) { /* count number of words in responce file */
      while (i<l && (responce[i] == ' '  || responce[i] == '\n' ||
                     responce[i] == '\r' || responce[i] == '\t')) ++i;
      if (i<l) ++n; /* next word found */
      while (i<l && responce[i] != ' '  && responce[i] != '\n' &&
                    responce[i] != '\r' && responce[i] != '\t') ++i;
   }
   /* create new argument vector */
   argvector = (char **)salloc((int)((*argc + n) * sizeof(char *)));
   /* copy origin arguments */
   for (i=0; i<*argc; i++) argvector[i] = (*argv)[i];
   *argv = argvector; /* i = *argc; */ *argc += n;

   /* append new arguments */
   n = 0;
   while (n < l) {
      while (n<l && (responce[n] == ' '  || responce[n] == '\n' ||
                     responce[n] == '\r' || responce[n] == '\t'))
         responce[n++] = '\0';
      if (n<l) argvector[i++] = responce+n; /* next word found */
      while (n<l && responce[n] != ' '  && responce[n] != '\n' &&
                    responce[n] != '\r' && responce[n] != '\t') ++n;
   }
}

static void stdhelp __ARGS__((void))
{
   (void)fprintf(stdout, note, SIGN_VERSION);
   (void)fprintf(stdout, help);
   (void)fflush (stdout);
#ifdef MSDOS
   if (ioctl(fileno(stdout),0) & 0x80) {/* not a disk file */
      (void)fprintf(stderr,"\nDo you want to see device list? ");
      (void)fflush (stderr);
      if (!YES_NO()) return;
   }
   (void)fprintf(stdout, devlist);
#endif
}

int main(argc, argv)
int argc; char *argv[];
{
   register i;
   register char *p;
   char *tapename;
   register k;
   static char m_tape[] = "Tar: tape device multiply defined\n";
   static char bad_lz[] = "Tar: compression method multiply defined\n";

   if (argc < 2) {
      stdhelp(); return ERRARG;
   }
#ifdef UNIX
   p_flag = FALSE;
#endif
#ifdef MSDOS
   deldrv   = FALSE;
   setdrive = FALSE;
   filemask = FA_SYSTEM+FA_HIDDEN+FA_RDONLY+FA_DIREC+FA_ARCH;
#endif
   a_flag = x_flag = t_flag = v_flag = u_flag = y_flag = w_flag = i_flag =
   o_flag = m_flag = d_flag = j_flag = s_flag = nonest = dslash = l_flag =
      FALSE;
   pktype = PKNONE;
   cblock = 0;
   tapename = NULL;
   myout = stdout;

   p = argv[1]; i = 2;
#ifdef MSDOS
   if (*p == '-' || *p == '/') ++p;
#else
   if (*p == '-') ++p;
#endif
   while (*p) {
      switch(*p>='A' && *p<='Z' ? *p + ('Z'-'z') : *p) {
#ifdef MSDOS
         case 'h':
         case '?': stdhelp();
                   return ERRARG;
#endif
         case 'a':
         case 'r': a_flag = TRUE; break;
         case 'c': c_flag = a_flag = TRUE; break;
         case 'u': u_flag = TRUE; break;
         case 'y': y_flag = a_flag = TRUE; break;
         case 'x': x_flag = TRUE; break;
         case 't': t_flag = TRUE; break;
         case 'v': v_flag = TRUE; break;
         case 'w': w_flag = TRUE; break;
         case 'i': i_flag = TRUE; break;
         case 'm': m_flag = TRUE; break;
         case 'o': o_flag = TRUE; break;
         case 'b': if (cblock) {
                      (void)fprintf(stderr,
                         "Tar: blocksize multiply defined\n");
                      return ERRARG;
                   }
                   if ((cblock = atoi(argv[i++])) < 1 ||
                        cblock > MAXBLOCK) {
                      (void)fprintf(stderr,
                         "Tar: bad blocksize (max is %d)\n", MAXBLOCK);
                      return ERRARG;
                   }
                   break;
         case 'f':
                   if (
#ifdef MSDOS
                       k_flag ||
#endif
                                 tapename) {
                      (void)fprintf(stderr, m_tape); return ERRARG;
                   }
                   tapename = argv[i++];
                   break;
         case 'n': nonest = TRUE;
#ifdef MSDOS
                   filemask = FA_SYSTEM+FA_HIDDEN+FA_RDONLY+FA_ARCH;
#endif
                   break;
         case 'e': if (pktype != PKNONE) {
                      (void)fprintf(stderr, bad_lz); return ERRARG;
                   }
                   pktype = PKLZH;
                   break;
         case '.': if (pktype != PKNONE) {
                      (void)fprintf(stderr, bad_lz); return ERRARG;
                   }
                   pktype = PKZIP; pklevel = 6; /* default */
                   if (*(p+1) >= '1' && *(p+1) <= '9') {
                      pklevel = *(p += 1) - '0';
                   }
                   break;
         case ',': if (pktype != PKNONE) {
                      (void)fprintf(stderr, bad_lz); return ERRARG;
                   }
                   pktype = PKpLZW;
                   goto getbits;
         case 'z': if (pktype != PKNONE) {
                      (void)fprintf(stderr, bad_lz); return ERRARG;
                   }
                   pktype = PKfLZW;
         getbits : pklevel = BITS;
                   if        (*(p+1) == '9') {
                      p += 1; pklevel = 9;
                   } else if (*(p+1) == '1') {
                      p += 2; pklevel = 10 + *p - '0';
                      if (pklevel < 10 || pklevel > BITS) {
                         (void)fprintf(stderr, "Tar: Invalid bits factor\n");
                         return ERRARG;
                      }
                   }
                   break;
         case 'l': l_flag = TRUE; break;
#ifdef UNIX
         case 'p': p_flag = TRUE; break;
#endif
#ifdef MSDOS
         case 'k': if (k_flag || tapename) {
                      (void)fprintf(stderr, m_tape); return ERRARG;
                   }
                   k_flag = 1;
                   if (argdisk(atoi(argv[i++])) != 0) {
                      (void)fprintf(stderr, "Tar: Invalid diskette size\n");
                      return ERRARG;
                   }
                   break;

#endif
#ifdef MSDOS
         case '\\':
#endif
#ifdef UNIX
         case '/':
#endif
                   dslash = TRUE; break;
#ifdef MSDOS
         case ':': deldrv = TRUE; break;
#endif
         case '0': case '1': case '2': case '3':
#ifdef UNIX
         case '4': case '5': case '6': case '7':
#endif
                   if (ndrive) {
                      (void)fprintf(stderr, m_tape); return ERRARG;
                   }
#ifdef MSDOS
                   setdrive = TRUE;
#endif
                   ndrive = *p & 7;
                   break;
         case 'd': d_flag = TRUE; break;
         case 'j': j_flag = TRUE; break;
         case 's': s_flag = TRUE; break;
         case '@': append(&argc, &argv, argv[i++]);
                   break;
         default : (void)fprintf(stderr, "Tar: bad option '%c'\n", *p);
                   return ERRARG;
      }
      ++p;
   }
   if (
#ifdef MSDOS
       !k_flag &&
#endif
                  !tapename) tapename = getenv("TAPE");

   if (u_flag && !x_flag) a_flag = TRUE;

   if (!a_flag && !x_flag && !t_flag && !d_flag) {
      (void)fprintf(stderr, "Tar: nothing to do\n"); return ERRARG;
   }
   if (a_flag || d_flag) {
      if (i >= argc) {
         (void)fprintf(stderr, "Tar: no files specified\n");
         return ERRARG;
      }
#ifndef USE_COMPRESS
      if (pktype == PKfLZW || pktype == PKpLZW) {
         (void)fprintf(stderr,
         "Tar: this restricted version does not support LZW compression\n");
         return ERRARG;
      }
#endif
      if (pktype == PKZIP) {
         (void)fprintf(stderr, "Tar: sorry, deflate unsupported yet\n");
         return ERRARG;
      }
   }
   if ((a_flag && (x_flag || t_flag)) ||  (x_flag && t_flag) ||
      (!a_flag && (j_flag || dslash)) || (!x_flag && m_flag) ||
       (d_flag && (c_flag || x_flag || t_flag)) ||
       (i_flag && (d_flag || a_flag))) {
      (void)fprintf(stderr, "Tar: incompatible options\n");
      return ERRARG;
   }
   if (m_flag && u_flag) {
      (void)fprintf(stderr, "Tar: 'm' and 'u' options ambiguous\n");
   }
   if (pktype == PKZIP && (a_flag || d_flag)) {
      (void)fprintf(stderr, "Tar: sorry, deflate unsupported yet\n");
      return ERRARG;
   }
   if ((k = initape(tapename)) != CORRECT) done(k);
   if ((io_buf=getbuf(cblock ? cblock*BLKSIZE : MAXBLOCK*BLKSIZE)) == NULL)
      done(ESMALL);
   io_2nd = io_buf;
   if (pktype != PKNONE) {
      if (pktype == PKpLZW || pktype == PKZIP) {/* Pipe compression */
         pksize = BLKSIZE;
         k = (d_flag || x_flag || t_flag) && (pk_inp=malloc(pksize))==NULL ||
             (d_flag || a_flag)           && (pk_out=malloc(pksize))==NULL;
      } else {/* Individual file(s) compression */
         pksize = PKSIZE;
         k = (d_flag || x_flag || (t_flag && pktype == PKfLZW)) &&
                                   (pk_out=malloc(pksize))==NULL ||
             (d_flag || a_flag) && (pk_inp=malloc(pksize))==NULL;
      }
      if (k) {/* Memory lack */
         if (!w_flag) {
            outmem(stderr);
         } else {
            (void)fprintf(stderr,
               "No memory for [de]compression. Continue? ");
            k = YES_NO();
            (void)fprintf(stderr, "\n");
            if (!k) done(ESMALL);
            if (pk_inp) { free(pk_inp); pk_inp = NULL; }
            pktype = PKNONE;
         }
      }
   }
   cbreak = FALSE;

   if ((k = runtape()) != CORRECT) done(k);
#ifdef UNIX
   myinp = stdin;
   if (!isatty(/* stdin */ 0) && (myinp = fopen("/dev/tty", "r")) == NULL) {
      (void)fprintf(myout,
         "Tar: warning: can\'t open terminal device, may be problems\n");
      myinp = stdin;
   }
#endif

#ifdef MSDOS
   if (a_flag && isfile) {
      p = tapename;
      if ((p[0]>='A' && p[0]<='Z' || p[0]>='a' && p[0]<='z') && p[1]==':') {
         k = p[0] < 'a' ? p[0] - ('A'-1) : p[0] - ('a'-1); p += 2;
      } else {
         k = 0;
      }
      if (p[0] == '/' || p[0] == '\\') {
         takename((archname = salloc(strlen(p) + 1)), p);
      } else {
         archname = salloc(strlen(p) + MAXPATH);
         *(int *)archname = '/';
         getcurdir(k, archname+1); takename(archname, archname);
         k = strlen(archname);
         archname[k++] = '/';
         takename(archname+k, p);
      }
   }
#endif
   argc -= i; argv += i;

   if (d_flag) {
      register struct header *h; register long l; register m;

      if (d_flag && !isfile) {
         (void)fprintf(stderr,
                       "Tar: delete option is for file archives only\n");
         return ERRARG;
      }
      duptape(tapename);

      m = FALSE;
      do {
         if ((k = gethead()) == ERROR) done(ERREAD);
         if (!k) continue;

         if        (a_flag) {
            struct stat s;

             if (inargs(argc, argv, hblock->name)) {
                if (u_flag) uplist();
                if (stat(hblock->name, &s)!=0) {
                   if (v_flag) {
                      (void)fprintf(myout, "Tar: can\'t access \'%s\'\n",
                                            hblock->name);
                   }
                } else if (!u_flag || s.st_mtime > st.st_mtime) {
                   delfile(); continue;
                }
             }
         } else {/* pure delete */
            if (inargs(argc, argv, hblock->name) ||
               ((hblock->filetype == TF_LNK || hblock->filetype == TF_SYM) &&
                inargs(argc, argv, hblock->linkname))) {
               m = TRUE; delfile(); continue;
            }
         }

         /* move file to output archive */
         for (h=steptape(), i=0; i<BLKSIZE/sizeof(int); i++) {
            ((int *)h)[i] = ((int *)hblock)[i];
         }
         if (usize()) {
            l = (st.st_size + (BLKSIZE-1)) / BLKSIZE;
            while (l-- > 0) {
               if ((hblock = readtape()) == NULL) done(ERREAD);
               for (h=steptape(), i=0; i<BLKSIZE/sizeof(int); i++) {
                  ((int *)h)[i] = ((int *)hblock)[i];
               }
            }
         }
      } while (k);

      if (a_flag) {
         l = lseek(hwrite, 0L, 1);
         putfiles(argc, argv);
         m = lseek(hwrite, 0L, 1) > l;
      }

      if (m) {/* archive was modified */
         endtape();
         if (unlink(tapename)) {
            (void)fprintf(myout, "Tar: can\'t delete \'%s\'\n", tapename);
            done(EWRITE);
         }
#ifdef UNIX
         if (link(scratch, tapename)) {
            (void)fprintf(myout,
               "Tar: can\'t link \'%s\' - data stay in \'%s\'\n",
               tapename, scratch);
            done(EWRITE);
         }
         if (unlink(scratch)) {
            (void)fprintf(myout, "Tar: can\'t delete scratch file\n");
         }
#endif

#ifdef MSDOS
         if (rename(scratch, tapename)) {
            (void)fprintf(myout, "Tar: can\'t rename \'%s\' to \'%s\'\n",
                                  scratch, tapename);
            done(EWRITE);
         }
#endif

#ifdef UNIX
         if (a_flag && l_flag) {
            register struct node *this;

            for (this=linkhead; this; this=this->next) {
               (void)fprintf(myout, "Tar: missed %d link(s) to \'%s\'\n",
                             this->info.data.count, this->name);
            }
         }
#endif
      } else {
         if (v_flag) (void)fprintf(myout, "Tar: archive unchanged\n");
         (void)close(hwrite);
         if (unlink(scratch)) {
            (void)fprintf(myout, "Tar: can\'t delete scratch file\n");
            done(EWRITE);
         }
      }
   } else if (a_flag) {
#ifdef MSDOS
      if (w_flag && c_flag && devtype == DEV_FLOP) {
         fprintf(stderr,
                 "\007Data on drive %c: would be destroyed. Continue ? ",
                 ndrive + 'A');
         if (!YES_NO()) done(ERRARG);
      }
#endif
      if (a_flag && !c_flag) {
         scantape(argc, argv, acctime); backtape();
      }
      set_sig();
      putfiles(argc, argv);
      endtape();
#ifdef UNIX
      if (l_flag) {
         register struct node *this;

         for (this=linkhead; this; this=this->next) {
            (void)fprintf(myout, "Tar: missed %d link(s) to \'%s\'\n",
                          this->info.data.count, this->name);
         }
      }
#endif
   } else if (x_flag) {
#ifdef UNIX
      (void)umask(0);
#endif
      scantape(argc, argv, extract);
   } else {/* if (t_flag) */
      allblock = allbytes = 0; allfiles = 0;
      scantape(argc, argv, catalog);
      if (v_flag) {
         (void)fprintf(myout,
            "\tTotal %u file(s) for %lu bytes in %lu tape blocks\n",
            allfiles, allbytes, ++allblock);
      }
   }
   _done(); return 0;
}
