{ Copyright (C) 1997 Wolfgang J Moeller. All rights reserved. NO WARRANTY! }

unit ethernet;

{ "packet driver" interface
{
{ w.j.m. dec 1996
{ 12-apr-1997 wjm: don't report CANT_SEND (hardware) error to client
{ 19-may-1997 wjm: add alternate eth_startup2(); fix eth_startup() interface.
}

interface

type
	lanaddr = record
		b: packed array[0..5] of byte;
	end;

function lanaddr_eql(var a1,a2: lanaddr): boolean;	{ compare lanaddr }
function lanaddr2str(var a: lanaddr): string;          	{ lanaddr -> ascii }

type
	eth_header = record		{ for reference only ... }
		dst: lanaddr;
		src: lanaddr;
		pty: word;		{ note problem with endian-ness }
	end;
const
	eth_hdr_size = 14;		{ supposedly = sizeof(eth_header) }
	eth_max_data = 1500;
	eth_max_buf = eth_hdr_size + eth_max_data;

var
	eth_lcladdr: lanaddr;	{ read-only, valid only after eth_start_rcv() }

function eth_startup2(int_lo,int_hi: byte): integer;
function eth_startup(pktd_int: byte): boolean;

function eth_start_rcv(var pty; ptylen: word; rcv_mode: word): boolean;
procedure eth_feed_rcv(p: pointer; siz: word);		{ provide rcv buffers }
function eth_receive(var len: word): pointer;
function eth_send(p: pointer; len: word): boolean;

type
	eth_pktd_stats = record
		packets_in: longint;
		packets_out: longint;
		bytes_in: longint;
		bytes_out: longint;
		errors_in: longint;
		errors_out: longint;
		packets_lost: longint;
	end;

{ works only after successful eth_start_rcv() }
function eth_statistics(var st: eth_pktd_stats): boolean;


implementation uses dos,wjmp,etherrcv;


{----- LANADDR stuff -----}

function lanaddr_eql(var a1,a2: lanaddr): boolean;
var
	i: integer;
begin
	lanaddr_eql := false;
	for i := 0 to 5 do if a1.b[i] <> a2.b[i] then exit;
	lanaddr_eql := true;
end;

function lanaddr2str(var a: lanaddr): string;
begin
	lanaddr2str := hex2(a.b[0]) + '-' + hex2(a.b[1]) + '-' +
		       hex2(a.b[2]) + '-' + hex2(a.b[3]) + '-' +
		       hex2(a.b[4]) + '-' + hex2(a.b[5]);
end;


{----- "raw" Packet Driver interface -----}

var
	pktdrv: byte;		{ interrupt # , to be filled in prior to use }

	if_class: byte;		{ filled in by pktd_driver_info() }
	if_type: word;
	if_number: byte;


function pktd_driver_info: boolean;
var
	reg: registers;
begin
	reg.ah := 1;
	reg.al := 255;
	intr(pktdrv,reg);

	if (reg.flags and FCarry) <> 0 then begin
		writeln('pktd_driver_info() error ',reg.dh);
		pktd_driver_info := false;
	end else begin
		if_class := reg.CH;
		if_type := reg.DX;   
		if_number := reg.CL;
		{ also:
			version		reg.BX
			functionality	reg.AL
			name		ptr(reg.DS,reg.SI)	-> asciz string
		}
		pktd_driver_info := true;
	end;
end;

function pktd_set_rcv_mode(rhandle: word; mode: word): boolean;
var
	reg: registers;
begin
	reg.ah := 20;
	reg.bx := rhandle;
	reg.cx := mode;
	intr(pktdrv,reg);
	if (reg.flags and FCarry) <> 0 then begin
		writeln('pktd_set_rcv_mode() error ',reg.dh);
		pktd_set_rcv_mode := false;
	end else begin
		pktd_set_rcv_mode := true;
	end;
end;

function pktd_get_rcv_mode(rhandle: word; var mode: word): boolean;
var
	reg: registers;
begin
	reg.ah := 21;
	reg.bx := rhandle;
	intr(pktdrv,reg);
	if (reg.flags and FCarry) <> 0 then begin
		writeln('pktd_get_rcv_mode() error ',reg.dh);
		pktd_get_rcv_mode := false;
	end else begin
		mode := reg.AX;
		pktd_get_rcv_mode := true;
	end;
end;

function pktd_get_statistics(rhandle: word; var st: eth_pktd_stats): boolean;
var
	reg: registers;
begin
	reg.ah := 24;
	reg.bx := rhandle;
	intr(pktdrv,reg);
	if (reg.flags and FCarry) <> 0 then begin
		writeln('pktd_get_statistics() error ',reg.dh);
		pktd_get_statistics := false;
	end else begin
		move(ptr(reg.DS,reg.SI)^,st,sizeof(st));
		pktd_get_statistics := true;
	end;
end;


type
	callback = procedure;	{ of a very special nature }

function pktd_access_type(var pty; ptylen: word; rcv: callback;
			 var rhandle: word): boolean;
{NOTE: on successful return, caller must _immediately_ set up an exit handler }
var
	reg: registers;
begin
	reg.ah := 2;
	reg.al := if_class;
	reg.bx := if_type;
	reg.dl := if_number;
	reg.cx := ptylen;
	reg.ds := seg(pty);
	reg.si := ofs(pty);
	reg.es := seg(rcv);
	reg.di := ofs(rcv);
	intr(pktdrv,reg);
	if (reg.flags and FCarry) <> 0 then begin
		writeln('pktd_access_type() error ',reg.dh);
		pktd_access_type := false;
	end else begin
		rhandle := reg.ax;
		pktd_access_type := true;
	end;
end;

function pktd_release_type(rhandle: word): boolean;
var
	reg: registers;
begin
	reg.ah := 3;
	reg.bx := rhandle;
	intr(pktdrv,reg);
	if (reg.flags and FCarry) <> 0 then begin
		writeln('pktd_release_type() error ',reg.dh);
		pktd_release_type := false;
	end else begin
		pktd_release_type := true;
	end;
end;

function pktd_send_pkt(p: pointer; len: word): boolean;
var
	reg: registers;
begin
	reg.ah := 4;
	reg.cx := len;
	reg.ds := seg(p^);
	reg.si := ofs(p^);
	intr(pktdrv,reg);
	if (reg.flags and FCarry) <> 0 then begin
		write('pktd_send_pkt() error ',reg.dh);
		if reg.dh = 12 then begin
			writeln(' (CANT_SEND) ignored');
			pktd_send_pkt := true;
		end else begin
			writeln;
			pktd_send_pkt := false;
		end;
	end else begin
		pktd_send_pkt := true;
	end;
end;

function pktd_get_address(rhandle: word; var a: lanaddr): boolean;
var
	reg: registers;
begin
	reg.ah := 6;
	reg.bx := rhandle;
	reg.cx := sizeof(a);		{ = 6, supposedly }
	reg.es := seg(a);
	reg.di := ofs(a);
	intr(pktdrv,reg);
	if (reg.flags and FCarry) <> 0 then begin
		writeln('pktd_get_address() error ',reg.dh);
		pktd_get_address := false;
	end else if reg.cx <> sizeof(a) then begin
		writeln('pktd_get_address() return length ',reg.cx,
			'(expected ',sizeof(a),')');
		pktd_get_address := false;
	end else begin
		pktd_get_address := true;
	end;
end;

{----- "cooked" Packet Driver interface -----}

const
	max_acc = 4;	{ # of handles managed }
type
	t_acc = record
		rhandle: word;
		orgmode,actmode: word;
		inuse: boolean;
	end;
var
        acc: array[1..max_acc] of t_acc;
	started: boolean;

procedure cleanup; far;  { exit handler }

var
	i: integer;
begin
	{ experience indicates that a non-standard "rcv_mode"
	  has to be reset prior to releasing the handle }

	for i := max_acc downto 1 do if acc[i].inuse then begin
		with acc[i] do begin
			if actmode <> orgmode then begin
				{ reset "rcv_mode" }
				if pktd_set_rcv_mode(rhandle,orgmode) then
					actmode := orgmode;
			end;

			{ release handle }
			if pktd_release_type(rhandle) then rhandle := 0;

			{ done }
			inuse := false;
		end;
	end;
end;


function eth_startup(pktd_int: byte): boolean;
var
	i: integer;
begin
	if started then fehler('eth_startup() called twice');

	{ copy argument }
	pktdrv := pktd_int;

	{ initialize local variables }
	for i := 1 to max_acc do acc[i].inuse := false;

	{ access packet driver }
	if pktd_driver_info then begin

		{ declare exit handler once for all }
		userex(cleanup);

		{ commit }
		started := true;
		eth_startup := true;

	end else begin
		eth_startup := false;
	end;
end;

{ Find lowest "packet driver" int.vector within range int_lo..int_hi,
{ invoke eth_startup() if found.
{ Return int.vector if o.k., 0 on nofind, -(int.vector) on eth_startup() error.
 }
function eth_startup2(int_lo,int_hi: byte): integer;
type
	t_pktdrvr = packed record
		fill: packed array[1..3] of byte;
		sign: packed array[1..8] of char;
	end;
	t_pktdrvrp = ^t_pktdrvr;
const
	signature: string[8] = 'PKT DRVR';	{ "packet driver" signature }
var
	vec: byte;
	i: integer;
	p: pointer;
	pp: t_pktdrvrp;
label
	mismatch;
begin
	for vec := int_lo to int_hi do begin
		getintvec(vec,p); pp := p;
		for i := 1 to length(signature) do begin
			if pp^.sign[i] <> signature[i] then goto mismatch;
		end;

		{ vector found }
		if eth_startup(vec) then
			eth_startup2 := vec
		else
			eth_startup2 := -vec;
		exit;

mismatch:	{ continue search }
	end;

	eth_startup2 := 0;
end;


function eth_start_rcv(var pty; ptylen: word; rcv_mode: word): boolean;
var
	i: integer;
label
	slot_ok,undo;
begin
	if not started then fehler('eth_start_rcv(): not started');

	for i := 1 to max_acc do if not acc[i].inuse then goto slot_ok;
	fehler('eth_start_rcv(): maximum number of protocols exceeded');

slot_ok:
	with acc[i] do begin
		{ default fields used by exit handler }
		orgmode := 0;
		actmode := 0;

		{ start receiver }
		if pktd_access_type(pty,ptylen,rcv,rhandle) then begin
			inuse := true;		{ activate exit handler }
		end else begin
			eth_start_rcv := false;
			exit;
		end;
		
		{ provide local address info, if possible }
		if not pktd_get_address(rhandle,eth_lcladdr) then nop;

		{ ggf. set non-standard rcv_mode }
		if rcv_mode <> 0 then begin
			if not pktd_get_rcv_mode(rhandle,orgmode) then
				goto undo;
			if not pktd_set_rcv_mode(rhandle,rcv_mode) then
				goto undo;
			actmode := rcv_mode;
		end;

		eth_start_rcv := true;
		exit;

undo:
		if pktd_release_type(rhandle) then inuse := false;
		eth_start_rcv := false;
	end;
end;

function eth_send(p: pointer; len: word): boolean;
begin
	if not started then fehler('eth_send(): not started');

	eth_send := pktd_send_pkt(p,len);
end;

procedure eth_feed_rcv(p: pointer; siz: word);
begin
	if not started then fehler('eth_feed_rcv(): not started');

	rcv_put(p,siz);
end;

function eth_receive(var len: word): pointer;
var
        p: pointer;
begin
	if not started then fehler('eth_receive(): not started');

	if rcv_get(p,len) then
        	eth_receive := p
        else
	        eth_receive := nil;
end;

function eth_statistics(var st: eth_pktd_stats): boolean;
begin
	if not started then fehler('eth_statistics(): not started');

	if acc[1].inuse then
		eth_statistics := pktd_get_statistics(acc[1].rhandle,st)
	else
		eth_statistics := false;
end;


{----- initialization -----}

const
	a0: lanaddr = (b: (0,0,0,0,0,0));

begin
	started := false;
	eth_lcladdr := a0;	{ say we don't know yet }
end.
