{ Copyright (C) 1997 Wolfgang J Moeller. All rights reserved. NO WARRANTY! }

{ "NDFP" server - using 60-06 ethernet protocol via DOS packet driver
  w.j.m. dec 1996 (E0ND*)
  06-jan-1997 wjm: use "ethernet" routines
  07-jan-1997 wjm: NDFP V3 (multi-packet + increased DATA_MAX)
  24-jan-1997 wjm: always allow for reading after a "write-only" file creation,
		   so ignore NDOM_CREATERW; add directory routines
  26-jan-1997 wjm: add logical disk I/O (enhanced DSK_CIO)
  17-feb-1997 wjm: add conditional LOG_E code from ETHNDS3E (inactive by default)
  21-feb-1997 wjm: send a dummy packet on startup, to alert switch or bridge
  01-mar-1997 wjm: fix erase() error handling
  06-mar-1997 wjm: fix(?) seek() & truncate() error handling
  16-mar-1997 wjm: no longer ignore "directory file" offsets;
  16-mar-1997 wjm: V4 - replace some former open() options by close() options;
		   add rename()
  20-mar-1997 wjm: re-structure the link-level protocol for clarity
  23-mar-1997 wjm: V5 - replace 31-bit offset & fsize by 40-bit (blk,byt) pairs;
		   add "truename" return with open()
  26-mar-1997 wjm: fix returned path for root directories; remove redundant
		   handling of NDOM_FATTR in NDIR5IO (changed from NDIR_IO);
		   return "expanded" file name from close/rename
  28-mar-1997 wjm: finally provide reasonable LOGGING (formerly LOG_E);
		   fix re-structured link-level protocol
  03-apr-1997 wjm: various minor edits; properly handle read/write errors
  04-apr-1997 wjm: (5.2.2) raise stack size (def=16k, process_rq() uses >11.5k)
  12-apr-1997 wjm: (5.3) do send the "dummy" packet every ~100 sec
  20-apr-1997 wjm: (5.4) add TIME operations
  19-may-1997 wjm: (5.4.1) automatically locate packet driver interrupt;
		   add startup message
  25-may-1997 wjm: (5.4.2) cctx list replaces cctxa[]
  25-may-1997 wjm: (5.4.3) Optional access protection via NDS_PERMIT file
		   (cf. NDS_UTIL.PAS). LOGGING code no longer conditional

Bugs:
	File names returned from open/create or close/rename don't necessarily
	reflect the true names as they appear on disk (e.g. fexpand() doesn't
	enforce 8.3). INT 21/AH=60 ("truename") might help ...
}

{$M 20480,0,655360}		{ stack size slightly raised from 16k }

program ETHNDS5(input,output);
uses dos,crt,wjmp,dsk_cio,ethernet,time64,nd_const,nds_util,ndir5io,nd_prot5;

const
	version = '5.4.3';

const				{ range for packet driver interrupt vector }
	pktdrv_lo = $60;
	pktdrv_hi = $7F;

const
	n_ctx = 10;		{ arbitrary maximum # of connection handles }

const
	winsiz = 8;		{ NDFP V2+ window size, 2 .. 8 }


type
	ndfp_pktptr = ^ndfp_pkt;

	ndfp_buffer = record
		hdr: eth_header;
		n: ndfp_pkt;
	end;
	ndfp_bufferptr = ^ndfp_buffer;

	{ client transaction state }
	ctra_state_t = (cts_idle,	{ (0) no context }
			  cts_rqmulti,	{ (1) partial request received }
			  cts_rpdone);	{ (2) reply done, retransmit possible }

	{ client slot state }
	cctx_flags_t = (
		ccf_inuse,		{ active }
		ccf_prealc);		{ pre-allocated, never disposed of }

	ndfp_client_ctxptr = ^ndfp_client_ctx;
	ndfp_client_ctx = record
		next: ndfp_client_ctxptr;	{ chain link }
		flags: set of cctx_flags_t;	{ slot state }
		tstate: ctra_state_t;		{ transaction state }
		macaddr: lanaddr;		{ client's MAC address }
		refcnt: byte;			{ # of open handles }
		rqmask: byte;			{ request NAK mask (cts 1) }
		cseq: word;			{ current seq# (cts 1&2) }
		op2: word;			{ current opcode w/o flags }
		rqhi: integer;			{ current rq nfrag = hi index }
		rqa: array[0..(winsiz-1)] of
			ndfp_bufferptr; 	{ receive slots }
		rphi: integer;			{ rp nfrag = hi index (cts 2) }
		rpa: array[0..(winsiz-1)] of
			ndfp_bufferptr; 	{ send slots }
		cname: permit_name_str;		{ client name (to be logged) }
	end;

	nds_ctx = record
		f: file;
		inuse: boolean;
		subh: byte;
		accmode: word;
		new_fattr: word;
		new_ftime: longint;
		chandle, shandle: word;
		client: ndfp_client_ctxptr;
	end;
	nds_ctxptr = ^nds_ctx;

	byteptr = ^byte;
	wordptr = ^word;
	pathstrptr = ^pathstr;
const
	NDFP_HDRSIZE = sizeof(ndfp_pkt) - NDFP_DATA_MAX;	{ shorthand }

const	{ initialized variables }
	log_all: boolean = false;		{ debugging switch }
	using_permit: boolean = false;		{ initially, no restrictions }
	cctxhead: ndfp_client_ctxptr = nil;	{ client ctx listhead }
var
	log_id: string[31];			{ "client/shnd/chnd" }
	ctxa: array[1..n_ctx] of nds_ctx;	{ one per file connection }


procedure init_contexts;	{ one-time init }
var
	j: integer;
begin
	for j := 1 to n_ctx do ctxa[j].inuse := false;
end;

function get_cctx(var a: lanaddr; var ccp: ndfp_client_ctxptr): boolean;
var
	i: integer;
begin
	ccp := cctxhead;
	while ccp <> nil do begin
		if lanaddr_eql(ccp^.macaddr,a) then begin
			ccp^.flags := ccp^.flags + [ccf_inuse];
			get_cctx := true;
			exit;
		end;
		ccp := ccp^.next;
	end;

	{ maybe allocate on the fly }
	if not using_permit then begin
		if sizeof(ccp^) > maxavail then begin
			writeln('get_cctx(): out of memory');	{?repeat?}
			get_cctx := false;
			exit;
		end;
		new(ccp);
		ccp^.next := cctxhead; cctxhead := ccp;
		for i := 0 to (winsiz - 1) do ccp^.rqa[i] := nil;
		for i := 0 to (winsiz - 1) do ccp^.rpa[i] := nil;
		ccp^.flags := [ccf_inuse];
		ccp^.refcnt := 0;
		ccp^.macaddr := a;
		ccp^.cname := '';
		ccp^.tstate := cts_idle;
		get_cctx := true;
	end else
		get_cctx := false;
end;

{ undo get_cctx, return true if cctx shouldn't be touched anymore }
function release_cctx(ccp: ndfp_client_ctxptr): boolean;
var
	i: integer;
	pred: ndfp_client_ctxptr;
begin
	if (ccp^.refcnt <> 0) or (ccf_inuse in ccp^.flags) then begin
		release_cctx := false;
		exit;
	end else begin
		for i := 0 to (winsiz - 1) do if ccp^.rpa[i] <> nil then begin
			dispose(ccp^.rpa[i]);
			ccp^.rpa[i] := nil;
		end;
		for i := 0 to (winsiz - 1) do if ccp^.rqa[i] <> nil then begin
			dispose(ccp^.rqa[i]);
			ccp^.rqa[i] := nil;
		end;

		release_cctx := true;
		if (ccf_prealc in ccp^.flags) then exit; 	{ preserve }

		pred := cctxhead;
		if pred = ccp then begin
			cctxhead := ccp^.next;
			dispose(ccp);
			exit;
		end else while pred <> nil do begin
			if pred^.next = ccp then begin
				cctxhead := ccp^.next;
				dispose(ccp);
				exit;
			end;
			pred := pred^.next;
		end;
		fehler('release_cctx(): pred(cctx) not found');
	end;
end;

function alloc_ctx(var cc: ndfp_client_ctx; var x: nds_ctxptr): boolean;
var
	i: integer;
begin
	for i := 1 to n_ctx do if not ctxa[i].inuse then begin
		x := addr(ctxa[i]);
		x^.inuse := true;
		x^.subh := 0;
		x^.shandle := $7700 + i;	{ some funny nonzero value }
		x^.client := addr(cc);
		inc(cc.refcnt);
		alloc_ctx := true;
		exit;
	end;
	alloc_ctx := false;
end;

procedure free_ctx(var ctx: nds_ctx);
begin
	if not ctx.inuse then fehler('free_ctx(): ctx not in use');

	if not (ccf_inuse in ctx.client^.flags) then
		fehler('free_ctx(): cctx not in use');
	if ctx.client^.refcnt <= 0 then
		fehler('free_ctx(): cctx.refcnt <= 0');

	dec(ctx.client^.refcnt);

	ctx.inuse := false;
end;

function find_ctx(var n: ndfp_pkt; var cc: ndfp_client_ctx; var x: nds_ctxptr):
	 boolean;
var
	i: integer;
begin
	i := n.shandle - $7700; 			{ cf. alloc_ctx() }
	if ((i >= 1) and (i <= n_ctx)) and{_then}
	   ctxa[i].inuse and
	   (ctxa[i].client = addr(cc)) and
	   (ctxa[i].chandle = n.chandle) and
	   (ctxa[i].shandle = n.shandle) then begin
		x := addr(ctxa[i]);
		find_ctx := true;
		exit;
	end;
	n.rp_sts := NDSS_IVHANDLE;
	n.ndata := 0;
	find_ctx := false;
end;

{ Note: do_reset() and final_reset() below also know about ctxa[] }

function do_close(var ctx: nds_ctx; mode: word; newpp: pathstrptr): word;
var
	sts: word;
	save_fm: byte;
label
	done;
begin
	if ctx.client^.cname <> '' then write(ctx.client^.cname)
	else write(lanaddr2str(ctx.client^.macaddr));
	write('/',hex4(ctx.shandle),'/',hex4(ctx.chandle),
	      ' # CLOSE(mode=',hex2(mode));
	if (mode and NDCM_RENAME) <> 0 then write(',newpath="',newpp^,'"');
	writeln(')');

	if (ctx.accmode and (NDOM_DISK or NDOM_DRV)) <> 0 then begin
		sts := dsk2sts('close',dsk_close(ctx.subh));
		goto done;
	end;

	if (ctx.accmode and NDOM_DIR) <> 0 then begin
		sts := ndir_close(ctx.subh,mode and NDCM_DELETE);
		if (mode and NDCM_DELETE) <> 0 then goto done;
	end else begin	{ _ignore_ FTIME for directory }
		sts := NDSS_OK;
		if (ctx.accmode and NDOM_FTIME) <> 0 then begin
			if (ctx.accmode and NDOM_WRITE) <> 0 then begin
				{ close & re-open read-only, so "ftime" sticks }
				save_fm := filemode;
				filemode := 0;	{ read-only }
				{$I-}
				reset(ctx.f,1);
				{$I+}
				filemode := save_fm;
				if io_failed('re-opening file') then
					sts := NDSS_OPENERR;
			end;
			if sts = NDSS_OK then begin
				setftime(ctx.f,ctx.new_ftime);
				sts := dos_sts('setftime');
			end;
		end;

		{$I-}
		Close(ctx.f);
		{$I+}
		if io_failed('closing file') then nop;
	end;

	if (sts = NDSS_OK) and ((ctx.accmode and NDOM_FATTR) <> 0) then begin
		{ file or directory }
		setfattr(ctx.f,ctx.new_fattr);
		sts := dos_sts('setfattr');
	end;

	if (sts = NDSS_OK) and ((mode and NDCM_DELETE) <> 0) then begin
		{ can't get here for a directory }
		{$I-}
		erase(ctx.f);
		{$I+}
		if io_failed('erase') then sts := NDSS_NOTDEL;
	end;

	if (sts = NDSS_OK) and ((mode and NDCM_RENAME) <> 0) then begin
		{ file or directory }
		{$I-}
		rename(ctx.f,newpp^);
		{$I+}
		if io_failed('rename') then sts := NDSS_NOTREN;
	end;
done:
	do_close := sts;
	free_ctx(ctx);
end;

procedure do_reset(var cc: ndfp_client_ctx; free_cc: boolean; p_ch: wordptr);
var
	i: integer;
begin
	if p_ch <> nil then for i := 1 to n_ctx do begin
		if ctxa[i].inuse and
		   (ctxa[i].client = addr(cc)) and
		   ((p_ch^ = 0) or (ctxa[i].chandle = p_ch^)) then begin
			if do_close(ctxa[i],0,nil) <> NDSS_OK then nop;
			{ always implies free_ctx() }
		end;
	end;

	if free_cc and (cc.refcnt = 0) then begin
		{ mark cctx for deallocation }
		cc.flags := cc.flags - [ccf_inuse];
	end;
end;

procedure final_reset; far;	{ exit handler }
const
	k_0: word = 0;
var
	ccp: ndfp_client_ctxptr;
begin
	ccp := cctxhead;
	while ccp <> nil do begin
		do_reset(ccp^,true,addr(k_0));
		ccp := ccp^.next;
	end;
end;

procedure ndfp_send(var cctx: ndfp_client_ctx; var w: ndfp_buffer);
var
	wlen: word;
begin
	{ always set "REPLY" bit }
	w.n.op := w.n.op or NDFP_OP_M_REPLY;

	{ compute packet length }
	wlen := sizeof(w.hdr) +
		max(NDFP_PKT_MINLEN,NDFP_HDRSIZE + w.n.ndata);

	{ fill in header }
	w.hdr.dst := cctx.macaddr;
	w.hdr.src := eth_lcladdr;
	w.hdr.pty := ndfp_ptype.w;

{$IFDEF DUMP_PKT}
	writeln(asctime,' Reply, l=',wlen);
	dump_16(addr(w),min(wlen,sizeof(eth_header) + NDFP_HDRSIZE));
	{ dump_16(addr(w),wlen); }
{$ENDIF}

	if not eth_send(addr(w),wlen) then begin
		writeln('?eth_send() error');
		halt(1);
	end;
end;

function do_seek(var ctx: nds_ctx; blkoff: longint; bytoff: word): word;
var
	sts: word;
	dskoff: dsk_t_off;
	newoff: longint;
begin
	do_seek := NDSS_OK;

	if (ctx.accmode and (NDOM_DISK or NDOM_DRV)) <> 0 then begin
		dskoff.boff := bytoff;
		dskoff.lbn := blkoff;
		do_seek := dsk2sts('seek',dsk_seek(ctx.subh,dskoff));
	end else begin
		newoff := off32(blkoff,bytoff);
		if newoff <> filepos(ctx.f) then begin
			seek(ctx.f,newoff);
{$IFDEF NOTDEF}
			do_seek := dos_sts('seek');
{$ELSE}
			if filepos(ctx.f) <> newoff then begin
				writeln('??? seek() failed, off=',newoff,
					', pos=',filepos(ctx.f));
				do_seek := NDSS_DOSFAIL;
			end;
{$ENDIF}
		end;
	end;
end;

procedure proc_open(var n: ndfp_pkt; var cc: ndfp_client_ctx);
var
	x: nds_ctxptr;
	save_fm: byte;
	dskdev: dsk_t_dev;
	fsize32: longint;
	path,epath: pathstr;
label
	open_ok, open_failed, open_done;
begin
	epath := '';

	if (n.chandle = 0) or (n.shandle <> 0) then begin
		n.rp_sts := NDSS_IVHANDLE;
		goto open_done;
	end;
	if	{ undefined mode bits }
	   ((n.w1.rq_open_mode and not
	     (NDOM_READ or NDOM_WRITE or NDOM_FATTR or NDOM_FTIME or
	      NDOM_DISK or NDOM_DRV or NDOM_DIR)) <> 0) or
		{ attr/time for read-only file or disk }
	   (((n.w1.rq_open_mode and (NDOM_FATTR or NDOM_FTIME)) <> 0) and
	    (((n.w1.rq_open_mode and (NDOM_READ or NDOM_WRITE)) = NDOM_READ) or
	     ((n.w1.rq_open_mode and (NDOM_DISK or NDOM_DRV)) <> 0))) or
		{ both DISK and DRV }
	   ((n.w1.rq_open_mode and (NDOM_DISK or NDOM_DRV)) =
	    (NDOM_DISK or NDOM_DRV)) then begin
		n.rp_sts := NDSS_IVPARAM;
		goto open_done;
	end;
	if (n.ndata > (sizeof(path) - 1)) then begin
		{ bad "path" length }
		n.rp_sts := NDSS_IVDATALEN;
		goto open_done;
	end;

	do_reset(cc,false,addr(n.chandle));
	{ if this changed anything, alloc_ctx() shouldn't fail }
	if not alloc_ctx(cc,x) then begin
		n.rp_sts := NDSS_NOROOM;
		goto open_done;
	end;

	x^.chandle := n.chandle;
	x^.accmode := n.w1.rq_open_mode;
	x^.new_fattr := n.w2.rq_open_fattr;
	x^.new_ftime := n.l1.rq_open_ftime;
	move(n.data,path[1],n.ndata);
	path[0] := chr(n.ndata);	{ set string length }

	if cc.cname <> '' then log_id := cc.cname
	else log_id := lanaddr2str(cc.macaddr);
	log_id := log_id + '/' + hex4(x^.shandle) + '/' + hex4(n.chandle);

{$IFDEF SHOW_EPATH}
	write(log_id,' # OPEN(mode=',hex2(x^.accmode),',path="',path,'"');
{$ELSE}
	writeln(log_id,' # OPEN(mode=',hex2(x^.accmode),',path="',path,'")');
{$ENDIF}

	if (x^.accmode and (NDOM_DISK or NDOM_DRV)) <> 0 then begin
		if (x^.accmode and NDOM_DISK) <> 0 then begin
			n.rp_sts := path2dsk(path,dskdev);
			if n.rp_sts <> NDSS_OK then begin
{$IFDEF SHOW_EPATH}
				writeln(')');
{$ENDIF}
				goto open_failed;
			end;
			case dskdev.devcls of
			  dsk_k_fd:	epath := '{Diskette';
			  dsk_k_hd:	epath := '{Harddisk';
			end;
			epath := epath + ' ' + hex2(dskdev.unitno) + 'h}';
		end else begin
			if length(path) <> 1 then begin
				n.rp_sts := NDSS_IVDATALEN;
{$IFDEF SHOW_EPATH}
				writeln(')');
{$ENDIF}
				goto open_failed;
			end;
			dskdev.devcls := dsk_k_drv;
			dskdev.letter := upcase(path[1]);
			epath := '{Drive ' + dskdev.letter + ':}';
		end;
{$IFDEF SHOW_EPATH}
		writeln('->',epath,')');
{$ENDIF}

		n.w1.rp_open_fattr := 0;
		n.w3.rp_open_fsizebyt := 0;
		n.rp_sts := dsk2sts('open',dsk_open(dskdev,x^.subh,
						    n.l0.rp_open_fsizeblk,
						    n.l1.rp_opendsk_geom));
		if n.rp_sts <> NDSS_OK then goto open_failed;

		{ n.l1.rp_open_ftime *overlaid* by "geometry" longword }

		goto open_ok;
	end;

	if (x^.accmode and NDOM_DIR) <> 0 then begin
		{ NDFP directory names never have a trailing delimiter, }
		{ but root directories aren't recognized as such without. }
		{ Append '\' prior to fexpand(), and remove it afterwards }
		epath := fexpand(path + '\');
		if length(epath) > 0 then dec(epath[0]);
						{ shorten string by 1 }
	end else begin
		{ file }
		epath := fexpand(path);
	end;

	{ NOTE: Apparently, as of Turbo Pascal V6, fexpand() isn't really
	{	the DOS "truename" function. It seems to only prepend drive
	{	and directory as needed (however correctly resolving 
	{	"drive:file.ext"), and truncates the result to fit a `pathstr'.
	 }

{$IFDEF NOTDEF}
	n.rp_sts := dos_sts('fexpand');	{ DOSerror not set by fexpand()! }
	if n.rp_sts <> NDSS_OK then begin
{$IFDEF SHOW_EPATH}
		writeln(')');
{$ENDIF}
		epath := '';
		goto open_failed;
	end;
{$ENDIF}
{$IFDEF SHOW_EPATH}
	writeln('->"',epath,'")');
{$ENDIF}

	assign(x^.f,epath);	{ file or directory(!) }

	if (x^.accmode and NDOM_DIR) <> 0 then begin
		n.rp_sts := ndir_open(epath,x^.accmode and
				       (NDOM_DIR or NDOM_READ or NDOM_WRITE),
				      x^.subh,n.w1.rp_open_fattr,
				      n.l1.rp_open_ftime);
		n.l0.rp_open_fsizeblk := 0;
		n.w3.rp_open_fsizebyt := 0;
		if n.rp_sts = NDSS_OK then goto open_ok else goto open_failed;
	end;

	if ((x^.accmode and NDOM_READ) = 0) and
	   ((x^.accmode and NDOM_WRITE) <> 0) then begin
		{ write-only -> create file }
		n.rp_sts := ndir_lookup(epath,n.w1.rp_open_fattr,
					fsize32,n.l1.rp_open_ftime);
		n.l0.rp_open_fsizeblk := fsize32 shr 9;
		n.w3.rp_open_fsizebyt := fsize32 and ((1 shl 9) - 1);

		case n.rp_sts of
		  NDSS_IVPATH: goto open_failed;
		  NDSS_OK: if (n.w1.rp_open_fattr and (Readonly or Sysfile or
					   Volumeid or Directory)) <> 0 then
			begin
				n.rp_sts := NDSS_NOACCESS;
				goto open_failed;
			end;
		end;

		n.rp_sts := NDSS_OK;
		{$I-}
		rewrite(x^.f,1);
		{$I+}
		if io_failed('creating file') then begin
			n.rp_sts := NDSS_CREATERR;
			goto open_failed;
		end;

		{ always allow for reading }
		x^.accmode := x^.accmode or NDOM_READ;

		n.w1.rp_open_fattr := 0;
		n.l1.rp_open_ftime := 0;
		n.l0.rp_open_fsizeblk := 0;
		n.w3.rp_open_fsizebyt := 0;
	end else begin
		{ open existing file }
		n.rp_sts := ndir_lookup(epath,n.w1.rp_open_fattr,
					fsize32,n.l1.rp_open_ftime);
		n.l0.rp_open_fsizeblk := fsize32 shr 9;
		n.w3.rp_open_fsizebyt := fsize32 and ((1 shl 9) - 1);

		if n.rp_sts <> NDSS_OK then goto open_failed;
		if (n.w1.rp_open_fattr and (Volumeid or Directory)) <> 0
		   then begin
			n.rp_sts := NDSS_NOACCESS;
			goto open_failed;
		end;

		save_fm := filemode;
		{ 0: read-only, 1: write-only, 2(default): read/write }
		if (x^.accmode and NDOM_WRITE) = 0 then filemode := 0;
		{$I-}
		reset(x^.f,1);
		{$I+}
		filemode := save_fm;
		if io_failed('opening file') then begin
			n.rp_sts := NDSS_OPENERR;
			goto open_failed;
		end;
	end;
open_ok:
	n.shandle := x^.shandle;
	goto open_done;

open_failed:
	free_ctx(x^);
	{ no need to preserve reply }
	do_reset(cc,true,nil);
open_done:
	n.ndata := length(epath);
	move(epath[1],n.data,n.ndata);
end;

procedure proc_close(var n: ndfp_pkt; var cc: ndfp_client_ctx);
var
	x: nds_ctxptr;
	newpath: pathstr;
label
	closeit,close_done;
begin
	newpath := '';

	if not find_ctx(n,cc,x) then exit;

	if	{ undefined mode bits }
	   ((n.w1.rq_close_mode and not
	     (NDCM_DELETE or NDCM_RENAME or NDCM_FATTR or NDCM_FTIME)) <> 0) or
		{ both DELETE and RENAME (FATTR is o.k. with DELETE!) }
	   (((n.w1.rq_close_mode and NDCM_DELETE) <> 0) and
	    ((n.w1.rq_close_mode and NDCM_RENAME) <> 0)) or
		{ any modifier with read-only file or disk }
	   ((n.w1.rq_close_mode <> 0) and
	    (((x^.accmode and (NDOM_READ or NDOM_WRITE)) = NDOM_READ) or
	     ((x^.accmode and (NDOM_DISK or NDOM_DRV)) <> 0))) then begin
		n.rp_sts := NDSS_IVPARAM;
		goto close_done;
	end;

	if (n.w1.rq_close_mode and NDCM_RENAME) <> 0 then begin
		if (n.ndata > (sizeof(newpath) - 1)) then begin
			{ bad "path" length }
			n.rp_sts := NDSS_IVDATALEN;
			goto close_done;
		end;

		move(n.data,newpath[1],n.ndata);
		newpath[0] := chr(n.ndata);	{ set string length }

		{ expect a fully qualified path here, no shortcuts }
		{ "related file parsing" is left to client }

		{ cf. NOTE in proc_open()! }
		newpath := fexpand(newpath);
	end;	

	{ ggf. override fattr/ftime specified earlier }
	if (n.w1.rq_close_mode and NDCM_FATTR) <> 0 then begin
		x^.new_fattr := n.w2.rq_close_fattr;
		x^.accmode := x^.accmode or NDOM_FATTR;
	end;
	if (n.w1.rq_close_mode and NDCM_FTIME) <> 0 then begin
		x^.new_ftime := n.l1.rq_close_ftime;
		x^.accmode := x^.accmode or NDOM_FTIME;
	end;

	n.rp_sts := do_close(x^,n.w1.rq_close_mode,addr(newpath));

	{ don't do_reset() since CLOSE can't be repeated }
close_done:
	n.ndata := length(newpath);
	move(newpath[1],n.data,n.ndata);
end;


procedure proc_time(var n: ndfp_pkt; var cc: ndfp_client_ctx);
var
	nt: t_numtim;
	l0,l1: longint;
label
	time_done;
begin
	if	{ undefined mode bits }
	   ((n.w1.rq_time_mode and not (NDTM_ADD or NDTM_SET)) <> 0) or
		{ both ADD and SET }
	   (((n.w1.rq_time_mode and NDTM_ADD) <> 0) and
	    ((n.w1.rq_time_mode and NDTM_SET) <> 0)) then begin
		n.rp_sts := NDSS_IVPARAM;
		goto time_done;
	end;

	n.rp_sts := NDSS_OK;	{ assume success }

	if (n.w1.rq_time_mode and NDTM_ADD) <> 0 then begin
		adjtim_64b(n.l0.rq_time_q_lo,n.l1.rq_time_q_hi);
	end else if (n.w1.rq_time_mode and NDTM_SET) <> 0 then begin
		cvt_64b1980_numtim(n.l0.rq_time_q_lo,n.l1.rq_time_q_hi,nt);
		if not set_numtim(nt) then n.rp_sts := NDSS_WRITERR;
	end;

	get_numtim(nt);
	cvt_numtim_64b1980(nt,n.l0.rp_time_q_lo,n.l1.rp_time_q_hi);

time_done:
	{ return n.data & n.ndata unchanged }
end;


function nds_process(var cc: ndfp_client_ctx; var n: ndfp_pkt;
		     idatap,odatap: byteptr): word;	{ # "odata" bytes }
var
	x: nds_ctxptr;
	ct1, ct2: word;
	odp: byteptr;
label
	read_done, write_done;
begin
	nds_process := 0;	{ default }

	case n.op of
	  NDFP_OP_RESET: begin
		do_reset(cc,true,addr(n.chandle));
		n.rp_sts := NDSS_OK;
		if NDFP_VERSION = NDFP_VERSION1 then begin
			n.w1.rp_reset_vers := NDFP_VERSION;
		end else begin
			n.w1.rp_reset_vers := NDFP_VERSION + winsiz;
		end;
			end;

	  NDFP_OP_OPEN: proc_open(n,cc);

	  NDFP_OP_READ: begin
		if not find_ctx(n,cc,x) then exit;
		if n.w1.rq_read_count > (winsiz * NDFP_DATA_MAX) then begin
			n.rp_sts := NDSS_IVDATALEN;
			exit;
		end;

		n.ndata := 0;
		if n.w1.rq_read_count > NDFP_DATA_MAX then
			odp := odatap
		else
			odp := addr(n.data);

		if (x^.accmode and NDOM_READ) = 0 then begin
			n.rp_sts := NDSS_IVMODE;
			exit;
		end;

		if log_all then writeln(log_id,' READ ',n.w1.rq_read_count:6,
					 ' @',n.l1.rq_read_offsetblk:11,
					 '.',j4(n.w3.rq_read_offsetbyt));

		if (x^.accmode and NDOM_DIR) <> 0 then begin
			n.rp_sts := ndir_read(x^.subh,
					      off32(n.l1.rq_read_offsetblk,
						    n.w3.rq_read_offsetbyt),
					      n.w1.rq_read_count,odp^,n.ndata);
			goto read_done;
		end;

		n.rp_sts := do_seek(x^,n.l1.rq_read_offsetblk,
				       n.w3.rq_read_offsetbyt);

		if (x^.accmode and (NDOM_DISK or NDOM_DRV)) <> 0 then begin
			if (n.rp_sts = NDSS_OK) and (n.w1.rq_read_count <> 0)
			   then begin
				n.rp_sts := dsk2sts('read',dsk_read(x^.subh,
						    n.w1.rq_read_count,odp^,
						    n.ndata));
			end;
			goto read_done;
		end;

		if (n.rp_sts = NDSS_OK) and
		   (off32(n.l1.rq_read_offsetblk,n.w3.rq_read_offsetbyt) >
		    filesize(x^.f)) then
			n.rp_sts := NDSS_EOF;
		if (n.rp_sts = NDSS_OK) and (n.w1.rq_read_count <> 0) then begin
			ct1 := n.w1.rq_read_count;
			if filesize(x^.f) < (filepos(x^.f) + ct1) then
				ct1 := filesize(x^.f) - filepos(x^.f);
			{$I-}
			blockread(x^.f,odp^,n.w1.rq_read_count,ct2);
			{$I+}
			if io_failed('blockread') or
			   (ct2 <> ct1) then n.rp_sts := NDSS_READERR;
			n.ndata := ct2;
		end;
read_done:
		n.w2.rp_read_boff := 0;
		n.w1.rp_read_count := n.ndata;
		if odp = odatap then nds_process := n.ndata;
			end;

	  NDFP_OP_WRITE: begin
		if not find_ctx(n,cc,x) then goto write_done;
		if ((n.nfrag = 0) and (n.ndata > NDFP_DATA_MAX)) or
		   ((n.nfrag <> 0) and
		    (n.w1.rq_write_count > (winsiz * NDFP_DATA_MAX))) then begin
			n.rp_sts := NDSS_IVDATALEN;
			goto write_done;
		end;
		if ((x^.accmode and NDOM_DIR) <> 0) or
		   ((x^.accmode and NDOM_WRITE) = 0) then begin
			n.rp_sts := NDSS_IVMODE;
			goto write_done;
		end;

		if n.nfrag = 0 then
			ct1 := n.ndata
		else
			ct1 := n.w1.rq_write_count;

		if log_all then writeln(log_id,' WRITE',n.w1.rq_write_count:6,
					 ' @',n.l1.rq_write_offsetblk:11,
					 '.',j4(n.w3.rq_write_offsetbyt));

		n.w1.rp_write_count := 0;

		n.rp_sts := do_seek(x^,n.l1.rq_write_offsetblk,
				       n.w3.rq_write_offsetbyt);

		if (n.rp_sts = NDSS_OK) and (ct1 <> 0) then begin
			if (x^.accmode and (NDOM_DISK or NDOM_DRV)) <> 0
			   then begin
				n.rp_sts := dsk2sts('write',dsk_write(x^.subh,
						    ct1,idatap^,
						    n.w1.rp_write_count));
				goto write_done;
			end;

			{$I-}
			blockwrite(x^.f,idatap^,ct1,ct2);
			{$I+}
			if io_failed('blockwrite') or
			   (ct2 <> ct1) then n.rp_sts := NDSS_WRITERR;
			n.w1.rp_write_count := ct2;
		end;
write_done:
		n.ndata := 0;
			end;

	  NDFP_OP_TRUNC: begin
		if not find_ctx(n,cc,x) then exit;
		if ((x^.accmode and NDOM_WRITE) = 0) or
		   ((x^.accmode and (NDOM_DISK or NDOM_DRV or NDOM_DIR)) <> 0)
		   then begin
			n.rp_sts := NDSS_IVMODE;
			exit;
		end;
		n.rp_sts := do_seek(x^,n.l1.rq_trunc_offsetblk,
				       n.w3.rq_trunc_offsetbyt);
		if n.rp_sts = NDSS_OK then begin
			truncate(x^.f);
{$IFDEF NOTDEF}
			n.rp_sts := dos_sts('truncate');
{$ELSE}

			if filesize(x^.f) <> off32(n.l1.rq_trunc_offsetblk,
						   n.w3.rq_trunc_offsetbyt)
			   then begin
				writeln('??? truncate() failed, off=',
					off32(n.l1.rq_trunc_offsetblk,
					      n.w3.rq_trunc_offsetbyt),
					', size=',filesize(x^.f));
				n.rp_sts := NDSS_DOSFAIL;
			end;
{$ENDIF}
		end;
			end;

	  NDFP_OP_CLOSE: proc_close(n,cc);

	  NDFP_OP_TIME: proc_time(n,cc);

	else begin
		n.rp_sts := NDSS_UNSUP;
		do_reset(cc,true,nil);	{ avoid denial-of-service }
			end;
	end {case};
end;


procedure process_rq(var cc: ndfp_client_ctx);
var
	rq: ndfp_pktptr;
	dboff: word;
	dbct: word;
	i: integer;
	datb: array[0..((winsiz * NDFP_DATA_MAX) - 1)] of byte;
label
	nds_done,ivproto;
begin
	{ point to _last_ rq packet }
	rq := addr(cc.rqa[cc.rqhi]^.n);

	{ multiple input packets? }
	if cc.rqhi > 0 then begin
		{ function must be WRITE }
		if rq^.op <> NDFP_OP_WRITE then goto ivproto;

		{ correct number of packets? }
		if (cc.rqhi + 1) <>
		   ((rq^.w1.rq_write_count + NDFP_DATA_MAX - 1) div
		    NDFP_DATA_MAX) then goto ivproto;

		{ data expected in reverse order! }
		{ verify packets & collect data into "datb" }
		dboff := 0;
		for i := cc.rqhi downto 0 do with cc.rqa[i]^.n do begin
			if not ((chandle = rq^.chandle) and
				(shandle = rq^.shandle) and
				(op = rq^.op) and
				(w1.rq_write_count = rq^.w1.rq_write_count) and
				(w2.rq_write_boff = dboff) and
				(l1.rq_write_offsetblk =
				 rq^.l1.rq_write_offsetblk) and
				(w3.rq_write_offsetbyt =
				 (rq^.w3.rq_write_offsetbyt + dboff)) and
				(ndata = min(NDFP_DATA_MAX,
					     rq^.w1.rq_write_count - dboff))
			       ) then
				goto ivproto;
			move(data,datb[dboff],ndata);
			inc(dboff,ndata);
		end;

		{ ok }
		dbct := nds_process(cc,rq^,addr(datb),addr(datb));
	end else
		dbct := nds_process(cc,rq^,addr(rq^.data),addr(datb));
nds_done:
	{ "rq^" has been transformed into 1st output packet,
	{ except for "nfrag" & "ifrag",
	{ and with "dbct" non-zero, "datb" has the data }

	{ move primary buffer into 1st output slot - "rq" still valid }
	if cc.rpa[0] <> nil then dispose(cc.rpa[0]);
	cc.rpa[0] := cc.rqa[cc.rqhi]; cc.rqa[cc.rqhi] := nil;

	if dbct = 0 then
		cc.rphi := 0
	else begin
		cc.rphi := ((dbct + NDFP_DATA_MAX - 1) div NDFP_DATA_MAX) - 1;

		if cc.rphi = 0 then begin
			{ simply move data }
			move(datb,rq^.data,dbct);
			rq^.ndata := dbct;
		end;
	end;

	rq^.ifrag := 0;
	rq^.nfrag := cc.rphi;

	if log_all then
		writeln(log_id,' --> cseq=',rq^.cseq,
			' nfrag=',rq^.nfrag,
			' sts=',hex4(rq^.rp_sts));

	if cc.rphi > 0 then begin	{ multi-packet output (dbct <> 0) }

		if cc.rphi >= winsiz then fehler('process_rq(): too much data');

		{ function must be READ }
		if (rq^.op and not NDFP_OP_M_REPLY) <> NDFP_OP_READ then
			fehler('process_rq(): multi-output, op <> READ');

		rq^.w1.rp_read_count := dbct;	{ [?] }

		dboff := 0;
		for i := 0 to cc.rphi do begin
			if cc.rpa[i] = nil then new(cc.rpa[i]);
			if i <> 0 then move(cc.rpa[0]^.n,
					    cc.rpa[i]^.n,NDFP_HDRSIZE);
			with cc.rpa[i]^.n do begin
				ifrag := i;
				w2.rp_read_boff := dboff;
				ndata := min(NDFP_DATA_MAX,dbct - dboff);
				move(datb[dboff],data,ndata);

				inc(dboff,ndata);
			end;

			ndfp_send(cc,cc.rpa[i]^);
		end;
	end else
		ndfp_send(cc,cc.rpa[0]^);
	exit;

ivproto:	{ fake error return from nds_process() }
	rq^.rp_sts := NDSS_IVPROTO;
	rq^.ndata := 0;
	dbct := 0;
	goto nds_done;
end;


procedure init_permit;		{ set up access restrictions from NDS_PERMIT }
var
	addr: lanaddr;
	name: permit_name_str;
	n: integer;
	ccp: ndfp_client_ctxptr;
begin
	using_permit := false;		{ enable cctx creation }

	if permit_open then begin	{ i.e. if NDS_PERMIT defined }
		n := 0;
		while permit_read(addr,name) do begin
			inc(n);
			writeln('Client',n:3,': ',lanaddr2str(addr),' ',name);
			if get_cctx(addr,ccp) then begin
				ccp^.flags := ccp^.flags + [ccf_prealc];
				ccp^.cname := name;
			end else
				fehler('Sorry, too many PERMIT clients');
		end;
		using_permit := true;	{ additional clients not accepted }
	end;
end;


const
	rcvmin = winsiz + 1;
	rcvmax = 10;		{ rcvmin .. 16 }
var
	rcvcount: integer;

procedure feed_rcv;
var
	bp: ndfp_bufferptr;
begin
	while rcvcount < rcvmin do begin
		new(bp);
		eth_feed_rcv(bp,sizeof(bp^));
		inc(rcvcount);
	end;
end;

procedure dispose_bufp(var bp: ndfp_bufferptr);
begin
	if rcvcount < rcvmax then begin
		eth_feed_rcv(bp,sizeof(bp^));
		inc(rcvcount);
	end else
		dispose(bp);
	bp := nil;
end;


{ for the benefit of bridges & switches, send a dummy packet to ourselves }
{ ??? do we have to send a broadcast packet [90-00 to CF-00-00-00-00-00] ??? }
procedure send_dummy;
var
	npk: ndfp_buffer;
begin
	npk.hdr.dst := eth_lcladdr;
	npk.hdr.src := eth_lcladdr;
	npk.hdr.pty := ndfp_ptype.w;
	npk.n.nfrag := 0;
	npk.n.ifrag := 0;
	npk.n.op := NDFP_OP_M_REPLY;	{ will always be ignored }
	npk.n.ndata := 0;

	if not eth_send(addr(npk),sizeof(npk.hdr) +
			max(NDFP_PKT_MINLEN,NDFP_HDRSIZE)) then
		writeln('?eth_send(dummy) error');
end;

procedure maybe_send_dummy;	{ ... if the time is right ... }
const
	seconds = 100;		{ repeat interval }
var
	hh,mm,ss,cc: word;
const		{ initialized variables }
	nsec: word = seconds;	{ trigger at 1st invocation, via ... }
	prev_ss: word = 61;	{ ... some non-second value here }
begin
	gettime(hh,mm,ss,cc);
	if ss <> prev_ss then begin
		prev_ss := ss;
		inc(nsec);
		if nsec >= seconds then begin
			nsec := 0;
			send_dummy;
		end;
	end;
end;

{ send special "NAK reject" packet to get client out of possible NAK loop }
procedure reject_nak(var cc: ndfp_client_ctx; rcp: ndfp_bufferptr);
var
	nak: ndfp_buffer;
begin
	writeln(log_id,' --> NAKREJ');
	move(rcp^.n,nak.n,NDFP_HDRSIZE);
	nak.n.nfrag := 0;
	nak.n.ifrag := 0;
	nak.n.op := (nak.n.op or NDFP_OP_M_REPLY) and (not NDFP_OP_M_NAK);
	nak.n.ndata := 0;
	nak.n.rp_sts := NDSS_IVNAK;
	{ paranoia }
	nak.n.w1.rp_read_count := 0;
	nak.n.w2.rp_read_boff := 0;
	nak.n.w3.rp_open_fsizebyt := 0;
	nak.n.l0.rp_open_fsizeblk := 0;
	nak.n.l1.rp_open_ftime := 0;

	ndfp_send(cc,nak);
end;

{ send NAK REPLY, asking client to retransmit some piece(s) of its request }
procedure send_nak(var cc: ndfp_client_ctx; rcp: ndfp_bufferptr);
var
	nak: ndfp_buffer;
begin
	writeln(log_id,' --> NAK');
	move(rcp^.n,nak.n,NDFP_HDRSIZE);
	nak.n.ifrag := cc.rqmask;
	nak.n.op := nak.n.op or NDFP_OP_M_NAK or NDFP_OP_M_REPLY;
	nak.n.ndata := 0;
	ndfp_send(cc,nak);
end;

{ reset client transaction state }
procedure reset_cts(var cc: ndfp_client_ctx);
var
	i: integer;
begin
	with cc do begin

		{ free send & receive slots }
		for i := 0 to rphi do if rpa[i] <> nil then
			dispose_bufp(rpa[i]);
		for i := 0 to rqhi do if rqa[i] <> nil then
			dispose_bufp(rqa[i]);

		rphi := -1;
		rqhi := -1;

		tstate := cts_idle;
	end;
end;

var
	i: integer;
	rcp: ndfp_bufferptr;
	rlen: word;
	ccp: ndfp_client_ctxptr;
	stats: eth_pktd_stats;
label
	mainloop,
	new_input,multi_input,do_process,
	discard_rcp_e,discard_rcp;
begin
	writeln('NDFP Server ',version,' (c) w.j.m 1997');

	i := eth_startup2(pktdrv_lo,pktdrv_hi);
	if i = 0 then begin
		writeln('No packet driver within range 0x',
			hex2(pktdrv_lo),' .. 0x',hex2(pktdrv_hi));
		halt(1);
	end;
	writeln('using packet driver at 0x',hex2(abs(i)));
	if i < 0 then begin
		writeln('eth_startup() failed');
		halt(1);
	end;

	if not eth_start_rcv(ndfp_ptype.b,2,0) then begin
		writeln('eth_start_rcv("',
			hex2(ndfp_ptype.b[0]),'-',hex2(ndfp_ptype.b[1]),
			'",2,0) failed');
		halt(1);
	end;

	writeln('Local server address: ',lanaddr2str(eth_lcladdr));

	init_contexts;
	userex(final_reset);

	init_permit;

	rcvcount := 0;
	feed_rcv;

mainloop:
	repeat begin
		if keypressed then case readkey of
		  'S','s',chr(13): begin
			if eth_statistics(stats) then with stats do
				writeln(asctime,
					'  p: ',packets_in,'/',packets_out,
					'  b: ',bytes_in,'/',bytes_out,
					'  e: ',errors_in,'/',errors_out,
					'  lost: ',packets_lost);
				end;
		  'L','l':	begin
			log_all := not log_all;
			writeln('[log_all=',log_all,']');
				end;
		  else
			exit;
		end;
		rcp := eth_receive(rlen);
		if rcp = nil then maybe_send_dummy;
	end until rcp <> nil;

	dec(rcvcount);

{$IFDEF DUMP_PKT}
	writeln(asctime,' Request, l=',rlen);
	dump_16(rcp,min(rlen,sizeof(eth_header) + NDFP_HDRSIZE));
	{ dump_16(rcp,rlen); }
{$ENDIF}

	log_id := '????/????';

	{ discard if bad length }
	if rlen < (sizeof(rcp^.hdr) +
		   max(NDFP_PKT_MINLEN,NDFP_HDRSIZE + rcp^.n.ndata)) then
		goto discard_rcp_e;

	{ silently discard if opcode has M_REPLY }
	if ((rcp^.n.op and NDFP_OP_M_REPLY) <> 0) then goto discard_rcp;

	{ find or create cctx based on enet-addr }
	if not get_cctx(rcp^.hdr.src,ccp) then begin
		writeln('?input from ',lanaddr2str(rcp^.hdr.src),
			' discarded: no cctx');			{?repeat?}
		goto discard_rcp;
	end;

	if ccp^.cname <> '' then log_id := ccp^.cname
	else log_id := lanaddr2str(ccp^.macaddr);
	log_id := log_id +
		  '/' + hex4(rcp^.n.shandle) + '/' + hex4(rcp^.n.chandle);

	case ccp^.tstate of
	  cts_idle:		{ no context }
		begin
			{ NAK not expected here }
			if (rcp^.n.op and NDFP_OP_M_NAK) <> 0 then begin
				reject_nak(ccp^,rcp);
				goto discard_rcp;
			end else begin
				if log_all then write(log_id,' 0<-');
				goto new_input;
			end;
		end;

	  cts_rqmulti:		{ partial request pending }
		begin
			{ NAK not expected here }
			if (rcp^.n.op and NDFP_OP_M_NAK) <> 0 then begin
				reject_nak(ccp^,rcp);
				reset_cts(ccp^);
				goto discard_rcp;
			end;

			{ reset if request mismatch }
			if (rcp^.n.cseq <> ccp^.cseq) or
			   (rcp^.n.op <> ccp^.op2) or
			   (rcp^.n.nfrag <> ccp^.rqhi) then begin
				{ mismatch, forget about current transaction }
				reset_cts(ccp^);
				if log_all then write(log_id,' ?<-');
				goto new_input;
			end;

			goto multi_input;
		end;

	cts_rpdone:		{ reply sent & ready for retransmission }
		begin
			{ retransmission request? }
			if (rcp^.n.op and NDFP_OP_M_NAK) <> 0 then begin

				{ reject NAK on mismatch }
				if (rcp^.n.cseq <> ccp^.cseq) or
				   ((rcp^.n.op and not NDFP_OP_M_NAK) <>
				    ccp^.op2) or
				   (rcp^.n.nfrag <> ccp^.rphi) then begin
					reject_nak(ccp^,rcp);
					reset_cts(ccp^);
					goto discard_rcp_e;
				end;

				{ NAK -> resend reply slots in ifrag mask }
				writeln(log_id,' 2<- NAK');
				for i := 0 to ccp^.rphi do begin
					if (rcp^.n.ifrag and (1 shl i)) <> 0
					   then
						ndfp_send(ccp^,ccp^.rpa[i]^);
				end;

			end else if (rcp^.n.cseq <> ccp^.cseq) or
				    (rcp^.n.op <> ccp^.op2) or
				    (rcp^.n.nfrag <> ccp^.rqhi) then begin
				if log_all then write(log_id,' 2<-');
				{ normal start of new transaction }
				reset_cts(ccp^);
				goto new_input;

			end else begin
				{ repeated request - resend 1st send slot }
				writeln(log_id,' 2<- retransmit');
				ndfp_send(ccp^,ccp^.rpa[0]^);
			end;
		end;
	end;
	goto discard_rcp;


	{ initiate new transaction }
new_input:	{ tstate == cts_idle }

	if log_all then
		writeln(' cseq=',rcp^.n.cseq,
			' nfrag=',rcp^.n.nfrag,
			' op=',hex4(rcp^.n.op));

	{ discard if nfrag is too large }
	if rcp^.n.nfrag >= winsiz then goto discard_rcp_e;

	{ establish cseq, op2, rqhi }
	ccp^.cseq := rcp^.n.cseq;
	ccp^.op2 := rcp^.n.op;		{ both M_REPLY and M_NAK are clear }
	ccp^.rqhi := rcp^.n.nfrag;

	{ with an unfragmented request, ignore ifrag (V1) & process right away }
	if rcp^.n.nfrag = 0 then begin
		if ccp^.rqa[0] <> nil then dispose_bufp(ccp^.rqa[0]);
		ccp^.rqa[0] := rcp;
		feed_rcv;
		goto do_process;	{ ==> cts_rpdone }
	end;

	{ init rqmask from nfrag }
	ccp^.rqmask := (1 shl (ccp^.rqhi + 1)) - 1;

	{ ==> cts_rqmulti }
	ccp^.tstate := cts_rqmulti;

	{ drop thru }

	{ handle fragmented request }
multi_input:	{ tstate == cts_rqmulti }

	{ discard fragment out of range }
	i := rcp^.n.ifrag;
	if i > ccp^.rqhi then goto discard_rcp_e;

	{ store in slot (superseding previous contents) }
	if ccp^.rqa[i] <> nil then dispose_bufp(ccp^.rqa[i]);
	ccp^.rqa[i] := rcp;
	feed_rcv;

	{ update rqmask }
	ccp^.rqmask := (ccp^.rqmask and not (1 shl i));

	{ do not react unless (ifrag = nfrag) }
	if rcp^.n.ifrag = rcp^.n.nfrag then begin

		{ if mask is zero, go process & reply }
		if ccp^.rqmask = 0 then goto do_process;

		{ NAK - ask for request retransmission }
		send_nak(ccp^,rcp);
	end;

	goto mainloop;	{ rcp saved }


	{ process request in rqa[0..(rqhi-1)] }
do_process:	{ tstate == cts_idle or tstate == cts_rqmulti }

	process_rq(ccp^);	{ send reply, fill in rphi }

	{ ==> cts_rpdone }
	ccp^.tstate := cts_rpdone;

	{ free request buffer(s), or maybe all of context }
	if not release_cctx(ccp) then begin
		if ccp^.op2 = NDFP_OP_RESET then begin
			reset_cts(ccp^);	{ don't remember RESET }
		end else for i := 0 to ccp^.rqhi do
			if ccp^.rqa[i] <> nil then dispose_bufp(ccp^.rqa[i]);
			{ and DON't reset ccp^.rqhi (used by retransmit logic) }
	end;
	goto mainloop;


	{ free input packet }
discard_rcp_e:
	writeln(log_id,' ?<- bad input');
discard_rcp:
	dispose_bufp(rcp);
	goto mainloop;
end.
