{ Copyright (C) 1997 Wolfgang J Moeller. All rights reserved. NO WARRANTY! }

unit ndir5io;

{ NDFP V5 directory operations
{
{ w.j.m. jan 1997 (NDIR_IO)
{ wjm 26-mar-1997: (NDIR5IO) remove `fattr' handling
{
{ Contrary to DOS practice, here we use (for consistency):
{	X:	root directory of drive X (NOT current directory)
{	X:\A	1st level subdirectory
{	X:\A\B	2nd level subdirectory
{	(etc.)
{
{ Directory specifications not in the above "absolute" format
{ are relative to "current" drive and/or directories; 
{ their use should be considered "unsupported" for now. 
}

interface

const
	NDIR_NAME_MAX = 8+1+3;
type
	ndir_t_record = record
		rlen: integer;		{ actual "record" length }
		struclev: word;		{ = 1 }
		ftime: longint;		{ file date/time }
		fsize: longint;		{ file size }
		fattr: word;		{ file attributes }
		nlen: integer;		{ length of file name }
		name: array[1..NDIR_NAME_MAX] of byte;
					{ file/dir name incl. period, ... }
					{ ... VolumeId without period }
	end;

{ state-less directory lookup (no wildcards [?]) }
function ndir_lookup(path: string;
		     var attr: word; var size,time: longint): word;

{ directory access via open/close }
function ndir_open(path: string; mode: word;
		   var hnd: byte; var oattr: word; var otime: longint): word;

function ndir_read(hnd: byte; offset: longint;
		   buflen: word; var buf; var retlen: word): word;

function ndir_close(hnd: byte; mode: word): word;


implementation
uses dos,wjmp,nd_const,nds_util;

const
	nctx = 3;	{ # client contexts = max.number of clients }

const
	NDIR_HDRLEN = sizeof(ndir_t_record) - NDIR_NAME_MAX;
						{ length of constant part }
type
	ndir_t_listptr = ^ndir_t_list;
	ndir_t_list = record
		next: ndir_t_listptr;
		loff: longint;		{ offset w/i virtual "directory file" }
		r: ndir_t_record;
	end;

	ndir_t_ctx = record
		inuse: boolean;
		fill_1: byte;
		openmode: word;
		list: ndir_t_listptr;	{ list of entries }
		lsize: longint;		{ length of virtual "directory file" }
		curlp: ndir_t_listptr;	{ nil, or "current" entry (shortcut) }
		dpath: pathstr;
	end;
	ndir_t_ctxptr = ^ndir_t_ctx;

var
	ctxa: array[1..nctx] of ndir_t_ctx;

function find_ctx(hnd: byte): ndir_t_ctxptr;
var
	h: byte;
begin
	h := hnd - $D0;			{ cf. ndir_open() below !! }

	if (h < 1) or
	   (h > nctx) or{_else}
	   not ctxa[h].inuse then
		find_ctx := nil
	else
		find_ctx := addr(ctxa[h]);
end;


function do_mkdir(p: string): word;
begin
	{$I-}
	mkdir(p);
	{$I+}
	if io_failed('mkdir') then
		do_mkdir := NDSS_CREATERR
	else
		do_mkdir := NDSS_OK;
end;

function do_rmdir(p: string): word;
begin
	{$I-}
	rmdir(p);
	{$I+}
	if io_failed('rmdir') then
		do_rmdir := NDSS_NOTDEL
	else
		do_rmdir := NDSS_OK;
end;


{ state-less directory lookup (no wildcards [?]) }
function ndir_lookup(path: string;
		     var attr: word; var size,time: longint): word;
var
	sr: searchrec;
label
	fake_dir;
begin
	{ current disk's root directory got to exist }
	if (length(path) = 0) then goto fake_dir;

	{ an existing disk's root directory should allow for lookup of '*.*' }
	if (length(path) = 2) and 
	   (path[2] = ':') and
	   (path[1] in ['A'..'Z','a'..'z']) then begin
		findfirst(path + '\*.*',AnyFile,sr);
		case doserror of
		  0,18:	goto fake_dir;
		end;

		{ assume invalid disk on any error }
		ndir_lookup := NDSS_IVPATH;
		exit;
	end;

	{ wildcards not allowed }
	if (pos('*',path) > 0) or
	   (pos('?',path) > 0) then begin
		ndir_lookup := NDSS_IVPATH;
		exit;
	end;

	{ should be able to "find" non-root directories & files }
	findfirst(path,AnyFile,sr);
	case doserror of
	  0:	begin
		attr := sr.attr;
		size := sr.size;
		time := sr.time;
		ndir_lookup := NDSS_OK;
		end;
	  3:				{ "path invalid" }
		ndir_lookup := NDSS_IVPATH;
	  5:				{ "no access" }
		ndir_lookup := NDSS_NOACCESS;
	  18:
		ndir_lookup := NDSS_FNF;
	else
		ndir_lookup := NDSS_DOSFAIL;
	end;
	exit;

fake_dir:	{ make up a "directory entry" for root directory }
	attr := Directory or SysFile;
	size := 0;
	time := 0;		{ do we know the date of FORMATting [?] }
	ndir_lookup := NDSS_OK;
end;


procedure make_rec(s: searchrec; var r: ndir_t_record);
begin
	{ void result for '.' and '..' }
	if (s.name = '.') or (s.name = '..') then begin
		r.rlen := 0;
	end else begin
		r.struclev := 1;
		r.ftime := s.time;
		r.fsize := s.size;
		r.fattr := s.attr;
		r.nlen := length(s.name);
		if (s.attr and VolumeId) <> 0 then begin	{ remove '.' }
			move(s.name[1],r.name[1],8);
			if r.nlen > 8 then begin
				move(s.name[10],r.name[9],3);
				dec(r.nlen);
			end;
		end else begin
			move(s.name[1],r.name[1],12);
		end;
		r.rlen := NDIR_HDRLEN + r.nlen;
	end;
end;

function make_list(var ctx: ndir_t_ctx): integer;
var
	sr: searchrec;
	p,q: ndir_t_listptr;
	doserr: word;
label
	next_entry;
begin
	{ read all of directory into dynamically allocated list }
	
	{ *assume* ctx.list = nil, ctx.curlp = nil, ctx.lsize = 0 }
	p := nil;

	findfirst(ctx.dpath + '\*.*',AnyFile,sr);
	doserr := doserror;
	case doserr of
	  0:				{ some file found }
		goto next_entry;		{ continue below }
	  5:
		make_list := NDSS_NOACCESS;
	  18: 				{ no file found }
		make_list := NDSS_OK;
	else				{ ??? }
		make_list := NDSS_DOSFAIL;
	end;
	exit;

next_entry:
	if maxavail < sizeof(q^) then begin
		make_list := NDSS_NOROOM;	{ out of dynamic memory }	
		exit;			{ caller is supposed to clean up }
	end;

	new(q);		{ no error expected }
	q^.next := nil;
	make_rec(sr,q^.r);

	{ NOTE: some memory could be saved by only allocating RLEN bytes [?] }

	q^.loff := ctx.lsize;
	inc(ctx.lsize,q^.r.rlen);

	if p = nil then ctx.list := q else p^.next := q;
	p := addr(q^.next);

	findnext(sr);
	doserr := doserror;
	case doserr of
	  0:					{ more work to do }
		goto next_entry;
	  18:					{ end of directory }
		make_list := NDSS_OK;
	else					{ ??? }
		make_list := NDSS_DOSFAIL;
	end;
end;

procedure free_list(var ctx: ndir_t_ctx);
var
	p,q: ndir_t_listptr;
begin
	{ release memory allocated by make_list() }

	p := ctx.list;

	ctx.list := nil;
	ctx.curlp := nil;
	ctx.lsize := 0;

	while p <> nil do begin
		q := p^.next;
		dispose(p);
		p := q;
	end;
end;


function ndir_open(path: string; mode: word;
		   var hnd: byte; var oattr: word; var otime: longint): word;
var
	h: byte;
	x: ndir_t_ctxptr;
	sts: word;
	tmpsize: longint;
begin
	hnd := 0;

	x := nil;
	h := 0; repeat
		inc(h);
		if not ctxa[h].inuse then x := addr(ctxa[h]);
	until (x <> nil) or (h >= nctx);
	if x = nil then begin
		ndir_open := NDSS_NOROOM;
		exit;
	end;

	x^.openmode := mode and (NDOM_READ or NDOM_WRITE);
	if (mode and not x^.openmode) <> NDOM_DIR then begin
		ndir_open := NDSS_IVPARAM;
		exit;
	end;

	if (length(path) > (sizeof(x^.dpath) - 1)) then begin
		ndir_open := NDSS_IVDATALEN;
		exit;
	end else begin
		x^.dpath := path;
	end;

	x^.list := nil;
	x^.curlp := nil;
	x^.lsize := 0;

	if (mode and (NDOM_READ or NDOM_WRITE)) = NDOM_WRITE then begin
		{ mkdir }
		sts := ndir_lookup(path,oattr,tmpsize,otime);
		if sts = NDSS_FNF then begin
			oattr := 0;
			otime := 0;
			sts := do_mkdir(x^.dpath);
		end else if (sts = NDSS_OK) and
			    ((oattr and Directory) = 0) then begin
			sts := NDSS_IVPATH;
		end; { else either error, or directory already exists };
		if sts = NDSS_OK then	{ allow for reading, like file I/O }
			x^.openmode := x^.openmode or NDOM_READ;
	end else begin
		sts := ndir_lookup(path,oattr,tmpsize,otime);
		if sts = NDSS_OK then if (oattr and Directory) = 0 then
			sts := NDSS_IVPATH;
	end;

	if (sts = NDSS_OK) and
	   ((x^.openmode and NDOM_READ) <> 0) then begin
		{ read all of directory }
		sts := make_list(x^);
		if sts <> NDSS_OK then free_list(x^);
	end;

	if sts = NDSS_OK then begin
		x^.inuse := true;
		hnd := h + $D0;			{ cf. find_ctx() above !! }
	end;
	ndir_open := sts;
end;

function ndir_read(hnd: byte; offset: longint;
		   buflen: word; var buf; var retlen: word): word;
type
	byte_array = array[0..65534] of byte;
	byte_arrayptr = ^byte_array;
var
	x: ndir_t_ctxptr;
	r: ndir_t_record;
	bp: byte_arrayptr;
	ct: integer;
begin
	retlen := 0;
	bp := addr(buf);

	x := find_ctx(hnd);
	if x = nil then begin
		ndir_read := NDSS_IVHANDLE;
		exit;
	end;

	if (x^.openmode and NDOM_READ) = 0 then begin
		ndir_read := NDSS_IVMODE;
		exit;
	end;

	{ at or beyond "eof"? }
	if offset = x^.lsize then begin
		ndir_read := NDSS_OK;
		x^.curlp := nil;
		exit;
	end else if offset > x^.lsize then begin
		ndir_read := NDSS_EOF;
		x^.curlp := nil;
		exit;
	end;

	if (x^.curlp = nil) or{_else}
	   (x^.curlp^.loff <> offset) then begin
		{ search for entry with desired offset }
		x^.curlp := x^.list;
		while (x^.curlp <> nil) and{_then}
		      (x^.curlp^.loff < offset) do begin
			x^.curlp := x^.curlp^.next;
		end;

		if (x^.curlp = nil) or{_else}
		   (x^.curlp^.loff <> offset) then begin
			{ only support 'seek' to entry boundaries! }
			ndir_read := NDSS_IVOFFSET;
			exit;
		end;		
	end;

	{ only transfer complete records! }
	while (x^.curlp <> nil) and{_then}
	      (x^.curlp^.r.rlen <= (buflen - retlen)) do begin
		move(x^.curlp^.r,bp^[retlen],x^.curlp^.r.rlen);
		inc(retlen,x^.curlp^.r.rlen);
		x^.curlp := x^.curlp^.next;
	end;
	ndir_read := NDSS_OK;
end;

{ always free handle }
function ndir_close(hnd: byte; mode: word): word;
var
	x: ndir_t_ctxptr;
	doserr: word;
label
	done;
begin
	x := find_ctx(hnd);
	if x = nil then begin
		ndir_close := NDSS_IVHANDLE;
		exit;
	end;

	if ((mode = NDCM_DELETE) and
	    ((x^.openmode and (NDOM_READ or NDOM_WRITE)) = NDOM_READ)) or
	   ((mode and not NDCM_DELETE) <> 0) then begin
		ndir_close := NDSS_IVMODE;
		goto done;
	end;

	if mode = NDCM_DELETE then begin
		{ rmdir() seems to work independent of attributes }
		ndir_close := do_rmdir(x^.dpath);
	end else
		ndir_close := NDSS_OK;

done:
	{ free_ctx() }
	free_list(x^);
	x^.inuse := false;
end;


{ initialization }
var
	i: integer;
begin
	for i := 1 to nctx do begin
		ctxa[i].inuse := false;
		ctxa[i].list := nil;
	end;
end.
