{ Copyright (C) 1997 Wolfgang J Moeller. All rights reserved. NO WARRANTY! }

unit nds_util;
 
{ miscellaneous NDFP server utility routines
{
{ w.j.m. jan 1997 ff.
{ 25-may-1997: add NDS_PERMIT routines, etc.
}

interface uses ethernet,dsk_cio;


function io_failed(s: string): boolean;

function dos_sts(s: string): word;

function path2dsk(path: string; var drv: dsk_t_dev): word;
function dsk2sts(s: string; ds: dsk_t_sts): word;

function asctime: string;

function off32(blkoff: longint; bytoff: word): longint;

const permit_envname = 'NDS_PERMIT';	{ environment name of 'PERMIT' file }
type permit_name_str = string[8];
function permit_open: boolean;		{ false, if envname undefined }
function permit_read(var addr: lanaddr; var name: permit_name_str): boolean;


implementation uses dos,wjmp,nd_const;


function io_failed(s: string): boolean;
var
	ios: integer;
begin
	ios := ioresult;
	if ios <> 0 then begin
		writeln('??? IOresult ',ios,' * ',s);
		io_failed := true;
	end else
		io_failed := false;
end;

function dos_sts(s: string): word;
var
	dose: word;
begin
	dose := doserror; doserror := 0;
	if dose <> 0 then begin
		write('??? DOSerror ',dose);
		case dose of
		  2: begin
			dos_sts := NDSS_FNF;
			write(' (file not found)');
			end;
		  3: begin
			dos_sts := NDSS_IVPATH;
			write(' (directory not found)');
			end;
		  5: begin
			dos_sts := NDSS_NOACCESS;
			write(' (access denied)');
			end;
		else
			dos_sts := NDSS_DOSFAIL;
		end;
		writeln(' * ',s);
	end else
		dos_sts := NDSS_OK;
end;

function path2dsk(path: string; var drv: dsk_t_dev): word;
begin
	if length(path) <> 1 then begin
		path2dsk := NDSS_IVDATALEN;
		exit;	
	end;
	case path[1] of
	  'A','B','a','b': drv.devcls := dsk_k_fd;
	  'C'..'Z','c'..'z': drv.devcls := dsk_k_hd;
	  else begin
		path2dsk := NDSS_IVPARAM;
		exit;
	  end;
	end;
	case path[1] of
	  'A','a': drv.unitno := 0;
	  'B','b': drv.unitno := 1;
	  'C'..'Z': drv.unitno := ord(path[1]) - ord('C');
	  'c'..'z': drv.unitno := ord(path[1]) - ord('c');
	end;
	path2dsk := NDSS_OK;
end;

function dsk2sts(s: string; ds: dsk_t_sts): word;
begin
	if ds <> dsksts_ok then begin
		write('??? DSK error ',ds);
		case ds of
	 	  dsksts_erd: begin
			write(' (read error)');
			dsk2sts := NDSS_READERR;
			end;
	 	  dsksts_ewr: begin
			write(' (write error)');
			dsk2sts := NDSS_WRITERR;
			end;
		  dsksts_inv: begin
			write(' (invalid arg)');
			dsk2sts := NDSS_IVPARAM;
			end;
		  dsksts_eom: begin
			write(' (end of medium)');
			dsk2sts := NDSS_EOF;
			end;
		  dsksts_ond: begin
			write(' (no such drive)');
			dsk2sts := NDSS_NODRIVE;
			end;
		  dsksts_onh: begin
			write(' (no free handle)');
			dsk2sts := NDSS_NOROOM;
			end;
		end;
		writeln(' * ',s);
	end else
		dsk2sts := NDSS_OK;
end;


function asctime: string;
var
	hh,mm,ss,cc: word;
begin
	gettime(hh,mm,ss,cc);
	asctime := j2(hh)+':'+j2(mm)+':'+j2(ss)+'.'+j2(cc);
end;


function off32(blkoff: longint; bytoff: word): longint;
var
	result: longint;
begin
	off32 := (blkoff shl 9) + bytoff;
end;


{----- NDS_PERMIT file -----}
{
	"#" or "!" start a comment (to end of line)
	1st nonblank field:
		address, either xx-xx-xx-xx-xx-xx, or Ph.IV DECnet area.node
	2nd nonblank field (optional):
		client name (1..8 characters)
}

{ decode base-B (2..16) unsigned integer 	}
{ return length of decoded (initial) substring	}
function atow(s: string; b: integer; var w: word): integer;
var
	i: integer;
	v: word;
begin
	atow := 0;
	w := 0;
	for i := 1 to length(s) do begin
		case s[i] of
		  '0'..'9':	v := ord(s[i]) - ord('0');
		  'a'..'f':	v := 10 + ord(s[i]) - ord('a');
		  'A'..'F':	v := 10 + ord(s[i]) - ord('A');
		  else exit;
		end;
		if v >= b then exit;
		atow := i;		{ valid digit found }
		w := b * w + v;
	end;
end;

procedure compress_pline(var s: string);	{ uncomment, trim, compress }
const
	ht = char(9);	 { TAB }
var
	i,l1,l2: integer;
	bol: boolean;
label
	skip_white,done;
begin
	l1 := length(s);
	l2 := 0;
	i := 0;
	bol := true;

skip_white:
	repeat
		inc(i);
	until (i > l1) or ((s[i] <> ' ') and (s[i] <> ht));
	if (i > l1) then goto done;

	if (s[i] = '!') or
	   (s[i] = '#') then goto done;		{ trailing comment }

	if (not bol) then begin
		inc(l2);
		if l2 <> i then s[l2] := ' ';
	end;

	bol := false;
	repeat
		if (s[i] = ' ') or
		   (s[i] = ht) then goto skip_white;

		if (s[i] = '!') or
		   (s[i] = '#') then goto done;		{ trailing comment }

		inc(l2); if l2 <> i then s[l2] := s[i];

		inc(i);
	until i > l1;

done:
	s[0] := chr(l2);	{ set new length }
end;

{ decode compressed PERMIT line }
function decode_pline(s: string; var a: lanaddr; var name: permit_name_str):
	boolean;
var
	i,ib,j1,j2,l: integer;
	area,node,v: word;
label
	not_decnet;
begin
	decode_pline := false;

	l := length(s);

	ib := l + 1;
	for i := 1 to l do if s[i] = ' ' then begin
		if (ib <= l) then exit;		{ at most 1 space permitted }
		ib := i;
	end;
	{ ib: index of delimiting space, or > l }

	if ib < l then name := copy(s,ib + 1,l - ib)
	else name := '';

	l := ib - 1;		{ now l = <length of 1st field> }

	{* try decoding DECnet "a.n" *}
	j1 := atow(s,10,area);
	if (j1 = 0) or
	   (j1 >= l) or{_else}
	   (s[j1 + 1] <> '.') then goto not_decnet;
	j2 := atow(copy(s,j1 + 2,l - j1 - 1),10,node);
	if (j2 = 0) or
	   (j1 + 1 + j2 <> l) then goto not_decnet;
	if (area < 1) or (area > 63) or
	   (node < 1) or (node > 1023) then goto not_decnet;
	v := 1024*area + node;	{ DECnet Ph.IV address }
	{ corresponding "physical" LAN address: AA-00-04-00-ll-hh }
	a.b[0] := $AA; a.b[1] := $00; a.b[2] := $04; a.b[3] := $00;
	a.b[4] := lo(v); a.b[5] := hi(v);
	decode_pline := true;
	exit;

not_decnet:

	{* decode "xx-xx-xx-xx-xx-xx" *}
	if l <> 12 + 5 then exit;
	for i := 0 to 5 do begin
		j1 := 3 * i;
		if (j1 > 0) and{_then} (s[j1] <> '-') then exit;
		if atow(copy(s,j1 + 1,2),16,v) <> 2 then exit;
		a.b[i] := lo(v);
	end;
	decode_pline := true;
end;

var
	permit_file: text;
	permit_failure: boolean;
	permit_done: boolean;

function permit_open: boolean;
var
	pfname: string;
begin
	permit_failure := false;
	permit_done := false;

	pfname := getenv(permit_envname);
	if (pfname = '') then begin
		writeln('Access not restricted (',permit_envname,' undefined)');
		permit_done := true;
		permit_open := false;
		exit;
	end else
		permit_open := true;

	permit_failure := false;
	{ An error reading the permit file will be FATAL! }

	assign(permit_file,pfname);
	{$I-}
	reset(permit_file);
	{$I+}
	if ioresult <> 0 then begin
		permit_failure := true;
		permit_done := true;
		writeln('Aborting due to missing PERMIT file: ',pfname);
		halt(80);
	end;
end;

function permit_read(var addr: lanaddr; var name: permit_name_str): boolean;
var
	line: string;
begin
	while not (permit_done or eof(permit_file)) do begin
		readln(permit_file,line);	
		compress_pline(line);
		if length(line) > 0 then begin
			if decode_pline(line,addr,name) then begin
				permit_read := true;
				exit;
			end else begin
				writeln('? unrecognized PERMIT line:');
				writeln(line);
				permit_failure := true;
			end;
		end;
	end;
	permit_done := true;

	if(permit_failure) then fehler('Aborting due to invalid PERMIT file');
	permit_read := false;
end;

end.
