/* CACHED tape I/O using only READ, WRITE, WRITEMARK, REWIND
 *
 * int ct_read(void *buffer,int size,int *trans)
 * int ct_write(void *buffer,int size)
 * int ct_bsr()		backspace 1 block
 * int ct_fsr()		forward space 1 block
 * int ct_wtm()		write tape mark
 * int ct_rew()		rewind
 * int ct_run()		rewind+unload *** invalidates cache ***
 *
 * Result
 *       0 => o.k.						** ok.
 *       1 => [invalid function or parameter list] ** NOT USED
 *       2 => EOF or EOT (EOF with READ & spacing, EOT with WRITE)
 *       3 => "permanent error" (any data error)
 *       4 => [invalid device id] ** NOT USED
 *       5 => [tape not attached] ** returned on open() error
 *       6 => write protection					** not yet impl
 *       7 => [invalid device attached] ** NOT USED
 *       8 => "incorrect length" ** ??? see BUGS ??? ** NOT USED
 *
 * BUGS: ** don't dare to call ct_read() with `size' < tape block size **
 *	 (yields undefined result due to braindamaged read() buffering)
 *
 *	data caching is very simple-minded, only a quick shot ...
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#ifdef TRACE
#define TRACE1(n,s) f_trace(n,s)
#else
#define TRACE1(n,s) (s)
#endif

extern int
  TAPREW(void),
  TAPRUN(void),
  TAPRD(void *,int,int *),
  TAPWTM(void),
  TAPWR(void * const,int);

/* global data */

typedef enum {CT_NULL,CT_EOC,CT_MARK,CT_DATA,CT_ERR} CTINFO;

typedef struct CTblock {
  int size;
  char data[4];	/* pro forma, dimension is actually dynamic */
} CTBLOCK;

typedef struct CTitem {
  struct CTitem *flink;
  struct CTitem *blink;
  int pos;
  CTINFO info;
  CTBLOCK b;	/* DIRTY: this part of CTITEM may be missing! */
} CTITEM;

/* size of CTITEM with info != CT_DATA */
static const int CTitemsize1 = sizeof(CTITEM) - sizeof(CTBLOCK);

/* size of CTITEM with info == CT_DATA, minus the contents of .b.size */
static const int CTitemsize2 = sizeof(CTITEM) - 4;

static CTITEM cteoc = {NULL,NULL,0,CT_EOC};	/* .b present, but not init. */
static CTITEM *ctbot = &cteoc;

static int ppos = 0;	/* "physical" tape position (0=BOT) */
static int vpos = 0;	/* "virtual" position (0=BOT) */

/* cache pointers point to highest CTITEM with .pos <= ?pos */
static CTITEM *pptr = &cteoc;	/* "physical" cache pointer */
static CTITEM *vptr = &cteoc;	/* "virtual" cache pointer */

/* 1st time init flag */
static int/*logical*/ initialized = 0;
			  
/* buffer used for positioning via TAPRD() */
static unsigned char fsrbuf[0xFFFF];
static int fsrlen;

/*****/

#ifdef TRACE
static int f_trace(char *n, int s)
{
  fprintf(stdout,"CT_%s (ppos=%d, vpos=%d) => %d\n",n,ppos,vpos,s);
  return s;
}
#endif

/*****/

/* private error reporting ... */
static void ct_abort(char *str, int sts)
{
  fprintf(stderr,"CT_abort: %s, status=%d\n",str,sts);
  fflush(stderr);
  abort();
}

/* add CTITEM '*itm' *before* another CTITEM '*loc' */
static void insert_item(CTITEM *itm, CTITEM *loc)
{
  CTITEM *q;

  assert(itm->pos < loc->pos);
  q = itm->blink = loc->blink;
  itm->flink = loc;
  loc->blink = itm;
  if(q) {
    assert(itm->pos > q->pos);
    q->flink = itm;	/* not first CTITEM */
  } else {
    ctbot = itm;	/* first CTITEM */
  }
}

/* remove CTITEM '*itm', return successor */
static CTITEM *delete_item(CTITEM *itm)
{
  CTITEM *q;

  assert(itm->flink != NULL && itm->info != CT_EOC);

  q = itm->flink;
  q->blink = itm->blink;
  if(itm->blink) {
    itm->blink->flink = q;
  } else {
    assert(ctbot == itm);
    ctbot = q;
  }
  free(itm);

  return q;
}

/* truncate cache at current position */
static void truncate_cache(void)
{
  assert(vpos == ppos && vptr == pptr);

  while(vptr->info != CT_EOC) {
    vptr = delete_item(vptr);
  }
  vptr->pos = vpos;
  pptr = vptr;

  assert(vptr == &cteoc);
}

/* create CTITEM of specified 'info' type */
static CTITEM *cache_mark(int pos)		/* ... CT_MARK */
{
  CTITEM *ip;

  ip = (CTITEM *)malloc(CTitemsize1);
  if(!ip) ct_abort("malloc",0);

  ip->pos = pos;
  ip->info = CT_MARK;

  return ip;
}

static CTITEM *cache_err(int pos)		/* ... CT_ERR */
{
  CTITEM *ip;

  ip = (CTITEM *)malloc(CTitemsize1);
  if(!ip) ct_abort("malloc",0);

  ip->pos = pos;
  ip->info = CT_ERR;

  return ip;
}

/* create CT_DATA item, or return NULL - cache policy implemented here */
/* first shot: cache every block of 80 bytes
 *             which looks like an ANSI label
 */
static CTITEM *cache_data(int pos,void *buf,int siz)
{
  CTITEM *ip;

  if(siz != 80) return NULL;
  if(strncmp(buf,"VOL",3) &&
     strncmp(buf,"HDR",3) &&
     strncmp(buf,"EOF",3) &&
     strncmp(buf,"EOV",3) &&
     strncmp(buf,"UHL",3) &&
     strncmp(buf,"UTL",3)) return NULL;

  ip = (CTITEM *)malloc(CTitemsize2 + siz);
  if(!ip) ct_abort("malloc",0);

  memcpy(ip->b.data,buf,siz);
  ip->b.size = siz;
  ip->pos = pos;
  ip->info = CT_DATA;

  return ip;
}

/* position tape to match "virtual" position - using TAPRD and TAPREW only */
static void move_to_vpos(void)
{
  int sts;
  CTITEM *ip;

  if(vpos < ppos) {
    sts = TRACE1("pos-TAPREW",TAPREW());
    if(sts != 0) ct_abort("TAPREW",sts);
    ppos = 0;
    pptr = ctbot;
  }
  if(ppos == vpos) {
    assert(pptr == vptr);
    return;
  }

  assert(vpos > ppos);
  while(vpos > ppos) {
    CTINFO curinf;

    assert(pptr->info != CT_NULL);
    curinf = (pptr->pos == ppos) ? pptr->info : CT_NULL;
    sts = TRACE1("pos-TAPRD",TAPRD(fsrbuf,sizeof(fsrbuf),&fsrlen));
    switch(sts) {
    case 0:
    case 8:
      switch(curinf) {
      case CT_MARK:
	ct_abort("pos.TAPRD expecting TAPEMARK",sts);
      case CT_ERR:
	assert(vptr != pptr);
	pptr = delete_item(pptr);
	curinf = CT_NULL;		/* forget about error */
	/* drop thru */
      case CT_NULL:
	if(sts == 0 &&
	   (ip = cache_data(ppos,fsrbuf,fsrlen))) insert_item(ip,pptr);
	break;
      case CT_DATA:
	if(sts == 8 ||
	   fsrlen != pptr->b.size ||
	   memcmp(fsrbuf,pptr->b.data,fsrlen)) ct_abort("pos.TAPRD [inconsistent data]",sts);
	break;
      default:
	abort();
      }
      break;

    case 2:
      if(curinf != CT_MARK) ct_abort("pos.TAPRD expecting DATA",sts);
      break;

    case 3:
      if(curinf == CT_MARK) ct_abort("pos.TAPRD expecting TAPEMARK",sts);
      if(curinf != CT_ERR) ct_abort("pos.TAPRD expecting DATA",sts);
      break;

    default:
      ct_abort("pos.TAPRD",sts);
    }

    if(curinf != CT_NULL) {
      pptr = pptr->flink;
    }
    ppos++;
  }
}

static void ct_init(void)	/* 1st time REWIND */
{
  if(!initialized) {
    int sts = TRACE1("init-TAPREW",TAPREW());
    if(sts != 0) ct_abort("init.TAPREW",sts);
    initialized = 1;
  }
}

/*****/

static int do_write(void * const buf, int siz)
{
  int sts;
  CTITEM *ip;

  assert(vpos <= vptr->pos);

  move_to_vpos();
  truncate_cache();
  assert(vptr->info == CT_EOC && vpos == vptr->pos);

  sts = buf ? TRACE1("TAPWR",TAPWR(buf,siz)) : TRACE1("TAPWTM",TAPWTM());
  switch(sts) {
  case 0:
    vptr->pos = ppos = ++vpos;
    if(ip = (buf ? cache_data(vpos - 1,buf,siz) : cache_mark(vpos - 1))) {
      insert_item(ip,vptr);
    }
    return 0;

  case 2:
    vptr->pos = ppos = ++vpos;		/* HOPE that the tape moved */
    if(!buf) insert_item(cache_mark(vpos - 1),vptr);
    return 2;

  case 3:
    vptr->pos = ppos = ++vpos;		/* HOPE that the tape moved */
    insert_item(buf ? cache_err(vpos - 1) : cache_mark(vpos - 1),vptr);
    return 3;

  case 6:
    return 6;				/* write protect => NOOP */
    					/* sorry for the lost cache ... */
  default:
    ct_abort(buf ? "TAPWR" : "TAPWTM",sts);
  }
}

int ct_write(void * const buf,int siz)
{
  ct_init();
  assert(buf != NULL);
  return do_write(buf,siz);
}

int ct_wtm(void)
{
  ct_init();
  return do_write(NULL,0);
}

int ct_read(void *buf, int siz, int *trp)
{
  int sts;
  CTITEM *ip;

  ct_init();
  assert(vpos <= vptr->pos);

  if(vptr->pos != vpos) {		/* not at EndOfCache */
    assert(vpos < vptr->pos);

    move_to_vpos();
    sts = TRACE1("TAPRD",TAPRD(buf,siz,trp));
    switch(sts) {
    case 0:
      ppos = ++vpos;
      if(ip = cache_data(vpos - 1,buf,*trp)) insert_item(ip,vptr);
      return 0;
    case 2:
      ct_abort("TAPRD expecting DATA",sts);
    case 3:
      ppos = ++vpos;			/* HOPE that the tape moved */
      insert_item(cache_err(vpos - 1),vptr);
      return 3;
    case 8:
      ppos = ++vpos;
      return 8;
    default:
      ct_abort("TAPRD",sts);
    }
  } else {
    switch(vptr->info) {

    case CT_DATA:
      if(vptr->b.size > siz) {
	sts = 8;
	*trp = siz;
      } else {
	sts = 0;
	*trp = vptr->b.size;
      }
      memcpy(buf,vptr->b.data,*trp);
      vpos++;
      vptr = vptr->flink;
      return sts;

    case CT_MARK:
      *trp = 0;
      vpos++;
      vptr = vptr->flink;
      return 2;

    case CT_ERR:		/* retry! */
      move_to_vpos();
      truncate_cache();
      assert(vptr->info == CT_EOC);
      /* drop thru */
    case CT_EOC:			/* at EndOfCache */	
      move_to_vpos();
      sts = TRACE1("TAPRD@eoc",TAPRD(buf,siz,trp));
      switch(sts) {
      case 0:
	vptr->pos = ppos = ++vpos;
	if(ip = cache_data(vpos - 1,buf,*trp)) insert_item(ip,vptr);
	return 0;
      case 2:
	vptr->pos = ppos = ++vpos;
	insert_item(cache_mark(vpos - 1),vptr);
	return 2;
      case 3:
	vptr->pos = ppos = ++vpos;	/* HOPE that the tape moved */
	insert_item(cache_err(vpos - 1),vptr);
	return 3;
      case 8:
	vptr->pos = ppos = ++vpos;
	return 8;
      default:
	ct_abort("TAPRD",sts);
      }

    default: abort();

    }
  }
}

int ct_bsr(void)
{
  ct_init();
  assert(vpos <= vptr->pos);

  if(vpos == 0) {	/* BOT => good status, NOOP */
    return 0;
  } else {
    assert(vpos > 0);
    vpos--;
    if(vptr->blink != NULL && vptr->blink->pos == vpos) {
      vptr = vptr->blink;
      switch(vptr->info) {
      case CT_MARK: return 2;
      case CT_ERR: return 3;
      case CT_DATA: return 0;
      default: abort();
      }
    } else {
      assert(vptr == ctbot || vpos > vptr->blink->pos);
      return 0;
    }
  }
}

int ct_fsr(void)
{
  int sts;
  CTITEM *ip;

  ct_init();
  assert(vpos <= vptr->pos);

  if(vptr->pos != vpos) {
    assert(vpos < vptr->pos);
    vpos++;
    return 0;
  } else {
    switch(vptr->info) {

    case CT_DATA:
      vpos++;
      vptr = vptr->flink;
      return 0;

    case CT_MARK:
      vpos++;
      vptr = vptr->flink;
      return 2;

    case CT_ERR:		/* retry! */
      move_to_vpos();
      truncate_cache();
      assert(vptr->info == CT_EOC);
      /* drop thru */
    case CT_EOC:
      move_to_vpos();
      sts = TRACE1("fsr-TAPRD",TAPRD(fsrbuf,sizeof(fsrbuf),&fsrlen));
      switch(sts) {
      case 0:
	vptr->pos = ppos = ++vpos;
	if(ip = cache_data(vpos - 1,fsrbuf,fsrlen)) insert_item(ip,vptr);
	return 0;
      case 2:
	vptr->pos = ppos = ++vpos;
	insert_item(cache_mark(vpos - 1),vptr);
	return 2;
      case 3:
	vptr->pos = ppos = ++vpos;	/* HOPE that the tape moved */
	insert_item(cache_err(vpos - 1),vptr);
	return 3;
      case 8:
	vptr->pos = ppos = ++vpos;
	return 0;
      default:
	ct_abort("TAPRD",sts);
      }

    default: abort();

    }
  }
}

int ct_rew(void)
{
  ct_init();

  vpos = 0;
  vptr = ctbot;
  return 0;
}

int ct_run(void)
{
  int sts;

  ct_init();

  sts = TRACE1("TAPRUN",TAPRUN());
  if(sts != 0) ct_abort("TAPRUN",sts);

  vpos = ppos = 0;
  vptr = pptr = ctbot;
  truncate_cache();

  initialized = 0;	/* must rewind new tape next time */
  return 0;
}
