/*
 ******************************************************************************
 ******************************************************************************
 *
 *  Licensed Materials - Property of IBM
 *
 *  IBM TCP/IP for DOS.
 *  Copyright (c) IBM Corp. 1992.  All rights reserved.
 *  US Government Users Restricted Rights - Use, duplication or
 *  disclosure restricted by GSA ADP Schedule contract with IBM Corp.
 *
 ******************************************************************************
 ******************************************************************************
 *
 *  $Author:   Terri_Waldron  $
 *  $Revision:   1.2  $
 *  $Modtime:   14 Jan 1993 15:50:12  $
 *
 *  $Header:   L:/INCLUDE/RSA/RPC/AUTH.H_V   1.2   14 Jan 1993 15:52:02   Terri_Waldron  $
 *
 *  $Log:   L:/INCLUDE/RSA/RPC/AUTH.H_V  $
 * 
 *    Rev 1.2   14 Jan 1993 15:52:02   Terri_Waldron
 * Made a few changes so that windows apps would compile when including 
 * RPC stuff. Bob Pombrio
 *
 *    Rev 1.1   13 Oct 1992 14:29:24   R_Kania
 * Microsoft C 7.00 changes
 *
 *    Rev 1.0   07 Oct 1992 12:09:46   RKania
 * Initial revision.
 *
 ******************************************************************************
 ******************************************************************************
 *
 * Copyright (c) 1990 by Sun Microsystems, Inc.
 *      @(#)auth.h      1.17 88/02/08 SMI
 *      @(#)auth.h      1.3  90/07/17 4.1NFSSRC SMI
 *
 ******************************************************************************
 ******************************************************************************
 */

/*
 ******************************************************************************
 * auth.h
 * Authentication interface.
 *
 * The data structures are completely opaque to the client.  The client
 * is required to pass a AUTH * to routines that create RPC "sessions".
 ******************************************************************************
 */
#ifndef _rpc_auth_h
#define _rpc_auth_h

#include <rpc/xdr.h>

#define MAX_AUTH_BYTES          400
#define MAXNETNAMELEN           255     /* Max length of network user's name  */

/*
 ******************************************************************************
 * Status returned from authentication check.
 ******************************************************************************
 */
enum auth_stat
  {
  /* No errors */
  AUTH_OK=0,                            /* Authentication successful          */

  /* Remote end errors */
  AUTH_BADCRED=1,                       /* Bogus credentials (seal broken)    */
  AUTH_REJECTEDCRED=2,                  /* Client should begin new session    */
  AUTH_BADVERF=3,                       /* Bogus verifier (seal broken)       */
  AUTH_REJECTEDVERF=4,                  /* Verifier expired or was replayed   */
  AUTH_TOOWEAK=5,                       /* Rejected due to security reasons   */

  /* Local end errors */
  AUTH_INVALIDRESP=6,                   /* Bogus response verifier            */
  AUTH_FAILED=7                         /* Some unknown reason                */
  };

union reserved
  {
  struct
    {
    unsigned long high;
    unsigned long low;
    } key;
  char c[8];
  };

typedef union reserved reserved;

/*
 ******************************************************************************
 * Authentication information.  Opaque to client.
 ******************************************************************************
 */
struct opaque_auth
  {
  enum_t oa_flavor;                     /* Flavor of auth                     */
  char *oa_base;                        /* Address of more auth stuff         */
  unsigned short oa_length;             /* Not to exceed MAX_AUTH_BYTES       */
  };

/*
 ******************************************************************************
 * Authentication handle, interface to client side authenticators.
 ******************************************************************************
 */
typedef struct AUTH AUTH;
struct AUTH
  {
  struct opaque_auth ah_cred;
  struct opaque_auth ah_verf;
  union reserved ah_key;
  struct auth_ops
    {
    void (*ah_nextverf)(AUTH *);
    int (*ah_marshal)(AUTH *,
                      XDR *);           /* Nextverf & serialize               */
    int (*ah_validate)(AUTH *,
                       struct opaque_auth*);
    int (*ah_refresh)(AUTH *);          /* Refresh credentials                */
    void (*ah_destroy)(AUTH *);         /* Destroy this structure             */
    } *ah_ops;
//  caddr_t ah_private;
  char *ah_private;
  };

/*
 ******************************************************************************
 * Authentication ops.
 * The ops and the auth handle provide the interface to the authenticators.
 *
 * AUTH *auth;
 * XDR  *xdrs;
 * struct opaque_auth verf;
 ******************************************************************************
 */
#define AUTH_NEXTVERF(auth) \
        ((*((auth)->ah_ops->ah_nextverf))(auth))
#define auth_nextverf(auth) \
        ((*((auth)->ah_ops->ah_nextverf))(auth))

#define AUTH_MARSHALL(auth, xdrs) \
        ((*((auth)->ah_ops->ah_marshal))(auth, xdrs))
#define auth_marshall(auth, xdrs) \
        ((*((auth)->ah_ops->ah_marshal))(auth, xdrs))

#define AUTH_VALIDATE(auth, verfp) \
        ((*((auth)->ah_ops->ah_validate))((auth), verfp))
#define auth_validate(auth, verfp) \
        ((*((auth)->ah_ops->ah_validate))((auth), verfp))

#define AUTH_REFRESH(auth) \
        ((*((auth)->ah_ops->ah_refresh))(auth))
#define auth_refresh(auth) \
        ((*((auth)->ah_ops->ah_refresh))(auth))

#define AUTH_DESTROY(auth) \
        ((*((auth)->ah_ops->ah_destroy))(auth))
#define auth_destroy(auth) \
        ((*((auth)->ah_ops->ah_destroy))(auth))

extern struct opaque_auth _null_auth;

/*
 ******************************************************************************
 * These are the various implementations of client side authenticators.
 ******************************************************************************
 * UNIX style authentication
 *
 * AUTH *authunix_create(machname,
 *                       uid,
 *                       gid,
 *                       len,
 *                       aup_gids);
 *
 * char *machname;
 * int uid;
 * int gid;
 * int len;
 * int *aup_gids;
 *
 * Creates and returns a UNIX-based authentication handle.
 ******************************************************************************
 */
extern AUTH *authunix_create(char *,
                             long,
                             long,
                             register int,
                             long *);

/*
 ******************************************************************************
 * Calls authunix_create() with default parameters.
 ******************************************************************************
 */
extern AUTH *authunix_create_default(); /* Takes no parameters                */

/*
 ******************************************************************************
 * Creates and returns a NULL RPC authentication handle.
 ******************************************************************************
 */
extern AUTH *authnone_create();         /* Takes no parameters                */

extern AUTH *authdes_create();

#define AUTH_NONE               0       /* No authentication                  */
#define AUTH_NULL               0       /* Backward compatibility             */
#define AUTH_UNIX               1       /* UNIX style (uid, gids)             */
#define AUTH_SHORT              2       /* Short hand unix style              */

#endif
