/**
 * @(#)FontChooser.java
 *
 * <code>FontChooser</code> idea is taken from 
 * Sun's documentations. The basic idea is to 
 * create a menu to let users choose the font 
 * face, size and style at once. By keeping a 
 * static boolean named <code>chooserOpen</code>, 
 * creating more than one <code>FontChooser</code> 
 * is prevented.
 * 
 * @author Gokhan Arici
 * @version 1.00 21.12.2009
 */
 
package bin;
 
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JComboBox;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.JButton;
import java.awt.event.*;
import java.awt.FlowLayout;
import java.awt.GraphicsEnvironment;
import java.awt.Font;

public class FontChooser extends JFrame {
	
	/**
	 * <code>serialVersionUID</code><br>
	 * Created by Eclipse.
	 */
	private static final long serialVersionUID = -2199129831318388166L;
	
	/**
	 * <code>chooserOpen</code><br>
	 * A boolean value. Must be checked while creating 
	 * a new FontChooser. If true, new FontChooser 
	 * should not be created.
	 */
	private static boolean chooserOpen=false;
	
	/**
	 * Creates the <code>FontChooser</code> window.
	 * @param callerObject A reference to the caller 
	 * <code>Display</code> object. This is needed, 
	 * because FontChooser calls the changeFont() 
	 * method of the caller <code>Display</code> 
	 * object to change its font.
	 * @param defaultFont Name of the current font.
	 * It is read from <code>Display</code> class.
	 */
	public FontChooser(final Display callerObject,String defaultFont) {
		// Set chooserOpen to true
		chooserOpen=true;
		// Make it unresizable
		setResizable(false);
		// Add a new WindowListener
		addWindowListener(new WindowAdapter() {
			// While closing ...
			public void windowClosing(WindowEvent we) {
				// ...set chooserOpen to false...
				chooserOpen=false;
				// ... and dispose.
				dispose();
			}
		});
		// Make the layout FlowLayout
		setLayout(new FlowLayout());
		// Add the label "Font Face"
		add(new JLabel(Language.getText(7)));
		// Get local graphics environment to use it...
        GraphicsEnvironment gEnv = GraphicsEnvironment.getLocalGraphicsEnvironment();
        // ... while getting available fonts.
        final JComboBox fonts = new JComboBox(gEnv.getAvailableFontFamilyNames());
        // Choose the current font by default
        fonts.setSelectedItem(defaultFont);
        // Show at most 5 rows
        fonts.setMaximumRowCount(5);
        // Add the combo box to the window
        add(fonts);
        // Add the label "Style"
        add(new JLabel(Language.getText(8)));
        // Create a new array of style names, "Plain", "Bold", "Italic", "Bold Italic"
        String[] styleNames = {Language.getText(9), Language.getText(10), Language.getText(11), Language.getText(12)};
        // Show them in another combo box
        final JComboBox styles = new JComboBox(styleNames);
        // Add it to the window
        add(styles);
        // Add the label "Size"
        add(new JLabel(Language.getText(13)));
        // Create a JSpinner for the sizes
        final JSpinner sizes = new JSpinner(new SpinnerNumberModel(12, 6, 36, 1));
        // Add it to the window
        add(sizes);
        // Create a new JButton
        JButton apply = new JButton(Language.getText(14));
        // Add an ActionListener...
        apply.addActionListener(new ActionListener(){
        	// ...which will...
        	public void actionPerformed(ActionEvent e){
        		// ... change the font to the selected one
				callerObject.changeFont(new Font((String)fonts.getSelectedItem(),
										styles.getSelectedIndex(),
										Integer.parseInt(sizes.getModel().getValue().toString()) ));
				// Set chooserOpen to false
		        chooserOpen=false;
		        // Dispose the window
		        dispose();
			}
		});
		// Add the button to the window
		add(apply);
		// Pack the window
		pack();
	}
	
	/**
	 * <code>isAlreadyOpen</code><br>
	 * Returns if another fontChooser is already open.
	 * @return chooserOpen's value.
	 */
	public static boolean isAlreadyOpen() {
		// Return the chooserOpen
		return chooserOpen;
	}
}
