/*
 * jNPad v0.3 - jNPad's an Simple Text Editor written in Java
 *
 * Copyright (C) 2014-2017  rgs
 *
 * Require JDK 1.6 (or later)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 *
 * Info, Questions, Suggestions & Bugs Report to rgsevero@gmail.com
 */

package jnpad.text.syntax;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;

import javax.swing.text.BadLocationException;
import javax.swing.text.Element;
import javax.swing.text.Segment;
import javax.swing.text.Utilities;

import jnpad.text.JNPadDocument;
import jnpad.text.syntax.JavaScheme.WordType;

/**
 * The Class JavaView.
 * 
 * @version 0.3
 * @since jNPad 0.1
 */
public class JavaView extends SyntaxView {
  JavaScheme scheme;

  /**
   * Instantiates a new java view.
   *
   * @param elem the elem
   * @param mini the mini
   */
  public JavaView(Element elem, boolean mini) {
    super(elem, mini);
    scheme = !mini ? (JavaScheme) ((JNPadDocument) getDocument()).getScheme() :
      (JavaScheme) ((JNPadDocument) getDocument()).getMiniScheme();
  }
  
  /**
   * Do draw unselected text.
   *
   * @param g the graphics context
   * @param x the starting X coordinate >= 0
   * @param y the starting Y coordinate >= 0
   * @param start the start
   * @param length the length
   * @return the int
   * @throws BadLocationException the bad location exception
   * @see jnpad.text.syntax.SyntaxView#doDrawUnselectedText(java.awt.Graphics, int, int, int, int)
   */
  @Override
  int doDrawUnselectedText(Graphics g, int x, int y, int start, int length) throws BadLocationException {
    Segment content = new Segment();
    // fill the segment.
    getDocument().getText(start, length, content);

    boolean inComment = isParagraphInComment(start);
    boolean inJavadoc = isParagraphInJavadoc(start);

    for (int wordIndex = 0; wordIndex < content.length();) {
      char indexedChar = content.charAt(wordIndex);

      if (Character.isJavaIdentifierStart(indexedChar) && !inComment && !inJavadoc) {
        String scannedIdentifier = scanIdentifier(content, wordIndex);
        int scannedIdentifierLength = scannedIdentifier.length();
        
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedIdentifierLength, text);
        final Font savedFont = g.getFont();
        final WordType type = scheme.getWordType(scannedIdentifier);
        g.setColor(scheme.getWordColor(type));
        g.setFont(scheme.getWordFont(type));
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedIdentifierLength;

        continue;
      }
      else if (Character.isDigit(indexedChar) && !inComment && !inJavadoc) {
        String scannedNumericLiteral = scanNumericLiteral(content, wordIndex);
        int scannedNumericLiteralLength = scannedNumericLiteral.length();

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedNumericLiteralLength, text);
        g.setColor(scheme.getNumberColor());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);

        wordIndex += scannedNumericLiteralLength;

        continue;
      }
      else if (isStringDelimiter(indexedChar) && !inComment && !inJavadoc) {
        int scannedStringLength = scanStringLiteral(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedStringLength, text);
        final Font savedFont = g.getFont();
        g.setColor(scheme.getStringColor());
        g.setFont(scheme.getStringFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        wordIndex += scannedStringLength;
        
        continue;
      }
      else if (isCharDelimiter(indexedChar) && !inComment && !inJavadoc) {
        int scannedCharLength = scanCharLiteral(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCharLength, text);
        g.setColor(scheme.getStringColor());
        g.setFont(scheme.getStringFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        
        wordIndex += scannedCharLength;
        
        continue;
      }
      else if (isSingleLineCommentStart(content, wordIndex) && !inComment && !inJavadoc) {
        int scannedCommentLength = scanSingleLineComment(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCommentLength, text);
        final Font savedFont = g.getFont();
        g.setColor(scheme.getCommentColor());
        g.setFont(scheme.getCommentFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedCommentLength;
        
        continue;
      }
      else if (isAnnotationStart(content, wordIndex) && inJavadoc &&
               (wordIndex < content.length() - 1)) {
        String scannedIdentifier = scanIdentifier(content, wordIndex + 1);
        int scannedIdentifierLength = scannedIdentifier.length() + 1;

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedIdentifierLength, text);
        final Font savedFont = g.getFont();
        if (scheme.isJavadocTag("@".concat(scannedIdentifier))) { //$NON-NLS-1$
          final Font f = scheme.getJavadocTagFont();
          g.setFont(f);
          g.setColor(scheme.getJavadocTagColor());
          if (f.isItalic()) {
            Graphics2D g2D = (Graphics2D) g;
            final Object oldTextAntialiasing = g2D.getRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING);
            g2D.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
            x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
            g2D.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, oldTextAntialiasing);
          }
          else {
            x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
          }
          // ---------
          if("param".equals(scannedIdentifier)) { //$NON-NLS-1$
            wordIndex += scannedIdentifierLength;
            scannedIdentifier = scanIdentifier(content, wordIndex + 1);
            scannedIdentifierLength = scannedIdentifier.length() + 1;
            text = getLineBuffer();
            getDocument().getText(start + wordIndex, scannedIdentifierLength, text);
            g.setColor(scheme.getJavadocParamColor());
            g.setFont(scheme.getJavadocParamFont());
            x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
          }
          // ---------
        }
        else {
          g.setFont(scheme.getJavadocFont());
          g.setColor(scheme.getJavadocColor());
          x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        }
        g.setFont(savedFont);

        wordIndex += scannedIdentifierLength;

        continue;
      }
      else if (isAnnotationStart(content, wordIndex) && !inComment && !inJavadoc &&
               (wordIndex < content.length() - 1)) {
        String scannedIdentifier = scanIdentifier(content, wordIndex + 1);
        int scannedIdentifierLength = scannedIdentifier.length() + 1;

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedIdentifierLength, text);
        final Font savedFont = g.getFont();
        if ("interface".equals(scannedIdentifier)) { // @interface //$NON-NLS-1$
          g.setColor(scheme.getKeywordColor());
          g.setFont(scheme.getKeywordFont());
        }
        else {
          g.setColor(scheme.getAnnotationColor());
          g.setFont(scheme.getAnnotationFont());
        }
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedIdentifierLength;
        
        continue;
      }
      else if (isJavadocEnd(content, wordIndex) && inJavadoc) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 2, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getJavadocFont());
        g.setColor(scheme.getJavadocColor());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        inJavadoc = false;
        wordIndex += 2;
        
        continue;
      }
      else if (isMultiLineCommentEnd(content, wordIndex) && inComment) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 2, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getCommentFont());
        g.setColor(scheme.getCommentColor());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        inComment = false;
        wordIndex += 2;
        
        continue;
      }
      else if (isJavadocStart(content, wordIndex)) {
        int scannedCommentLength = scanMultiLineComment(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCommentLength, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getJavadocFont());
        g.setColor(scheme.getJavadocColor());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        inJavadoc = true;
        wordIndex += scannedCommentLength;
        
        continue;
      }
      else if (isMultiLineCommentStart(content, wordIndex)) {
        int scannedCommentLength = scanMultiLineComment(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCommentLength, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getCommentFont());
        g.setColor(scheme.getCommentColor());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        inComment = true;
        wordIndex += scannedCommentLength;
        
        continue;
      }
      else if (isBracket(indexedChar) && !inComment && !inJavadoc) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 1, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getBracketFont());
        g.setColor(scheme.getBracketColor());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        wordIndex++;

        continue;
      }
      else if (isOperator(indexedChar) && !inComment && !inJavadoc) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 1, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getOperatorFont());
        g.setColor(scheme.getOperatorColor());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        wordIndex++;
        
        continue;
      }
      
      final Font savedFont = g.getFont();
      Color color;
      if (inJavadoc) {
        g.setFont(scheme.getJavadocFont());
        color = scheme.getJavadocColor();
      }
      else if (inComment) {
        g.setFont(scheme.getCommentFont());
        color = scheme.getCommentColor();
      }
      else {
        color = scheme.getTextColor();
      }
      Segment text = getLineBuffer();
      getDocument().getText(start + wordIndex, 1, text);
      g.setColor(color);
      x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
      g.setFont(savedFont);
      
      wordIndex++;
    }

    return x;
  }
  
  /**
   * Do draw selected text.
   *
   * @param g the graphics context
   * @param x the starting X coordinate >= 0
   * @param y the starting Y coordinate >= 0
   * @param start the start
   * @param length the length
   * @return the int
   * @throws BadLocationException the bad location exception
   * @see jnpad.text.syntax.SyntaxView#doDrawSelectedText(java.awt.Graphics, int, int, int, int)
   */
  @Override
  int doDrawSelectedText(Graphics g, int x, int y, int start, int length) throws BadLocationException {
    Segment content = new Segment();
    // fill the segment.
    getDocument().getText(start, length, content);

    boolean inComment = isParagraphInComment(start);
    boolean inJavadoc = isParagraphInJavadoc(start);

    g.setColor(scheme.getSelectedTextColor());

    for (int wordIndex = 0; wordIndex < content.length();) {
      char indexedChar = content.charAt(wordIndex);

      if (Character.isJavaIdentifierStart(indexedChar) && !inComment && !inJavadoc) {
        String scannedIdentifier = scanIdentifier(content, wordIndex);
        int scannedIdentifierLength = scannedIdentifier.length();
        
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedIdentifierLength, text);
        final Font savedFont = g.getFont();
        final WordType type = scheme.getWordType(scannedIdentifier);
        g.setFont(scheme.getWordFont(type));
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedIdentifierLength;

        continue;
      }
      else if (isStringDelimiter(indexedChar) && !inComment && !inJavadoc) {
        int scannedStringLength = scanStringLiteral(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedStringLength, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getStringFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        wordIndex += scannedStringLength;
        
        continue;
      }
      else if (isCharDelimiter(indexedChar) && !inComment && !inJavadoc) {
        int scannedCharLength = scanCharLiteral(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCharLength, text);
        g.setFont(scheme.getStringFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        
        wordIndex += scannedCharLength;
        
        continue;
      }
      else if (isSingleLineCommentStart(content, wordIndex) && !inComment && !inJavadoc) {
        int scannedCommentLength = scanSingleLineComment(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCommentLength, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getCommentFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);
        
        wordIndex += scannedCommentLength;
        
        continue;
      }
      else if (isAnnotationStart(content, wordIndex) && inJavadoc &&
               (wordIndex < content.length() - 1)) {
        String scannedIdentifier = scanIdentifier(content, wordIndex + 1);
        int scannedIdentifierLength = scannedIdentifier.length() + 1;

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedIdentifierLength, text);
        final Font savedFont = g.getFont();
        if (scheme.isJavadocTag("@".concat(scannedIdentifier))) { //$NON-NLS-1$
          final Font f = scheme.getJavadocTagFont();
          g.setFont(f);
          if (f.isItalic()) {
            Graphics2D g2D = (Graphics2D) g;
            final Object oldTextAntialiasing = g2D.getRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING);
            g2D.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
            x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
            g2D.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, oldTextAntialiasing);
          }
          else {
            x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
          }
          // ---------
          if("param".equals(scannedIdentifier)) { //$NON-NLS-1$
            wordIndex += scannedIdentifierLength;
            scannedIdentifier = scanIdentifier(content, wordIndex + 1);
            scannedIdentifierLength = scannedIdentifier.length() + 1;
            text = getLineBuffer();
            getDocument().getText(start + wordIndex, scannedIdentifierLength, text);
            g.setFont(scheme.getJavadocParamFont());
            x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
          }
          // ---------
        }
        else {
          g.setFont(scheme.getJavadocFont());
          x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        }
        g.setFont(savedFont);

        wordIndex += scannedIdentifierLength;
        
        continue;
      }
      else if (isAnnotationStart(content, wordIndex) && !inComment && !inJavadoc &&
             (wordIndex < content.length() - 1)) {
        String scannedIdentifier = scanIdentifier(content, wordIndex + 1);
        int scannedIdentifierLength = scannedIdentifier.length() + 1;

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedIdentifierLength, text);
        final Font savedFont = g.getFont();
        if ("interface".equals(scannedIdentifier)) { // @interface //$NON-NLS-1$
          g.setFont(scheme.getKeywordFont());
        }
        else {
          g.setFont(scheme.getAnnotationFont());
        }
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        wordIndex += scannedIdentifierLength;
        
        continue;
      }
      else if (isJavadocEnd(content, wordIndex) && inJavadoc) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 2, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getJavadocFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        inJavadoc = false;
        wordIndex += 2;
        
        continue;
      }
      else if (isMultiLineCommentEnd(content, wordIndex) && inComment) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 2, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getCommentFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        inComment = false;
        wordIndex += 2;
        
        continue;
      }
      else if (isJavadocStart(content, wordIndex)) {
        int scannedCommentLength = scanMultiLineComment(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCommentLength, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getJavadocFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        inJavadoc = true;
        wordIndex += scannedCommentLength;
        
        continue;
      }
      else if (isMultiLineCommentStart(content, wordIndex)) {
        int scannedCommentLength = scanMultiLineComment(content, wordIndex);

        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, scannedCommentLength, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getCommentFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        inComment = true;
        wordIndex += scannedCommentLength;
        
        continue;
      }
      else if (isBracket(indexedChar) && !inComment && !inJavadoc) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 1, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getBracketFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        wordIndex++;

        continue;
      }
      else if (isOperator(indexedChar) && !inComment && !inJavadoc) {
        Segment text = getLineBuffer();
        getDocument().getText(start + wordIndex, 1, text);
        final Font savedFont = g.getFont();
        g.setFont(scheme.getOperatorFont());
        x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
        g.setFont(savedFont);

        wordIndex++;
        
        continue;
      }
      
      final Font savedFont = g.getFont();
      if (inJavadoc) {
        g.setFont(scheme.getJavadocFont());
      }
      else if (inComment) {
        g.setFont(scheme.getCommentFont());
      }
      Segment text = getLineBuffer();
      getDocument().getText(start + wordIndex, 1, text);
      x = Utilities.drawTabbedText(text, x, y, g, this, start + wordIndex);
      g.setFont(savedFont);
      
      wordIndex++;
    }

    return x;
  }
  
  /**
   * Checks if is paragraph in comment.
   *
   * @param startOffset the start offset
   * @return true, if is paragraph in comment
   */
  boolean isParagraphInComment(int startOffset) {
    if (startOffset > 0) {
      Element root = getDocument().getDefaultRootElement();
      Element previousElement = root.getElement(root.getElementIndex(startOffset - 1));
      if (previousElement.getAttributes().containsAttribute(JNPadDocument.MULTILINE_COMMENT, JNPadDocument.MULTILINE_COMMENT))
        return true;
    }
    return false;
  }

  /**
   * Checks if is paragraph in javadoc.
   *
   * @param startOffset the start offset
   * @return true, if is paragraph in javadoc
   * @since 0.3
   */
  boolean isParagraphInJavadoc(int startOffset) {
    if (startOffset > 0) {
      Element root = getDocument().getDefaultRootElement();
      Element previousElement = root.getElement(root.getElementIndex(startOffset - 1));
      if (previousElement.getAttributes().containsAttribute(JNPadDocument.JAVADOC, JNPadDocument.JAVADOC))
        return true;
    }
    return false;
  }

  /**
   * Checks if is single line comment start.
   *
   * @param s the Segment
   * @param index the index
   * @return true, if is single line comment start
   * @see jnpad.text.syntax.SyntaxView#isSingleLineCommentStart(javax.swing.text.Segment, int)
   */
  @Override
  boolean isSingleLineCommentStart(Segment s, int index) {
    return s.charAt(index) == '/' && index + 1 < s.length() && s.charAt(index + 1) == '/';
  }
  
  /**
   * Checks if is multi line comment start.
   *
   * @param s the Segment
   * @param index the index
   * @return true, if is multi line comment start
   * @see jnpad.text.syntax.SyntaxView#isMultiLineCommentStart(javax.swing.text.Segment, int)
   */
  @Override
  boolean isMultiLineCommentStart(Segment s, int index) {
    return s.charAt(index) == '/' && index + 1 < s.length() && s.charAt(index + 1) == '*';
  }

  /**
   * Checks if is multi line comment end.
   *
   * @param s the Segment
   * @param index the index
   * @return true, if is multi line comment end
   * @see jnpad.text.syntax.SyntaxView#isMultiLineCommentEnd(javax.swing.text.Segment, int)
   */
  @Override
  boolean isMultiLineCommentEnd(Segment s, int index) {
    return s.charAt(index) == '*' && index + 1 < s.length() && s.charAt(index + 1) == '/';
  }

  /**
   * Checks if is javadoc start.
   *
   * @param s the Segment
   * @param index the index
   * @return true, if is javadoc start
   * @since 0.3
   */
  boolean isJavadocStart(Segment s, int index) {
    return s.charAt(index) == '/' && index + 1 < s.length() && s.charAt(index + 1) == '*'
      && index + 2 < s.length() && s.charAt(index + 2) == '*';
  }

  /**
   * Checks if is javadoc end.
   *
   * @param s the Segment
   * @param index the index
   * @return true, if is javadoc end
   * @since 0.3
   */
  boolean isJavadocEnd(Segment s, int index) {
    return isMultiLineCommentEnd(s, index);
  }
  
  /**
   * Checks if is annotation start.
   *
   * @param s the Segment
   * @param index the index
   * @return true, if is annotation start
   */
  boolean isAnnotationStart(Segment s, int index) {
    return (s.charAt(index) == '@');
  }

  /**
   * Scan multi line comment.
   *
   * @param s the Segment
   * @param index the index
   * @return the int
   * @see jnpad.text.syntax.SyntaxView#scanMultiLineComment(javax.swing.text.Segment, int)
   */
  @Override
  int scanMultiLineComment(Segment s, int index) {
    int commentLength;
    boolean starFound = false;
    for (commentLength = 0; commentLength < (s.length() - index); commentLength++) {
      char commentChar = s.charAt(index + commentLength);
      if (starFound && commentChar == '/') {
        commentLength++;
        break;
      }
      starFound = false;
      if (commentChar == jnpad.util.Utilities.LF)
        break;
      else if (commentChar == '*')
        starFound = true;
    }
    return commentLength;
  }  
  
  /**
   * Gets the scheme.
   *
   * @return the scheme
   * @see jnpad.text.syntax.SyntaxView#getScheme()
   */
  @Override
  SyntaxScheme getScheme() {
    return scheme;
  }
}
