/**
 * $Id: UserEditInterface.java,v 1.4 2001/09/21 02:48:24 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.client;

import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.io.IOException;
import java.util.Vector;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
import javax.swing.SwingUtilities;

import redlight.hotline.*;
import redlight.utils.DebuggerOutput;

/**
  * The account info interface.
  */
public class UserEditInterface implements Child,
                                          ActionListener,
                                          WindowListener,
                                          Runnable {

    Machine rlm;
    Container contentPane;
    JTextField login, password, nickname;
    JTextArea infoHelpText;
    JFrame f;
    JScrollPane scrollPane;
    JMenuItem deleteItems, downloadItems, openInNewWindow;
    JButton cancel, save, delete;
    JPanel bitPanel;
    HLProtocol.AccountInfo account;
    boolean create;
    Vector checkboxes;
    Thread getter;
    String newAccount;

    /* An evil magic value. */

    String defaultPasswordString = "0123456789";
    String[] privilegeDescriptions = { 
        "Can rename folders",
        "Can delete folders",
        "Can create folders",
        "Can move files",
        "Can rename files",
        "Can download files",
        "Can upload files",
        "Can delete files",
        "Can delete users",
        "Can create users",
        null,
        null,
        null,
        "Can send chat",
        "Can read chat",
        "Can move folders",
        "Cannot be disconnected",
        "Can disconnect users",
        "Can post news",
        "Can read news",
        null,
        null,
        "Can modify users",
        "Can read users",
        "Can make aliases",
        "Can view dropboxes",
        "Can comment folders",
        "Can comment files",
        "Don't show agreement",
        "Can use any name",
        "Can upload anywhere",
        "Can get user info"
    };

    public UserEditInterface(Machine machine,
                             String newAccount) {

        this(machine, null, false);        
        this.newAccount = newAccount;
        login.setText(newAccount);
        getter = new Thread(this);
        getter.start();


    }

    public void run() {
        
        try {

            account = 
                rlm.getHLC().getAccountInfo(newAccount);
            
            if(account != null) {
                
                infoHelpText.setText("Modify users handle, password, or privileges. To delete this user, hit Delete.");
                login.setText(account.login);
                nickname.setText(account.nick);
                setPrivileges(account.privileges);
                password.setEnabled(true);
                nickname.setEnabled(true);
                login.setEnabled(true);
                save.setEnabled(true);
                delete.setEnabled(true);
                
            } else {

                close();

            }

        } catch(InterruptedException e) {
        } catch(IOException e) {

            close();

        } catch(HLException e) {

            close();

        }

    }

    public UserEditInterface(Machine machine, 
                             HLProtocol.AccountInfo account, 
                             final boolean create) {
        
        if(account == null) {

            HLProtocol hlp = new HLProtocol();
            account = hlp.new AccountInfo();

        }

        rlm = machine;
        this.create = create;
        this.account = account;
        checkboxes = new Vector();

        Font guiFont = (Font) Main.rlo.getProperty("Font.gui");

        f = new JFrame((create ? "Create" : "Modify") + 
                       " account (" + rlm.getServerName() + ")");
        contentPane = f.getContentPane();
        GridBagLayout gbl = new GridBagLayout();
        GridBagConstraints gbc = new GridBagConstraints();
        JPanel textfieldPanel = new JPanel();
        JLabel l1 = new JLabel("Login: ");
        JLabel l2 = new JLabel("Password: ");
        JLabel l3 = new JLabel("Nickname: ");
        JPanel userInfoPanel = new JPanel();

        if(create) {
            
            infoHelpText = new JTextArea("To create a new user, type the new user's login, password and default handle and hit Save.", 2, 20);
            
        } else {
            
            infoHelpText = new JTextArea("Retrieving user info, please wait ...\n", 2, 20);
        
        }

        JPanel buttonPanel = new JPanel();
        JPanel deleteButtonPanel = new JPanel();
        JPanel otherButtonPanel = new JPanel();
        bitPanel = new JPanel();
        JScrollPane scrollPane = new JScrollPane(bitPanel);
        login = new JTextField(account.login);
        password = new JPasswordField(create ? "" : defaultPasswordString);
        password.addKeyListener(new KeyListener() {
                
                boolean firstKey = create;
                
                public void keyPressed(KeyEvent e) {}
                public void keyReleased(KeyEvent e) {}
                public void keyTyped(KeyEvent e) {

                    if(firstKey == false) {
                        
                        password.setText("");
                        password.setCaretPosition(0);
                        
                        firstKey = true;
                        
                    }

                }

            });
        nickname = new JTextField(account.nick);
        cancel = new JButton("Cancel");
        save = new JButton("Save");
        delete = new JButton("Delete");

        buttonPanel.registerKeyboardAction(this, "Cancel", KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0), JComponent.WHEN_IN_FOCUSED_WINDOW);
	buttonPanel.setLayout(new BorderLayout());
	bitPanel.setLayout(new GridBagLayout());
	bitPanel.setForeground(rlm.getSchemeColor("foreground"));
	bitPanel.setBackground(rlm.getSchemeColor("background"));
	contentPane.setLayout(gbl);
	otherButtonPanel.setLayout(new FlowLayout(FlowLayout.RIGHT));
	textfieldPanel.setLayout(new GridBagLayout());
        userInfoPanel.setLayout(new GridBagLayout());

        Border emptyBorder = 
            new EmptyBorder((Insets) Main.rlo.getProperty("BorderInsets"));
        Border lineBorder = 
            BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(), (create ? "Create" : "Modify") + " account");
        Border infoBorder = 
            BorderFactory.createCompoundBorder(lineBorder, emptyBorder);
        lineBorder = 
            BorderFactory.createEtchedBorder();
        Border privSpacerBorder = 
            BorderFactory.createCompoundBorder(emptyBorder, lineBorder);
        lineBorder =
            BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(), "Privileges");
        Border privBorder = 
            BorderFactory.createCompoundBorder(lineBorder, privSpacerBorder);

        userInfoPanel.setBorder(infoBorder);
        scrollPane.setBorder(privBorder);

	deleteButtonPanel.setLayout(new FlowLayout(FlowLayout.LEFT));
        cancel.setMnemonic('C');
        save.setMnemonic('S');

	cancel.addActionListener(this);
	save.addActionListener(this);
	delete.addActionListener(this);

	if(!create) {

            delete.setMnemonic('D');
            login.setBorder(BorderFactory.createEmptyBorder());
	    login.setEditable(false);
            login.setOpaque(false);
            login.setText(newAccount);

        }

        infoHelpText.setEditable(false);
        infoHelpText.setOpaque(false);
        infoHelpText.setLineWrap(true);
        infoHelpText.setWrapStyleWord(true);

        if(!create) {

            password.setEnabled(false);
            nickname.setEnabled(false);
            save.setEnabled(false);
            delete.setEnabled(false);
            login.setEnabled(false);

        }

        /* Organize text fields. */

	gbc.gridy = 0;
        gbc.ipadx = 4;
        gbc.ipady = 4;
	gbc.fill = gbc.BOTH;
	gbc.anchor = gbc.EAST;
	gbc.weightx = 0;
	textfieldPanel.add(l1, gbc);	
	gbc.gridy = 1;
	textfieldPanel.add(l2, gbc);
	gbc.gridy = 2;
	textfieldPanel.add(l3, gbc);
	gbc.weightx = 1;
	gbc.gridwidth = gbc.REMAINDER;
	gbc.gridx = 1;
	gbc.fill = gbc.BOTH;
	gbc.gridy = 0;
	textfieldPanel.add(login, gbc);
	gbc.gridy = 1;
	textfieldPanel.add(password, gbc);
	gbc.gridy = 2;
	textfieldPanel.add(nickname, gbc);
	gbc.anchor = gbc.EAST;
	gbc.fill = gbc.BOTH;
	gbc.gridx = 0;
	gbc.gridy = 0;
	gbc.weightx = 1;
	gbc.gridwidth = gbc.REMAINDER;

        /* Add help text to bordered panel. */

        userInfoPanel.add(infoHelpText, gbc);

        /* Add text fields to bordered panel. */

        gbc.gridy = 1;
	userInfoPanel.add(textfieldPanel, gbc);

	gbc.anchor = gbc.EAST;
	gbc.gridx = 0;
	gbc.gridy = gbc.RELATIVE;
	gbc.weightx = 1;
        Insets oldInsets = gbc.insets;
        gbc.insets = new Insets(0, 4, 0, 4);
        gbc.ipady = 0;

        /* Create 32 checkboxes for the privileges. Hide some of the
           reserved ones. And add them to the panel. */

	for(int i = 0; i < 32; i++) {

	    String description = privilegeDescriptions[i] == null ?
		"reserved" : privilegeDescriptions[i];
	    JCheckBox b = new JCheckBox(description);

	    if(privilegeDescriptions[i] == null)
		b.setVisible(false);
            
            if(!create)
                b.setEnabled(false);

	    checkboxes.addElement(b);
	    b.setFont((Font) Main.rlo.getProperty("Font.list"));
	    b.setForeground(rlm.getSchemeColor("foreground"));
	    b.setBackground(rlm.getSchemeColor("background"));
	    bitPanel.add(b, gbc);
	}

        gbc.ipady = 4;

	if(create) 

            /* Reasonable defaults. */

            setPrivileges(HLProtocol.AccountInfo.CAN_DOWNLOAD_FILES |
                          HLProtocol.AccountInfo.CAN_UPLOAD_FILES |
                          HLProtocol.AccountInfo.CAN_SEND_CHAT |
                          HLProtocol.AccountInfo.CAN_READ_CHAT |
                          HLProtocol.AccountInfo.CAN_READ_NEWS |
                          HLProtocol.AccountInfo.CAN_POST_NEWS |
                          HLProtocol.AccountInfo.CAN_USE_ANY_NAME);

	gbc.gridx = 0;
        gbc.gridy = 2;
	gbc.weightx = 1;
        gbc.weighty = 1;
	gbc.fill = GridBagConstraints.BOTH;
	gbc.gridwidth = 2;
        gbc.insets = oldInsets;
        //	userInfoPanel.add(scrollPane, gbc);

	if(!create)
	    deleteButtonPanel.add(delete);

	otherButtonPanel.add(cancel);
	otherButtonPanel.add(save);
	buttonPanel.add(deleteButtonPanel, BorderLayout.WEST);
	buttonPanel.add(otherButtonPanel, BorderLayout.EAST);

        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.weightx = 0;
        gbc.weighty = 0;
        gbc.fill = gbc.BOTH;
        gbc.anchor = gbc.NORTHEAST;
        gbc.insets = (Insets) Main.rlo.getProperty("BorderInsets");
        contentPane.add(userInfoPanel, gbc);

        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.weightx = 1;
        gbc.weighty = 1;
        gbc.fill = gbc.BOTH;
        gbc.anchor = gbc.NORTHEAST;
        gbc.insets = (Insets) Main.rlo.getProperty("BorderInsets");
        contentPane.add(scrollPane, gbc);

        gbc.gridx = 0;
        gbc.gridy = 2;
	gbc.weightx = 0; 
        gbc.weighty = 0;
	gbc.fill = gbc.BOTH;
	gbc.anchor = gbc.NORTHEAST;
        gbc.insets = new Insets(0, 8, 0, 8); /* keep a little to the right & left */
	contentPane.add(buttonPanel, gbc);

        SwingUtilities.getRootPane(f).setDefaultButton(save);
        
	f.pack();

	f.setSize(300, 400);

	Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
	Dimension w = f.getSize();

	f.setLocation((d.width / 2) - (w.width / 2), 
		      (d.height / 2) - (w.height / 2));

	f.addWindowListener(this);
	f.show();

        if(create)
            login.requestFocus();
        else
            password.requestFocus();

	rlm.getInterface().registerChild(this);

    }
    
    void setPrivileges(long privs) {
	long bit = 1;
	JCheckBox cb;

	for(int i=0; i<checkboxes.size(); i++) {

	    cb = (JCheckBox) checkboxes.elementAt(i);
            cb.setEnabled(true);

	    if((long) (privs & bit) == bit) 
		cb.setSelected(true);

	    bit *= 2;

	}

    }


    long getPrivileges() {
	long val = 0;
	long bit = 1;
	JCheckBox cb;

	for(int i=0; i<checkboxes.size(); i++) {

	    cb = (JCheckBox) checkboxes.elementAt(i);

	    if(cb.isVisible()) 
		if(cb.isSelected()) 
		    val |= bit;

	    bit *= 2;

	}

	return val;

    }

    /**
     * The following methods implement ActionListener.
     */
    public void actionPerformed(ActionEvent e) {

	try {

	    if(e.getActionCommand().equals("Save")) {

		long privileges = getPrivileges();

		if(create) {

		    rlm.getHLC().requestAccountCreate(login.getText(), password.getText(), nickname.getText(), privileges);

		} else {

                    /* This is evil, better to just show password to
                       user instead of relying on the
                       defaultPasswordString magic value. */

		    String pw;
		    if(password.getText().equals(defaultPasswordString))
			pw = null;
		    else
			pw = password.getText();

		    rlm.getHLC().requestAccountModify(login.getText(), pw, nickname.getText(), privileges);
                    
		}

	    } else if(e.getActionCommand().equals("Delete")) {

		rlm.getHLC().requestAccountDelete(login.getText());

	    }

	} catch (IOException ex) {}

	close();

    }

    /**
     * The following methods implement WindowListener.
     */
    public void windowActivated(WindowEvent e) { }
    public void windowDeactivated(WindowEvent e) { }
    public void windowClosed(WindowEvent e) { }

    public void windowClosing(WindowEvent e) {

	close();

    }

    public void windowIconified(WindowEvent e) { }
    public void windowDeiconified(WindowEvent e) { }
    public void windowOpened(WindowEvent e) {}

    /**
     * The following methods implement Child.
     */
    public void close() {

	rlm.getInterface().unregisterChild(this);

        if(getter != null)
            getter.interrupt();

        SwingUtilities.invokeLater(new Runnable() {

                public void run() {

                    f.dispose();
                    
                }

            });

    }

    /**
     * Invoked when a display property changes.
     */
    public void displayPropertyChanged(String what, Object property) {

	if(what.equals("listfont")) { 
	    
	    for(int i=0; i<checkboxes.size(); i++) 
		((JCheckBox) checkboxes.elementAt(i)).setFont((Font) property);
	    
	} else {

	    for(int i=0; i<checkboxes.size(); i++) {

		((JCheckBox) checkboxes.elementAt(i)).setBackground(rlm.getSchemeColor("background"));
		((JCheckBox) checkboxes.elementAt(i)).setForeground(rlm.getSchemeColor("foreground"));
		bitPanel.setBackground(rlm.getSchemeColor("background"));
		bitPanel.setForeground(rlm.getSchemeColor("foreground"));

	    }
	    
	}

	f.repaint();

    }
    
}

