/**
 * $Id: HLBookmarkFile.java,v 1.2 2001/09/05 04:01:55 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.hotline;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.Hashtable;

/**
 * A class to manipulate Hotline bookmark files.
 */
public class HLBookmarkFile {
    RandomAccessFile raf;
    Hashtable props;
    File file;

    /**
     * Constructs a HLBookmark object for the specified 
     * bookmark file.
     * @param f a bookmark file
     */
    public HLBookmarkFile(File f) {
	file = f;
	props = new Hashtable();
    }

    /**
     * Reads the properties from the bookmark file.
     */
    public void load() throws IOException {
	try {
	    raf = new RandomAccessFile(file, "rw");
	    byte[] magic = new byte[4];
	    raf.readFully(magic, 0, 4);
	    if(new String(magic).equals("HTsc"))
		if(raf.readShort() == 1) {
		    int offset = 135;
		    props.put("login", getString(offset));
		    offset += 34;
		    props.put("password", getString(offset));
		    offset += 34;
		    props.put("address", getString(offset));
		}
		else
		    throw new IOException(file.toString()+": unsupported version");
	    else
		throw new IOException(file.toString()+": not a bookmark file");
	} catch (IOException e) {
	    throw e;
	} finally {
	    if(raf != null) raf.close();
	}
    }

    /**
     * Stores the properties in the bookmark file.
     */
    public void store() throws IOException {
	RandomAccessFile raf = null;
	try {
	    raf = new RandomAccessFile(file, "rw");
	    raf.write(new String("HTsc").getBytes());
	    raf.writeShort(1);
	    raf.write(new byte[129]);
	    raf.write(createString("login"));
	    raf.write(createString("password"));
	    raf.write(createString("address"));
	    int pos = (int) raf.getFilePointer();
	    raf.write(new byte[460 - pos]);
	} catch (IOException e) {
	    throw e;
	} finally {
	    if(raf != null) raf.close();
	}
    }
    
    /**
     * Returns the file that this HLBookmark object
     * refers to.
     */
    public File getFile() {
        return file;
    }

    /**
     * Returns a property. Supported properties are:<BR>
     * <DL><DT><I>login</I>
     * <DD>The server login name.
     * <DT><I>password</I>
     * <DD>The server password.
     * <DT><I>address</I>
     * <DD>The server address (form: xxx.xxx.xxx.xxx:port)
     * </DL>
     * @return the requested property or null
     */     
    public String get(String property) {
	return (String) props.get(property);
    }
	
    /**
     * Stores a property. See the {@link #get get} method for a list of the
     * supported properties.
     * @param property the property to store
     * @param the value of the property
     */
    public void put(String property, String value) {
	props.put(property, value);
    }

    private String getString(int offset) throws IOException {
	raf.seek(offset);
	byte[] s = new byte[raf.readByte()];
	raf.readFully(s, 0, s.length);
	return new String(s);
    }
    private byte[] createString(String t) {
	String prop = (String) props.get(t);
        int len = prop.length() > 32 ? 33 : prop.length();
	byte[] buf = new byte[34];
	buf[0] = (byte) len;
	System.arraycopy(prop.getBytes(), 0, buf, 1, len);
	return buf;
    }

    /**
     * Returns a string representation of this HLBookmarkFile.
     */
    public String toString() {
        return "HLBookmarkFile[" + file.toString() + "]";
    }

}

