/**
 * $Id: HLClient.java,v 1.17 2001/10/05 14:19:07 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.hotline;

import java.net.*;
import java.io.*;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Vector;

import redlight.utils.ToArrayConverters;
import redlight.utils.DebuggerOutput;
import redlight.utils.Meter;
import redlight.utils.InterruptableInputStream;
import redlight.macfiles.MacFile;
import redlight.utils.MacFileUtils;

/**
 * This class provides a high-level API for communicating with Hotline
 * servers.<P>
 *
 * Let's start out with a simple example:<P> 
 *
 * <BLOCKQUOTE>
 * <PRE>
 * <FONT color="#ff0000">// redlight.hotline.* example program.
 * //
 * // This program demonstrates how to connect to a Hotline
 * // server and show some bad manners.</FONT>
 * import redlight.hotline.*;
 * import java.io.*;
 *
 * public class HLDemo {
 *
 *     public static void main(String args[]) {
 *         HL h = new HL("hlserver.com");
 *     }
 *
 * }
 *
 * <FONT color="#ff0000">
 * // We extend {@link redlight.hotline.HLClientAdapter} so we can register
 * // ourselves as a {@link redlight.hotline.HLClientListener} to receive
 * // events from the server. </FONT>
 * class HL extends {@link redlight.hotline.HLClientAdapter} {
 *     HLClient hlc;
 *
 *     HL(String server) {
 *         try {
 *             hlc = new HLClient(server,                  <FONT color="#ff0000">// Server address.</FONT>
 *                                          {@link redlight.hotline.HLProtocol#HTLS_TCPPORT},  <FONT color="#ff0000">// Port number.</FONT>
 *                                          "guest",       <FONT color="#ff0000">// Login.</FONT>
 *                                          "",            <FONT color="#ff0000">// Password.</FONT>
 *                                          "HLDemo",      <FONT color="#ff0000">// Nickname.</FONT>
 *                                          (short) 0);    <FONT color="#ff0000">// Icon number.</FONT>
 *
 *             <FONT color="#ff0000">
 *             // This is where we register ourselves as the handler for
 *             // server events.</FONT>
 *             hlc.addHLClientListener(this);
 *
 *             <FONT color="#ff0000">// Open the connection to the server.</FONT>
 *             hlc.connect();
 *
 *             <FONT color="#ff0000">// Send the user information to the server.</FONT>
 *             hlc.login();
 *
 *             <FONT color="#ff0000">// Show off.</FONT>
 *             demo1();
 *             demo2();
 *             demo3();
 *
 *             <FONT color="#ff0000">// Stay alive for 20 seconds.</FONT>
 *             Thread.currentThread().sleep(20000);
 *
 *         } catch(InterruptedException e) {
 *             <FONT color="#ff0000">// May be thrown by the sleep() call above.</FONT>
 *         } catch(FailedLoginException e) {
 *             <FONT color="#ff0000">// May be thrown from {@link #login}.</FONT>
 *             System.out.println("Could not login to the server: " + e.getMessage());
 *         } catch(IOException e) {
 *             <FONT color="#ff0000">// May be thrown from {@link #connect} or {@link #login}.</FONT>
 *             System.out.println("A network error occurred: " + e.getMessage());
 *         } catch(HLException e) {
 *             <FONT color="#ff0000">// May be thrown in some of the HLClient methods.</FONT>
 *             System.out.println("A Hotline server error occurred: " + e.getMessage());
 *         } finally {
 *             try {
 *                 hlc.close();
 *             } catch(IOException e) {}
 *         }
 *     }
 *
 *     <FONT color="#ff0000">// Retrieve the contents of the root folder.</FONT>
 *     void demo1() throws IOException, {@link redlight.hotline.HLException} {
 *         {@link HLProtocol.FileListComponent}[] files = hlc.getFileList(":");
 *         for(int i=0; i&lt;files.length; i++) 
 *             System.out.println(new String(files[i].fname));
 *     }
 *
 *     <FONT color="#ff0000">// Send everybody a message.</FONT>
 *     void demo2() throws IOException, {@link redlight.hotline.HLException} {
 *         {@link HLProtocol.UserListComponent}[] users = hlc.getUserList();
 *         for(int i=0; i&lt;users.length; i++)
 *             hlc.sendMessage(users[i].sock, "Hi there!");
 *     }
 *
 *     <FONT color="#ff0000">// Write silly text to chat.</FONT>
 *     void demo3() throws IOException {
 *         hlc.sendChat("I am silly");
 *     }
 *
 *     <FONT color="#ff0000">// Inherited from {@link redlight.hotline.HLClientAdapter}.
 *     // This method is invoked when chatting is received.</FONT>
 *     public void handleChat(String msg) {
 *         System.out.println("chat received: "+msg);
 *     }
 *
 *     <FONT color="#ff0000">// Inherited from {@link redlight.hotline.HLClientAdapter}.
 *     // This method is invoked when a message is received.</FONT>
 *     public void handleMessage(int s, String n, String msg) {
 *         System.out.println("message from "+n+": "+msg);
 *         try {
 *             <FONT color="#ff0000">// Return the message that was received to the sender.</FONT>
 *             hlc.sendMessage(s, msg);
 *         } catch(IOException e) {}
 *     }
 *
 * }</PRE>
 * </BLOCKQUOTE>
 *
 * <B>Tasks and transaction IDs</B><P>
 * 
 * A task is a client-initiated transaction. Examples of tasks are
 * logging in, sending a private message and getting a
 * filelist. Hotline identifies different tasks using a transaction
 * ID. <p>
 *
 * This transaction ID can be used for example to wait for completion
 * of a task, using the {@link #waitFor} method. The {@link #waitFor}
 * method blocks until a reply has been received for the task
 * identified by the transaction ID. The contents of the task reply is
 * then returned as a generic Object, which must be typecast to the
 * desired type depending on the task performed. For this reason, a
 * number of convenience functions (with the prefix <i>get</i>) are
 * provided, which perform the typecasting for you automatically.<p>
 *
 * Note that Hotline servers do not send a reply for chat tasks (that
 * is, those created using the {@link #sendChat} and {@link
 * #sendChatAction} methods) unless an error occurs.<p>
 * 
 * <B>Retrieving information: get vs. request</B><P>
 *
 * Some methods exist in both a <TT>request</TT> and a <TT>get</TT> variant.
 * The <TT>get</TT> version fetches the requested information from the server
 * and returns it. The <TT>request</TT> version, by contrast, only requests 
 * the information something and returns immediately. This leaves handling up 
 * to any registered HLClientListeners. <P>
 *
 * The <TT>get</TT> version will block until the data has been received and
 * return it. This is frequently convenient, because it bypasses the need 
 * to register an HLClientListener.<P>
 *
 * Note that the <TT>get</TT> variants can return <TT>null</TT> when an 
 * error occurs. Also, they throw an exception when the connection to
 * the server is closed (either by using the {@link #close} method, or 
 * because the connection with the server was terminated).<P>
 *
 * <B>Files and pathname conventions</B><P>
 *
 * Hotline is originally a Macintosh application. This has some 
 * implications for the way directories and files are handled.<P>
 *
 * Instead of using the familiar slash (/) to separate directories, the Mac
 * uses a colon (:). Thus, a Hotline path to the file <TT>Fubar</TT> might
 * be <TT>"Foo:Bar:Fubar"</TT>. For maximum flexibility you can use the
 * {@link HLProtocol#DIR_SEPARATOR} constant instead of :.<P>
 * 
 * Also, the Mac concept of a file is different from Windows and Unix
 * platforms. Most importantly, Mac files differ in that they can
 * contain two separate data streams, called the data fork and the
 * resource fork. This does not easily translate to Java's file
 * flat-file model. In addition, the Macintosh filesystem tracks some
 * meta-data about files that simply does not exist on other
 * platforms, such as type and creator codes. <p>
 *
 * Therefore, the file transfer API uses {@link
 * redlight.macfiles.MacFile} objects to represent the local file. The
 * application can choose between a number of strategies for dealing
 * with these discrepancies by instantiating the appropriate {@link
 * redlight.macfiles.MacFile} subclass.<p>
 *
 * <B>Newlines</B><P>
 * 
 * Again, Hotline is originally a Macintosh application. The Macintosh uses
 * \r (ASCII 0x0d) to indicate a line ending, while Java has adopted the Unix
 * convention of \n (ASCII 0x0a) for line endings. This means that text
 * received from the server (e.g. messages, chat, news, file comments) may 
 * need to have have \r replaced by \n. You can use the 
 * {@link redlight.utils.TextUtils#findAndReplace TextUtils.findAndReplace} 
 * function to do this, e.g.
 * <BR>
 * <BLOCKQUOTE>
 * <PRE>
 * String converted = TextUtils.findAndReplace(someString, "\r", "\n");</PRE>
 * </BLOCKQUOTE>
 *
 * Conversely, when sending text to a Hotline server, you may need to replace
 * \n by \r.<P>
 * 
 * <b>Threading and event handling</b><p>
 *
 * A HLClient uses a seperate thread for reading packets
 * from the server and a seperate thread for writing packets to the
 * server. In addition, every packet read from the server is
 * dispatched in a separate dispatcher thread.<p>
 * 
 * To terminate all these running threads, it is important to always
 * call the {@link #close} method when you are done.<p>
 * 
 * Most of the HLClient object methods are thread-safe, with the
 * notable exception of {@link #connect} and {@link #login}. Generally
 * thread-safe, but with some restrictions, is {@link #setBlocked}.<p>
 *
 * <B>File transfer</B><P>
 *
 * Transferring files is done using the 
 * {@link #requestFileUpload} and 
 * {@link #requestFileDownload} methods. Both methods share three
 * common parameters:
 * <UL>
 * <LI>a {@link redlight.macfiles.MacFile} object specifying the local file</LI>
 * <LI>a String specifying the remote file/path</LI>
 * <LI>a {@link redlight.utils.Meter} object that monitors the transfer progress</LI>
 * </UL>
 *
 * <U>The <TT>MacFile</TT> object</U><P>
 *
 * The MacFile object describes a Macintosh file, which can
 * contain information that "conventional" (e.g. Unix) files do not have.<P>
 * Creating a MacFile from an ordinary File is accomplished simply
 * by passing the File as an argument to the MacFile
 * constructor:<P>
 *
 * <BLOCKQUOTE><PRE>
 * File f = new File("thefile");
 * MacBinaryMacFile mf = new MacBinaryMacFile(f); // Can also be SplitMacFile or NativeMacFile.
 * </PRE></BLOCKQUOTE>
 *
 * Note that a MacFile may actually consist of multiple files on disk
 * or have it's name changed slightly (e.g. a {@link
 * redlight.macfiles.MacBinaryMacFile} creates a file with ".bin"
 * appended to the filename). See the MacFile documentation for more
 * info. <P>
 *
 * <U>The <TT>Meter</TT> object</U><P>
 *
 * The <TT>Meter</TT> object monitors the transfer progress. Interrupting a 
 * transfer is also done through the <TT>Meter</TT> object. For example:<P>
 *
 * <BLOCKQUOTE>
 * <PRE>
 * <FONT color="#ff0000">
 * // A simple example of a Meter object that monitors a transfer.
 * // The stopTransfer method demonstrates how to stop a transfer.</FONT>
 *
 * public class TransferMeter implements {@link redlight.utils.Meter} {
 *     {@link redlight.utils.MeterSource} transferrer;
 *
 *     void stopTransfer() {
 *          <FONT color="#ff0000">// Interrupts the transfer.</FONT>
 *         transferrer.interrupt();
 *
 *     }
 *
 *     <FONT color="#ff0000">// Following methods implement Meter.</FONT>
 *     public void startMeter(String s, int size) {
 *         System.out.println("transfer started, file = "+s+", size = "+size);
 *     }
 *
 *     public void progressMeter(int done) {
 *         System.out.println("transfer progress = "+done);
 *     }
 *
 *     public void stopMeter() {
 *         System.out.println("transfer completed successfully");
 *     }
 *
 *     public void stopMeterWithError(Throwable t) {
 *         System.out.println("transfer completed abnormally: "+t.getMessage());
 *     }
 *
 *     public void setMeterSource({@link redlight.utils.MeterSource} ms) {
 *         transferrer = ms;
 *     }
 *
 *     public MeterSource getMeterSource() {
 *         return transferrer;
 *     }
 *
 * }
 * </PRE>
 * </BLOCKQUOTE>
 * <P>
 *
 * Note that you cannot depend on the {@link #close} method to stop
 * transfers that are in progress. A file transfer <i>has to be</i>
 * stopped by calling the {@link redlight.utils.MeterSource#interrupt}
 * method from the {@link redlight.utils.Meter} that monitors that
 * transfer (see the stopTransfer method in the example
 * above).
 * <P> */
public class HLClient {

    /* Communications. */

    InetAddress	host;
    int	port;
    Socket socket;
    DataOutputStream output;
    DataInputStream input;
    ReaderThread reader;
    HLClientDispatcher hlr;
    Hashtable tasks, transfers;

    /* Vector containing the event listeners for this client. */

    Vector hle;

    /* Setup defaults for login data. */

    private static String default_nick = "Red Light library";
    private static String default_login = "";
    private static String default_password = "";
    private static short default_icon = 2015;
    String nick = default_nick;
    String login = default_login;
    String password = default_password;
    int icon = default_icon;

    /* Internal state flags and communication variables. */

    boolean connected = false, blocked = false;
    int	transaction = 2, serverVersion = 0;
    Integer closeLock = new Integer(0);

    /* Create these objects once instead of creating one for every
       transaction. */

    HLProtocol hlp;

    /**
     * Prepare a connection to the specified server.
     * @param server the server name to log into.
     */
    public HLClient(String server) throws IOException {

	this(InetAddress.getByName(server), HLProtocol.HTLS_TCPPORT);

    }

    /**
     * Prepare a connection to the specified host address.
     * @param host the IP address of the host to log into.
     */
    public HLClient(InetAddress host) throws IOException {

	this(host, HLProtocol.HTLS_TCPPORT);

    }

    /**
     * Prepare a connection to the specified server at the specified port.
     * @param server the server name
     * @param port the port number
     */
    public HLClient(String server, int port) throws IOException {

	this(InetAddress.getByName(server), 
	     port, 
	     default_login, 
	     default_password, 
	     default_nick, 
	     default_icon);

    }

    /**
     * Prepare a connection to the specified host at the specified port.
     * @param host the IP address of the host to log into.
     * @param port the port number
     */
    public HLClient(InetAddress host, int port) throws IOException {

	this(host, 
	     port, 
	     default_login, 
	     default_password, 
	     default_nick, 
	     default_icon);

    }

    /**
     * Prepare a connection to the specified server at the specified port
     * with specified user credentials. Usually "guest" and "" (empty string)
     * will suffice as credentials.
     * @param server the server name
     * @param port the port number
     * @param login the hotline user login
     * @param password the hotline user password
     * @param nick the hotline user nickname
     * @param icon the hotline user icon
     */
    public HLClient(String server, 
                    int port, 
                    String login, 
                    String password, 
                    String nick, 
                    short icon) throws UnknownHostException, IOException {

	this(InetAddress.getByName(server), 
	     port, 
	     login, 
	     password, 
	     nick, 
	     icon);

    }

    /**
     * Prepare a connection to the specified host at the specified port
     * with specified user credentials. Usually "guest" and "" (empty string)
     * will suffice as credentials.
     * @param host the IP address of the host to log into.
     * @param port the port number
     * @param l the hotline user login
     * @param p the hotline user password
     * @param n the hotline user nickname
     * @param i the hotline user icon
     */
    public HLClient(InetAddress h, 
                    int pr, 
                    String l, 
                    String ps, 
                    String n, 
                    short i) throws UnknownHostException, IOException {
        
	host = h;
	port = pr;
	login = l;
	password = ps;
	nick = n;
	icon = i;
	tasks = new Hashtable();
	transfers = new Hashtable();
	hle = new Vector();
        hlp = new HLProtocol();
        hlr = new HLClientDispatcher(this);

    }

    /* Connecting, logging in, disconnecting */

    /**
     * Establishes a connection to a Hotline server. This entails
     * opening a communications socket and dispatching two threads
     * to take care of reading / writing data to the server. This method
     * is not thread-safe.
     * @throws BadMagicException if the server does not respond with
     *         the expected magic value.
     */
    public void connect() throws IOException, BadMagicException {

	if(!connected) {

            connected = true;

            DebuggerOutput.debug("HLClient.connect: opening socket ...");
	    socket = new Socket(host, (char) port);
            DebuggerOutput.debug("HLClient.connect: socket opened");
            socket.setSoTimeout(1000);
            socket.setSoLinger(true, 5);

	    output = new DataOutputStream(socket.getOutputStream());
            input = new DataInputStream(new InterruptableInputStream(socket.getInputStream()));

	    output.writeBytes(HLProtocol.HTLC_MAGIC);
	    output.flush();
            DebuggerOutput.debug("HLClient.connect: wrote client magic");

	    byte[] buf = new byte[HLProtocol.HTLS_MAGIC_LEN];
	    input.readFully(buf, 0, HLProtocol.HTLS_MAGIC_LEN);
            DebuggerOutput.debug("HLClient.connect: read server magic");

	    if(!HLProtocol.HTLS_MAGIC.equals(new String(buf))) {

                socket.close();
                output = null;
                input = null;
	        throw new BadMagicException("not a Hotline server: bad magic");

            }

	    reader = new ReaderThread(this, input);
            reader.setBlocked(blocked);
            reader.setName("HLClient ReaderThread " + host.toString());
	    reader.start();

	    DebuggerOutput.debug("HLClient.connect: ("+getHost().toString()+").");

	}

    }

    /**
     * Log in with the credentials given during construction
     * or the default credentials.
     */
    public void login() throws IOException, HLException, InterruptedException {

	login(login, password, nick, icon);

    }
	
    /**
     * Log in with specified credentials. When the credentials are
     * incorrect, FailedLoginException is thrown. As a special case of
     * login failure, on >= 1.5 servers, when one of the registered
     * event listeners rejects the agreement, an
     * AgreementRejectedException is thrown. For < 1.5 servers, the
     * agreement is basically ignored; see also {@link
     * redlight.hotline.HLClientListener#handleAgreement}.
     * This method is not thread-safe.
     * @param l an account name
     * @param p the account password
     * @param n the nickname to use
     * @param ic the icon to use 
     */
    public void login(String login, 
                      String password, 
                      String nick, 
                      int icon) throws IOException, HLException, InterruptedException {

        if(login == null || password == null || nick == null)
            throw new IllegalArgumentException("null argument");

	this.login = login;
	this.password = password;
	this.nick = nick;
	this.icon = icon;

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hlp.new DataComponent(HLProtocol.HTLC_DATA_LOGIN, HLProtocol.invert(login.getBytes())),
                hlp.new DataComponent(HLProtocol.HTLC_DATA_PASSWORD, HLProtocol.invert(password.getBytes())),
                hlp.new DataComponent(HLProtocol.HTLC_DATA_NICK, nick.getBytes()),
                hlp.new DataComponent(HLProtocol.HTLC_DATA_ICON, ToArrayConverters.intToByteArray(icon)),
                
            };
        
        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_LOGIN, 
                           nextTrans(), 
                           dataComponents);

	HLTask task = createTask(packet, false);

        DebuggerOutput.debug("HLClient: writing login header");

        packet.write(output);

        /* Now wait for the server reply. */

        DebuggerOutput.debug("HLClient: waiting for reply from login header");

        try {

            task.getData();

        } finally {

            disposeTask(packet.header.trans);

        }

        /* If we did not get an error, look at the server version. For
           > 1.5 servers, we need to wait until we receive the
           agreement. */

        if(serverVersion >= 150) {

            try {

                hlr.waitForAgreement();
                
                if(hlr.agreementAccepted) {

                    sendAgree();

                } else {

                    throw new AgreementRejectedException("agreement rejected");

                }

            } catch(InterruptedException e) {

                throw new HLException("wait for agreement interrupted");

            }
            
        }

    }

    /**
     * Close the connection with the server. This method must always
     * be called in order to cleanly terminate running threads. In
     * particular, it must be called from the
     * {@link HLClientListener#handleDisconnect} method. This method
     * first closes the socket to the server. Then, all pending
     * tasks are cancelled, which causes any threads waiting on
     * results from the getXXX() and waitFor() methods to get an
     * exception.
     */
    public void close() throws IOException {

        setBlocked(false);
        
        /* Interrupt the reader to let it know that it is being
           closed. */

        if(reader != null)
            reader.interrupt();

        /* Close the socket to wake up the reader thread. */
        
        socketClose();

        /* Wait for the reader thread to die. */

        if(reader != null) {

            try {
                
                DebuggerOutput.debug("HLClient.close: joining reader");
                reader.join();
                
            } catch(InterruptedException e) {}
            
        }
        
        internalDisconnect();

        DebuggerOutput.debug("HLClient.close: ("+getHost().toString()+").");

    }

    private synchronized void socketClose() throws IOException {

	if(socket != null) {

            DebuggerOutput.debug("HLClient.close: closing socket");
	    socket.close();
            socket = null;

        }

        /* InterruptableStream overrides close to interrupt any
           pending reads. */

        if(input != null) {

            DebuggerOutput.debug("HLClient.close: closing input");
            input.close();
            input = null;

        }

    }

    /* Event handler */
    
    /**
     * Adds an event handler.
     * @param h the HLClientListener object to add
     */
    public synchronized void addHLClientListener(HLClientListener h) {

	hle.addElement(h);

    }

    /**
     * Removes an event handler.
     * @param h the HLClientListener object to remove
     */
    public synchronized void removeHLClientListener(HLClientListener h) {

	hle.removeElement(h);

    }

    /* Query methods */

    /**
     * Returns the server version.
     * @return the server version (e.g. 100, 150, ...)
     */
    public int getServerVersion() {

	return serverVersion;

    }

    /**
     * Returns the host.
     * @return an InetAddress object
     */
    public InetAddress getHost() {

	return host;

    } 
	
    /**
     * Returns the port.
     * @return the port number
     */
    public int getPort() {

	return port;

    } 

    /**
     * Returns the login for this connection.
     * @return login name for this connection.
     */
    public String getLogin() {

	return login;

    }

    /**
     * Returns the password for this connection.
     * @return password for this connection.
     */
    public String getPassword() {

	return password;

    }

    /**
     * Returns the nick for this connection.
     * @return nick for this connection.
     */
    public String getNick() {

	return nick;

    }

    /**
     * Returns the icon for this connection.
     * @return icon for this connection.
     */
    public int getIcon() {

	return icon;

    }

    /**
     * Queries connection state.
     * @return true if the connection with the server is active.
     */
    public synchronized boolean isConnected() {

	return connected;

    }

    /** 
     * Queries whether this connection is blocked.
     * @return true or false
     */
    public boolean isBlocked() {

	return blocked;

    }

    /* Modifier methods */

    /**
     * Sets the server version.
     * @param v the version.
     */
    void setServerVersion(int v) {

	serverVersion = v;

    }

    /**
     * This method controls whether event processing is active or
     * not. If the argument is true, then processing of incoming
     * server events is suspended. This is useful in the case where
     * you need to add a new event listener, and need to be sure that
     * it does not "miss" any events. It is OK to call this method
     * at any time, except during {@link #connect}.
     * @param b true to block, false to unblock 
     */
    public void setBlocked(boolean b) {

        blocked = b;

        if(reader != null)
            reader.setBlocked(b);

    }
    
    /* Server commands */

    /**
     * Requests file (or folder) info.
     * @param path the file to get info on
     * @return transaction id
     */
    public int requestFileInfo(String path) throws IOException {

        return requestFileInfo(path, true);

    }

    public int requestFileInfo(String path, 
                               boolean disposeWhenReplyReceived) throws IOException {

        if(path == null)
            throw new IllegalArgumentException("path == null");

        int trans = nextTrans();

        HLProtocol.DataComponent[] dataComponents = 
            ComponentFactory.createPathComponents(path, 
                                                  HLProtocol.HTLC_DATA_DIR, 
                                                  true);

        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_FILE_GETINFO, 
                           trans, 
                           dataComponents);

	createTask(packet, disposeWhenReplyReceived);
	packet.write(output);

	return packet.header.trans;

    }

    /**
     * Requests the information for the specified account.
     * @param login the account to request information for
     * @return transaction id
     */
    public int requestAccountInfo(String login) throws IOException {

        return requestAccountInfo(login, true);

    }

    public int requestAccountInfo(String login,
                                  boolean disposeWhenReplyReceived) throws IOException {

        if(login == null)
            throw new IllegalArgumentException("login == null");

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                hlp.new DataComponent(HLProtocol.HTLC_DATA_LOGIN, login.getBytes()),
            };
        
        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_ACCOUNT_READ, 
                           nextTrans(),
                           dataComponents);

	createTask(packet, disposeWhenReplyReceived);
	packet.write(output);

	return packet.header.trans;

    }

    /**
     * Upload a file. To monitor the upload progress, you must create
     * a {@link redlight.utils.Meter} object and supply it as the third
     * argument. <p>
     * The resume flag indicates whether or not this upload should
     * resume an earlier, partially completed upload. However, this 
     * is only possible if the file already partially exists on the 
     * server. You will get a task error if you try to resume a file
     * that does not already exist.<p>
     * The proper course of action for an application is to first 
     * check (using {@link #getFileList} for example) whether the file
     * exists on the server, then to set the resume flag appropriately.
     * @param local the local file
     * @param remotePath the destination path on the server
     * @param meter the non-null Meter object that monitors this transfer
     * @param resume if true, resumes partially completed transfer
     * @return transaction id
     * @throws IllegalArgumentException when meter object is null.  
     */
    public int requestFileUpload(MacFile local, 
                                 String remotePath, 
                                 Meter meter,
                                 boolean resume) throws IOException {

        return requestFileUpload(local, remotePath, meter, resume, true);

    }

    public int requestFileUpload(MacFile local, 
                                 String remotePath, 
                                 Meter meter,
                                 boolean resume,
                                 boolean disposeWhenReplyReceived) throws IOException {

        if(meter == null) 
	    throw new IllegalArgumentException("meter object cannot be null");

        if(remotePath == null || local == null) 
            throw new IllegalArgumentException("null argument");

        remotePath = remotePath + HLProtocol.DIR_SEPARATOR + 
            local.getFile().getName();

        HLProtocol.DataComponent[] pathComponents = 
            ComponentFactory.createPathComponents(remotePath, 
                                                  HLProtocol.HTLC_DATA_DIR, 
                                                  true);

        /* Create and serialize this object because we need to know
           it's length in advance -- apparently Hotline expects the
           length of download reply in it's entirety (excluding the
           HTXF block). Changes here mean changes in
           HLClientDispatcher.java. (There is also a race here; need
           to add the fti to the tx Transfer object and reuse in
           HLClientDispatcher). */

        HLProtocol.FileTransferInfo fti = 
            hlp.new FileTransferInfo(local.getFile().getName(),
                                     local.getType(),
                                     local.getCreator(),
                                     local.getComment(),
                                     local.getCreationDate(),
                                     local.getModificationDate(),
                                     local.getFinderFlags());
                                     
        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[resume ? 4 : 3];

        dataComponents[0] = hlp.new DataComponent(HLProtocol.HTLC_DATA_XFERSIZE, ToArrayConverters.intToByteArray(16 /* DATA block length */ + 16 /* MACR block length */ + fti.data.length + (int) local.getResourceSize() + (int) local.getDataSize()));
        dataComponents[1] = pathComponents[1];
        dataComponents[2] = pathComponents[0];

        if(resume)           
            dataComponents[3] = hlp.new DataComponent(HLProtocol.HTLC_DATA_RESUME, ToArrayConverters.shortToByteArray(1));

        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_FILE_PUT, 
                           nextTrans(), 
                           dataComponents);

	Transfer tx = new Transfer();
	tx.local = local;
	tx.remote = remotePath;
	tx.local_data_size = local.getDataSize();
	tx.local_rsrc_size = local.getResourceSize();
	tx.remote_data_size = tx.remote_rsrc_size = 0;
	tx.meter = meter;

	createTask(packet, disposeWhenReplyReceived);
	createTransfer(packet, tx);
	packet.write(output);

	return packet.header.trans;

    }

    /**
     * Grab the data fork of a file.
     * The meter and inputPipe arguments may not be null.
     */
    public int requestDataForkDownload(String path,
                                       PipedInputStream inputPipe,
                                       Meter meter) throws IOException {
        
        if(inputPipe == null)
            throw new IllegalArgumentException("inputPipe == null");

        return requestForkOrFileDownload(path,
                                         null,
                                         inputPipe,
                                         meter,
                                         HLProtocol.DL_DATA_FORK,
                                         true);
        
    }
    
    /**
     * Download a file. To monitor the download progress, you must
     * create a {@link redlight.utils.Meter} object and supply it as
     * the third argument. <p>
     * If the local file already exists, then <TT>requestFile</TT>
     * assumes it should resume the transfer.
     * @param path the file to download
     * @param local the local file to retrieve to
     * @param meter the (non-null) Meter object that meters this transfer
     * @return transaction id
     * @throws IllegalArgumentException when meter object is null.  
     */
    public int requestFileDownload(String path, 
                                   MacFile local, 
                                   Meter meter) throws IOException {

        return requestFileDownload(path, local, meter, true);

    }

    public int requestFileDownload(String path, 
                                   MacFile local, 
                                   Meter meter,
                                   boolean disposeWhenReplyReceived) throws IOException {

        if(local == null)
            throw new IllegalArgumentException("local == null");

        return requestForkOrFileDownload(path, 
                                         local, 
                                         null,
                                         meter, 
                                         HLProtocol.DL_WHOLE_FILE, 
                                         disposeWhenReplyReceived);

    }

    private int requestForkOrFileDownload(String path, 
                                          MacFile local, 
                                          PipedInputStream inputPipe,
                                          Meter meter,
                                          int forkOrFile,
                                          boolean disposeWhenReplyReceived) throws IOException {

	if(meter == null) 
	    throw new IllegalArgumentException("meter object cannot be null");

        if(path == null) 
            throw new IllegalArgumentException("null argument");

        HLProtocol.DataComponent[] pathComponents = 
            ComponentFactory.createPathComponents(path, 
                                                  HLProtocol.HTLC_DATA_DIR, 
                                                  true);

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {

                hlp.new DataComponent(HLProtocol.HTLC_DATA_RESUME, 
                                      ToArrayConverters.shortToByteArray(2)),
                pathComponents[1],
                pathComponents[0],

            };

        if(forkOrFile != HLProtocol.DL_DATA_FORK) 
            dataComponents[0] = hlp.new ResumeTransferComponent(local.getDataFork().size(), local.getResourceFork().size());

        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_FILE_GET, 
                           nextTrans(), 
                           dataComponents);

	Transfer tx = new Transfer();
        tx.name = path;
	tx.local = local;
        tx.inputPipe = inputPipe;
	tx.remote = path;

        if(forkOrFile != HLProtocol.DL_DATA_FORK) {

            tx.local_data_size = (int) local.getDataSize();
            tx.local_rsrc_size = (int) local.getResourceSize();

        }

	tx.remote_data_size = tx.remote_rsrc_size = 0;
	tx.meter = meter;
        tx.forkOrFile = forkOrFile;

	createTask(packet, disposeWhenReplyReceived);
	createTransfer(packet, tx);
        packet.write(output);

	return packet.header.trans;

    }
	
    /**
     * Requests the file list.
     * @param path the path whose contents to retrieve
     * @return transaction id
     */
    public int requestFileList(String path) throws IOException {

        return requestFileList(path, true);

    }

    public int requestFileList(String path,
                               boolean disposeWhenReplyReceived) throws IOException {

        if(path == null)
            throw new IllegalArgumentException("path == null");

        HLProtocol.DataComponent[] pathComponents = 
            ComponentFactory.createPathComponents(path, 
                                                  HLProtocol.HTLC_DATA_DIR, 
                                                  false);

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {

                pathComponents[0],

            };

        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_DIR_LIST, 
                           nextTrans(), 
                           dataComponents);

	createTask(packet, disposeWhenReplyReceived);
	packet.write(output);

	return packet.header.trans;

    }

    /**
     * Requests the user list.
     * @return transaction id
     */
    public int requestUserList() throws IOException {

        return requestUserList(true);

    }

    public int requestUserList(boolean disposeWhenReplyReceived) throws IOException {

        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_USER_LIST, 
                           nextTrans(), 
                           null);

	createTask(packet, disposeWhenReplyReceived);
	packet.write(output);

	return packet.header.trans;

    }

    /**
     * Requests the news.
     * @return transaction id
     */
    public int requestNews() throws IOException {

        return requestNews(true);

    }

    public int requestNews(boolean disposeWhenReplyReceived) throws IOException {

        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_NEWS_GET, 
                           nextTrans(), 
                           null);

	createTask(packet, disposeWhenReplyReceived);
	packet.write(output);

	return packet.header.trans;

    } 
	
    /**
     * Requests user info.
     * @param sock sock of user to get info on
     * @return transaction id
     */
    public int requestUserInfo(int sock) throws IOException {

        return requestUserInfo(sock, true);

    }

    public int requestUserInfo(int sock, 
                               boolean disposeWhenReplyReceived) throws IOException {

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {

                hlp.new DataComponent(HLProtocol.HTLC_DATA_SOCKET, 
                                      ToArrayConverters.intToByteArray(sock)),

            };

        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_USER_GETINFO, 
                           nextTrans(), 
                           dataComponents);

	createTask(packet, disposeWhenReplyReceived);
	packet.write(output);

	return packet.header.trans;

    }

    /** 
     * Requests account information and returns it; returns 
     * null on error. Note that the password field in the 
     * {@link redlight.hotline.HLProtocol.AccountInfo}
     * structure will always contain garbage. If you wish to
     * modify an existing account but not change the password,
     * you must supply <TT>null</TT> as the password in 
     * {@link redlight.hotline.HLClient#sendAccountChange}.
     * This function blocks until the requested data has been
     * received. 
     * @return account information
     */
    public HLProtocol.AccountInfo getAccountInfo(String login) throws IOException, HLException, InterruptedException {

	Object o = waitFor(requestAccountInfo(login, false));
	return (HLProtocol.AccountInfo) o;
        
    }

    /**
     * Requests info on a file and returns it; returns null on error.
     * This function blocks until the requested data has been
     * received. 
     * @param path the filesystem object to get info on
     * @return information on the filesystem object
     */
    public HLProtocol.FileInfo getFileInfo(String path) throws IOException, HLException, InterruptedException {

	Object o = waitFor(requestFileInfo(path, false));
	return (HLProtocol.FileInfo) o;

    }

    /**
     * Requests the file list and returns it; returns null on error.
     * This function blocks until the requested data has been
     * received. 
     * @param path the path to retrieve
     * @return an array of files
     */
    public HLProtocol.FileListComponent[] getFileList(String path) throws IOException, HLException, InterruptedException {

	Object o = waitFor(requestFileList(path, false));
        return (HLProtocol.FileListComponent[]) o;

    }

    /**
     * Requests the user list and returns it; returns null on error.
     * This function blocks until the requested data has been
     * received.
     * @return an array of users
     */
    public HLProtocol.UserListComponent[] getUserList() throws IOException, HLException, InterruptedException {

	Object o = waitFor(requestUserList(false));
	return (HLProtocol.UserListComponent[]) o;

    }

    /**
     * Requests the news and returns it; returns null on error.
     * This function blocks until the requested data has been
     * received.
     * @return the news
     */
    public String getNews() throws IOException, HLException, InterruptedException {

	Object o = waitFor(requestNews(false));
	return (String) o;

    }
	
    /**
     * Requests user info and returns it; returns null on error.
     * This function blocks until the requested data has been
     * received.
     * @return user info
     */
    public String getUserInfo(int sock) throws IOException, HLException, InterruptedException {

	Object o = waitFor(requestUserInfo(sock, false));
	return (String) o;
		
    }

    /**
     * Changes the name and comment of a file (or directory).
     * @param path the file to change
     * @param newName the new name of the file
     * @param newComment the new comment of the file
     */
    public int requestFileInfoChange(String path, 
                                     String newName, 
                                     String newComment) throws IOException {

        return requestFileInfoChange(path, newName, newComment, true);

    }

    public int requestFileInfoChange(String path, 
                                     String newName, 
                                     String newComment,
                                     boolean disposeWhenReplyReceived) throws IOException {

        if(path == null || newName == null || newComment == null)
            throw new IllegalArgumentException("null argument");

        HLProtocol.DataComponent[] pathComponents = 
            ComponentFactory.createPathComponents(path, 
                                                  HLProtocol.HTLC_DATA_DIR, 
                                                  true);

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {

                pathComponents[1],
                pathComponents[0],
                hlp.new DataComponent(HLProtocol.HTLC_DATA_FILE_RENAME, newName.getBytes()),
                hlp.new DataComponent(HLProtocol.HTLS_DATA_FILE_COMMENT, newComment.getBytes()),

            };

        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_FILE_SETINFO, 
                           nextTrans(), 
                           dataComponents);

	createTask(packet, disposeWhenReplyReceived);
	packet.write(output);

	return packet.header.trans;

    }

    /**
     * This is a dummy method to let > 1.5 servers know
     * that the agreement has been accepted or something...
     * @return transaction id.
     */
    void sendAgree() throws IOException {
        
        /*
        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hlp.new DataComponent(HLProtocol.HTLC_DATA_NICK, nick.getBytes()),
                hlp.new DataComponent(HLProtocol.HTLC_DATA_ICON, ToArrayConverters.charToByteArray((char) icon)),
                
            };
        */

        /* FIXME: Must do a createTask() here. */

        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_AGREE, 
                           nextTrans(), 
                           null);

        packet.write(output);

    }

    /**
     * Creates an alias (like a symbolic link).  The below example
     * creates an alias "testfile" in the root directory, which points
     * to the file "testfile" in the directory "test":<P>
     * <TT>hlc.sendFileMakeAlias(":test:testfile", ":");</TT><BR>
     * @param source the object on the server to make an alias to
     * @param target the location of the alias
     * @return transaction id 
     */
    public int requestFileMakeAlias(String source, 
                                    String target) throws IOException {

        return requestFileMakeAlias(source, target, true);

    }

    public int requestFileMakeAlias(String source, 
                                    String target,
                                    boolean disposeWhenReplyReceived) throws IOException {

        return requestFileMoveOrMakeAlias(HLProtocol.HTLC_HDR_FILE_MAKE_ALIAS, 
                                          source, 
                                          target,
                                          disposeWhenReplyReceived);

    }

    /**
     * Moves a file.
     * @param source the file on the server to move
     * @param target the location to move the file to
     * @return transaction id
     */
    public int requestFileMove(String source, 
                               String target) throws IOException {

        return requestFileMove(source, target, true);

    }

    public int requestFileMove(String source, 
                               String target,
                               boolean disposeWhenReplyReceived) throws IOException {

        return requestFileMoveOrMakeAlias(HLProtocol.HTLC_HDR_FILE_MOVE, 
                                          source, 
                                          target,
                                          disposeWhenReplyReceived);

    }

    /**
     * Moves or links a file.
     * @param kind HLProtocol.HTLC_HDR_FILE_MAKE_ALIAS or 
     * HLProtocol.HTLC_HDR_FILE_MOVE.
     * @param source the source file.
     * @param target the target file.
     */
    private int requestFileMoveOrMakeAlias(int kind,
                                           String source, 
                                           String target,
                                           boolean disposeWhenReplyReceived) throws IOException {

        if(source == null || target == null)
            throw new IllegalArgumentException("null argument");

        HLProtocol.DataComponent[] sourcePathComponents = 
            ComponentFactory.createPathComponents(source, 
                                                  HLProtocol.HTLC_DATA_DIR, 
                                                  true);

        HLProtocol.DataComponent[] targetPathComponents = 
            ComponentFactory.createPathComponents(target, 
                                                  HLProtocol.HTLC_DATA_DIR_RENAME, 
                                                  false);
        
        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                sourcePathComponents[1],
                sourcePathComponents[0],
                targetPathComponents[0],
                
            };

        HLProtocol.Packet packet = 
            hlp.new Packet(kind, 
                           nextTrans(), 
                           dataComponents);

	createTask(packet, disposeWhenReplyReceived);
	packet.write(output);

	return packet.header.trans;

    }

    /**
     * Deletes a file on the server.
     * @param path the file on the server to delete
     * @return transaction id
     */
    public int requestFileDelete(String path) throws IOException {

        return requestFileDelete(path, true);

    }

    public int requestFileDelete(String path,
                                 boolean disposeWhenReplyReceived) throws IOException {

        HLProtocol.DataComponent[] pathComponents = 
            ComponentFactory.createPathComponents(path, 
                                                  HLProtocol.HTLC_DATA_DIR, 
                                                  true);
        
        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                pathComponents[1],
                pathComponents[0],
                
            };

        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_FILE_DELETE, 
                           nextTrans(), 
                           dataComponents);

	createTask(packet, disposeWhenReplyReceived);
	packet.write(output);

	return packet.header.trans;

    }

    /**
     * Creates a directory.
     * @param path the directory to create on the server
     * @return transaction id
     */
    public int requestDirectoryCreate(String path) throws IOException {

        return requestDirectoryCreate(path, true);

    }

    public int requestDirectoryCreate(String path,
                                      boolean disposeWhenReplyReceived) throws IOException {

        HLProtocol.DataComponent[] pathComponents = 
            ComponentFactory.createPathComponents(path, 
                                                  HLProtocol.HTLC_DATA_DIR, 
                                                  true);
        
        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                pathComponents[1],
                pathComponents[0],
                
            };

        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_DIR_CREATE, 
                           nextTrans(), 
                           dataComponents);

	createTask(packet, disposeWhenReplyReceived);
	packet.write(output);

	return packet.header.trans;

    }
	
    /**
     * Changes our nickname and icon. The server may simply ignore
     * us if it doesn't like the new settings. Otherwise the server
     * sends a "user change" message. Either way, the nick and icon
     * as returned by {@link #getNick} and {@link getIcon} are 
     * changed by this method.
     * @param n the new nick name for ourselves
     * @param ic the new icon for ourselves
     */
    public void sendUserChange(String nick, 
                               int icon) throws IOException {

        if(nick == null)
            throw new IllegalArgumentException("nick == null)");

        this.nick = nick;
        this.icon = icon;

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hlp.new DataComponent(HLProtocol.HTLC_DATA_ICON, ToArrayConverters.charToByteArray((char) icon)),
                hlp.new DataComponent(HLProtocol.HTLC_DATA_NICK, nick.getBytes()),
                
            };

        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_USER_CHANGE,
                           nextTrans(), 
                           dataComponents);

	packet.write(output);

    }

    /**
     * Deletes the specified account.
     * @param login the account to delete
     * @return transaction id
     */
    public int requestAccountDelete(String login) throws IOException {

        return requestAccountDelete(login, true);

    }

    public int requestAccountDelete(String login,
                                    boolean disposeWhenReplyReceived) throws IOException {

        if(login == null)
            throw new IllegalArgumentException("login == null)");
        
        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hlp.new DataComponent(HLProtocol.HTLC_DATA_LOGIN, HLProtocol.invert(login.getBytes())),
                
            };
        
        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_ACCOUNT_DELETE,
                           nextTrans(), 
                           dataComponents);
        
	createTask(packet, disposeWhenReplyReceived);
	packet.write(output);
        
	return packet.header.trans;

    }

    /**
     * Modifies an existing user account. If password is <TT>null</TT>,
     * then the existing password is not changed.
     * @param login the login name
     * @param password the new password
     * @param nick the new nickname
     * @param privileges the new privileges 
     * (see {@link redlight.hotline.HLProtocol}
     * @return transaction id
     */
    public int requestAccountModify(String login, 
                                    String password,
                                    String nick, 
                                    long privileges) throws IOException {
        
        return requestAccountModify(login, password, nick, privileges, true);

    }

    public int requestAccountModify(String login, 
                                    String password,
                                    String nick, 
                                    long privileges,
                                    boolean disposeWhenReplyReceived) throws IOException {
        
	return requestAccountCreateOrModify(login, 
                                            password, 
                                            nick, 
                                            privileges, 
                                            HLProtocol.HTLC_HDR_ACCOUNT_MODIFY,
                                            disposeWhenReplyReceived);

    }

    /**
     * Create a new user account.
     * @param login the login name
     * @param password the new password
     * @param nick the new nickname
     * @param privileges the new privileges 
     * (see {@link redlight.hotline.HLProtocol.AccountInfo}
     * @return transaction id
     */
    public int requestAccountCreate(String login, 
                                    String password,
                                    String nick, 
                                    long privileges) throws IOException {

        return requestAccountCreate(login, password, nick, privileges, true);

    }

    public int requestAccountCreate(String login, 
                                    String password,
                                    String nick, 


                                    long privileges,
                                    boolean disposeWhenReplyReceived) throws IOException {

	return requestAccountCreateOrModify(login, 
                                            password, 
                                            nick, 
                                            privileges, 
                                            HLProtocol.HTLC_HDR_ACCOUNT_CREATE,
                                            disposeWhenReplyReceived);

    }

    /**
     * Internal method for modifying / creating a user account.
     */
    private int requestAccountCreateOrModify(String login, 
                                             String password,
                                             String nick, 
                                             long privileges,
                                             int packetType,
                                             boolean disposeWhenReplyReceived) throws IOException {

        if(login == null || nick == null)
            throw new IllegalArgumentException("null argument");
        
        byte[] p = (password == null ? 
                    new byte[1] : 
                    HLProtocol.invert(password.getBytes()));
                    
        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hlp.new DataComponent(HLProtocol.HTLC_DATA_LOGIN, HLProtocol.invert(login.getBytes())),
                hlp.new DataComponent(HLProtocol.HTLC_DATA_PASSWORD, p),
                hlp.new DataComponent(HLProtocol.HTLC_DATA_NICK, nick.getBytes()),
                hlp.new DataComponent(HLProtocol.HTLS_DATA_PRIVILEGES, ToArrayConverters.swapByteArray(ToArrayConverters.longToByteArray(privileges))),
                
            };
        
        HLProtocol.Packet packet = 
            hlp.new Packet(packetType,
                           nextTrans(), 
                           dataComponents);
        
	createTask(packet, disposeWhenReplyReceived);
	packet.write(output);
        
	return packet.header.trans;

    }
    
    /**
     * Sends a string to the chat. HL servers do not consider chat to
     * be a task. hxd on the other hand does consider chat to be a task, 
     * and will send a task error object in case we do not have permission
     * to send that. What's strange however is that while hxd will
     * send a task error for a failed chat request, it will not send
     * a task complete object for a successfull chat request. 
     * @param s the string to send
     */
    public void sendChat(String s) throws IOException {

        if(s == null)
            throw new IllegalArgumentException("s == null");

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hlp.new DataComponent(HLProtocol.HTLC_DATA_CHAT, s.getBytes()),
                
            };
        
        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_CHAT,
                           nextTrans(), 
                           dataComponents);
        
	packet.write(output);
        
    }
	
    /**
     * Sends an action string to the chat. 
     * @param s the action to send
     * @return transaction id 
     */
    public void sendChatAction(String s) throws IOException {

        if(s == null)
            throw new IllegalArgumentException("s == null");

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hlp.new DataComponent(HLProtocol.HTLC_DATA_CHAT, s.getBytes()),
                hlp.new DataComponent(HLProtocol.HTLC_DATA_OPTION, ToArrayConverters.shortToByteArray(1)),
                
            };
        
        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_CHAT,
                           nextTrans(), 
                           dataComponents);

	packet.write(output);

    }
	
    /**
     * Create a private chat with another user.
     * @param sock the sock of the user to privchat with
     * @return transaction id
     */
    public int requestPrivateChatCreate(int sock) throws IOException {

        return requestPrivateChatCreate(sock, true);

    }

    public int requestPrivateChatCreate(int sock,
                                        boolean disposeWhenReplyReceived) throws IOException {

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hlp.new DataComponent(HLProtocol.HTLC_DATA_SOCKET, ToArrayConverters.intToByteArray(sock)),
                
            };
        
        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_PRIVCHAT_CREATE,
                           nextTrans(), 
                           dataComponents);
        
	createTask(packet, disposeWhenReplyReceived);
	packet.write(output);
        
	return packet.header.trans;

    }

    /**
     * Invite another user to join a private chat.
     * @param pcref the reference of the private chat
     * @param sock the user to invite to join
     * @return transaction id
     */
    public int requestPrivateChatInvite(int pcref, 
                                        int sock) throws IOException {

        return requestPrivateChatInvite(pcref, sock, true);

    }

    public int requestPrivateChatInvite(int pcref, 
                                        int sock,
                                        boolean disposeWhenReplyReceived) throws IOException {
        
        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hlp.new DataComponent(HLProtocol.HTLC_DATA_SOCKET, ToArrayConverters.intToByteArray(sock)),
                hlp.new DataComponent(HLProtocol.HTLC_DATA_CHAT_REF, ToArrayConverters.intToByteArray(pcref)),
                
            };
        
        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_PRIVCHAT_INVITE,
                           nextTrans(), 
                           dataComponents);
        
	createTask(packet, disposeWhenReplyReceived);
	packet.write(output);
        
	return packet.header.trans;

    }

    /**
     * Send a request to join a private chat.
     * @param pcref the reference of the private chat
     * @return transaction id
     */
    public int requestPrivateChatJoin(int pcref) throws IOException {

        return requestPrivateChatJoin(pcref, true);

    }

    public int requestPrivateChatJoin(int pcref,
                                      boolean disposeWhenReplyReceived) throws IOException {

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hlp.new DataComponent(HLProtocol.HTLC_DATA_CHAT_REF, ToArrayConverters.intToByteArray(pcref)),
                
            };
        
        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_PRIVCHAT_JOIN,
                           nextTrans(), 
                           dataComponents);
        
	createTask(packet, disposeWhenReplyReceived);
	packet.write(output);
        
	return packet.header.trans;

    }

    /**
     * Decline an invitation to a private chat.
     * @param pcref the reference of the private chat
     * @return transaction id
     */
    public void sendPrivateChatDecline(int pcref) throws IOException {

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hlp.new DataComponent(HLProtocol.HTLC_DATA_CHAT_REF, ToArrayConverters.intToByteArray(pcref)),
                
            };
        
        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_PRIVCHAT_DECLINE,
                           nextTrans(), 
                           dataComponents);
        
	packet.write(output);
        
    }
	
    /**
     * Leaves a private chat.
     * @param pcref the reference of the private chat
     * @return transaction id
     */
    public void sendPrivateChatLeave(int pcref) throws IOException {

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hlp.new DataComponent(HLProtocol.HTLC_DATA_CHAT_REF, ToArrayConverters.intToByteArray(pcref)),
                
            };
        
        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_PRIVCHAT_LEAVE,
                           nextTrans(), 
                           dataComponents);
        
	packet.write(output);
        
    }
	
    /**
     * Sends text to a private chat.
     * @param pcref the reference of the private chat
     * @param s the text to send
     * @return transaction id
     */
    public void sendPrivateChat(int pcref, 
                                String s) throws IOException {

        if(s == null)
            throw new IllegalArgumentException("s == null");

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hlp.new DataComponent(HLProtocol.HTLC_DATA_CHAT, s.getBytes()),
                hlp.new DataComponent(HLProtocol.HTLC_DATA_CHAT_REF, ToArrayConverters.intToByteArray(pcref)),
                
            };
        
        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_CHAT,
                           nextTrans(), 
                           dataComponents);
        
	packet.write(output);
        
    }

    /**
     * Sends an action to a private chat.
     * @param pcref the reference of the private chat
     * @param s the action to send
     * @return transaction id
     */
    public void sendPrivateChatAction(int pcref, 
                                      String s) throws IOException {

        if(s == null)
            throw new IllegalArgumentException("s == null");

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hlp.new DataComponent(HLProtocol.HTLC_DATA_OPTION, ToArrayConverters.shortToByteArray(1)),
                hlp.new DataComponent(HLProtocol.HTLC_DATA_CHAT, s.getBytes()),
                hlp.new DataComponent(HLProtocol.HTLC_DATA_CHAT_REF, ToArrayConverters.intToByteArray(pcref)),
                
            };
        
        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_CHAT,
                           nextTrans(), 
                           dataComponents);
        
	packet.write(output);
        
    }
	
    /**
     * Post a news article.
     * @param s the news post
     * @return transaction id
     */
    public int requestNewsPost(String s) throws IOException {

        return requestNewsPost(s, true);

    }

    public int requestNewsPost(String s,
                               boolean disposeWhenReplyReceived) throws IOException {
        
        if(s == null)
            throw new IllegalArgumentException("s == null");

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hlp.new DataComponent(HLProtocol.HTLC_DATA_NEWS_POST, s.getBytes()),
                
            };
        
        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_NEWS_POST,
                           nextTrans(), 
                           dataComponents);
        
        createTask(packet, disposeWhenReplyReceived);
	packet.write(output);
        
        return packet.header.trans;

    }
	
    /**
     * Send a message to another user.
     * @param s the message
     * @return transaction id
     */
    public int requestMessageSend(int sock, 
                                  String s) throws IOException {

        return requestMessageSend(sock, s, true);

    }

    public int requestMessageSend(int sock, 
                                  String s,
                                  boolean disposeWhenReplyReceived) throws IOException {
        
        if(s == null)
            throw new IllegalArgumentException("s == null");

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hlp.new DataComponent(HLProtocol.HTLC_DATA_SOCKET, ToArrayConverters.intToByteArray(sock)),
                hlp.new DataComponent(HLProtocol.HTLC_DATA_MSG, s.getBytes()),
                
            };
        
        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_MSG,
                           nextTrans(), 
                           dataComponents);
        
	createTask(packet, disposeWhenReplyReceived);
	packet.write(output);
        
	return packet.header.trans;

    }
	
    /**
     * Kick a user.
     * @param sock the user to kick
     * @return transaction id
     */
    public int requestUserKick(int sock) throws IOException {

        return requestUserKick(sock, true);

    }

    public int requestUserKick(int sock,
                               boolean disposeWhenReplyReceived) throws IOException {

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hlp.new DataComponent(HLProtocol.HTLC_DATA_SOCKET, ToArrayConverters.intToByteArray(sock)),
                
            };
        
        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_USER_KICK,
                           nextTrans(), 
                           dataComponents);
        
	createTask(packet, disposeWhenReplyReceived);
	packet.write(output);
        
	return packet.header.trans;

    }
	
    /**
     * Kicks and bans a user.
     * @param sock the user to ban
     * @return transaction id
     */
    public int requestUserKickBan(int sock) throws IOException {

        return requestUserKickBan(sock, true);

    }

    public int requestUserKickBan(int sock,
                                  boolean disposeWhenReplyReceived) throws IOException {

        HLProtocol.DataComponent[] dataComponents = 
            new HLProtocol.DataComponent[] {
                
                hlp.new DataComponent(HLProtocol.HTLC_DATA_BAN, ToArrayConverters.shortToByteArray(1)),
                hlp.new DataComponent(HLProtocol.HTLC_DATA_SOCKET, ToArrayConverters.intToByteArray(sock)),
                
            };
        
        HLProtocol.Packet packet = 
            hlp.new Packet(HLProtocol.HTLC_HDR_USER_KICK,
                           nextTrans(), 
                           dataComponents);
        
	createTask(packet, disposeWhenReplyReceived);
	packet.write(output);
        
	return packet.header.trans;

    }
	
    /* Utility methods */

    /**
     * Waits for the specified task to complete and returns it's
     * result.  For this to work <i>reliably</i>, the task that is
     * being waited for <i>must</i> have been created with the
     * <tt>disposeWhenReplyReceived</tt> flag set to <tt>false</tt>.
     * If the task number does not exist, returns null.
     * @param id the transaction id to wait for.
     * @return a data object
     * @throws IllegalArgumentException when the task specified by id
     * has automatic dispose turned on.
     */
    public Object waitFor(int id) throws HLException, IOException, InterruptedException {

        Object o = null;

        try {

            HLTask task = getTask(id);
            
            if(task.disposeWhenReceived)
                throw new IllegalArgumentException("aborted attempted wait for task that has disposeWhenReceived flag set to true; unwise.");
            
            o = task.getData();

            DebuggerOutput.debug("HLClient.waitFor[" + id + "]: got data.");

        } finally {

            //            if(lockTask(id)) {
                
            //                DebuggerOutput.debug("HLClient.waitFor[" + id + "]: disposing task[" + id + "].");
                disposeTask(id);
                
                //            } else {

                //                DebuggerOutput.debug("HLClient.waitFor[" + id + "]: task locked, won't dispose.");

                //            }

        }

        return o;

    }

    /**
     * Invokes handleDisconnect() on registered HLClientListeners.
     */
    void fireDisconnect(String msg) {

        DebuggerOutput.debug("HLClient.fireDisconnect: accepted (reason = " + msg + ")");
        
        for(Enumeration en = hle.elements(); en.hasMoreElements(); ) {
            
            HLClientListener l = (HLClientListener) en.nextElement();
            DebuggerOutput.debug("HLClient.fireDisconnect: invoking handleDisconnect on " + l);
            l.handleDisconnect(msg);
            
        }
        
    }

    private synchronized void internalDisconnect() {

        if(!connected) {
            
            DebuggerOutput.debug("HLClient.internalDisconnect: already disconnected.");
            return;
            
        }
        
        /* Wake up any threads still waiting for an agreement or a
           task reply. */
        
        hlr.notifyAgreementAborted();
        
        DebuggerOutput.debug("HLClient.internalDisconnect: terminating tasks");
        terminateAllTasks("connection closed");
        DebuggerOutput.debug("HLClient.internalDisconnect: tasks terminated");
        
        connected = false;
        transaction = 2;
        
    }

    /* Following methods are about task handling. Might as well make
       this into a separate class. */

    void terminateAllTasks(String msg) {

	Enumeration k = tasks.keys();

	while(k.hasMoreElements()) {

            int trans = ((Integer) k.nextElement()).intValue();

            if(lockTask(trans)) {
                
                /* If we can't lock the task, then the dispatcher
                   got to this task first. That's OK. Just let it
                   process the task reply. */

                try {
                    
                    DebuggerOutput.debug("terminateAllTasks: terminating task " + getTask(trans).toString());
                    
                    HLTask task = getTask(trans);
                    task.setError(new HLException(msg));

                    /* Have to dispose this task here because the
                       dispatcher won't be able to. */

                    if(task.disposeWhenReceived)
                        disposeTask(trans);

                } catch(HLTaskNotFoundException e) {

                    DebuggerOutput.stackTrace(e);

                }
                
            }

	}

    }

    synchronized boolean lockTask(int trans) {
        boolean lockSucceeded = false;
        HLTask task;

        /* It's possible that the task was disposed before we could
           get to it. That's not possible here, because disposeTask()
           is synchronized. */

        if(!tasks.containsKey(new Integer(trans))) {

            DebuggerOutput.debug("lockTask[" + trans + "]: task does not exist");
            return false;

        }

        try {

            task = getTask(trans);
            
            if(!task.isLocked()) {
                
                DebuggerOutput.debug("lockTask[" + trans + "]: task not locked, locking task ...");
                task.lock();
                DebuggerOutput.debug("lockTask[" + trans + "]: task locked.");
                lockSucceeded = true;
                
            } else {
                
                DebuggerOutput.debug("lockTask[" + trans + "]: task already locked, not locking");
                
            }
        
        } catch(HLTaskNotFoundException e) {

            DebuggerOutput.stackTrace(e);

        }

        return lockSucceeded;

    }

    synchronized int nextTrans() {

	return transaction++;

    }

    synchronized HLTask createTask(HLProtocol.Packet packet, 
                                   boolean disposeWhenReplyReceived) throws IOException {
        
        if(!connected)
            throw new IOException("not connected");

	HLTask task = new HLTask(packet.header.id, disposeWhenReplyReceived);
	tasks.put(new Integer(packet.header.trans), task);
	DebuggerOutput.debug("createTask: created task[" + packet.header.trans + "]: " + task);

        return task;

    }
    
    synchronized void disposeTask(int trans) throws HLTaskNotFoundException {

	DebuggerOutput.debug("disposeTask: disposing task [" + trans + "] ... ");
        DebuggerOutput.debug("disposeTask: task list = " + tasks);

	Transfer xf = getTransfer(trans);

	if(xf != null)
	    disposeTransfer(trans);

	if(tasks.containsKey(new Integer(trans))) {

            DebuggerOutput.debug("disposeTask: disposed task [" + trans + "].");
	    tasks.remove(new Integer(trans));

	} else {

	    DebuggerOutput.debug("disposeTask: no such task ["+trans+"]");
            throw new HLTaskNotFoundException("no such task: " + trans);

        }

    }

    synchronized HLTask getTask(int trans) throws HLTaskNotFoundException {

        if(tasks.containsKey(new Integer(trans)))
            return (HLTask) tasks.get(new Integer(trans));
        else
            throw new HLTaskNotFoundException("no such task: " + trans);

    }

    synchronized void createTransfer(HLProtocol.Packet packet, Transfer tx) {

        createTransfer(packet.header.trans, tx);

    }

    synchronized void createTransfer(int trans, Transfer tx) {

        DebuggerOutput.debug("HLClient.createTransfer: transfer = "+trans);
	transfers.put(new Integer(trans), tx);

    }
    
    synchronized void disposeTransfer(int trans) {

        Transfer xf = (Transfer) transfers.get(new Integer(trans));

        if(xf != null) {

            if(xf.local != null) {

                try {
                    
                    xf.local.close();
                    
                } catch(IOException e) {
                    
                    DebuggerOutput.stackTrace(e);
                    
                }

            }

            /*
            if(xf.inputPipe != null) {

                try {

                    //                    xf.inputPipe.close();

                } catch(IOException e) {

                    DebuggerOutput.stackTrace(e);

                }

            }
            */
                
            transfers.remove(new Integer(trans));
        
        }

    }

    synchronized Transfer getTransfer(int trans) {

	return (Transfer) transfers.get(new Integer(trans));

    }

    /**
     * Returns a string representation of this object.
     * @return a string representation.
     */
    public synchronized String toString() {
	String s = "HLClient";

	if(host != null) {

	    s += host.toString();
	    s += connected ? " [connected]" : " [disconnected]";

	    if(connected)
		s += " tasks: " + tasks;

	} else {

	    s += " no host";

	}

	return s;

    }
    
}

/**
 * Reads events from the server and dispatches new threads
 * to handle them.
 */
class ReaderThread extends Thread {
    HLClient hlc;
    DataInputStream input;
    boolean blocked = false;
    HLProtocol.Packet packet;

    ReaderThread(HLClient h, DataInputStream in) {

        super();
        hlc = h;
        input = in;
        DebuggerOutput.debug("ReaderThread["+this+"] created");

    }

    /**
     * This method controls whether event listeners are called or
     * not. If the argument is true, then event listeners will not be
     * called until setBlocked is invoked again with a false argument.
     * @param b true to block, false to unblock 
     */
    synchronized void setBlocked(boolean b) {

	blocked = b;

	if(blocked == false)
	    notify();

    }

    /**
     * Blocks until {@link #setBlocked} is called with a 
     * false parameter.
     */
    synchronized void honourBlock() throws InterruptedException {

	if(blocked) {

	    DebuggerOutput.debug("ReaderThread["+this+"]: waiting for unblock");

            while(blocked)
                wait(5000);

	}
	
    }

    /**
     * This is the main input loop where all incoming
     * data is read and dispatched.
     */
    public void run() {

        DebuggerOutput.debug("ReaderThread["+this+"]: running");

	try {

	    while(!this.isInterrupted()) {

                honourBlock();
                                
		packet = hlc.hlp.new Packet(input);

                DebuggerOutput.debug("ReaderThread: got packet " + packet.toString());
        
                if(packet.header.id == HLProtocol.HTLS_HDR_TASK) {

                    if(hlc.lockTask(packet.header.trans)) {
                        
                        HLTask t = hlc.getTask(packet.header.trans);
                        
                        if(t.type == HLProtocol.HTLC_HDR_FILE_GET ||
                           t.type == HLProtocol.HTLC_HDR_FILE_PUT) {
                            
                            DebuggerOutput.debug("ReaderThread["+this+"]: task[" + packet.header.trans + "]: dispatching file transfer in separate thread."); 
                            
                            FileTransferThread ft = new FileTransferThread(hlc, packet);
                            ft.start();

                        } else {

                            DebuggerOutput.debug("ReaderThread["+this+"]: task[" + packet.header.trans + "]: dispatching..."); 
                            hlc.hlr.dispatch(packet);

                        }
                        
                    } else {
                        
                        DebuggerOutput.debug("ReaderThread["+this+"]: task[" + packet.header.trans + "]: cannot lock, will not dispatch."); 
                        
                    }

                } else {

                    DebuggerOutput.debug("ReaderThread["+this+"]: dispatching...");
                    hlc.hlr.dispatch(packet);

                }


	    }
        
	} catch(Exception e) {

            DebuggerOutput.debug("ReaderThread["+this+"]: exception during event dispatch:");
            DebuggerOutput.stackTrace(e);

            String reason = e.toString();
            
            if(e.getMessage() != null)
                reason = e.getMessage();
            
            hlc.terminateAllTasks(reason);

            /* If we were explicitly interrupted, then we do not want
               to fire an unexpected disconnect event. */

            if(!isInterrupted()) {

                DebuggerOutput.debug("ReaderThread["+this+"]: calling fireDisconnect ...");
                hlc.fireDisconnect(reason);
                
            }

	}

        DebuggerOutput.debug("ReaderThread["+this+"] exiting");

    }

}

class FileTransferThread extends Thread {
    HLProtocol.Packet packet;
    HLClient hlc;

    FileTransferThread(HLClient hlc, HLProtocol.Packet packet) {

        this.hlc = hlc;
        this.packet = packet;

        setName("FileTransferThread[task[" + packet.header.trans + "]]");
        setPriority(Thread.MIN_PRIORITY);

    } 

    public void run() {
        
        hlc.hlr.dispatch(packet);
        
    }

}


