/**
 * $Id: SplitMacFile.java,v 1.4 2001/10/01 19:46:34 groomed Exp $
 *
 * Copyright (C) 1998-2001 groomed <groomed@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package redlight.macfiles;

import java.io.*;
import java.util.Calendar;

import redlight.utils.TextUtils;
import redlight.utils.ToArrayConverters;
import redlight.utils.Meter;
import redlight.utils.MeterSource;
import redlight.utils.DebuggerOutput;

/**
 * SplitMacFile provides a mechanism to store the 
 * data fork and the resource fork of a Mac file in
 * two seperate files.
 * A SplitMacFile creates two files to store a Macintosh file;
 * one for the data fork and another (with the extension .rsrc)
 * for the resource fork.
 * If either one of these files is empty on close(), the
 * empty file is deleted.
 */
public class SplitMacFile extends MacFile implements FilenameFilter {

    File infoFile;

    /**
     * Constructs a SplitMacFile from a given File. This
     * creates two files f.getName() and f.getName() + ".rsrc"
     * if they don't exist.
     * @param f the file
     * @param perms the permissions; MacFile.READ_PERM or MacFile.WRITE_PERM.
     */
    public SplitMacFile(File f, Integer perms) throws IOException {

	super(f, perms);
        
        dataFork = new SplitFork(this, permissions, "data", "");
        resourceFork = new SplitFork(this, permissions, "rsrc", ".rsrc");

        infoFile = new File(getFile().getParent(), "." + getFile().getName() + ".info");
        readHeader();

    }

    /**
     * Reads metafile info from a .info file.
     */
    public void readHeader() throws IOException {

        comment = "";
            
        if(!infoFile.exists())
            return;

        try {

            LineNumberReader lr = new LineNumberReader(new FileReader(infoFile));
            boolean readComment = false, readType = false, readCreator = false, readSection = false, readFinderflags = false, skipSection = false;
            
            for(String line = lr.readLine(); line != null; line = lr.readLine()) {
                
                if(line.startsWith("[")) {

                    skipSection = false;
                    readComment = false;
                    readType = false;
                    readCreator = false;
                    readFinderflags = false;

                    if(line.trim().toLowerCase().equals("[comment]")) {

                        readComment = true;
                    
                    } else if(line.trim().toLowerCase().equals("[type]")) {

                        readType = true;
                    
                    } else if(line.trim().toLowerCase().equals("[creator]")) {

                        readCreator = true;

                    } else if(line.trim().toLowerCase().equals("[finderflags]")) {

                        readFinderflags = true;

                    } else {

                        DebuggerOutput.debug("SplitMacFile.readHeader: warning: " + infoFile + ": Skipping unknown section '" + line + "'");
                        skipSection = true;

                    }
                    
                    continue;

                } else {

                    if(skipSection)
                        continue;
                       
                    if(readComment) {

                        comment += line + "\r";
                        
                    } else if(readFinderflags) {

                        try {

                            finderFlags = Integer.parseInt(line);
                            
                        } catch(NumberFormatException e) {
                            
                            DebuggerOutput.debug("SplitMacFile.readHeader: warning: section '" + finderFlags + "' contains garbage");
                            
                        }

                        readFinderflags = false;

                    } else if(readType || readCreator) {
                        
                        if(line.length() == 4) {
                            
                            if(readType)
                                type = line;
                            else if(readCreator)
                                creator = line;
                            
                        } else {
                            
                            DebuggerOutput.debug("SplitMacFile.readHeader: warning: " + infoFile + ": Type or creator has wrong size (must be 4 bytes)");
                            skipSection = true;

                        }
                        
                        /* Type, creator never occupy more than one
                           line. */

                        readType = false;
                        readCreator = false;                            
                        
                    }

                }
                
            }
            
            lr.close();

        } catch(IOException e) {

            if(DebuggerOutput.on)
                e.printStackTrace();

        }

    }

    /**
     * No function in SplitMacFile.
     */
    public void writeHeader(String name, 
			    String type,
			    String creator,
			    String comments, 
			    Calendar created, 
			    Calendar modifed, 
			    int finderFlags) throws IOException {

        /* Try not to create .info files if we don't strictly have
           to. */

        if((comments == null || comments.trim().equals("")) &&
           (type == null || type.trim().equals("")) &&
           (creator == null || creator.trim().equals("")) &&
           finderFlags == 0)
            return;

        FileWriter fw = new FileWriter(infoFile);

        DebuggerOutput.debug("SplitMacFile.writeHeader: fileWriter: " + fw);

        if(comment != null && !comment.trim().equals("")) {

            fw.write("[Comment]" + System.getProperty("line.separator"));
            fw.write(TextUtils.findAndReplace(comments, "\r", System.getProperty("line.separator")) + System.getProperty("line.separator"));

        }

        if(type != null && !type.trim().equals("")) {

            fw.write("[Type]" + System.getProperty("line.separator"));
            fw.write(type + System.getProperty("line.separator"));

        }

        if(creator != null && !creator.trim().equals("")) {

            fw.write("[Creator]" + System.getProperty("line.separator"));
            fw.write(creator + System.getProperty("line.separator"));

        }

        if(finderFlags != 0) {

            fw.write("[Finderflags]" + System.getProperty("line.separator"));
            fw.write(finderFlags + System.getProperty("line.separator"));

        }

        fw.close();

        comment = comments;

        return;

    }
   
    /**
     * No function in SplitMacFile.
     */
    public void cleanup() throws IOException {}

    /**
     * Closes the data fork and the resource fork files.
     * If either the data fork or the resource fork is 
     * empty, it is deleted automatically.
     */
    public void close() throws IOException {

        if(file.exists() && file.isDirectory())
            return;

        resourceFork.close();
        dataFork.close();

    }

    /**
     * Determines whether either the data fork or the 
     * resource fork exist.
     * @return true if either the data fork or the
     * resource fork exist.
     */
    public boolean exists() {

        if(file.exists() && file.isDirectory())
            return true;

	return resourceFork.exists() | dataFork.exists();

    }

    /**
     * Deletes data fork and resource fork of file.
     * @return true if either the data fork or the
     * resource fork could be deleted.
     */
    public boolean delete() {

	return resourceFork.delete() | dataFork.delete();

    }

    /**
     * Renames file.
     * @return false
     */
    public boolean renameTo(File f) {

        super.renameTo(f);

        infoFile.renameTo(new File(f.getParent(), "." + f.getName() + ".info"));
        infoFile = new File(f.getParent(), "." + f.getName() + ".info");

        if(file.exists() && file.isDirectory()) {

            if(file.renameTo(f)) {

                file = f;
                return true;
                
            } else {
                
                return false;
                
            }
            
        } else {

            ((SplitFork) dataFork).renameTo(f);
            ((SplitFork) resourceFork).renameTo(new File(f.getParent(), "." + f.getName() + ".rsrc"));

        }

        file = f;

        return true;

    }

    /**
     * Returns the size of the data fork.
     * @return size of data fork.
     */
    public long getDataSize() {

        return dataFork.size();

    }

    /**
     * Returns the size of the resource fork.
     * @return size of resource fork.
     */
    public long getResourceSize() {

        return resourceFork.size();

    }

    /**
     * Returns the list of files inside a directory, from the
     * perspective of a SplitMacFile. This list is constructed as
     * follows:<p>
     *
     * 1. Let LIST be an array of files as returned by File.list().<br>
     * 2. Remove from LIST all files R where R matches A.rsrc,<br>
     *    if there also exists a file A.<br>
     * 3. Remove from LIST all files R where R matches R.info.<br>
     * 4. Remove from LIST all files R where size(R) == 0.<br>
     * 5. For all remaining R, set the filename to A.<p>
     * 
     * @return an array of files (if this MacFile is a directory).
     */
    public String[] list() {

        String[] list = file.list(this);

        if(list != null) {

            for(int i = 0; i < list.length; i++)
                if(list[i].startsWith(".") && list[i].endsWith(".rsrc"))
                    list[i] = list[i].substring(1, list[i].length() - 5);

        } else {

            list = new String[0];

        }

        return list;

    }

    /**
     * Enforces 2, 3, 4 (see above).
     */
    public boolean accept(File dir, String name) {

        File f = new File(dir, name);

        DebuggerOutput.debug("SplitMacFile.accept: accepting (default = deny): " + f);

        if(name.equals("..rsrc") || name.equals("..info"))
            return false;

        if(name.startsWith(".") && name.endsWith(".rsrc") && new File(dir, name.substring(1, name.length() - 5)).exists())
            return false;

        if(name.startsWith(".") && name.endsWith(".info"))
            return false;

        if(!f.isDirectory() && f.length() == 0)
            return false;

        DebuggerOutput.debug("SplitMacFile.accept: accepted " + f);

        return true;

    }

}
