
/**
 *  Classroom Scheduler
 *  Copyright (C) 2004 Colin Archibald, Ph.D.
 *  https://sourceforge.net/projects/cr-scheduler/
 *
 *  Licensed under the Academic Free License version 2.0
 */

package application;

import java.awt.*;
import java.awt.print.*;
import javax.swing.*;
import java.util.*;

import resources.*;
/**
 * Look at a schedule in a graphical, time chart format
 */

public class ViewGraphSched extends JComponent implements Constants, Printable, Pageable {
    private Schedule schedule = Schedule.getSchedule();
    
    private Professor viewProf;
    private Classroom viewRoom;
    
    public ViewGraphSched(Object obj) {
        if (obj instanceof Professor)
            viewProf = (Professor) obj;
        else if (obj instanceof Classroom)
            viewRoom = (Classroom) obj;
        
        Font mono = new Font("Monospaced", Font.PLAIN, 10);
        
        setFont(mono);
        setBackground(Color.white);
    }
    
    public void setViewProf(Professor viewProf) {
        this.viewProf = viewProf;
        viewRoom = null;
        repaint();
    }
    
    public void setViewRoom(Classroom viewRoom) {
        this.viewRoom = viewRoom;
        viewProf = null;
        repaint();
    }
    
    public void paint(Graphics g) {
        Graphics2D g2D = (Graphics2D)g;   // Get a 2D device context
        Rectangle r = new Rectangle(6 * 85, 900);
        draw(g2D, r);
    }
    
    // there are 16 hours in the school day. Put ticks on the schedule
    
    private void placeTicks(Graphics g, int left, int top, int bottom) {
        
        String[] hours =  {"7AM", "8AM","9AM","10AM","11AM", "12PM",
        "1PM","2PM","3PM","4PM","5PM","6PM","7PM","8PM","9PM","10PM"
        };
        
        double gap = (bottom - top) / 15.0;
        double tickPos = top;
        
        for (int i = 0; i <= 15; i++) {
            g.drawString(hours[i], left - 25, (int)tickPos + 3);
            g.drawLine(left - 3, (int)tickPos, left, (int)tickPos);
            tickPos += gap;
        }
    }
    
    public void draw(Graphics g, Rectangle r) {
        
        //        System.out.println("in draw, r is: " + r);
        
        //  we will take 30 from the X direction to put the ticks.
        
        Rectangle withOffset = new Rectangle(
        (int) r.getX() + 30,
        (int) r.getY(),
        (int) r.getWidth() - 30,
        (int) r.getHeight());
        r = withOffset;
        
        
        setPreferredSize(new Dimension(r.width, r.height + 75));
        setBackground(Color.white);
        
        Font	titleFont = new Font("Times New Roman", Font.BOLD, 10);
        Font	regFont = new Font("Times New Roman", Font.PLAIN, 8);
        FontMetrics fm = g.getFontMetrics(titleFont);    // Get font metrics
        
        // Draw the outline for a room or a prof
        
        int	    colWidth = r.width / 6;
        float   heightRatio = 1.0f;
        
        int	    topMargin = (int) r.getY() + fm.getHeight();
        int 	leftMargin = (int) r.getX();
        
        g.setFont(titleFont);
        g.drawRect(leftMargin, topMargin, r.width, r.height - fm.getHeight());
        
        g.drawLine(leftMargin, topMargin + fm.getHeight(),
        leftMargin + colWidth * 6, topMargin + fm.getHeight());
        g.drawString("Monday", leftMargin + 2, -3 + topMargin + fm.getHeight());
        g.drawString("Tuesday", leftMargin + 2 + colWidth, -3 + topMargin + fm.getHeight());
        g.drawString("Wednesday", leftMargin + 2 + colWidth * 2, -3 + topMargin + fm.getHeight());
        g.drawString("Thursday", leftMargin + 2 + colWidth * 3, -3 + topMargin + fm.getHeight());
        g.drawString("Friday", leftMargin + 2 + colWidth * 4, -3 + topMargin + fm.getHeight());
        g.drawString("Saturday", leftMargin + 2 + colWidth * 5, -3 + topMargin + fm.getHeight());
        
        g.setFont(regFont);
        placeTicks(g, leftMargin, topMargin + (int)fm.getHeight(),
        (int) (r.getHeight() + topMargin - fm.getHeight()) );
        
        for (int j = 1; j <= 5; j++) // draw 6 columns for the days of week
        {
            g.drawLine(leftMargin + colWidth * j,
            topMargin,
            leftMargin + colWidth * j,
            topMargin + r.height);
        }
        
        // the rectangle for the minutes in the schedule
        
        Rectangle minutes = new Rectangle(leftMargin,
        topMargin + fm.getHeight(),
        r.width,
        r.height - fm.getHeight() * 2);
        
        fm = g.getFontMetrics(regFont);
        
        // ask a prof or room to draw its schedule
        if (!(viewProf == null)) {
            g.setFont(titleFont);
            g.drawString(schedule.getScheduleName() + "  Schedule for Professor: " +
            viewProf.toString(),
            leftMargin, topMargin - fm.getHeight() / 2);
            
            g.setFont(regFont);
            g.drawString(schedule.updated(),
            leftMargin + r.width -
            fm.stringWidth(schedule.updated()),
            topMargin - fm.getHeight() / 2);
            viewProf.draw(g, minutes);	// pass a rectangle?
        }
        
        if (!(viewRoom == null)) {
            g.setFont(titleFont);
            g.drawString(schedule.getScheduleName() + " Schedule for Room: " +
            viewRoom.toString(),
            leftMargin, topMargin - fm.getHeight() / 2);
            g.setFont(regFont);
            g.drawString(schedule.updated(),
            leftMargin + r.width -
            fm.stringWidth(schedule.updated()),
            topMargin - fm.getHeight() / 2);
            g.setFont(regFont);
            viewRoom.draw(g, minutes);	// pass a rectangle?
        }
    }
    
    
    /** Print one room per page with a header that includes the schedule's name
     */
    
    public int print(Graphics g, PageFormat pageFormat, int pageIndex) {
        if (pageIndex > 0) {
            return NO_SUCH_PAGE;    // done printing
        }
        pageFormat = getPageFormat(0);
        draw(g, new Rectangle(
        (int)(pageFormat.getPaper().getImageableX()),
        (int)(pageFormat.getPaper().getImageableY()),
        (int)(pageFormat.getPaper().getImageableWidth()),
        (int)(pageFormat.getPaper().getImageableHeight())));
        
        //        System.out.println("in print:x y width height\n " +
        //            (int)(pageFormat.getPaper().getImageableX()) + " " +
        //            (int)(pageFormat.getPaper().getImageableY()) + " " +
        //            (int)(pageFormat.getPaper().getImageableWidth()) + " " +
        //            (int)(pageFormat.getPaper().getImageableHeight())
        //        );
        
        return PAGE_EXISTS;
    }
    
    // Always one page
    public int getNumberOfPages() {
        return 1;
    }
    
    // Return the Printable object that will render the page
    public Printable getPrintable(int pageIndex) {
        return this;
    }
    
    public PageFormat getPageFormat(int pageIndex) {
        // Get the default page format and its Paper object.
        PageFormat pageFormat = PrinterJob.getPrinterJob().defaultPage();
        Paper paper = pageFormat.getPaper();
        paper.setImageableArea(36, 36, 540, 720); // half inch margins
        pageFormat.setPaper(paper);        // Restore the paper
        return pageFormat;                                    // Return the page format
    }
}





