/**
 *  Classroom Scheduler
 *  Copyright (C) 2004 Colin Archibald, Ph.D.
 *  https://sourceforge.net/projects/cr-scheduler/
 *
 *  Licensed under the Academic Free License version 2.0
 */

package resources;

import java.awt.*;
import java.io.*;
import java.util.*;

/**
 * Class declaration
 *
 */

public class SchedCourse implements Comparable, Serializable {
    static final long serialVersionUID = 911;
    
    public static final int PROFESSOR = 0;
    public static final int CLASSROOM = 1;
    
    private Professor professor;
    private Classroom classroom;
    private Course    course;
    private TimeSlot  timeSlot;
    private String    crn = "";
    private Conflict conflict = null;
    private Textbook book = null;
    private String note = "";
    
    private final static int BY_COURSE = 1;
    private final static int BY_PROFESSOR = 2;
    private final static int BY_ROOM = 3;
    private final static int BY_START_TIME = 4;
    
    private static int sortKey = BY_COURSE;
    
    /**
     * Constructor declaration
     *
     *
     * @param Professor professor
     * @param Classroom classroom
     * @param Course course
     * @param TimeSlot timeSlot
     *
     */
    
    public SchedCourse(Professor professor, Classroom classroom, Course course,
    TimeSlot timeSlot) {
        this.professor = professor;
        this.classroom = classroom;
        this.course = course;
        this.timeSlot = timeSlot;
    }
    public SchedCourse(Professor professor, Classroom classroom, Course course,
    TimeSlot timeSlot, String crn) {
        this.professor = professor;
        this.classroom = classroom;
        this.course = course;
        this.timeSlot = timeSlot;
        this.crn = crn;
    }  
      
    public static void setSortKey(int key) {
        if (key >= BY_COURSE && key <= BY_ROOM) {
            sortKey = key;
        }
    }
    
    public void setProfessor(Professor professor) {
        this.professor = professor;
    }
    
    public void setClassroom(Classroom classroom) {
        this.classroom = classroom;
    }
    
    public void setCourse(Course course) {
        this.course = course;
    }
    
    public void setTimeSlot(TimeSlot timeSlot) {
        this.timeSlot = timeSlot;
    }
    
    public void setCrn(String crn){
        this.crn = crn;
    }
    public void setTextbook(Textbook book){
        this.book = book;
    }
    public void setNote(String note){
        this.note = note;
    }
 
    public Professor getProfessor() {
        return professor;
    }
    
    public Classroom getClassroom() {
        return classroom;
    }
    
    public Course getCourse() {
        return course;
    }
    
    public TimeSlot getTimeSlot() {
        return timeSlot;
    }
    
    public void setConflict(Conflict c) {
        conflict = c;
    }
    
    public Conflict getConflict() {
        return conflict;
    }
    
    public String getCrn(){
        return crn;
    }
    
    public Textbook getTextbook(){
        return book;
    }
    
    public String getNote(){
        return note;
    }

    /**
     * provide comparison for sorting the list of scheduled courses
     */
    
    public int compareTo(Object obj) {
        if(!(obj instanceof SchedCourse)){
            return 0;
        }
        SchedCourse sched2 = (SchedCourse) obj;
        if (sortKey == BY_COURSE)
            return course.compareTo(sched2.course);
        if (sortKey == BY_PROFESSOR)
            return professor.compareTo(sched2.professor);
        if (sortKey == BY_ROOM)
            return classroom.compareTo(sched2.classroom);
        if (sortKey == BY_START_TIME)
            return timeSlot.getStartTime().compareTo(sched2.timeSlot.getStartTime());
        
        return 0;  // should never execute
    }
    public boolean equals(Object obj){
        if(!(obj instanceof SchedCourse)){
            return false;
        }
        SchedCourse that = (SchedCourse) obj;
        if(this.classroom.equals(that.classroom)
            && this.course.equals(that.course)
            && this.timeSlot.equals(that.timeSlot)
            && this.crn.equals(that.crn))
            return true;
        return false;
    }
            
    
        /*
         *    private Professor professor;
    private Classroom classroom;
    private Course    course;
    private TimeSlot  timeSlot;
    private String    crn = "";
    private Conflict conflict = null;
    private Textbook book = null;
    private String note = "";*/
        
    public String toString() {

        StringBuffer hold = new StringBuffer(100);
        char space = ' ';
        int crnWidth = 6;
        int courseWidth = 40;
        int professorWidth = 20;
        int timeSlotWidth = 20;
        int roomWidth = 12;
        
        hold.append(crn);
        for (int i = crn.length(); i < crnWidth; i++) {
            hold.append(space);
        }       
        hold.append(course.toString());
        
        for (int i = course.toString().length(); i < courseWidth; i++) {
            hold.append(space);
        }
        
        String slot = timeSlot.toString();
        
        hold.append(slot);
        
        for (int i = timeSlot.toString().length(); i < timeSlotWidth; i++) {
            hold.append(space);
        }
        
        hold.append(classroom.toString());
        
        for (int i = classroom.toString().length(); i < roomWidth; i++) {
            hold.append(space);
        }
        
        hold.append(professor.toString());
        
        for (int i = professor.toString().length(); i < professorWidth; i++) {
            hold.append(space);
        }
        
        if (conflict != null)
            hold.append("  ***");
        
        return hold.toString();
    }    // end toString
    
    
    // draw the graphical view of the course
    // Scaled by the rectangle passed in for either screen or printing
    
    /**
     * @param g
     * @param whatChart indicates either classroom ro professor chart
     */
    
    public void drawGraphical(Graphics g, int whatChart, Rectangle drawArea) {
        // calculate the location of the rectangle within the drawing area
        
        int colWidth = drawArea.width / 6;
        int left = drawArea.x;
        int top = drawArea.y;
        
        double heightRatio = drawArea.height / 900.0;
        
        boolean[] days = timeSlot.getDaysOfWeek();
        
        for (int i = 2; i < days.length; i++) {
            if (days[i]) {
                Rectangle r = new Rectangle((i-2) * colWidth + left,
                (int) (top + timeSlot.getStartTime().getcrMinutes() * heightRatio),
                colWidth,
                (int) (timeSlot.getMinutes() * heightRatio));
                
                
                //System.out.println("in SchedCourse: " + r);
                g.drawRect(r.x, r.y, r.width, r.height);
                drawText(g, r, whatChart);
            }
        }
    }
    
    // drawing for the chart view
    
    public void drawText(Graphics g, Rectangle r, int whatChart) {
        FontMetrics fm = g.getFontMetrics();
        int height;
        int textX = r.x + 1;
        height = fm.getHeight();
        int textY = r.y + height - 3;
        
        g.drawString(course.getField() + " " +course.getCourseNumber() + " " +
        course.getSectionPrefix() + course.getSectionNumber(), textX, textY);
        textY += height;
        int countChars = charsThatFit(course.getCourseName(), r, fm);
        g.drawString(course.getCourseName().substring(0, countChars),
        textX, textY);
        textY += height;
        
        int wid = 0;
        if (whatChart != PROFESSOR) {
            g.drawString(professor.getLastName()+ " " , textX, textY);
            wid = fm.stringWidth(professor.getLastName());
        }
        if (whatChart != CLASSROOM) {
            g.drawString(classroom.toString() + " ", textX, textY);
            wid = fm.stringWidth(classroom.toString());
        }
        g.drawString(" " + timeSlot.toTimeString(), textX + wid, textY);
    }
    
    // How many chars of the string fit in colWidth?
    private int charsThatFit(String s, Rectangle r, FontMetrics fm) {
        int wid = 0, charCount = 0;
        
        while (wid < r.width && charCount < s.length()) {
            wid += fm.charWidth(s.charAt(charCount++));
        }
        
        if (wid > r.width) {
            charCount--;
        }
        return charCount;
    }
    
    /**
     * @returns true if this course overlaps the time slot of the the parameter
     */
    public boolean overlap(SchedCourse sc) {
        return timeSlot.overlap(sc.timeSlot);
    }
    
        /* this is the test for the resources: Professor, Course,
        Classroom, Sched.  	*/
    
    
    public static void main(String[] a) {
        
        // make a prof
        
        Professor colin = new Professor("Colin", "Archibald", Professor.FULL_TIME);
        System.out.println(" a prof: " + colin);
        
        // make a classroom
        
        Classroom room = new Classroom("7", "129", 24);
        System.out.println(" a room: " + room);
        
        // make a course
        
        Course course = new Course("COP", "2700", "Intro to Java", "W", 1, 3);
        System.out.println(" a course: " + course);
        
        // make a timeSlot
        
        CRTime start = null, stop = null;
        boolean days[] = {false,false, true, false, true, true, true, false};
        
        try {
            start = new CRTime(299);
            stop = new CRTime(350);
        }
        catch (CRTimeException crtE) {
            System.out.println(crtE);
        }
        
        TimeSlot slot = new TimeSlot(days, start, stop);
        
        System.out.println("a time slot: " + slot);
    }
}


