/*
 * $Id: TargetableAction.java,v 1.2 2004/09/17 23:24:57 davidson1 Exp $
 *
 * Copyright 2004 Sun Microsystems, Inc., 4150 Network Circle,
 * Santa Clara, California 95054, U.S.A. All rights reserved.
 */

package org.jdesktop.swing.actions;

import java.awt.event.ActionEvent;
import java.awt.event.ItemEvent;

import javax.swing.Action;
import javax.swing.Icon;

/**
 * A class that represents a dynamically targetable action. The invocation of this
 * action will be dispatched to the <code>TargetManager</code>
 * <p>
 * You would create instances of this class to let the TargetManager handle the
 * action invocations from the ui components constructed with this action.
 * The TargetManager could be configured depending on application state to
 * handle these actions. Generally, this class doesn't need to be subclassed.
 *
 * @see TargetManager
 * @author Mark Davidson
 */
public class TargetableAction extends AbstractActionExt {

    public TargetableAction() {
        this("action");
    }

    public TargetableAction(String name) {
        super(name);
    }

    /**
     * @param name display name of the action
     * @param command the value of the action command key
     */
    public TargetableAction(String name, String command) {
        super(name, command);
    }

    /**
     * @param name display name of the action
     * @param command the value of the action command key
     * @param icon icon to display
     */
    public TargetableAction(String name, String command, Icon icon) {
        super(name, command, icon);
    }

    public TargetableAction(String name, Icon icon) {
        super(name, icon);
    }

    // Callbacks...

    /**
     * Callback for command actions. This event will be redispatched to
     * the target manager along with the value of the Action.ACTION_COMMAND_KEY
     *
     * @param evt event which will be forwarded to the TargetManager
     * @see TargetManager
     */
    public void actionPerformed(ActionEvent evt) {
        if (!isStateAction()) {
            // Do not process this event if it's a toggle action.
            TargetManager manager = TargetManager.getInstance();
            manager.doCommand(getActionCommand(), evt);
        }
    }

    /**
     * Callback for toggle actions. This event will be redispatched to
     * the target manager along with value of the the Action.ACTION_COMMAND_KEY
     *
     * @param evt event which will be forwarded to the TargetManager
     * @see TargetManager
     */
    public void itemStateChanged(ItemEvent evt) {
        // Update all objects that share this item
        boolean newValue;
        boolean oldValue = isSelected();

        if (evt.getStateChange() == ItemEvent.SELECTED) {
            newValue = true;
        } else {
            newValue = false;
        }

        if (oldValue != newValue) {
            setSelected(newValue);

            TargetManager manager = TargetManager.getInstance();
            manager.doCommand(getActionCommand(), evt);
        }
    }

    public String toString() {
        return super.toString();
    }
}
