/* $Id: RowTokenizer.java,v 2.4 1999/07/22 20:57:57 borg Exp $ */
/* Copyright  1999 George Reese, All Rights Reserved */
package com.imaginary.sql.msql;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Iterator;

/**
 * Represents a row parsed straight from mSQL. It takes in a row of raw data
 * from a connection to mSQL and parses it into a list of columns values.
 * <BR>
 * Last modified $Date: 1999/07/22 20:57:57 $
 * @version $Revision: 2.4 $
 * @author George Reese (borg@imaginary.com)
 */
public class RowTokenizer implements ParsedRow {
    private boolean       complete        = false;
    private ArrayList     columns         = new ArrayList();
    private String        encoding        = "8859_1";
    private MsqlLog       log             = null;
    private MsqlException parseException  = null;
    private byte[]        rawData         = null;
    private int           size            = -1;
    
    public RowTokenizer(byte[] data, String enc, int ll) {
        super();
        log = new MsqlLog(ll, this);
        encoding = enc;
        rawData = data;
        try {
            log.log("constructor", MsqlLog.DRIVER, "Parsing row " +
                    new String(data, encoding));
        }
        catch( Exception e ) {
            log.log("constructor", MsqlLog.ERROR, "Encoding " + enc +
                    " not supported: " + e.getMessage());
        }
        Thread t = new Thread() {
            public void run() {
                parse();
            }
        };

        t.start();
    }

    public Iterator columns() throws MsqlException {
        return new ColumnIterator(this);
    }

    public String get(int i) throws MsqlException {
        if( parseException != null ) {
            throw parseException;
        }
        synchronized( columns ) {
            if( columns.size() < (i+1) ) {
                if( complete ) {
                    throw new IndexOutOfBoundsException();
                }
                while( !complete ) {
                    try { columns.wait(); }
                    catch( InterruptedException e ) { }
                }
            }
        }
        return (String)columns.get(i);
    }

    public boolean hasColumn(int col) throws MsqlException {
        if( parseException != null ) {
            throw parseException;
        }
        synchronized( columns ) {
            if( col < size ) {
                return true;
            }
            while( !complete ) {
                if( col < columns.size() ) {
                    return true;
                }
                try { columns.wait(); }
                catch( InterruptedException e ) { }
            }
        }
        return (col < columns.size());
    }
		    
    // UNICODE: not Unicode compat... will break if mSQL ever goes unicode
    private void parse() {
        int start = 0;
        while( true ) {
            start = readColumn(start);
            if( start == -1 ) {
                synchronized( columns ) {
                    complete = true;
                    columns.notifyAll();
                }
                break;
            }
        }
    }
	
    private int readColumn(int start) {
        int ind = start;
        byte[] tmp;
        int len;
        
        if( rawData.length <= start ) {
            return -1;
        }
        while( rawData[ind] != ':' &&
               rawData[ind] != '\r' &&
               rawData[ind] != '\n' ) {
            ind++;
        }
        if( ind == start ) {
            synchronized( columns ) {
                columns.add(null);
                columns.notifyAll();
                return ++ind;
            }
        }
        tmp = new byte[ind-start];
        System.arraycopy(rawData, start, tmp, 0, tmp.length);
        ind++;
        try {
            String str;

            // mSQL data lengths will always be 8859-1
            str = new String(tmp, "8859_1");
            len = Integer.parseInt(str);
        }
        catch( UnsupportedEncodingException e ) {
            String str = new String(tmp);
            
            len = Integer.parseInt(str);
        }
        if( len == -2 ) {
            synchronized( columns ) {
                columns.add(null);
                columns.notifyAll();
            }
        }
        else if( len == 0 ) {
            synchronized( columns ) {
                columns.add(null);
                columns.notifyAll();
            }
        }
        else {
            tmp = new byte[len];
            System.arraycopy(rawData, ind, tmp, 0, len);
            ind = ind + len;
            try {
                synchronized( columns ) {
                    columns.add(new String(tmp, encoding));
                    columns.notifyAll();
                }
            }
            catch( UnsupportedEncodingException e ) {
                synchronized( columns ) {
                    columns.add(new String(tmp));
                    columns.notifyAll();
                }
            }
        }
        return ind;
    }
    
    public void set(int i, String val) {
        columns.set(i, val);
    }
    
    public int size() throws MsqlException {
        if( parseException != null ) {
            throw parseException;
        }
        synchronized( columns ) {
            while( size == -1 ) {
                try { columns.wait(1500); }
                catch( InterruptedException e ) { }
            }
        }
        return size;
    }

    public String toString() {
        try {
            return new String(rawData, "8859_1");
        }
        catch( UnsupportedEncodingException e ) {
            return super.toString();
        }
    }
}
