MODULE AscString;
(* ========================================================================= *)
(*                                                                           *)
(*  String handling module for the .NET to Gardens Point Component Pascal    *)
(*  Symbols tool.                                                            *)
(*  This module uses strings terminated by null character.                   *)
(*      Copyright (c) Siu-Yuen Chan 2001.                                    *)
(*                                                                           *)
(*  Note: If str is a string produced by this module,                        *)
(*        the LEN(str) returns the length with the '\0' terminator character *)
(*        The only exception is the TrimNull() with remove the '\0' from     *)
(*        an input str.                                                      *)
(* ========================================================================= *)

CONST
    Greater* = 1;       (* String compare result, Greater Than *)
    Equal* = 0;         (* String compare result, Equal*)
    Less* = -1;         (* String compare result, Less Than *)
    NotExist* = -1;     (* StrChr result, if character is not exist in string *)

TYPE
    CharOpen* = POINTER TO ARRAY OF CHAR;

VAR
    NullString*: CharOpen;
    Null32String*: CharOpen;
    i: INTEGER;


PROCEDURE ToChrOpen*(IN str: ARRAY OF CHAR): CharOpen;
VAR
    i, len, len1: INTEGER;
    rslt: CharOpen;
BEGIN
    (* find the length of the actual null terminated str1 *)
    len := LEN(str); len1 := 0;
    WHILE (len1 < len) & (str[len1] # 0X) & (str[len1] <= 7FX) DO INC(len1); END;

    (* process str as it is a non-null terminated string *)
    IF (len1 > 0) & ((str[len1-1] = 0X) OR (str[len1-1] > 7FX)) THEN DEC(len1); END;

    (* copy all characters from str to rslt *)
    NEW(rslt, len1 + 1);
    FOR i := 0 TO len1-1 DO rslt[i] := str[i]; END;
    rslt[len1] := 0X;
    RETURN rslt;

END ToChrOpen;


PROCEDURE ToCap*(IN str: CharOpen): CharOpen;
(* Substitute all the occurence of small letters in a string with corresponding capital letters *)
VAR
    i, len: INTEGER;
    rslt: CharOpen;
BEGIN
    len := LEN(str); NEW(rslt, len);
    FOR i := 0 TO len-1 DO rslt[i] := CAP(str[i]); END;

    RETURN rslt;
END ToCap;


PROCEDURE TrimNull*(IN str: CharOpen): CharOpen;
(* Returns the input str without the null termination character. *)
VAR
    i, len: INTEGER;
    rslt: CharOpen;
BEGIN
    len := LEN(str); NEW(rslt, len-1);
    ASSERT(str[len-1] = 0X);
    FOR i := 0 TO len-2 DO rslt[i] := str[i]; END;

    RETURN rslt;
END TrimNull;


PROCEDURE StrSubChr*(IN str: CharOpen; ch1, ch2: CHAR): CharOpen;
(* Pre: str is a null terminated string, and contains only ASCII characters 0-127.
   Substitute all the occurence of a character (ch1) in a string (str) with another character (ch2) *)
VAR
    i, len: INTEGER;
    rslt: CharOpen;
BEGIN
    (* copy str to rslt with all occurance of ch1 replaced by ch2 *)
    len := LEN(str); NEW(rslt, len);
    FOR i := 0 TO len-1 DO
        IF str[i] # ch1 THEN rslt[i] := str[i]; ELSE rslt[i] := ch2; END;
    END; (* FOR *)
    RETURN rslt;
END StrSubChr;


PROCEDURE StrCatChr*(IN str: CharOpen; ch: CHAR): CharOpen;
(* Pre: str is a null terminated string, and contains only ASCII characters 0-127.
   Append a character (ch) to the end of a string (str) *)
VAR
    i, len: INTEGER;
    rslt: CharOpen;
BEGIN
    len := LEN(str); NEW(rslt, len+1);
    FOR i := 0 TO len-2 DO rslt[i] := str[i]; END;
    rslt[len-1] := ch; rslt[len] := 0X;
    RETURN rslt;
END StrCatChr;


PROCEDURE StrCat*(IN str1: CharOpen; IN str2: CharOpen): CharOpen;
(* Pre: str1 and str2 are null terminated strings, and contain only ASCII characters 1-127.
   Append a string (str2) to the end of another string (str1) *)
VAR
   i, len, len1, len2: INTEGER;
   rslt: CharOpen;
BEGIN
    len1 := LEN(str1); len2 := LEN(str2);
    len := len1 + len2 - 1;
    NEW(rslt, len);
    FOR i := 0 TO len1-2 DO rslt[i] := str1[i]; END;
    FOR i := 0 TO len2-2 DO rslt[len1-1+i] := str2[i]; END;
    rslt[len-1] := 0X;
    RETURN rslt;
END StrCat;

(*
PROCEDURE StrStr*(IN str1: CharOpen; IN str2: CharOpen): BOOLEAN;
(* Pre: str1 and str2 are null terminated strings, and contain only ASCII characters 1-127.
   Search for the occurence of a string (str2) in another string (str1), 
   returns TRUE if found, else returns FALSE *)
VAR
    i, j, len1, len2: INTEGER;
    equal: BOOLEAN;
BEGIN
    len1 := LEN(str1); len2 := LEN(str2);
    IF len2 > len1 THEN RETURN FALSE END;
    i := 0; j := 0; equal := FALSE;
    WHILE (i < len1) & (j < len2) DO
        IF str2[j] = str1[i] THEN
            INC(i); INC(j); equal := TRUE;
        ELSE
            IF ~equal THEN INC(i); END;
            j := 0; equal := FALSE;
        END;
    END; (* WHILE *)
    RETURN (j = len2);
END StrStr;
*)

PROCEDURE StrStr*(IN str1: CharOpen; IN str2: CharOpen): INTEGER;
(* Pre: str1 and str2 are null terminated strings, and contain only ASCII characters 1-127.
   Search for the occurence of a string (str2) in another string (str1), 
   returns the first character position of str2 if found, else returns NotExist *)
VAR
    i, j, len1, len2, idx: INTEGER;
    equal: BOOLEAN;
BEGIN
    len1 := LEN(str1)-1; len2 := LEN(str2)-1;    (* not compare the terminating null character *)
    IF len2 > len1 THEN RETURN NotExist END;
    i := 0; j := 0; idx := i; equal := FALSE;
    WHILE (i < len1) & (j < len2) DO
        IF str2[j] = str1[i] THEN
            INC(i); INC(j); equal := TRUE;
        ELSE
            IF ~equal THEN INC(i); END;
            j := 0; idx := i; equal := FALSE;
        END;
    END; (* WHILE *)
    IF (j = len2) THEN
        RETURN idx;
    ELSE
        RETURN NotExist;
    END; (* IF *)
END StrStr;


PROCEDURE StrChr*(IN str: CharOpen; ch: CHAR): INTEGER;
(* Pre: str is a null terminated string, and contains only ASCII characters 0-127.
   Forward search for the occurence of a character (ch) in a string (str), 
   returns the position of the character if found, else returns NotExist *)
VAR i: INTEGER;
BEGIN
    FOR i := 0 TO LEN(str)-1 DO
        IF str[i] = ch THEN RETURN i END;
    END; (* FOR *)
    RETURN NotExist;
END StrChr;


PROCEDURE StrRChr*(IN str: CharOpen; ch: CHAR): INTEGER;
(* Pre: str is a null terminated string, and contains only ASCII characters 0-127.
   Reverse search for the occurence of a character in a string, 
   returns the position of the character if found, else returns NotExist *)
VAR i: INTEGER;
BEGIN
    FOR i := LEN(str)-1 TO 0 BY - 1 DO
        IF str[i] = ch THEN RETURN i END;
    END; (* FOR *)
    RETURN NotExist;
END StrRChr;


PROCEDURE StrCmp*(IN str1: CharOpen; IN str2: CharOpen): INTEGER;
(* Pre: str1 and str2 are null terminated strings, and contain only ASCII characters 1-127.
   Compare two strings and return whether the first string (str1) is greater,
   equal, or less than the second string (str2) *)
VAR
    i, len, len1, len2: INTEGER;
BEGIN
    i := 0; len1 := LEN(str1); len2 := LEN(str2);
    IF len1 < len2 THEN len := len1 ELSE len := len2 END;
    WHILE (i < len) DO
        IF str1[i] > str2[i] THEN RETURN Greater;
        ELSIF str1[i] < str2[i] THEN RETURN Less;
        ELSE i := i + 1;
        END; (* IF *)
    END; (* WHILE *)
    IF len1 > len2 THEN RETURN Greater;
    ELSIF len1 < len2 THEN RETURN Less;
    ELSE RETURN Equal;
    END; (* IF *)
END StrCmp;


PROCEDURE SubStr*(IN str: CharOpen; idx1, idx2: INTEGER): CharOpen;
(* Pre: str is a null terminated string, and contains only ASCII characters 0-127.
   Retrieve a substring from a string, given where to start and end *)
VAR i, dlen, slen: INTEGER;
    rslt: CharOpen;
BEGIN
    (* returns null string if idx1 > idx2 *)
    IF (idx1 > idx2) THEN RETURN NullString END;

    (* returns null string is idx1 is out of last character *)
    slen := LEN(str);
    IF (idx1 >= slen) THEN NEW(rslt,1); rslt[0] := 0X; RETURN rslt; END;

    (* set idx2 to last character if idx2 is out of last character *)
    IF idx2 >= slen THEN idx2 := slen - 1 END;

    IF (idx2 = slen-1) & (str[idx2] = 0X) THEN DEC(idx2) END;    (* idx2 is the last null character *)
    dlen := idx2 - idx1 + 1;
    NEW(rslt, dlen+1); rslt[dlen] := 0X;
    FOR i := idx1 TO idx2 DO rslt[i-idx1] := str[i] END;

    RETURN rslt;
END SubStr;


BEGIN
    NEW(NullString, 1); NullString[0] := 0X;
    NEW(Null32String, 32);
    FOR i := 0 TO 31 DO Null32String[i] := 0X END;
END AscString.
