/**
 * Title:        Comedia Beans
 * Description:  Calendar Bean
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 *
 * Changes:
 * (05 aug 2001) - Added Enabled property, which turns on/off user interface.
 */

package org.comedia.beans;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.border.*;
import java.text.*;
import java.util.*;
import org.comedia.ui.*;
import org.comedia.event.*;

/**
 * Visual component for Gregorian Calendar presentation.
 * Shows year, months and days of the month in a table.
 * Default date is current but you can change it on the screen
 * or using Date property.
 * Bean can be successfuly resized. The view of this bean is
 * similar to <b>Windows Date/Time Properties</b> application.
 * <p>
 * <center><img src="CCalendar.gif"></center>
 */
public class CCalendar extends JComponent implements ActionListener,
  CSpinListener, FocusListener {

  final private int startYear = 1900;
  final private int endYear = 2100;
  private boolean enabled = true;
  GregorianCalendar calendar = new GregorianCalendar();
  DateFormatSymbols format = new DateFormatSymbols();
  BorderLayout borderLayout1 = new BorderLayout();
  JPanel monthPanel = new JPanel();
  JComboBox monthComboBox;
  CSpinEdit yearComboBox = new CSpinEdit();
  GridLayout gridLayout1 = new GridLayout();
  Border border1;
  JPanel dayPanel = new JPanel();
  BorderLayout borderLayout2 = new BorderLayout();
  Border border2;
  TitledBorder titledBorder1;
  CalendarPanel calendarPanel = new CalendarPanel();

  /**
   * Default calendar constructor. Sets current date
   */
  public CCalendar() {
    try {
      jbInit();
    }
    catch(Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * Constructs this bean and sets specified date.
   * @param date specified date for calendar.
   */
  public CCalendar(Date date) {
    this();
    setDate(date);
  }

  /**
   * Gets selected date of this calendar.
   * @result selected date of this calendar.
   */
  public Date getDate() {
    return calendar.getTime();
  }

  /**
   * Sets a date for this calendar.
   * Methods redraws the bean on the screen after setting.
   * @param date specified date for this calendar.
   */
  public void setDate(Date date) {
    calendar.setTime(date);
    monthComboBox.setSelectedIndex(calendar.get(Calendar.MONTH));
    yearComboBox.setValue(calendar.get(Calendar.YEAR));
    calendarPanel.repaint();
  }

  /**
   * Gets Enabled property value.
   * @result enabled boolean value.
   */
  public boolean isEnabled() {
    return enabled;
  }

  /**
   * Sets Enabled property value and switch user interface.
   * @param enabled a new property value.
   */
  public void setEnabled(boolean enabled) {
    this.enabled = enabled;
    yearComboBox.setEnabled(enabled);
    monthComboBox.setEnabled(enabled);
  }

  /**
   * Constracts User Interface of the bean.
   * Used by JBuilder designers.
   */
  private void jbInit() throws Exception {
    monthComboBox = new JComboBox(format.getMonths());
    monthComboBox.setSelectedIndex(calendar.get(Calendar.MONTH));
    monthComboBox.addActionListener(this);
    yearComboBox.setMinimum(startYear);
    yearComboBox.setMaximum(endYear);
    yearComboBox.setValue(calendar.get(Calendar.YEAR));
    yearComboBox.getTextField().addActionListener(this);
    yearComboBox.getTextField().addFocusListener(this);
    yearComboBox.getTextField().setHorizontalAlignment(JLabel.LEFT);
    yearComboBox.getSpinButton().addSpinListener(this);

    border1 = BorderFactory.createEmptyBorder(11,11,11,11);
    border2 = BorderFactory.createEmptyBorder(0,11,11,11);
    titledBorder1 = new TitledBorder("");
    this.setLayout(borderLayout1);
    monthPanel.setLayout(gridLayout1);
    gridLayout1.setColumns(2);
    gridLayout1.setHgap(11);
    gridLayout1.setVgap(5);
    monthPanel.setBorder(border1);
    dayPanel.setLayout(borderLayout2);
    dayPanel.setBorder(border2);
    this.add(monthPanel, BorderLayout.NORTH);
    monthPanel.add(monthComboBox, null);
    monthPanel.add(yearComboBox, null);
    this.add(dayPanel, BorderLayout.CENTER);
    dayPanel.add(calendarPanel, BorderLayout.CENTER);
  }

  /**
   * Processes actions from User Interface.
   * Accepts changes of year and month and redraws bean on the screen.
   */
  public void actionPerformed(ActionEvent e) {
    int month = monthComboBox.getSelectedIndex();
    int year = yearComboBox.getValue();
    int day = calendar.get(Calendar.DAY_OF_MONTH);
    calendar.set(Calendar.DAY_OF_MONTH, 1);
    calendar.set(Calendar.YEAR, year);
    calendar.set(Calendar.MONTH, month);
    if (day > calendar.getActualMaximum(Calendar.DAY_OF_MONTH))
      calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMaximum(Calendar.DAY_OF_MONTH));
    else
      calendar.set(Calendar.DAY_OF_MONTH, day);
    calendarPanel.repaint();
  }

  /**
   * Performs event when this table box gets a focus.
   * @param e an object which described occured event.
   */
  public void focusGained(FocusEvent e) {
  }

  /**
   * Performs event when this table box lost a focus.
   * @param e an object which described occured event.
   */
  public void focusLost(FocusEvent e) {
    actionPerformed(null);
  }

  /**
   * If this listener is registered with a Spinner, this method will
   * be called when the control is performed.
   * @param event the CSpinEvent providing information about the 'spin'.
   */
  public void spinEventPerformed(CSpinEvent event) {
    actionPerformed(null);
  }

  /**
   * Presents a table with days. Allows to make a day selection.
   */
  private class CalendarPanel extends JPanel implements MouseListener {

    /**
     * Default class constructor.
     */
    public CalendarPanel() {
      this.addMouseListener(this);
    }

    /**
     * Performs days selection for this table.
     * @param e performed mouse event.
     */
    public void mousePressed(MouseEvent e) {
      Dimension d = this.getSize();
      int dx = d.width / 7;
      int dy = d.height / 7;
      int c = e.getX() / dx;
      int r = e.getY() / dy;
      if (r == 0 || !enabled) return;
      r--;
      int dn = (r * 7) + c + findFirstDay();
      if (dn > 0 && dn <= calendar.getActualMaximum(Calendar.DAY_OF_MONTH)) {
        calendar.set(Calendar.DAY_OF_MONTH, dn);
        repaint();
      }
    }

    /**
     * Skips the <code>mouseReleased</code> event.
     * @param e performed mouse event.
     */
    public void mouseReleased(MouseEvent e) {
    }

    /**
     * Skips the <code>mouseEntered</code> event.
     * @param e performed mouse event.
     */
    public void mouseEntered(MouseEvent e) {
    }

    /**
     * Skips the <code>mouseExited</code> event.
     * @param e performed mouse event.
     */
    public void mouseExited(MouseEvent e) {
    }

    /**
     * Skips the <code>mouseClicked</code> event.
     * @param e performed mouse event.
     */
    public void mouseClicked(MouseEvent e) {
    }

    /**
     * Finds a font size which better fits the specified message to given sizes.
     * @param g current graphical context.
     * @param s string message.
     * @param w width of the rectangle.
     * @param h height of the rectangle.
     */
    private Font findFont(Graphics g, String s, int w, int h) {
      int size = 6;
      Font font = new Font(this.getFont().getName(), this.getFont().getStyle(), size);
      while (true) {
        FontMetrics m = g.getFontMetrics(font);
        if (m.getHeight() > h || m.stringWidth(s) > w)
          break;
        size++;
        font = new Font(this.getFont().getName(), this.getFont().getStyle(), size);
      }
      return font;
    }

    /**
     * Finds a location of the first day of the month.
     */
    private int findFirstDay() {
      int w = calendar.get(Calendar.DAY_OF_WEEK);
      int d = calendar.get(Calendar.DAY_OF_MONTH);
      while (true) {
        if (d <= 1 && w == calendar.getFirstDayOfWeek()) break;
        d--; w--;
        w = (w < 1) ? 7 : w;
      }
      return d;
    }

    /**
     * Paints this days table on the screen.
     * Colors can not be changed in current version of this bean.
     * @param g current graphical context.
     */
    public void paint(Graphics g) {
      Dimension d = this.getSize();
      int dx = d.width / 7;
      int dy = d.height / 7;
      Font font = findFont(g, "Www", dx, dy);
      g.setFont(font);
      int selectedDay = calendar.get(Calendar.DAY_OF_MONTH);

      g.setColor(Color.white);
      g.fillRect(0, 0, d.width, d.height);
      g.setColor(Color.gray);
      g.fillRect(0, 0, d.width, dy);

      int x, y, dn = findFirstDay();
      int w = calendar.getFirstDayOfWeek();
      String s;
      FontMetrics metrics = g.getFontMetrics(font);
      for(int i = 0; i < 7; i++) {
        for(int j = 0; j < 7; j++) {
          x = j * dx;
          y = i * dy;

          if (i == 0) {
            g.setColor(Color.white);
            s = format.getShortWeekdays()[w];
            w++;
            w = (w > 7) ? 1 : w;
          } else {
            if (dn == selectedDay) {
              g.setColor(Color.blue);
              g.fillRect(x, y, dx, dy);
            }
            if (dn == selectedDay)
              g.setColor(Color.white);
            else
              g.setColor(Color.black);
            if (dn > 0 && dn <= calendar.getActualMaximum(Calendar.DAY_OF_MONTH))
              s = new Integer(dn).toString();
            else
              s = "";
            dn++;
          }
          x += (dx - metrics.stringWidth(s)) / 2;
          y += (dy - metrics.getHeight()) / 2;
          g.drawString(s, x, y + metrics.getAscent());
        }
      }
      g.draw3DRect(0, 0, d.width-1, d.height-1, false);
    }
  }

  /**
   * Shows this bean in separate application frame on the screen.
   * It allows to use this bean as stand along application.
   */
  public static void main(String[] args) {
    try {
//      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
      UIManager.setLookAndFeel("com.sun.java.swing.plaf.motif.MotifLookAndFeel");
    }
    catch(Exception e) {}
    CCalendar calendar = new CCalendar();
    JFrame frame = new JFrame("Gregorian Calendar");
    frame.setContentPane(calendar);
    frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    frame.setSize(300, 300);
    frame.setLocation(300, 300);
    frame.show();
  }

}