/**
 * Title:        Comedia Beans
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.game;

/**
 * Presents a table of high scores for game beans.
 */
public class CHighScores {
  /**
   * The maximum table length.
   */
  final static public int MAX_SCORES = 5;

  /**
   * The table players name contents.
   */
  private String[] players = new String[MAX_SCORES];

  /**
   * The table players score contents.
   */
  private int[] scores = new int[MAX_SCORES];

  /**
   * The current length of this score table.
   */
  private int length = 0;

  /**
   * Constructs this table with default values.
   */
  public CHighScores() {
  }

  /**
   * Gets the current table length.
   * @result the current table length.
   */
  public int getLength() {
    return length;
  }

  /**
   * Gets payer name by index.
   * @param index an index of a player.
   * @result a player name.
   */
  public String getPlayerName(int index) {
    return players[index];
  }

  /**
   * Gets payer score by index.
   * @param index an index of a player.
   * @result a player score.
   */
  public int getPlayerScore(int index) {
    return scores[index];
  }

  /**
   * Gets the minimum score in the table.
   * @result the minimum score in this table.
   */
  public int getMinimumScore() {
    if (length == 0)
      return 0;
    return scores[length - 1];
  }

  /**
   * Checks if this table full.
   * @result <code>TRUE</code> if table full and <code>FALSE</code> otherwise.
   */
  public boolean isFull() {
    return (length >= MAX_SCORES);
  }

  /**
   * Clears this score table.
   */
  public void clear() {
    length = 0;
  }

  /**
   * Adds a new score into this table
   */
  public void addScore(String player, int score) {
    // Checks can it be added
    if (isFull() && score < getMinimumScore())
      return;
    // Find position inside table
    int pos = length;
    if (score > getMinimumScore()) {
      for (int i = length-1; i >= 0; i--) {
        if (scores[i] >= score)
          break;
        pos = i;
      }
    }
    // Insert a new score
    for (int i = length-1; i >= pos; i--) {
      if (i+1 < MAX_SCORES) {
        scores[i+1] = scores[i];
        players[i+1] = players[i];
      }
    }
    scores[pos] = score;
    players[pos] = player;
    if (length < MAX_SCORES) length++;
  }

  /**
   * Formats a score by specified index.
   * @param index a score index.
   * @param length a player name length.
   * @result a formatted score.
   */
  public String formatScore(int index, int length) {
    int end = players[index].length();
    if (end > length) end = length;
    String result = players[index].substring(0, end);
    while (result.length() <= length+1)
      result += " ";
    return result + scores[index];
  }
}