/**
 * Title:        Comedia Beans
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.ui;

import java.awt.*;
import javax.swing.*;
import java.awt.event.*;
import javax.swing.event.*;
import org.comedia.event.*;

/**
 * Implements a spin button control.
 * <p><img src="CSpinButton.gif">
 * <p>Usage example:
 * <pre>
 * JFrame frame = new JFrame("Comedia CSpinButton Test");
 * final CSpinButton spin = new CSpinButton();
 * final JTextField text = new JTextField("0                 ");
 *
 * frame.getContentPane().setLayout(new FlowLayout());
 * frame.getContentPane().add(text);
 * frame.getContentPane().add(spin);
 *
 * spin.addSpinListener(new CSpinListener() {
 *     public void spinEventPerformed(CSpinEvent event) {
 *       text.setText(new Integer(spin.getValue()).toString());
 *     }
 *   });
 * </pre>
 */
public class CSpinButton extends JComponent implements KeyListener,
  AdjustmentListener {
  /**
   * The list of listeners.
   **/
  protected EventListenerList listenerList = new EventListenerList();

  /**
   * The scroll control.
   */
  private JScrollBar scrollBar = new JScrollBar(JScrollBar.VERTICAL, 1, 0, 0, 2);

  /**
   * The maximum value.
   */
  private int maximum = 100;

  /**
   * The minimum value.
   */
  private int minimum = 0;

  /**
   * The counter value.
   */
  private int value = 0;

  /**
   * Constructs this control with default properties.
   */
  public CSpinButton() {
    this.setLayout(new BorderLayout());
    this.setRequestFocusEnabled(true);
    this.setPreferredSize(new Dimension(16, 21));
    this.addKeyListener(this);

    this.add(scrollBar);
    scrollBar.addAdjustmentListener(this);
    scrollBar.setRequestFocusEnabled(false);
  }

  /**
  * Adds a listener to the list that's notified each time a change
  * to the spin value occurs.
  * @param l the TableModelListener.
  */
  public void addSpinListener(CSpinListener l) {
    listenerList.add(CSpinListener.class, l);
  }

  /**
  * Removes a listener from the list that's notified each time a
  * change to the spin occurs.
  * @param l the TableModelListener.
  */
  public void removeSpinListener(CSpinListener l) {
    listenerList.remove(CSpinListener.class, l);
  }

  /**
  * Forwards the given notification event to all
  * <code>SpinListeners</code> that registered
  * themselves as listeners for this table model.
  * @param e the event to be forwarded
  */
  public void fireSpinChanged(CSpinEvent e) {
    // Guaranteed to return a non-null array
    Object[] listeners = listenerList.getListenerList();
    // Process the listeners last to first, notifying
    // those that are interested in this event
    for (int i = listeners.length-2; i>=0; i-=2) {
      if (listeners[i] == CSpinListener.class) {
        ((CSpinListener)listeners[i+1]).spinEventPerformed(e);
      }
    }
  }

  /**
   * Checks is focus of the links is traversable.
   * @result <code>TRUE</code> always.
   */
  public boolean isFocusTraversable() {
    return true;
  }

  /**
   * Checks is focus of the links is managable.
   * @result <code>TRUE</code> always.
   */
  public boolean isManagingFocus() {
    return true;
  }

  /**
   * Performs UP event for spin box.
   */
  private void performSpinUp() {
    if (value < maximum) value++;
    this.fireSpinChanged(new CSpinEvent(this, CSpinEvent.UP));
  }

  /**
   * Performs DOWN event for spin box.
   */
  private void performSpinDown() {
    if (value > minimum) value--;
    this.fireSpinChanged(new CSpinEvent(this, CSpinEvent.DOWN));
  }

  /**
   * Performs event from scrollbar control.
   * @param e description of the event.
   */
  public void adjustmentValueChanged(AdjustmentEvent e) {
    switch (e.getValue()) {
      case 0:
        performSpinUp();
        break;
      case 2:
        performSpinDown();
        break;
    }
    e.getAdjustable().setValue(1);
    this.requestFocus();
  }

  /**
   * Performs char typed event.
   * @param e the event description.
   */
  public void keyTyped(KeyEvent e) {
  }

  /**
   * Performs key pressed event.
   * @param e the event description.
   */
  public void keyPressed(KeyEvent e) {
    int keycode = e.getKeyCode();
    if (keycode == e.VK_UP ||  keycode == e.VK_PAGE_UP || keycode == e.VK_HOME) {
//      upButton.doClick();
      performSpinUp();
//      scrollBar.mouseDown(new MouseEvent(this, Event.MOUSE_DOWN, 0, 0, 2, 2, 1, false),
//        2, 2);
//      scrollBar.mouseUp(new MouseEvent(this, Event.MOUSE_UP, 0, 0, 2, 2, 1, false),
//        2, 2);
      e.consume();
    } else if (keycode == e.VK_DOWN ||  keycode == e.VK_PAGE_DOWN
      || keycode == e.VK_END) {
//      downButton.doClick();
      performSpinDown();
      e.consume();
    }
  }

  /**
   * Performs button released event.
   * @param e the event description.
   */
  public void keyReleased(KeyEvent e) {
  }

  /**
   * Gets the current spin value.
   * @result the current spin value.
   */
  public int getValue() {
    return value;
  }

  /**
   * Sets a new spin value.
   * @param value a new spin value.
   */
  public void setValue(int value) {
    this.value = value;
  }

  /**
   * Gets the current spin maximum value.
   * @result the current spin maximum value.
   */
  public int getMaximum() {
    return maximum;
  }

  /**
   * Sets a new spin maximum value.
   * @param value a new spin maximum value.
   */
  public void setMaximum(int maximum) {
    this.maximum = maximum;
  }

  /**
   * Gets the current spin minimum value.
   * @result the current spin minimum value.
   */
  public int getMinimum() {
    return minimum;
  }

  /**
   * Sets a new spin minimum value.
   * @param value a new spin minimum value.
   */
  public void setMinimum(int minimum) {
    this.minimum = minimum;
  }

  /**
   * Sets a new enable status for this control.
   * @param enabled <code>TRUE</code> to enable the control and
   *   <code>FALSE</code> to disable.
   */
  public void setEnabled(boolean enabled) {
    scrollBar.setEnabled(enabled);
    super.setEnabled(enabled);
  }

  /**
   * The main routine to run this module as standalone application.
   */
  public static void main(String[] args) {
    try {
//      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
//      UIManager.setLookAndFeel("com.sun.java.swing.plaf.motif.MotifLookAndFeel");
    }
    catch(Exception e) {}

    JFrame frame = new JFrame("Comedia CSpinButton Test");
    final CSpinButton spin = new CSpinButton();
    final JTextField text = new JTextField("0                 ");

    frame.getContentPane().setLayout(new FlowLayout());
    frame.getContentPane().add(text);
    frame.getContentPane().add(spin);

    spin.addSpinListener(new CSpinListener() {
        public void spinEventPerformed(CSpinEvent event) {
          text.setText(new Integer(spin.getValue()).toString());
        }
      });
//    spin.setEnabled(false);

    frame.setLocation(100, 100);
    frame.setSize(150, 100);
    frame.setDefaultCloseOperation(frame.EXIT_ON_CLOSE);
    frame.show();
  }
}