/**
 * Title:        Comedia Beans
 * Copyright:    Copyright (c) 2001
 * Company:      Capella Development Group
 * @author Sergey Seroukhov
 * @version 1.0
 */

package org.comedia.ui;

import java.awt.*;
import javax.swing.*;
import java.awt.event.*;
import javax.swing.event.*;
import org.comedia.event.*;
import org.comedia.text.*;

/**
 * Implements a numeric spin editor control.
 * <p><img src="CSpinEdit.gif">
 * <p>Usage example:
 * <pre>
 * JFrame frame = new JFrame("Comedia CSpinButton Test");
 * CSpinEdit spinEdit = new CSpinEdit();
 * spinEdit.setValue(50);
 *
 * frame.getContentPane().setLayout(new FlowLayout());
 * frame.getContentPane().add(spinEdit);
 * </pre>
 */
public class CSpinEdit extends JComponent implements CSpinListener, KeyListener,
  ActionListener, FocusListener {
  /**
   * The text editor.
   */
  private JTextField text = new JTextField();

  /**
   * The spin button.
   */
  private CSpinButton spin = new CInternalSpin();

  /**
   * The numeric document object.
   */
  private CNumericDocument document = new CNumericDocument();

  /**
   * The maximum value.
   */
  private int maximum = 100;

  /**
   * The minimum value.
   */
  private int minimum = 0;

  /**
   * Constructs this control with default properties.
   */
  public CSpinEdit() {
    this.setLayout(new BorderLayout());
    this.add(text, BorderLayout.CENTER);
    this.add(spin, BorderLayout.EAST);
    this.setPreferredSize(new Dimension(70, 21));

    spin.setRequestFocusEnabled(false);
    spin.addSpinListener(this);

    text.setDocument(document);
    text.setText("0");
    text.setHorizontalAlignment(text.RIGHT);
    text.addKeyListener(this);
    text.addActionListener(this);
    text.addFocusListener(this);

//    this.setBorder(text.getBorder());
//    text.setBorder(null);
  }

  /**
   * Gets a value from the text editor.
   * @result an integer value from the text editor.
   */
  private int getTextValue() {
    String temp = text.getText();
    int result = minimum;
    try {
      result = new Integer(temp).intValue();
    } catch (Exception e) {}
    return result;
  }

  /**
   * Sets a new integer value to text editor.
   * @param value a new integer value to text editor.
   */
  private void setTextValue(int value) {
    if (value > maximum) value = maximum;
    if (value < minimum) value = minimum;
    text.setText(new Integer(value).toString());
  }

  /**
   * Implements a spin without focus management.
   */
  private class CInternalSpin extends CSpinButton {
    public boolean isFocusTraversable() {
      return false;
    }
  }

  /**
   * Performs event when this table box gets a focus.
   * @param e an object which described occured event.
   */
  public void focusGained(FocusEvent e) {
  }

  /**
   * Performs event when this table box lost a focus.
   * @param e an object which described occured event.
   */
  public void focusLost(FocusEvent e) {
    setTextValue(getTextValue());
  }

  /**
   * Performs actions from user interface.
   * @param e description of the action.
   */
  public void actionPerformed(ActionEvent e) {
    int value = getTextValue();
    setTextValue(value);
  }

  /**
   * Performs char typed event.
   * @param e the event description.
   */
  public void keyTyped(KeyEvent e) {
  }

  /**
   * Performs key pressed event.
   * @param e the event description.
   */
  public void keyPressed(KeyEvent e) {
    int keycode = e.getKeyCode();
    if (keycode == e.VK_UP ||  keycode == e.VK_PAGE_UP || keycode == e.VK_HOME) {
      spin.keyPressed(e);
      e.consume();
    } else if (keycode == e.VK_DOWN ||  keycode == e.VK_PAGE_DOWN
      || keycode == e.VK_END) {
      spin.keyPressed(e);
      e.consume();
    }
  }

  /**
   * Performs button released event.
   * @param e the event description.
   */
  public void keyReleased(KeyEvent e) {
  }

  /**
   * If this listener is registered with a Spinner, this method will
   * be called when the control is performed.
   * @param event the CSpinEvent providing information about the 'spin'.
   */
  public void spinEventPerformed(CSpinEvent event) {
    int value = getTextValue();
    if (event.getSpinCommand() == event.UP)
      setTextValue(value + 1);
    if (event.getSpinCommand() == event.DOWN)
      setTextValue(value - 1);
  }

  /**
   * Gets the current spin value.
   * @result the current spin value.
   */
  public int getValue() {
    return getTextValue();
  }

  /**
   * Sets a new spin value.
   * @param value a new spin value.
   */
  public void setValue(int value) {
    setTextValue(value);
  }

  /**
   * Gets the current spin maximum value.
   * @result the current spin maximum value.
   */
  public int getMaximum() {
    return maximum;
  }

  /**
   * Sets a new spin maximum value.
   * @param value a new spin maximum value.
   */
  public void setMaximum(int maximum) {
    if (maximum <= minimum)
      maximum = minimum + 1;
    this.maximum = maximum;
    setTextValue(getTextValue());
  }

  /**
   * Gets the current spin minimum value.
   * @result the current spin minimum value.
   */
  public int getMinimum() {
    return minimum;
  }

  /**
   * Sets a new spin minimum value.
   * @param value a new spin minimum value.
   */
  public void setMinimum(int minimum) {
    if (minimum >= maximum)
      minimum = maximum - 1;
    this.minimum = minimum;
    setTextValue(getTextValue());
  }

  /**
   * Sets a new enable status for this control.
   * @param enabled <code>TRUE</code> to enable the control and
   *   <code>FALSE</code> to disable.
   */
  public void setEnabled(boolean enabled) {
    spin.setEnabled(enabled);
    text.setEnabled(enabled);
    super.setEnabled(enabled);
  }

  /**
   * Gets the text field control of this component.
   * @result the text field control.
   */
  public JTextField getTextField() {
    return text;
  }

  /**
   * Gets the spin button control of this component.
   * @result the spin button control.
   */
  public CSpinButton getSpinButton() {
    return spin;
  }

  /**
   * The main routine to run this module as standalone application.
   */
  public static void main(String[] args) {
    try {
//      UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
//      UIManager.setLookAndFeel("com.sun.java.swing.plaf.motif.MotifLookAndFeel");
    }
    catch(Exception e) {}

    JFrame frame = new JFrame("Comedia CSpinButton Test");
    CSpinEdit spinEdit = new CSpinEdit();
    spinEdit.setValue(50);

    frame.getContentPane().setLayout(new FlowLayout());
    frame.getContentPane().add(spinEdit);
//    frame.getContentPane().add(new JTextField("XXX"));

    frame.setLocation(100, 100);
    frame.setSize(150, 100);
    frame.setDefaultCloseOperation(frame.EXIT_ON_CLOSE);
    frame.show();
  }
}