// copyright 2001-2002 by The Mind Electric

package electric.xml.xpath;

import java.io.*;
import electric.util.*;
import electric.xml.*;

/**
 * <tt>AttributePredicate</tt> processes XPath "[key='value']" and "name[key='value']" sub-expressions.
 *
 * @author <a href="http://www.themindelectric.com">The Mind Electric</a>
 */

public class AttributePredicate extends NameNode
  {
  private String key;
  private String value;
  private boolean not;

  // ********** CONSTRUCTION ************************************************

  /**
   * @param string
   * @throws XPathException
   */
  public AttributePredicate( String string )
    throws XPathException
    {
    try
      {
      int openBracket = string.indexOf( '[' );

      if( openBracket > 0 ) // leading name is optional
        setName( string.substring( 0, openBracket ) );

      int index = openBracket + 2;
      int equals = string.indexOf( '=', index );

      if( equals == -1 ) // [@key]
        {
        int closeBracket = string.indexOf( ']', index );

        if( closeBracket == -1 )
          throw new XPathException( "missing ] in " + string );

        key = string.substring( index, closeBracket ).trim();
        }
      else
        {
        if( string.charAt( equals - 1 ) == '!' ) // [@key!='value']
          {
          not = true;
          key = string.substring( index, equals - 1 ).trim();
          }
        else // [@key='value']
          {
          key = string.substring( index, equals ).trim();
          }

        Lex lex = new Lex( string.substring( equals + 1 ) );
        lex.skipWhitespace();
        int ch = lex.read();

        if( ch == '\"' )
          value = lex.readToPattern( "\"", Lex.CONSUME | Lex.HTML );
        else if( ch == '\'' )
          value = lex.readToPattern( "'", Lex.CONSUME | Lex.HTML );
        else
          throw new XPathException( "missing quote at start of attribute value" );

        lex.readChar( ']' );
        }
      }
    catch( IOException exception )
      {
      throw new XPathException( exception.toString() );
      }
    }

  // ********** MATCHING ****************************************************

  /**
   * @param xpath
   * @param node
   * @param list
   * @throws XPathException
   */
  public void addNodes( TMEXPath xpath, Node node, NodeList list )
    throws XPathException
    {
    NodeList selected = new NodeList();

    for( Elements elements = ((Parent) node).getElements(); elements.hasMoreElements(); )
      {
      Element element = elements.next();
      String attributeValue = element.getAttributeValue( key );

      if( value == null )
        {
        if( attributeValue == null )
          continue;
        }
      else if( not )
        {
        if( value.equals( attributeValue ) )
          continue;
        }
      else
        {
        if( !value.equals( attributeValue ) )
          continue;
        }

      if( name == null )
        selected.append( new Selection( element ) );
      else if( element.hasName( getNamespace( xpath, node ), name ) )
        selected.append( new Selection( element ) );
      }

    addNextNodes( xpath, new Nodes( selected ), list );
    }
  }