/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.designcockpit.ide;

import javax.swing.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;

/**
 * Manages a list of recently opened files.<br><br>
 * use <code>setMaxFiles<code> to limit the number of files in the list.<br>
 * Use <code>setMenu<code> to associate a <code>MruMenu</code> menu, and <code>setShowMnemonics</code>
 * to automatically add number/letter mnemonics to the entries on a menu.<br><br>
 * Whenever you open a file, invoke the <code>fileOpened</code> method to update the list and associated menu.
 */
class MruList {
    private ArrayList list;
    private int maxFiles;
    private MruMenu menu;
    private boolean showMnemonics;
    private String mnemonics = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";

    /** Creates an <code>MruList</code> instance, defining the maximum files.
     *
     * @param maxFiles the number of files to allow in the list
     */
    public MruList(int maxFiles) {
        list = new ArrayList(maxFiles);
        this.maxFiles = maxFiles;
    }

    /** Creates an <code>MruList</code> instance, defining the maximum files and whether mnemonics should be automatically generated.
     *
     * @param maxFiles the number of files to allow in the list
     * @param bShowMnemonics true, if nmemonics should be automatically generated
     */
    public MruList(int maxFiles, boolean bShowMnemonics) {
        this(maxFiles);
        this.showMnemonics = bShowMnemonics;
    }

    /** Notifies the list that a file has been opened. If a menu is associated, then it will be updated
     *
     * @param sFile name is the file opened
     */
    public void fileOpened (String sFile) {
        if (isInList(sFile))
            removeFromList(sFile);
        updateMenu();
    }

    /** Notifies the list that a file has been closed. If a menu is associated, then it will be updated
     *
     * @param sFile name is the file opened
     */
    public void fileClosed (String sFile) {
        if (isInList(sFile))
            removeFromList(sFile); // just in case it got in list because fileOpened was never called when it was opened (it should have been)
        if (numberOfFiles() >= maxFiles)
            removeOldestFile();
        insertAtTop(sFile);
        updateMenu();
    }

    /**
     * Switches on Mnemonics for the associated menu.
     * @param bShowMnemonics If true, mnemonics are automatically generated and displayed on the associated menu.
     */
    public void setShowMnemonics(boolean bShowMnemonics) {
        this.showMnemonics = bShowMnemonics;
        updateMenu();
    }

    /**
     * Defines the list of mnemonics that will be used when the list is displayed in the associted menu
     * @param mnemonics a String comprising the charactes to be used. If you don't send this message, the default is "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ", which means that the first menu item will have a mnemonic of "0" and the eleveth will be "A". Mnemonics will be assigned until there are no mone left, in which case items will have no mnemonic
     */
    public void setMnemonics(String mnemonics) {
        this.mnemonics = mnemonics;
        updateMenu();
    }



    /**
     * Removes all entries from the list and associated menu.
     */
    public void clearList () {
        list.clear();
        updateMenu();
    }

    private void  insertAtTop (String sFile) {
        list.add(0, sFile);
    }

    /**
     * Determines whether a file is in the list
     * @param sFile the fuly-qualified filename to check
     * @return Returns true if the file is in the list
     */
    private boolean isInList (String sFile) {
        return list.contains(sFile);
    }

    private void removeFromList (String sFile) {
        list.remove(sFile);
    }

    private void removeOldestFile() {
        list.remove(list.size() -1);
    }
    /**
     * Sets a new maximum number of files that can be in the list. If the list is made smaller, then any existing items outside the new range will be removed.
     * @param iNewMaxFiles
     */
    public void setMaxFiles(int iNewMaxFiles) {
        list.ensureCapacity(iNewMaxFiles);
        if (iNewMaxFiles < maxFiles)
            list.subList(iNewMaxFiles, numberOfFiles()).clear();
        list.trimToSize();
        maxFiles = iNewMaxFiles;
        updateMenu();
    }

    /**
     * Maximum number of files in the list
     * @return Returns the number of files that can be in the list
     */
    public int getMaxFiles () {
        return maxFiles;
    }

    /**
     * Determines the number of files currently in the list
     * @return Returns the number of files in the list
     */
    public int numberOfFiles() {
        return list.size();
    }

    /**
     * @return Returns the <code>ArrayList</code> object used internaly to store the list.
     */
    public ArrayList getArrayList() {
        return list;
    }

    /**
     * Defines the <code>MruMenu</code> instance that is associated with this MRU list of files.
     * @param menu the <code>MruMenu</code> instance
     */
    public void setMruMenu (MruMenu menu) {
        this.menu = menu;
        updateMenu();
    }
    /**
     * Updates the associated menu whenever a change is made to the MRU list
     */
    private void updateMenu() {
        int iPosition = 0;

        if (menu == null) return;  // do nothing if no menu

        menu.removeAll();

        if (numberOfFiles()==0) return ; // nothing to add

        Iterator li = list.iterator();
        String sFile;
        JMenuItem[] items = new JMenuItem[numberOfFiles()];
        while (li.hasNext()) {
            sFile = (String)li.next();
            File file = new File(sFile);
            String sMenuText = "";
            if (showMnemonics && iPosition <mnemonics.length()) {
                String sMnemonic = mnemonics.substring(iPosition, iPosition +1 );
                sMenuText = sMnemonic + "  " + file.getPath();
            } else {
                sMenuText = file.getPath();
            }

            items[iPosition] = new JMenuItem(sMenuText);
            items[iPosition].setToolTipText(sFile);
            items[iPosition].addActionListener(menu);
            if (showMnemonics && iPosition <mnemonics.length()) {
                char c = mnemonics.charAt(iPosition);
                items[iPosition].setMnemonic(c);
            }

            menu.add(items[iPosition]);
            iPosition++; // = li.next();
        }
        JSeparator mruSeparator = new JSeparator();
        JMenuItem mruClear = new JMenuItem("Clear");
        mruClear.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent ae) {
                clearList();
            }
        });
        menu.add(mruSeparator);
        menu.add(mruClear);

    }
}







