/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.designcockpit.ide;

import ch.abacus.lib.ui.renderer.common.*;

import javax.swing.event.TableModelEvent;
import javax.swing.table.AbstractTableModel;
import javax.swing.*;
import java.util.Iterator;
import java.util.Set;
import java.util.TreeMap;


class MetaPropertyDescriptorCollection {
    MetaPropertyDescriptor[] theData;
    int iEntryCount = 0;
    TreeMap theEntries = new TreeMap();

    MetaPropertyDescriptor get (int iRow) {
        if (iRow < iEntryCount)
            return theData[iRow];
        return null;
    }

    void add (MetaPropertyDescriptor thePropertyDetail) {
        String sKey = thePropertyDetail.getName();
        if (theEntries.get(sKey) == null) {
            theEntries.put(sKey, thePropertyDetail);
            iEntryCount++;
        }
    }

    void convert () {
        if (iEntryCount > 0) {
            Set theKeySet = theEntries.keySet();
            Iterator theIterator = theKeySet.iterator();
            theData = new MetaPropertyDescriptor[iEntryCount];
            int i = 0;
            while (theIterator.hasNext()) {
                String thePropertyName = (String) theIterator.next();
                theData[i++] = (MetaPropertyDescriptor) theEntries.get(thePropertyName);
            }
        }
    }

    int size () {
        return iEntryCount;
    }
}

/**
 * Created by IntelliJ IDEA.
 * User: michael
 * Date: Jun 27, 2003
 * Time: 6:32:11 PM
 * To change this template use Options | File Templates.
 */
public class PropertyTableModel extends AbstractTableModel {

    MetaObject theMetaObject = null;
    MetaClass theMetaClass = null;
    MetaPropertyDescriptorCollection theClassProperties = null;
    PropertyTable theTable = null;
    int iPropertyCount = 0;
    int[] propArrayIndex = null;
    private final SuperDesignCockpit theCockpit;

    public PropertyTableModel (SuperDesignCockpit theCockpit) {
        this.theCockpit = theCockpit;
    }

    public void setArrayIndex (int row, int value) {
        propArrayIndex[row] = value;
//      System.out.println("Changed the index of row " + row + " to " + value);
        // Tell the table that it has changed.
        TableModelEvent event = new TableModelEvent(this, row, row);
        theTable.tableChanged(event);
    }

    public int getRowCount () {
        if (theMetaObject != null)
            return iPropertyCount;
        return 0;
    }

    public int getColumnCount () {
        return 2;
    }

    int getArrayIndexFromString (String sIndex) {
        int iMatch1 = sIndex.lastIndexOf('[');
        if (iMatch1 != -1) {
            int iMatch2 = sIndex.lastIndexOf(']');
            if (iMatch2 != -1) {
                String sTest = sIndex.substring(iMatch1 + 1, iMatch2);
                if (sTest.trim().length() != 0)
                    return (new Integer(sTest)).intValue();
            }
        }
        return -1;
    }

    public int getArrayIndex (int row) {
        String sIndex = (String) getValueAt(row, 0);
        return getArrayIndexFromString(sIndex);
    }

    /**
     * isNameChangePermitted is used to determine whether a component can be renamed.
     * Component names need to be unique within a project, so a rename is not permitted if
     * another component is already using the name you request.
     * @param sNewName String - the name to change to
     * @return boolean - true indicates the name change is permitted; false indicates it is not.
     */
    private boolean isNameChangePermitted (String sNewName) {
        MetaObject mo; // reference to the metaobject with sNewName name. If name is not used, then it will be null.
        mo = theTable.theDesignCockpit.getDesignProject().findObject(sNewName);

        return (mo == null);
    }

    public void setValueAt (Object theNewValue, int row, int column) {
        if (propertyValueChangeDisallowed(theNewValue, row))
            return;

        MetaObject theMetaObject = theTable.theDesignCockpit.getDesignProject().getLastObjectSelected();
        MetaPropertyDescriptor theProperty = theClassProperties.get(row);
        LanguageManager lm = theCockpit.theLanguageManager;

        if (theProperty.getName().equalsIgnoreCase("Name")) {
            // We need to verify that the name is not already used. If not, we have to disqualify the change...
            String sNewName = (String) theNewValue;
            if (isNameChangePermitted(sNewName) == false) {
                MetaObject metaObject = theTable.theDesignCockpit.getDesignProject().findObject(sNewName);
                if ( !metaObject.getName().equalsIgnoreCase(theMetaObject.getName()) ) {
                    MetaObject metaObjectParent = metaObject.theParentObject;
                    String sMessage;
                    if (metaObjectParent != null) {
                        sMessage = lm.expandMessage("DuplcateObjectNames.hasParent.label", "That name is already used by another object, which is a child of \"{0}\"", metaObject.theParentObject.getName());
                    } else {
                        sMessage = lm.getMessage("DuplcateObjectNames.noParent.label", "That name is already used by another object, the project's first object");
                    }
                    String sTitle = lm.getMessage("DuplcateObjectNames.label", "Duplicate Object Name");
                    JOptionPane.showMessageDialog(this.theTable.theDesignCockpit, sMessage, sTitle, JOptionPane.ERROR_MESSAGE);
                    return;
                }
            }
        }

        MetaConstantGroup theConstantTable = theMetaObject.theDesignProject.findConstantGroup(theProperty.getIncludeTypeConstants());
        MetaConstantGroupCollection theConstantTableGroup = null;
        if (theConstantTable != null) {
            if (theNewValue instanceof String) {
                String sMatch = theConstantTable.get((String) theNewValue);
                if (sMatch != null)
                    theNewValue = sMatch;
            }
        } else {
            theConstantTableGroup = theMetaObject.theDesignProject.findConstantGroupCollection(theProperty.getIncludeTypeConstants());
            if (theConstantTableGroup != null) {
                if (theNewValue instanceof String) {
                    String sMatch = theConstantTableGroup.get((String) theNewValue);
                    if (sMatch != null)
                        theNewValue = sMatch;
                }
            }
        }

        if (column == 0) {
        } else {
            String sPropertyName = theProperty.getName();
            theMetaObject.theDesignProject.setProjectChangedState(true);
            if (theMetaObject.isFrame() || theMetaObject.isAbalet()) {
                if (sPropertyName.startsWith("Location"))
                    theNewValue = new String("0");
            }
            // indexed values are handled in the helper.
            MetaPropertyValueEx pvOldValue = theMetaObject.getPropertyValue(sPropertyName, 0);
            String sOldValue = null;
            if (pvOldValue != null)
                sOldValue = pvOldValue.getStringValue();
            String sNewReverseLookup = null;
            String sOldReverseLookup = null;
            if (theConstantTable != null) {
                sNewReverseLookup = theConstantTable.getKey((String) theNewValue);
                sOldReverseLookup = theConstantTable.getKey(sOldValue);
            } else if (theConstantTableGroup != null) {
                sNewReverseLookup = theConstantTableGroup.getKey((String) theNewValue);
                sOldReverseLookup = theConstantTableGroup.getKey(sOldValue);
            } else if (theNewValue.equals(sOldValue) == false) {
                propertyEdited(theMetaObject, sPropertyName, theProperty.getOrder(), theNewValue);
                return;
            }
            if ((theNewValue.equals(sOldValue) == false) &&
                    (sNewReverseLookup != null) &&
                    (sNewReverseLookup.equals(sOldReverseLookup) == false)) {
                propertyEdited(theMetaObject, sPropertyName, theProperty.getOrder(), theNewValue);
            }
        }
    }

    /**
     * Determines whether a property change should be disallowed. <br>
     * If the property is an int and has no constants connected, then the value is checked to ensure
     * that it is an int. If not, the change is aboorted and the value reverts to its prior value<p>
     * Note: this could be extended to check for valid constants, etc.
     * @param theNewValue - not currently used
     * @param iRow - the row of the property in the property table. Thsi is used to retrieve internal info about the property
     * @return true, if the property change should be disallowed; false to accept the property change
     */
    private boolean propertyValueChangeDisallowed(Object theNewValue, int iRow) {
        //MetaObject theMetaObject = theTable.theDesignCockpit.getDesignProject().getLastObjectSelected();
        MetaPropertyDescriptor theProperty = theClassProperties.get(iRow);
        if (theProperty.getClassName().equalsIgnoreCase("int") && theProperty.getIncludeTypeConstants() == null) {
            if (theNewValue.getClass() == String.class) {
                try {
                    Integer.parseInt((String)theNewValue);
                } catch (NumberFormatException e) {
                    return true; // not an int, so disallow the change!
                }
            }
        }
        return false;
    }

    public MetaPropertyDescriptor getPropertyAt (int row) {
        MetaPropertyDescriptor theProperty = theClassProperties.get(row);
        return theProperty;
    }


    public Object getValueAt (int row, int column) {
        MetaPropertyDescriptor theProperty = theClassProperties.get(row);
        if (theProperty == null)
            return null;
        String sPropertyName = theProperty.getName();
        if (column == 0) {
            if (theProperty.getName().equals("TableName"))
                return "TableName = [Right Click]";
            if (theProperty.bAllowMultipleValues == false)
                return theProperty.getName();
            if ((theProperty.getHelperClass() != null) && (theProperty.getHelperClass().length() > 0))
                return (theProperty.getName() + " = [Right Click]");
            return (theProperty.getName());
        } else {
            if (theMetaObject.isFrame() || theMetaObject.isAbalet()) {
                if (sPropertyName.startsWith("Location"))
                    return new String("0");
            }
            int iDotPos = sPropertyName.indexOf(".");
            String sKey = sPropertyName;
            if (iDotPos != -1)
                sKey = sPropertyName.substring(0, iDotPos);
            String sType = theProperty.getClassName();
            MetaProperty theProp = theMetaObject.getPropertyMetadata(sKey);
            if (theProp == null) {
                if ((sType.endsWith("oolean") == false) && (theProperty.getDefaultValue() != null)) {
                    // translate constants
                    String sDefault = theProperty.getDefaultValue();
                    Object oMatch = theProperty.getExpandedValue(theMetaObject.theDesignProject, sDefault);
                    return oMatch;

                } else
                    return null;
            }
            int iIndex = 0;
            MetaPropertyValueEx pvValue = theMetaObject.getPropertyValue(sPropertyName, iIndex);
            //  Check if there is a constant group for the property class.
            MetaConstantGroup theConstantTable = theMetaObject.theDesignProject.findConstantGroup(theProperty.getIncludeTypeConstants());
            if (theConstantTable != null) {
                if (pvValue != null) {
                    String sConstantValue = pvValue.getLocalString();
                    String sMatch = theConstantTable.getKey(sConstantValue);
                    if (sMatch != null) {
                        if (sType.endsWith("oolean"))
                            return new Boolean(sMatch);
                        return sMatch;
                    }
                    return sConstantValue;
                }
            } else {
                MetaConstantGroupCollection theConstantTableCollection = theMetaObject.theDesignProject.findConstantGroupCollection(theProperty.getIncludeTypeConstants());
                if (theConstantTableCollection != null) {
                    if (pvValue != null) {
                        String sConstantValue = pvValue.getLocalString();
                        String sMatch = theConstantTableCollection.getKey(sConstantValue);
                        if (sMatch != null) {
                            if (sType.endsWith("oolean"))
                                return new Boolean(sMatch);
                            return sMatch;
                        }
                        return sConstantValue;
                    }
                }

            }
            //  If there is maybe we need to translate.
            if (pvValue == null) {
                Object oValue = theProperty.getDefaultValue();
                if (oValue == null)
                    oValue = theProperty.getValueChoice(0);
                pvValue = new MetaPropertyValueEx(sType, oValue, true, theMetaObject.theDesignProject, theProperty.bAllowMultipleLanguages);
            } else {
                Object oValue = pvValue.getNativeValue();
                return oValue;
            }
            String sRetVal = pvValue.getStringValue();
            if (sType.endsWith("oolean"))
                return new Boolean(sRetVal);
            else
                return sRetVal;
        }
    }

    public Class getColumnClass (int iColumn) {
        String theClassName = "";
        return theClassName.getClass();
    }

    public Class getCellClass (int iRow, int iColumn) {
        String sTextString = "";
        Class cls = sTextString.getClass();
        MetaPropertyDescriptor theProperty = theClassProperties.get(iRow);
        if (iColumn != 0) {  // Column 0 is always string.
            String sClass = theProperty.getClassName().trim();
            if ((sClass.endsWith("Boolean")) || (sClass.equals("boolean"))) {
                Object obj = new Boolean(true);
                return obj.getClass();
            }
            // Left justify all cells.
//          else if ((sClass.equals("java.lang.Integer")) || (sClass.equals("int"))) {
//              Object obj = new Integer( 0 );
//              return obj.getClass();
//          }
        }
        return cls;
    }

    public boolean isCellEditable (int row, int column) {
        if (column == 0) {
//          MetaPropertyDescriptor theProperty = getPropertyAt(row);
//          if (theProperty == null)
//            Object oName = getValueAt(row, column);
//            if (oName instanceof String) {
//                if (((String)oName).indexOf("F4")==-1)
//                    return false;
//                else
//                    return true;
//            }
            return false;
//          MetaProperty thePropertyValue =
//                theMetaObject.getPropertyMetadata(theProperty.getName());
//          if (thePropertyValue==null)
//              return false;
//          return theProperty.bAllowMultipleValues;
        } else {
            MetaPropertyDescriptor theProperty = getPropertyAt(row);
            return (theProperty.isReadOnly() == false);
        }
    }

    public String getColumnName (int iColumn) {
        if (iColumn == 0)
            return theCockpit.theLanguageManager.getMessage("PropertyInspector.PropertyName.label", "Property");
        else
            return theCockpit.theLanguageManager.getMessage("PropertyInspector.PropertyValue.label", "Value");
    }

    protected MetaPropertyDescriptorCollection formatClassTable (MetaPropertyDescriptorCollection theClassProperties, MetaPropertyDescriptor theProperty, String thePrefix) {
        if (thePrefix != null) {
            int iDotPos = theProperty.getName().indexOf(".");
            if (iDotPos == -1)  // not already processed (this can get called many times)
                theProperty.setName(thePrefix + "." + theProperty.getName());
        }
        if (theProperty.bVisible) {
            theClassProperties.add(theProperty);
        }
        if (theProperty.theFirstSubproperty != null) {
            MetaPropertyDescriptor theSubprop = theProperty.theFirstSubproperty;
            while (theSubprop != null) {
                theClassProperties = formatClassTable(theClassProperties, theSubprop, theProperty.getName());
                theSubprop = theSubprop.getNextProperty();
            }
        }
        return theClassProperties;
    }

    protected void Load (PropertyTable objTable) {
        theTable = objTable;
        theMetaObject = theTable.theDesignCockpit.getDesignProject().getLastObjectSelected();
        if (theMetaObject == null)
            return;
        theMetaClass = theMetaObject.theClass;
        iPropertyCount = theMetaClass.getPropertyCount(true);
        theClassProperties = new MetaPropertyDescriptorCollection();
        while (theMetaClass != null) {
            MetaPropertyDescriptor theProperty = theMetaClass.getMetadata().theFirstProperty;
            while (theProperty != null) {
                theClassProperties = formatClassTable(theClassProperties, theProperty, null);
                theProperty = theProperty.getNextProperty();
            }
            theMetaClass = theMetaClass.getParentClass();
        }

        theClassProperties.convert();
        iPropertyCount = theClassProperties.size();

        propArrayIndex = new int[iPropertyCount];
        for (int i = 0; i < iPropertyCount; i++) propArrayIndex[i] = 0;
    }

    protected void clear () {
        iPropertyCount = 0;
        theClassProperties = null;
    }

    public void propertyEdited (MetaObject theMetaObject, String sPropertyName, int iOrder, Object theNewValue) {
        theTable.theDesignCockpit.getDesignProject().theUndoRedoController.createChangePropertyEvent(theMetaObject, sPropertyName, 0, 0);
        theMetaObject.setPropertyValue(sPropertyName, 0, iOrder, theNewValue, false);
        theMetaObject.refresh();
        if (theMetaObject.theClass.getMetadata().containerType.getContainerType() != MetaContainerType.CONTAINER_FRAME) {
            if (theMetaObject.theVisualObject != null)
                theMetaObject.theVisualObject.repaint();
        } else {
            if (theMetaObject.iMode == AbaMetaDataUser.DESIGN_MODE) {
                theTable.theDesignCockpit.getDesignProject().getMetaDataUser().propertyEditedRepaint(theMetaObject);
            }
        }
    }


}
