/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.lib.ui.renderer.common;

import com.sun.image.codec.jpeg.JPEGCodec;
import com.sun.image.codec.jpeg.JPEGEncodeParam;
import com.sun.image.codec.jpeg.JPEGImageEncoder;

import javax.imageio.ImageIO;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;


public class JOImageField extends JComponent
{
    private boolean editable;
    Image workImage;
    String lastSelectedPath = null;

    private Image image;

    static String defaultImageString = "http://www.java.com/im/logo_footer.gif";

    // The URL of the image.
    public String ImageLocation = "<Existing Image>";

    private boolean debug = true;

    // Width and height of the component
    private int width, height;

    public boolean isExplicitSize()
    {
        return explicitSize;
    }

    public void setExplicitSize(boolean explicitSize)
    {
        this.explicitSize = explicitSize;
    }

    /**
     * Determines if it will be sized automatically.
     */
    private boolean explicitSize = false;
    private int explicitWidth = 0, explicitHeight = 0;

    private MediaTracker tracker;
    private static int lastTrackerID = 0;
    private int currentTrackerID;
    private boolean doneLoading = false;

    private Container parentContainer;


    /**
     * Create an JOImageField with the default image.
     */
    public JOImageField()
    {
        this(defaultImageString);

        //byte[] buf = getBytes();
        //setBytes(buf);
        //paint(getGraphics());
    }

    /**
     * Create an JOImageField using the image at URL
     * specified by the string.
     *
     * @param imageURLString A String specifying the
     *                       URL of the image.
     */
    public JOImageField(String imageURLString)
    {
        this(makeURL(imageURLString));
    }

    /**
     * Create an JOImageField using the image at URL
     * specified.
     *
     * @param imageURL The URL of the image.
     */
    public JOImageField(URL imageURL)
    {
        this(loadImage(imageURL));
        ImageLocation = imageURL.toExternalForm();
    }

    public JOImageField(byte[] imageBytes)
    {
        this(loadImage(imageBytes));
    }

    public JOImageField(InputStream in) throws IOException
    {
        this(loadImage(in));
    }

    /**
     * Create an JOImageField using the image in the file
     * in the specified directory.
     *
     * @param imageDirectory Directory containing image
     * @param file           Filename of image
     */
    public JOImageField(URL imageDirectory, String file)
    {
        this(makeURL(imageDirectory, file));
        ImageLocation = file;
    }

    public void selectImageFile()
    {
        String filename = "." + File.separator;
        if (lastSelectedPath != null)
            filename = lastSelectedPath;
        JFileChooser fc = new JFileChooser(new File(filename));
        int returnVal = fc.showOpenDialog(null);
        if (returnVal != JFileChooser.APPROVE_OPTION)
            return;
        File selFile = fc.getSelectedFile();
        try
        {
            lastSelectedPath = selFile.toURL().toExternalForm();
            setImageLocation(lastSelectedPath);
        }
        catch (MalformedURLException e)
        {
            e.printStackTrace();  //To change body of catch statement use File | Settings | File Templates.
        }
    }

    class SelectActionListener implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            selectImageFile();
            //paint(getGraphics());
        }
    }

    class SelectURLActionListener implements ActionListener
    {
        public void actionPerformed(ActionEvent e)
        {
            String retstr = JOptionPane.showInputDialog("Edit", getImageLocation());
            if (retstr != null)
                setImageLocation(retstr);
        }
    }

    class ImageMouseListener extends MouseAdapter
    {
        public void mouseClicked(MouseEvent evt)
        {
            evt.getComponent().requestFocus();

            JMenuItem item;
            JPopupMenu menu = new JPopupMenu();
            if (isEditable())
            {
                item = new JMenuItem("Select");
                item.addActionListener(new SelectActionListener());
                menu.add(item);

                item = new JMenuItem("Edit");
                item.addActionListener(new SelectURLActionListener());
                menu.add(item);

                item = new JMenuItem("Cut");
                //item.addActionListener(actionListener);
                menu.add(item);
            }

            item = new JMenuItem("Copy");
            //item.addActionListener(actionListener);
            menu.add(item);

            if (isEditable())
            {
                item = new JMenuItem("Paste");
                //item.addActionListener(actionListener);
                menu.add(item);
            }

            menu.show(evt.getComponent(), evt.getX(), evt.getY());
        }
    }

    class ImageFocusListener extends FocusAdapter
    {
        public void focusGained(FocusEvent e)
        {
            paint(getGraphics());
        }

        public void focusLost(FocusEvent e)
        {
            paint(getGraphics());
        }
    }


    /**
     * Create an JOImageField using the image specified.
     *
     * @param img The image
     */
    public JOImageField(Image img)
    {
        setImage(img);
        addMouseListener(new ImageMouseListener());
        addFocusListener(new ImageFocusListener());
    }

    public void waitForImage(boolean doLayout)
    {
        if (!doneLoading)
        {
            debug("[waitForImage] - Resizing and waiting for "
                    + ImageLocation);
            try
            {
                tracker.waitForID(currentTrackerID);
            }
            catch (InterruptedException ie)
            {
            }
            catch (Exception e)
            {
                System.out.println("Error loading "
                        + ImageLocation + ": "
                        + e.getMessage());
                e.printStackTrace();
            }
            if (tracker.isErrorID(0))
                new Throwable("Error loading image "
                        + ImageLocation).printStackTrace();
            doneLoading = true;
            if (explicitWidth != 0)
                width = explicitWidth;
            else
                width = image.getWidth(this) + 2 * getInsets().left;
            if (explicitHeight != 0)
                height = explicitHeight;
            else
                height = image.getHeight(this) + 2 * getInsets().top;
            setSize(width, height);
            debug("[waitForImage] - " + ImageLocation + " is "
                    + width + "x" + height + ".");

            // If no parent, you are OK, since it will have
            // been resized before being added. But if
            // parent exists, you have already been added,
            // and the change in size requires re-layout.
            if (((parentContainer = getParent()) != null)
                    && doLayout)
            {
                setBackground(parentContainer.getBackground());
                parentContainer.doLayout();
            }
        }
    }

    public void setSize(int width, int height)
    {
        if (!doneLoading)
        {
            explicitSize = true;
            if (width > 0)
                explicitWidth = width;
            if (height > 0)
                explicitHeight = height;
        }
        super.setSize(width, height);
    }

    public void setBounds(int x, int y, int width, int height)
    {
        if (!doneLoading)
        {
            explicitSize = true;
            if (width > 0)
                explicitWidth = width;
            if (height > 0)
                explicitHeight = height;
        }
        super.setBounds(x, y, width, height);
    }

    public void centerAt(int x, int y)
    {
        debug("Placing center of " + ImageLocation + " at ("
                + x + "," + y + ")");
        setLocation(x - width / 2, y - height / 2);
    }

    public synchronized boolean contains(int x, int y)
    {
        return ((x >= 0) && (x <= width)
                && (y >= 0) && (y <= height));
    }

    // This method returns true if the specified image has transparent pixels
    public static boolean hasAlpha(Image image)
    {
        // If buffered image, the color model is readily available
        if (image instanceof BufferedImage)
        {
            BufferedImage bimage = (BufferedImage) image;
            return bimage.getColorModel().hasAlpha();
        }

        // Use a pixel grabber to retrieve the image's color model;
        // grabbing a single pixel is usually sufficient
        PixelGrabber pg = new PixelGrabber(image, 0, 0, 1, 1, false);
        try
        {
            pg.grabPixels();
        }
        catch (InterruptedException e)
        {
        }

        // Get the image's color model
        ColorModel cm = pg.getColorModel();
        return cm.hasAlpha();
    }

    // This method returns a buffered image with the contents of an image
    public static BufferedImage toBufferedImage(Image image)
    {
        if (image instanceof BufferedImage)
        {
            return (BufferedImage) image;
        }

        // This code ensures that all the pixels in the image are loaded
        image = new ImageIcon(image).getImage();

        // Determine if the image has transparent pixels; for this method's
        // implementation, see e661 Determining If an Image Has Transparent Pixels
        boolean hasAlpha = hasAlpha(image);

        // Create a buffered image with a format that's compatible with the screen
        BufferedImage bimage = null;
        GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
        try
        {
            // Determine the type of transparency of the new buffered image
            int transparency = Transparency.OPAQUE;
            if (hasAlpha)
            {
                transparency = Transparency.BITMASK;
            }

            // Create the buffered image
            GraphicsDevice gs = ge.getDefaultScreenDevice();
            GraphicsConfiguration gc = gs.getDefaultConfiguration();
            bimage = gc.createCompatibleImage(image.getWidth(null), image.getHeight(null), transparency);
        }
        catch (HeadlessException e)
        {
            // The system does not have a screen
        }

        if (bimage == null)
        {
            // Create a buffered image using the default color model
            int type = BufferedImage.TYPE_INT_RGB;
            if (hasAlpha)
            {
                type = BufferedImage.TYPE_INT_ARGB;
            }
            bimage = new BufferedImage(image.getWidth(null), image.getHeight(null), type);
        }

        // Copy image to buffered image
        Graphics g = bimage.createGraphics();

        // Paint the image onto the buffered image
        g.drawImage(image, 0, 0, null);
        g.dispose();

        return bimage;
    }

    public void paint(Graphics g)
    {
        if (!doneLoading)
            waitForImage(true);
        else
        {
            if (hasFocus())
            {
                Kernel kernel = new Kernel(3, 3, new float[]{-1, -1, -1, -1, 9, -1, -1, -1, -1});
                BufferedImageOp op = new ConvolveOp(kernel);
                BufferedImage bufferedImage = op.filter(toBufferedImage(image), null);
                workImage = Toolkit.getDefaultToolkit().createImage(bufferedImage.getSource());
            }
            else
                workImage = image;

            if (explicitSize)
                g.drawImage(workImage, getInsets().left, getInsets().top, width - (getInsets().left * 2), height - (getInsets().top * 2), this);
            else
                g.drawImage(workImage, getInsets().left, getInsets().top, this);
        }
    }

    public Dimension getPreferredSize()
    {
        if (!doneLoading)
            waitForImage(false);
        return (super.getPreferredSize());
    }

    public Dimension getMinimumSize()
    {
        if (!doneLoading)
            waitForImage(false);
        return (super.getMinimumSize());
    }


    /**
     * Draws a rectangle with the specified OUTSIDE
     * left, top, width, and height.
     * Used to draw the border.
     */
    protected void drawRectOLD(Graphics g,
                               int left, int top,
                               int width, int height,
                               int lineThickness,
                               Color rectangleColor)
    {
        g.setColor(rectangleColor);
        for (int i = 0; i < lineThickness; i++)
        {
            g.drawRect(left, top, width, height);
            if (i < lineThickness - 1)
            {  // Skip last iteration
                left = left + 1;
                top = top + 1;
                width = width - 2;
                height = height - 2;
            }
        }
    }

    /**
     * Calls System.out.println if the debug variable
     * is true; does nothing otherwise.
     *
     * @param message The String to be printed.
     */
    protected void debug(String message)
    {
        if (debug)
            System.out.println(message);
    }

    /**
     * Creates the URL with some error checking.
     */
    private static URL makeURL(String s)
    {
        URL u = null;
        try
        {
            if (!s.toLowerCase().startsWith("http:") &&
                    !s.toLowerCase().startsWith("file:") &&
                    !s.toLowerCase().startsWith("ftp:"))
                s = "file:/" + s;

            u = new URL(s);
        }
        catch (MalformedURLException mue)
        {
            System.out.println("Bad URL " + s + ": " + mue);
            mue.printStackTrace();
        }
        return (u);
    }

    /**
     * Creates the URL with some error checking.
     */
    private static URL makeURL(URL directory,
                               String file)
    {
        URL u = null;
        try
        {
            u = new URL(directory, file);
        }
        catch (MalformedURLException mue)
        {
            System.out.println("Bad URL " +
                    directory.toExternalForm() +
                    ", " + file + ": " + mue);
            mue.printStackTrace();
        }
        return (u);
    }

    /**
     * Loads an image from a url.
     * <p/>
     * The data must be in some image format, such as GIF or JPEG,
     * that is supported by this toolkit.
     */
    public static Image loadImage(URL url)
    {
        return (Toolkit.getDefaultToolkit().getImage(url));
    }

    /**
     * Loads an image from a url string.
     * <p/>
     * The data must be in some image format, such as GIF or JPEG,
     * that is supported by this toolkit.
     */
    public static Image loadImage(String surl)
    {
        return loadImage(makeURL(surl));
    }

    /**
     * Loads an image from array of bytes.
     * <p/>
     * The data must be in some image format, such as GIF or JPEG,
     * that is supported by this toolkit.
     *
     * @param imageBytes an array of bytes, representing
     *                   image data in a supported image format.
     * @return an image.
     */
    public static Image loadImage(byte[] imageBytes)
    {
        return Toolkit.getDefaultToolkit().createImage(imageBytes); //new ImageIcon(imageBytes).getImage();
    }

    /**
     * Loads an image from a stream.
     * <p/>
     * The data must be in some image format, such as GIF or JPEG,
     * that is supported by this toolkit.
     */
    public static Image loadImage(InputStream in) throws IOException
    {
        return ImageIO.read(in);
    }

    /**
     * The Image (ImageIcon, etc.) associated with the JOImageField.
     */
    public Image getImage()
    {
        return (image);
    }

    /**
     * Loads an image from array of bytes.
     * <p/>
     * The data must be in some image format, such as GIF or JPEG,
     * that is supported by this toolkit.
     *
     * @param imageBytes an array of bytes, representing
     *                   image data in a supported image format.
     */
    public void setBytes(byte[] imageBytes)
    {
        setImage(loadImage(imageBytes));
    }

    /**
     * Return array of bytes in JPEG format.
     *
     * @return
     */
    public byte[] getBytes()
    {
        BufferedImage bi = toBufferedImage(image);
        //BufferedOutputStream out = new BufferedOutputStream(new ByteArrayOutputStream());
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        JPEGImageEncoder encoder = JPEGCodec.createJPEGEncoder(out);
        JPEGEncodeParam param = encoder.getDefaultJPEGEncodeParam(bi);
        //param.setQuality(100.0f, false);
        encoder.setJPEGEncodeParam(param);
        try
        {
            encoder.encode(bi);
            out.close();
        }
        catch (IOException e)
        {
            e.printStackTrace();  //To change body of catch statement use File | Settings | File Templates.
        }

        return out.toByteArray();
    }

    /**
     * Gets the width
     */
    public int getWidth()
    {
        return (width);
    }

    /**
     * Gets the height
     */
    public int getHeight()
    {
        return (height);
    }

    //----------------------------------------------------
    /**
     * Has the JOImageField been given an explicit size?
     * This is used to decide if the image should be
     * stretched or not. This will be true if you
     * call resize or reshape on the JOImageField before
     * adding it to a Container. It will be false
     * otherwise.
     */
    protected boolean hasExplicitSize()
    {
        return (explicitSize);
    }

    /**
     * Returns the string representing the URL that
     * will be used if none is supplied in the
     * constructor.
     */
    public static String getDefaultImageString()
    {
        return (defaultImageString);
    }

    public static void setDefaultImageString(String file)
    {
        defaultImageString = file;
    }

    /**
     * Returns the string representing the URL
     * of image.
     */
    public String getImageLocation()
    {
        return (ImageLocation);
    }

    public void setImageLocation(String file)
    {
        ImageLocation = makeURL(file).toExternalForm();
        setImage(loadImage(ImageLocation));
    }

    public void setImage(Image img)
    {
        image = img;
        tracker = new MediaTracker(this);
        currentTrackerID = lastTrackerID++;
        tracker.addImage(image, currentTrackerID);
    }

    public boolean isDebugging()
    {
        return (debug);
    }

    public void setIsDebugging(boolean debug)
    {
        this.debug = debug;
    }

    /**
     * Returns the boolean indicating whether this
     * <code>TextComponent</code> is editable or not.
     *
     * @return the boolean value
     * @see #setEditable
     */
    public boolean isEditable()
    {
        return editable;
    }

    /**
     * Sets the specified boolean to indicate whether or not this
     * <code>TextComponent</code> should be editable.
     * A PropertyChange event ("editable") is fired when the
     * state is changed.
     *
     * @param b the boolean to be set
     * @see #isEditable
     */
    public void setEditable(boolean b)
    {
        if (b != editable)
        {
            boolean oldVal = editable;
            editable = b;
            if (editable)
            {
                setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
            }
            else
            {
                setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
            }
            enableInputMethods(editable);
            firePropertyChange("editable", Boolean.valueOf(oldVal), Boolean.valueOf(editable));
            repaint();
        }
    }

}

