/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.lib.ui.renderer.common;

import java.awt.*;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;

/**
 * <p>Title: uifactory</p>
 * <p>Description: </p>
 * <p>Copyright: Copyright (c) 2001</p>
 * <p>Company: Abacus Research</p>
 * @author Michael Gouker (Cagey Logic)
 * @version 1.0
 */

public class MetaConstantGroup {
    public MetaConstantGroup theNextSibling;
    public MetaConstantGroup thePrevSibling;
    public String sClassName;
    public MetadataDispenser theMetadataDispenser;
    HashMap theConstants;
    HashMap theValues;
    ArrayList theSeqConstants = new ArrayList(0);

    public MetaConstantGroup(String sClassName, MetadataDispenser theMetadataDispenser) {
        theConstants = new HashMap(1);
        theValues = new HashMap(1);
        this.sClassName = sClassName;
        this.theMetadataDispenser = theMetadataDispenser;
    }

    /**
     *
     * The sequential list is used for filling out a list of choices like in a combobox.
     * @return the list of choices.
     */
    public ArrayList getConstantList() {
        return theSeqConstants;
    }

    // Allows multiple values for same key. First registered will be default.
    public void set(String sName, String sValue) {
        Object oStoredKey = get(sName);
        if (oStoredKey == null)
            theConstants.put(sName, sValue);
        theValues.put(sValue, sName);
        if (theSeqConstants.contains(sName) == false) {
            theSeqConstants.add(sName);
        }
    }

    public String get(String sName) {
        if (sName == null)
            return null;
        return (String) theConstants.get(sName);
    }

    public String getKey(String sValue) {
        if (sValue == null)
            return null;
        return (String) theValues.get(sValue);
    }

    /**
     * Removes all constant pairs from the Group.<p>
     * This is used when the list of constants needs to be rebuilt in, for example, the Metadata Editor.
     */
    public void clearConstants() {
        theConstants.clear();
        theValues.clear();
        theSeqConstants.clear();
    }

    /**
     * Removes a MetaConstantGroup from all Group-Collections
     */
    public void removeGroupFromCollections() {
        MetaConstantGroupCollection collection = theMetadataDispenser.getFirstMetaConstantGroupCollection();
        while (collection != null) {
            collection.removeGroup(this);
            collection = collection.theNextSibling;
        }
    }


    static public Class resolveClass(Object object, MetadataDispenser theMetadataDispenser, ClassLoader theLoader) throws HammerException {
        if (object == null)
            return null;
        if (object instanceof String) {
            String sDescription = (String) object;
            sDescription = sDescription.trim();
            int iBean = sDescription.indexOf("bean:");
            int iConstruct = sDescription.indexOf("construct:");
            int iInvoke = sDescription.indexOf("invoke:");

            if ((iBean != 0) && (iConstruct != 0) && (iInvoke != 0)) { // must be at zeroth position.
                // check if it is a font or a color
                Font testFont = MetaConstantGroup.getFontFromString(sDescription);
                if (testFont != null)
                    return java.awt.Font.class;
                Color testColor = MetaConstantGroup.getColorFromString(sDescription);
                if (testColor != null)
                    return java.awt.Color.class;
                return String.class;
            } else if (iBean > -1) {
                sDescription = sDescription.substring(iBean + 5);
                Object obj = getObjectFromBeanDescription(sDescription, theMetadataDispenser, theLoader);
                return obj.getClass();
            } else if (iInvoke > -1) {
                sDescription = sDescription.substring(iInvoke + 7);
                return getClassFromInvokeDescription(sDescription, theMetadataDispenser, theLoader);
            } else {
                sDescription = sDescription.substring(iConstruct + 10);
                Object obj = getObjectFromConstructionDescription(sDescription, theMetadataDispenser, theLoader);
                return obj.getClass();
            }
        } else
            return object.getClass();
    }

    static public Object resolve(Object object, MetadataDispenser theMetadataDispenser, ClassLoader theLoader) throws HammerException {
        if (object == null)
            return object;
        if (object instanceof String) {
            String sDescription = (String) object;
            sDescription = sDescription.trim();
            //int iLeft = sDescription.indexOf("[");
            //boolean bColorTest = sDescription.startsWith("$") | sDescription.startsWith("0x");
//            if ((iLeft == -1) && (bColorTest == false))
//                return object;  // short cut to make it faster.
            int iBean = sDescription.indexOf("bean:");
            int iConstruct = sDescription.indexOf("construct:");
            int iInvoke = sDescription.indexOf("invoke:");

            if ((iBean != 0) && (iConstruct != 0) && (iInvoke != 0)) { // must be at zeroth position.
                // check if it is a font or a color
                Font testFont = MetaConstantGroup.getFontFromString(sDescription);
                if (testFont != null)
                    return testFont;
                Color testColor = MetaConstantGroup.getColorFromString(sDescription);
                if (testColor != null)
                    return testColor;
                return object;
            } else if (iBean > -1) {
                sDescription = sDescription.substring(iBean + 5);
                return getObjectFromBeanDescription(sDescription, theMetadataDispenser, theLoader);
            } else if (iInvoke > -1) {
                sDescription = sDescription.substring(iInvoke + 7);
                return getObjectFromInvokeDescription(sDescription, theMetadataDispenser, theLoader);
            } else {
                sDescription = sDescription.substring(iConstruct + 10);
                return getObjectFromConstructionDescription(sDescription, theMetadataDispenser, theLoader);
            }
        } else
            return object;
    }


    static public Method getMethod(Method[] methods, String sKey) {
        for (int i = 0; i < methods.length; i++) {
            if (sKey.equals(methods[i].getName())) {
                Class[] theParams = methods[i].getParameterTypes();
                if (theParams.length == 1) {
                    return methods[i];
                }
            }
        }
        return null;
    }

    /**
     * getObjectFromBeanDescription
     *
     *    This method returns an object from a bean string description.
     *    This is the format of a bean description:
     *
     *    <class>[<propname=value>{,}]
     *
     *
     * @param sDescription
     * @return
     */

    static public Object getObjectFromBeanDescription(String sDescription, MetadataDispenser theMetadataDispenser, ClassLoader theLoader) throws HammerException {
        Object obj = null;
        if ((sDescription != null) && (sDescription.trim().length() > 0)) {
            int lfbr = sDescription.indexOf("[");
            int rtbr = sDescription.indexOf("]");
            if ((lfbr == -1) || (rtbr == -1))
                return null;
            String sClass = sDescription.substring(0, lfbr);

            // now parse out the arguments. Args are like this.  x=val.
            // x is identifier name for bean property.
            // val is either a number or a quoted string.
            // args are separated by commas.
            ArrayList fncs = new ArrayList(0);
            ArrayList vals = new ArrayList(0);
            String sArgList = sDescription.substring(lfbr + 1, rtbr);
            int eq = sArgList.indexOf("=");
            while (eq != -1) {
                String sName = sArgList.substring(0, eq);
                sName = "set" + sName.substring(0, 0).toUpperCase() + sName.substring(1);
                fncs.add(sName);
                sArgList = sArgList.substring(eq + 1);
                sArgList = sArgList.trim();
                int iEnd = sArgList.indexOf(",");  // assume unquoted.
                String sValue = "";
                if (sArgList.startsWith("\"")) {
                    iEnd = sArgList.indexOf("\"");
                    if (iEnd == -1)
                        return null;
                    sValue = sArgList.substring(1, iEnd);  // skip quotes.
                    iEnd = sArgList.indexOf(",", iEnd);
                    iEnd++;
                } else {
                    if (iEnd == -1)
                        sValue = sArgList;
                    else
                        sValue = sArgList.substring(0, iEnd);
                    iEnd++;
                }
                vals.add(sValue);
                if (iEnd != -1)
                    sArgList = sArgList.substring(iEnd);
                eq = sArgList.indexOf("=");
            }
            // now create the object.
            try {
                Class cls = Class.forName(sClass, true, theLoader);
                obj = cls.newInstance();
                // now apply the properties.
                Method[] methods = cls.getMethods();
                int iPropCount = fncs.size();
                for (int i = 0; i < iPropCount; i++) {
                    Method m = getMethod(methods, (String) fncs.get(i));
                    Object[] param = new Object[1];
                    param[0] = MetaConstantGroup.resolve(vals.get(i), theMetadataDispenser, theLoader);
                    m.invoke(obj, param);
                }
                return obj;
            } catch (Exception e1) {
                throw new HammerException(HammerException.CANNOT_CREATE_CONSTANT, sDescription);
            }
        }
        return null;
    }

    /**
     * getObjectFromConstructionDescription
     *
     *    This method returns an object from a construction description.
     *    This is the format of a construction description:
     *
     *    <class>[<type=value>{,}]
     *
     *
     * @param sDescription
     * @return
     */

    static public Object getObjectFromConstructionDescription(String sDescription, MetadataDispenser theMetadataDispenser, ClassLoader theLoader) throws HammerException {
        Object obj = null;
//      javax.swing.border.EtchedBorder.
        if ((sDescription != null) && (sDescription.trim().length() > 0)) {
            int lfbr = sDescription.indexOf("[");
            int rtbr = sDescription.indexOf("]");
            if ((lfbr == -1) || (rtbr == -1))
                return null;
            String sClass = sDescription.substring(0, lfbr);

            // now parse out the arguments. Args are like this.  x=val.
            // x is identifier name for bean property.
            // val is either a number or a quoted string.
            // args are separated by commas.
            ArrayList types = new ArrayList(0);
            ArrayList vals = new ArrayList(0);
            String sArgList = sDescription.substring(lfbr + 1, rtbr);
            int eq = sArgList.indexOf("=");
            while (eq != -1) {
                String sType = sArgList.substring(0, eq);
                types.add(sType);
                sArgList = sArgList.substring(eq + 1);
                sArgList = sArgList.trim();
                int iEnd = sArgList.indexOf(",");  // assume unquoted.
                String sValue = "";
                if (sArgList.startsWith("\"")) {
                    iEnd = sArgList.indexOf("\"");
                    if (iEnd == -1)
                        return null;
                    sValue = sArgList.substring(1, iEnd);  // skip quotes.
                    iEnd = sArgList.indexOf(",", iEnd);
                    iEnd++;
                } else {
                    if (iEnd == -1)
                        sValue = sArgList;
                    else
                        sValue = sArgList.substring(0, iEnd);
                    iEnd++;
                }
                vals.add(sValue);
                if (iEnd != -1)
                    sArgList = sArgList.substring(iEnd);
                eq = sArgList.indexOf("=");
            }
            // now create the object.
            try {
                Class cls = Class.forName(sClass, true, theLoader);
                // now apply the properties.
                int iParamCount = types.size();
                Class[] theParamClasses = new Class[iParamCount];
                Object[] theParamObjects = new Object[iParamCount];
                for (int i = 0; i < iParamCount; i++) {
                    String sType = (String) types.get(i);
                    String sValue = (String) vals.get(i);
                    int iType = MetaPropertyValueEx.translateType(sType);
                    if (iType != MetaPropertyValueEx.TYPE_UNDEFINED) {
                        Object oValue = null;
                        if (iType == MetaPropertyValueEx.TYPE_INT) {
                            int iTestDot = sValue.lastIndexOf(".");
                            if (iTestDot != -1) {
                                String sClassName = sValue.substring(0, iTestDot);
                                String sField = sValue.substring(iTestDot + 1);
                                try {
                                    oValue = MetadataProvider.GetFieldValueAsObject(sClassName, sField);
                                } catch (Exception e1) {
                                    // this is for field values that are defined in the Java clases. If this fails, we want to continue searching
                                }
                            }
                        }
                        Class clsParam = MetaPropertyValueEx.getTypeAsClass(iType);
                        theParamClasses[i] = clsParam;
                        if (oValue != null)
                            theParamObjects[i] = oValue;
                        else
                            theParamObjects[i] = MetaPropertyValueEx.getObjectForValue(iType, sValue);
                    } else {
                        // Class can be a constant group.
                        MetaConstantGroup theGroup = theMetadataDispenser.findConstantGroup(sType);
                        if (theGroup != null) {
                            String sStoredValue = theGroup.get(sValue);
                            Object oStoredValue = MetaConstantGroup.resolve(sStoredValue, theMetadataDispenser, theLoader);
                            theParamClasses[i] = MetaConstantGroup.resolveClass(sStoredValue, theMetadataDispenser, theLoader);
                            theParamObjects[i] = oStoredValue;
                        }
                        MetaConstantGroupCollection theGroupCollection = theMetadataDispenser.findConstantGroupCollection(sType);
                        if (theGroupCollection != null) {
                            String sStoredValue = theGroupCollection.get(sValue);
                            Object oStoredValue = MetaConstantGroup.resolve(sStoredValue, theMetadataDispenser, theLoader);
                            theParamClasses[i] = MetaConstantGroup.resolveClass(sStoredValue, theMetadataDispenser, theLoader);
                            theParamObjects[i] = oStoredValue;
                        } else {    // Ok, no class found.  Pass as string and pray.
                            theParamClasses[i] = Class.forName("java.lang.String");
                            theParamObjects[i] = sValue;
                        }
                    }
                }
                Constructor theConstructor = cls.getConstructor(theParamClasses);
                obj = theConstructor.newInstance(theParamObjects);
                return obj;
            } catch (Exception e1) {
                throw new HammerException(HammerException.CANNOT_CREATE_CONSTANT, sDescription);
            }
        }
        return null;
    }

    private static Class getClassFromInvokeDescription(String sDescription, MetadataDispenser theMetadataDispenser, ClassLoader theLoader) throws HammerException {
        int iCastClassBeg = 0;
        int iCastClassEnd = sDescription.indexOf(',');

        String sCastClassName = sDescription.substring(iCastClassBeg, iCastClassEnd);
        Class clsCasted = null;
        try {
            if ((sCastClassName != null) && (sCastClassName.length() > 1))
                clsCasted = theLoader.loadClass(sCastClassName);
        } catch (ClassNotFoundException e1) {
            throw new HammerException(HammerException.CANNOT_CAST_CONSTANT, "Cannot cast constant of " + sDescription + " to unknown class " + sCastClassName);
        }
        return clsCasted;
    }

    private static Object getObjectFromInvokeDescription(String sDescription, MetadataDispenser theMetadataDispenser, ClassLoader theLoader) throws HammerException {
        String sClass = null;
        String sMethod = null;
        String sArgList = null;
        int iPosStart, iPosEnd; // position within sDescription
        // The return value will be casted to the correct class.
        // We do this for all invocations now, but there may actually not be a need to cast some in the future.
        // TODO: Make this more clever so that it can be optional.
        int iCastClassBeg = 0;
        int iCastClassEnd = sDescription.indexOf(',');

        String sCastClassName = sDescription.substring(iCastClassBeg, iCastClassEnd);
        Class clsCasted = null;
        try {
            if ((sCastClassName != null) && (sCastClassName.length() > 1))
                clsCasted = theLoader.loadClass(sCastClassName);
        } catch (ClassNotFoundException e1) {
            throw new HammerException(HammerException.CANNOT_CAST_CONSTANT, "Cannot cast constant of " + sDescription + " to unknown class " + sCastClassName);
        }

        iPosStart = iCastClassEnd + 1;
        iPosEnd = sDescription.indexOf(',', iCastClassEnd + 1);
        if (iPosEnd == -1) return null; // error condition

        sClass = sDescription.substring(iPosStart, iPosEnd);
        //sClass = "ch.abacus.lib.ui.plaf.AbacusLookAndFeel";
        iPosStart = ++iPosEnd;

        iPosEnd = sDescription.indexOf('[', iPosStart);
        if (iPosEnd == -1) return null; // error condition
        sMethod = sDescription.substring(iPosStart, iPosEnd);
        iPosStart = ++iPosEnd;
        iPosEnd = sDescription.indexOf(']', iPosStart);
        if (iPosEnd == -1) return null; // error condition
        sArgList = sDescription.substring(iPosStart, iPosEnd);

        Class theClass = null; // load it!
        try {
            theClass = theLoader.loadClass(sClass);
            // now parse out the arguments. Args are like this.  x=val.
            // x is identifier name for bean property.
            // val is either a number or a quoted string.
            // args are separated by commas.
            ArrayList types = new ArrayList(0);
            ArrayList vals = new ArrayList(0);
            int eq = sArgList.indexOf("=");
            while (eq != -1) {
                String sType = sArgList.substring(0, eq);
                types.add(sType);
                sArgList = sArgList.substring(eq + 1);
                sArgList = sArgList.trim();
                int iEnd = sArgList.indexOf(",");  // assume unquoted.
                String sValue = "";
                if (sArgList.startsWith("\"")) {
                    iEnd = sArgList.indexOf("\"");
                    if (iEnd == -1)
                        return null;
                    sValue = sArgList.substring(1, iEnd);  // skip quotes.
                    iEnd = sArgList.indexOf(",", iEnd);
                    iEnd++;
                } else {
                    if (iEnd == -1)
                        sValue = sArgList;
                    else
                        sValue = sArgList.substring(0, iEnd);
                    iEnd++;
                }
                vals.add(sValue);
                if (iEnd != -1)
                    sArgList = sArgList.substring(iEnd);
                eq = sArgList.indexOf("=");
            }

            Object[] objects = new Class[types.size()];
            Class[] classes = new Class[vals.size()];

            for (int i = 0; i < vals.size(); i++) {
                int iType = MetaPropertyValueEx.translateType((String) types.get(i));

                String sValue = (String) vals.get(i);
                objects[i] = resolve(sValue, theMetadataDispenser, theLoader);
                classes[i] = resolveClass(sValue, theMetadataDispenser, theLoader);
//                if (iType == MetaPropertyValueEx.TYPE_UNDEFINED)
//                    classes[i] = theLoader.loadClass((String) types.get(i));
//                else
//                    classes[i] = MetaPropertyValueEx.getTypeAsClass(iType);
            }

            Method m = null;
            m = theClass.getMethod(sMethod, classes);
            Object obj = m.invoke(null, objects);
            // cast object to expected class if it is the wrong type.
            // we do this so we get a specific class cast exception rather than
            // a invocation exception.
            if (clsCasted != null) {
                Object retval = obj;
                try {  // force cast.
                    retval = MetadataProvider.createBogusObject(clsCasted);
                    retval = obj;
                } catch (ClassCastException e1) {
                    throw new HammerException(HammerException.CANNOT_CAST_CONSTANT, "Cannot cast constant of " + sDescription + " to " + sCastClassName);
                }
                return retval;
            } else
                return obj;
        } catch (Exception e) {
            throw new HammerException(HammerException.CANNOT_CREATE_CONSTANT, sDescription);
        }
    }


    public static Font getFontFromString(String sFont) throws HammerException {
        if ((sFont != null) && (sFont.trim().length() != 0)) {
            int iPos1 = sFont.indexOf("family=");
            if (iPos1 == -1)
                return null;
            int iPos2 = sFont.indexOf(",name");
            if (iPos2 == -1)
                return null;
            int iPos3 = sFont.indexOf("style=");
            if (iPos3 == -1)
                return null;
            int iPos4 = sFont.indexOf("size=");
            if (iPos4 == -1)
                return null;
            int iLength = sFont.length();
            String sFamily = sFont.substring(iPos1 + 7, iPos2);
            String sFontName = sFont.substring(iPos2 + 6, iPos3 - 1);
            String sStyle = sFont.substring(iPos3 + 6, iPos4 - 1);
            String sSize = sFont.substring(iPos4 + 5, iLength - 1);
            int iStyleParam = Font.PLAIN;
            if (sStyle.indexOf("bold") != -1) {
                if (sStyle.indexOf("italic") != -1)
                    iStyleParam = Font.BOLD + Font.ITALIC;
                else
                    iStyleParam = Font.BOLD;
            } else if (sStyle.indexOf("italic") != -1)
                iStyleParam = Font.ITALIC;
            Integer iSize = null;
            try {
                iSize = new Integer(sSize);
            } catch (java.lang.NumberFormatException e1) {
                throw new HammerException(HammerException.CANNOT_CREATE_CONSTANT, "Bad number in " + sFont);
            }
            Font theReturnValue = null;
            if (iSize != null)
                theReturnValue = new Font(sFamily, iStyleParam, iSize.intValue());
            return theReturnValue;
        }
        return null;
    }

    public static Color getColorFromString(String sColor) {
        // May be a rgb declaraion: new color(r,g,b)
        // May be a toString value: java.awt.color(r=0, g=0, b=0)
        // May be something like white, blue, green...
        // May be a integer or a hex value.
        if (sColor == null)
            return null;
        if (sColor.trim().length() == 0)
            return null;
        if (sColor.startsWith("$")) {
            sColor = "0x" + sColor.substring(1);
        }
        if (sColor.startsWith("0x")) {
            int iRgb = Integer.valueOf(sColor.substring(2), 16).intValue();
            return new Color(iRgb);
        }
        if (sColor.indexOf("new") != -1) { // Just return it (assume they know what they are doing.
            int iPos1 = sColor.indexOf("(");
            if (iPos1 == -1) return null;
            sColor = sColor.substring(iPos1 + 1);
            int iPos2 = sColor.indexOf(",");
            if (iPos2 == -1) return null;
            String sRed = sColor.substring(0, iPos2);
            sColor = sColor.substring(iPos2 + 1);
            int iPos3 = sColor.indexOf(",");
            if (iPos3 == -1) return null;
            String sGreen = sColor.substring(0, iPos3);
            sColor = sColor.substring(iPos3 + 1);
            int iPos4 = sColor.indexOf(")");
            if (iPos4 == -1) return null;
            String sBlue = sColor.substring(0, iPos4);
            Integer iRed = new Integer(sRed.trim());
            Integer iGreen = new Integer(sGreen.trim());
            Integer iBlue = new Integer(sBlue.trim());
            return new Color(iRed.intValue(), iGreen.intValue(), iBlue.intValue());
        } else if (sColor.indexOf("java.awt.Color") != -1) {
            int iPos1 = sColor.indexOf("r=");
            int iPos2 = sColor.indexOf(",g=");
            int iPos3 = sColor.indexOf(",b=");
            int iLength = sColor.length();
            Integer iRed = new Integer(sColor.substring(iPos1 + 2, iPos2));
            Integer iGreen = new Integer(sColor.substring(iPos2 + 3, iPos3));
            Integer iBlue = new Integer(sColor.substring(iPos3 + 3, iLength - 1));
            return new Color(iRed.intValue(), iGreen.intValue(), iBlue.intValue());
        } else {
            Color clrTest = new Color(0, 0, 0);
            if (sColor != null) {
                sColor = sColor.trim();
                if (!sColor.equals("")) {
                    try {
                        return (Color) MetadataProvider.GetFieldValueAsObject(clrTest, sColor);
                    } catch (Exception e1) {
                        // this is for field values that are defined in the Java clases. If this fails, we want to continue searching
                    }
                }
            }
        }
        return null;
    }

    /**
     * Make escape sequences legal, by doubling up backslashes when they are not
     * escape sequences.  Also puts backslash before single quote or double quote
     * if not already present.  This is necessary for the compiler not to generate
     * bad compiler errors.
     *
     * @param sTestValue
     * @return
     */
    public static String preserveEscapeSequences(String sTestValue) {
        String sNewString = "";
        if (sTestValue == null)
            return null;
        int iBackslash = sTestValue.indexOf('\\');
        int iLength = sTestValue.length();
        String sEscapes = "tnrfb\'\"\\";
        int iLastMatch = 0;
        while (iBackslash != -1) {
            if (iBackslash + 1 < iLength) {
                char c = sTestValue.charAt(iBackslash + 1);
                int iFindEscape = sEscapes.indexOf(c);
                if (iFindEscape == -1) {
                    // test if it is a hex or an octal
                    boolean bFoundEscape = false;
                    if (iBackslash + 3 < iLength) {
                        // octal
                        char c2 = sTestValue.charAt(iBackslash + 2);
                        char c3 = sTestValue.charAt(iBackslash + 3);
                        if ((c >= '0') && (c <= '7') &&
                                (c2 >= '0') && (c2 <= '7') &&
                                (c3 >= '0') && (c3 <= '7')) {
                            iBackslash += 3;
                            bFoundEscape = true;
                        } else if (iBackslash + 5 < iLength) { // check for hex
                            char c4 = sTestValue.charAt(iBackslash + 4);
                            char c5 = sTestValue.charAt(iBackslash + 5);
                            if ((c == 'u') &&
                                    (((c2 >= '0') && (c2 <= '9')) || ((c2 >= 'a') && (c2 <= 'f')) || ((c2 >= 'A') && (c2 >= 'F'))) &&
                                    (((c3 >= '0') && (c3 <= '9')) || ((c3 >= 'a') && (c3 <= 'f')) || ((c3 >= 'A') && (c3 >= 'F'))) &&
                                    (((c4 >= '0') && (c4 <= '9')) || ((c4 >= 'a') && (c4 <= 'f')) || ((c4 >= 'A') && (c4 >= 'F'))) &&
                                    (((c5 >= '0') && (c5 <= '9')) || ((c5 >= 'a') && (c5 <= 'f')) || ((c5 >= 'A') && (c5 >= 'F')))) {
                                iBackslash += 5;
                                bFoundEscape = true;
                            }
                        }

                    }
                    if (bFoundEscape) {
                        sNewString = sNewString + sTestValue.substring(iLastMatch, iBackslash);
                    } else {
                        sNewString = sNewString + sTestValue.substring(iLastMatch, iBackslash);
                        sNewString += '\\';  // add a back slash.
                    }
                } else {
                    iBackslash++;
                    // an escape sequence - just add it to new string.
                    sNewString = sNewString + sTestValue.substring(iLastMatch, iBackslash);
                }
            } else { // a backslash at the end - just add an extra backslash and we are done
                sNewString = sNewString + sTestValue.substring(iLastMatch, iBackslash);
                sNewString += '\\';  // add a back slash.
            }
            iLastMatch = iBackslash;
            iBackslash = sTestValue.indexOf('\\', iBackslash + 1);
        }
        // add leftover characters.
        sNewString = sNewString + sTestValue.substring(iLastMatch);
        return sNewString;
    }

    public static String protectQuotes (String sText) {
        if (sText == null)
            return null;
        int iQuotePos = sText.indexOf('\"');
        if (iQuotePos == -1)
            return sText; // nothing to change

        int iLength = sText.length();
        StringBuffer buffer = new StringBuffer(iLength +16);

        for (int i = 0; i <sText.length(); i++) {
            char c = sText.charAt(i);
            if (c == '"')
                buffer.append('\\');
            buffer.append(c);
        }
        return buffer.toString();
    }

    public static void main(String[] args) {
        String a = MetaConstantGroup.preserveEscapeSequences(null);
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("");
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("abcdefg");
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("\\abcdefg");
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("abc\\defg");
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("abcdefg\\");        //*
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("\\\\abcdefg");      //*
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("abcd\\\\efg");      //*
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("abcdefg\\\\");      //*
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("\tabcdefg");
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("abc\tdefg");
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("abcdefg\t");
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("\'abcdefg");
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("abc\'defg");
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("abcdefg\'");
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("\333abcdefg");
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("abc\333defg");
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("abcdefg\333");
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("\uABACabcdefg");
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("abc\uABAC333defg");
        System.out.println(a);
        a = MetaConstantGroup.preserveEscapeSequences("abcdefg\uABAC");
        System.out.println(a);
    }

}