/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.lib.ui.renderer.common;

import java.io.BufferedReader;
import java.io.StringReader;
import java.lang.reflect.Method;
import java.util.ArrayList;

/**
 * Title:        UIFactoryBlueprint
 * Description:  Writes the source code for all objects and stores the metadata.
 * Copyright:    Copyright (c) 2001
 * Company:      Abacus Research
 * @author Michael Gouker (Cagey Logic)
 * @version 1.0
 */


class UIFactoryBlueprintProperty {
    String sPropertyName;
    ArrayList objValues = new ArrayList(10);  // 10 property values is a good guess.  Not too much memory.
    UIFactoryBlueprintProperty theNextProperty = null;

    public UIFactoryBlueprintProperty(String sName) {
        sPropertyName = sName;
    }

    public void addValue(Object oValue) {
        objValues.add(oValue);
    }

    public Object getValue(int iIndex) {
        if (iIndex < getValueCount())
            return objValues.get(iIndex);
        else
            return null;
    }

    public int getValueCount() {
        return objValues.size();
    }

    public void setValue(int iIndex, Object oValue) {
        objValues.set(iIndex, oValue);
    }
}

/**
 * UIFactoryBlueprint class - This class contains the definition of how to build an object.
 * It is used by the Assembly line to store properties that are later queried in code generation.
 * It also contains all class-specific code generators.
 */

public class UIFactoryBlueprint {

    /**
     * Class object of the target object to be generated.
     */

    Class cClassObjectGone = null;  // not used.

    String theClassName = "";

    /**
     * Identifier of object to be generated.
     */

    String sIdentifier;

    /**
     *   Property names and vector values.  Each property consists of a name followed by a value.
     */

    UIFactoryBlueprintProperty theFirstProperty = null;
    UIFactoryBlueprintProperty theLastProperty = null;

    /**
     * Allows literal code to be inserted after property declarations.  The code is not
     * checked for syntax - it is just copied into the automatically generated source.
     */

    String sCustomCode;

    /**
     * Declare as member - If set, do the declaration as a member of the main class.
     */

    boolean bDeclareAsMember = true;


    /**
     * Allows parameter list to be stored for construction
     */

    String sConstructorParameterList = "";

    /**
     * Allows code to precede the constructor for creation of params
     */

    String sPreConstructionCode = "";

    /**
     * sContainerName - String - Defines name of container for assignment of layout
     */

    String sContainerName = "";

    /**
     * sLayoutName - String - Defines name of Layout for assignment of layout
     */

    String sLayoutName = "";


    /**
     * Activate as popup.  Adds code to master listener to activate this object.
     */

    boolean bActivateAsPopup = false;

    /**
     * Add action listener.
     */

    boolean bAddActionListener = false;

    /**
     * Add window event listener.
     */

    boolean bAddWindowEventListener = false;

    /**
     * Target of action event.
     */

    String sActionListenerTarget = "";

    /**
     * Class of target object.
     */

    String sActionListenerClass = "";

    /**
     * Message to send to target object.
     */

    String sActionListenerMessage = "";

    /**
     * objParentMeta - This is the meta object that represents the parent object.
     */

    Object objParentMeta = null;

    /**
     * theObject - the link to the design project's metaobject description.
     */

    MetaObject theObject;

    /**
     * theDiscriminators - used to build listeners.  Each element is a ListenerDiscriminator.
     */
    public ArrayList theDiscriminators;

    /**
     * findProperty - This method finds a blueprint property entry for a given
     * property name.  It returns null, if the property cannot be found.
     */

    public UIFactoryBlueprintProperty findProperty(String sPropertyName) {
        UIFactoryBlueprintProperty theProperty = theFirstProperty;
        while (theProperty != null) {
            if (theProperty.sPropertyName.equals(sPropertyName))
                return theProperty;
            theProperty = theProperty.theNextProperty;
        }
        return null;
    }

    /**
     * getPropertyValue - Find the property value with the matching name.
     *    @param sPropertyName - String - the name of the property.
     * return an object.
     */
    public Object getPropertyValue(String sPropertyName, int iIndex) {
        UIFactoryBlueprintProperty theProperty = findProperty(sPropertyName);
        if (theProperty != null)
            return theProperty.getValue(iIndex);
        else
            return null;
    }

    public int getPropertyValueCount(String sPropertyName) {
        UIFactoryBlueprintProperty theProperty = findProperty(sPropertyName);
        int iCount = 0;
        if (theProperty != null)
            iCount = theProperty.getValueCount();
        return iCount;
    }

    public Integer getIntegerPropertyValue(String sPropertyName, int iIndex) {
        Object objReturned = getPropertyValue(sPropertyName, iIndex);
        if (objReturned instanceof Integer)
            return (Integer) objReturned;
        if (objReturned instanceof Boolean) {
            if (objReturned.equals(new Boolean(true)))
                return new Integer(1);
            else
                return new Integer(0);
        }
        if (objReturned instanceof String) {
            try {
                Integer retval = new Integer((String) objReturned);
                return retval;
            } catch (java.lang.NumberFormatException e1) {
                return new Integer(0);
            }
        } else
            return (Integer) objReturned;  // this should throw exception
    }

    public Boolean getBooleanPropertyValue(String sPropertyName, int iIndex) {
        Object objReturned = getPropertyValue(sPropertyName, iIndex);
        if (objReturned instanceof Boolean)
            return (Boolean) objReturned;
        if (objReturned instanceof String)
            return new Boolean((String) objReturned);
        return null;
    }

    public String getMultilinePropertyValue(String sPropertyName, int iIndex) {
        String sPropertyValue = (String) getPropertyValue(sPropertyName, iIndex);
        if (sPropertyValue != null) {
            StringReader theReader = new StringReader(sPropertyValue);
            BufferedReader theBufferedReader = new BufferedReader(theReader);
            String sReturnValue = null;
            try {
                while (true) {
                    String sThisLine = theBufferedReader.readLine();
                    if (sThisLine == null)
                        break;
                    if (sThisLine.length() != 0) {
                        if (sReturnValue != null) {
                            sReturnValue = sReturnValue + "+\"";
                            sReturnValue = sReturnValue + sThisLine;
                        } else
                            sReturnValue = "\"" + sThisLine;
                        sReturnValue = sReturnValue + "\"";
                    }
                }
            } catch (java.io.IOException e1) {
            }
            if (sReturnValue == null)
                return "";
            return sReturnValue;
        }
        return "";
    }

    /**
     * addProperty - Adds a property to the object.
     * @param sPropertyName
     */

    public UIFactoryBlueprintProperty addProperty(String sPropertyName) {
        UIFactoryBlueprintProperty theProperty = new UIFactoryBlueprintProperty(sPropertyName);
        if (theFirstProperty == null)
            theFirstProperty = theProperty;
        else
            theLastProperty.theNextProperty = theProperty;
        theLastProperty = theProperty;
        return theProperty;
    }

    /**
     * setPropertyValue - Sets the property value of a given name.
     *    @param sPropertyName - the name of the property.
     *    @param iIndex - the value number to set.
     *    @param oTheValue - the value - an object.
     * return an object.
     */
    public void setPropertyValue(String sPropertyName, int iIndex, Object oTheValue) {
        UIFactoryBlueprintProperty theProperty = findProperty(sPropertyName);
        if (theProperty == null)
            theProperty = addProperty(sPropertyName);
        if (theProperty != null) {
            int iValueCount = theProperty.getValueCount();
            if (iIndex < iValueCount)
                theProperty.setValue(iIndex, oTheValue);
            else if (iIndex == iValueCount)
                theProperty.addValue(oTheValue);
        }
    }

    /**
     * getParentMeta - Return the parent meta object.
     * Returns an Object.
     */

    public Object getParentMeta() {
        return objParentMeta;
    }

    /**
     *  getActivateAsPopup - Tells whether the object is activated as a popup action
     *  returns a Boolean.
     */

    public Boolean getActivateAsPopup() {
        return new Boolean(bActivateAsPopup);
    }

    /**
     *  getDeclareAsMember - Tells whether the object is to be declared as a member of the
     *  main class - returns a Boolean.
     */

    public Boolean getDeclareAsMember() {
        return new Boolean(bDeclareAsMember);
    }


    /**
     *  getAddWindowEventListener - Tells whether the object uses a window event listener
     *  returns a Boolean.
     */

    public Boolean getAddWindowEventListener() {
        return new Boolean(bAddWindowEventListener);
    }

    /**
     *  getAddActionListener - Tells whether the object uses an actionlistener object
     *  returns a Boolean.
     */

    public Boolean getAddActionListener() {
        return new Boolean(bAddActionListener);
    }

    /**
     *  getActionListenerTarget - Tells name of the object that receives the action.
     *  returns a string.
     */

    public String getActionListenerTarget() {
        return sActionListenerTarget;
    }

    /**
     *  getActionListenerClass - Tells name of the class of the object that receives the action.
     *  returns a string.
     */

    public String getActionListenerClass() {
        return sActionListenerClass;
    }

    /**
     *  getActionListenerMessage - Tells name of message to send to target object.
     *  returns a string.
     */

    public String getActionListenerMessage() {
        return sActionListenerMessage;
    }

    /**
     *  getContainerName - Tells name of the container of the object.";
     *  returns a string.
     */
    public String getContainerName() {
        return sContainerName;
    }

    /**
     *  getLayoutName - Tells name of the Layout of the object.
     *  returns a string.
     */
    public String getLayoutName() {
        return sLayoutName;
    }


    /**
     * setParentMeta - Sets the parent meta object.
     */

    public void setParentMeta(Object theObject) {
        objParentMeta = theObject;
    }

    /**
     *  setActivateAsPopup - Tells whether the object is activated as a popup action
     *  @param bFlag - a Boolean - sets the property state.
     */

    public void setActivateAsPopup(Boolean bFlag) {
        bActivateAsPopup = bFlag.booleanValue();
    }

    /**
     *  setAddWindowEventListener - Tells whether the object uses a window event listener
     *  @param bFlag - a Boolean - sets the property state.
     */

    public void setAddWindowEventListener(Boolean bFlag) {
        bAddWindowEventListener = bFlag.booleanValue();
    }

    /**
     *  setAddActionListener - Tells whether the object uses an actionlistener object
     *  @param bFlag - a Boolean - sets the property state.
     */

    public void setAddActionListener(Boolean bFlag) {
        bAddActionListener = bFlag.booleanValue();
    }


    /**
     *  setDeclareAsMember - Tells whether the object should be declared as a member of
     * the main class.
     *  @param bFlag - a Boolean - sets the property state.
     */
    public void setDeclareAsMember(Boolean bFlag) {
        bDeclareAsMember = bFlag.booleanValue();
    }

    /**
     *  setContainerName - The name of the container of this object
     *  @param sContainer - the name of the container.
     */

    public void setContainerName(String sContainer) {
        sContainerName = sContainer;
    }

    /**
     *  setLayoutName - The name of the Layout of this object
     *  @param sLayout - the name of the Layout.
     */

    public void setLayoutName(String sLayout) {
        sLayoutName = sLayout;
    }

    /**
     *  setActionListenerTarget - The name of the object that receives the action
     *  @param sTarget - the name of the object to send the message to.
     */

    public void setActionListenerTarget(String sTarget) {
        sActionListenerTarget = sTarget;
    }

    /**
     *  setActionListenerClass - The name of the class of the object that receives the action
     *  @param sTargetClass - the name of the class of the object to send the message to.
     */

    public void setActionListenerClass(String sTargetClass) {
        sActionListenerClass = sTargetClass;
    }

    /**
     *  setActionListenerMessage - The name of the message to send to the target
     *  @param sTargetMessage - the name of the message.
     */
    public void setActionListenerMessage(String sTargetMessage) {
        sActionListenerMessage = sTargetMessage;
    }

    /**
     * Command the object to generate its code.
     * @param outputJava  Code generator that writes source to an embedded print writer.
     * @param sName  The identifier that names the object in the source.  This parameter
     * is necessary because the object name is hashed by Java.
     * Note:  For a control, a declaration is made, the control is attached to the
     * content pane of the current frame, properties are assigned, and custom code
     * is added, if it is present.
     */

    /**
     * Tell the object to store its custom code for later generation.
     * @param sCode  Code added after properties are set.  Passed through the system unprocessed.
     */

    public void setCustomCode(String sCode) {
        sCustomCode = sCode;
    }

    /**
     * Tell the object to store the code before the declaration.
     * @param sCode  Code added after properties are set.  Passed through the system unprocessed.
     */
    public void setPreConstructionCode(String sCode) {
        sPreConstructionCode = sCode;
    }


    /**
     * Allow the object to store param list for construction.
     * @param sParams  Parameter list is used for construction and then passed on to code generation.
     */
    public void setConstructorParameterList(String sParams) {
        sConstructorParameterList = sParams;
    }

    /**
     * Allow the object to store param list for construction.
     * returns sParamList  Parameter list is used for construction and then passed on to code generation.
     */

    public String getConstructorParameterList() {
        return sConstructorParameterList;
    }


    /**
     * generateCode - This is the basic method that generates code for all objects.
     *
     \    *
     */

    public static String formatColorString(String sColor) {
        // May be a rgb declaraion: new color()
        // May be a toString value: java.awt.color(r=0, g=0, b=0)
        // May be something like white, blue, green...
        if (sColor == null)
            return "";
        if (sColor.trim().length() == 0)
            return "";
        if (sColor.indexOf("new") != -1)  // Just return it (assume they know what they are doing.
            return sColor;
        if (sColor.indexOf("java.awt.Color") != -1) {
            int iPos1 = sColor.indexOf("r=");
            int iPos2 = sColor.indexOf(",g=");
            int iPos3 = sColor.indexOf(",b=");
            int iLength = sColor.length();
            String sRed = sColor.substring(iPos1 + 2, iPos2);
            String sGreen = sColor.substring(iPos2 + 3, iPos3);
            String sBlue = sColor.substring(iPos3 + 3, iLength - 1);
            return ("new Color(" + sRed + "," + sGreen + "," + sBlue + ")");
        }
        return "Color." + sColor;
    }

    public void SwingCodeGenerator(UIFactoryCodeGenerator outputJava, String sName) throws UIFactoryException {
        String sBackgroundColor = (String) getPropertyValue("Background", 0);
        String sForegroundColor = (String) getPropertyValue("Foreground", 0);
        String sSize = (String) getPropertyValue("Size", 0);
        String sLocation = (String) getPropertyValue("Location", 0);

        boolean bSkipDeclaration = false;
        Boolean bTest = getBooleanPropertyValue("NoDeclaration", 0);
        if (bTest != null)
            bSkipDeclaration = bTest.booleanValue();

        sBackgroundColor = formatColorString(sBackgroundColor);
        sForegroundColor = formatColorString(sForegroundColor);

        // Implement fonts!
        String sFont = (String) getPropertyValue("Font", 0);
        String sFontName = null;
        if ((sFont != null) && (sFont.trim().length() != 0)) {
            int iPos1 = sFont.indexOf("family=");
            int iPos2 = sFont.indexOf(",name");
            int iPos3 = sFont.indexOf("style=");
            int iPos4 = sFont.indexOf("size=");
            int iLength = sFont.length();
            String sFamily = sFont.substring(iPos1 + 7, iPos2);
            sFontName = sFont.substring(iPos2 + 6, iPos3 - 1);
            String sStyle = sFont.substring(iPos3 + 6, iPos4 - 1);
            String sFontSize = sFont.substring(iPos4 + 5, iLength - 1);
            String sStyleParam = "Font.PLAIN";
            if (sStyle.indexOf("bold") != -1) {
                if (sStyle.indexOf("italic") != -1)
                    sStyleParam = "Font.BOLD + Font.ITALIC";
                else
                    sStyleParam = "Font.BOLD";
            } else if (sStyle.indexOf("italic") != -1)
                sStyleParam = "Font.ITALIC";

            sFontName = "font$$$" + sName;
            // Write font definition.
            outputJava.println("Font " + sFontName + " = new Font(\"" + sFamily + "\"," + sStyleParam + "," + sFontSize + ");");
        }

        if (bSkipDeclaration) {
            if ((sBackgroundColor != null) && (sBackgroundColor.trim().length() != 0))
                outputJava.println("setBackground(" + sBackgroundColor + ");");
            if ((sForegroundColor != null) && (sForegroundColor.trim().length() != 0))
                outputJava.println("setForeground(" + sForegroundColor + ");");
            outputJava.println("setLocation(" + sLocation + ");");
            outputJava.println("setSize(" + sSize + ");");
        } else {
            if ((sBackgroundColor != null) && (sBackgroundColor.trim().length() != 0))
                outputJava.println(sName + ".setBackground(" + sBackgroundColor + ");");
            if ((sForegroundColor != null) && (sForegroundColor.trim().length() != 0))
                outputJava.println(sName + ".setForeground(" + sForegroundColor + ");");
            outputJava.println(sName + ".setLocation(" + sLocation + ");");
            outputJava.println(sName + ".setSize(" + sSize + ");");
            if (sFontName != null)
                outputJava.println(sName + ".setFont(" + sFontName + ");");
        }
    }

    public void ControlCodeGenerator(UIFactoryCodeGenerator outputJava, String sName) throws UIFactoryException {
        SwingCodeGenerator(outputJava, sName);
        String sText = (String) getPropertyValue("Text", 0);
        if (sText != null)
            outputJava.println(sName + ".setText(\"" + sText + "\");");
    }

    public void PanelCodeGenerator(UIFactoryCodeGenerator outputJava, String sName) throws UIFactoryException {
        String sBackgroundColor = (String) getPropertyValue("Background", 0);
        String sForegroundColor = (String) getPropertyValue("Foreground", 0);
        String sSize = (String) getPropertyValue("Size", 0);
        String sLocation = (String) getPropertyValue("Location", 0);

        boolean bSkipDeclaration = false;
        Boolean bTest = getBooleanPropertyValue("NoDeclaration", 0);
        if (bTest != null)
            bSkipDeclaration = bTest.booleanValue();

        sBackgroundColor = formatColorString(sBackgroundColor);
        sForegroundColor = formatColorString(sForegroundColor);


        if (bSkipDeclaration) {
            if ((sBackgroundColor != null) && (sBackgroundColor.trim().length() != 0))
                outputJava.println("setBackground(" + sBackgroundColor + ");");
            if ((sForegroundColor != null) && (sForegroundColor.trim().length() != 0))
                outputJava.println("setForeground(" + sForegroundColor + ");");
            outputJava.println("setLocation(" + sLocation + ");");
            outputJava.println("setSize(" + sSize + ");");
        } else {
            if ((sBackgroundColor != null) && (sBackgroundColor.trim().length() != 0))
                outputJava.println(sName + ".setBackground(" + sBackgroundColor + ");");
            if ((sForegroundColor != null) && (sForegroundColor.trim().length() != 0))
                outputJava.println(sName + ".setForeground(" + sForegroundColor + ");");
            outputJava.println(sName + ".setLocation(" + sLocation + ");");
            outputJava.println(sName + ".setSize(" + sSize + ");");
        }
    }

    public void FrameCodeGenerator(UIFactoryCodeGenerator outputJava, String sName) throws UIFactoryException {
        boolean bSkipDeclaration = false;
        Boolean bTest = getBooleanPropertyValue("NoDeclaration", 0);
        if (bTest != null)
            bSkipDeclaration = bTest.booleanValue();
        SwingCodeGenerator(outputJava, sName);
        String sText = (String) getPropertyValue("Text", 0);  // Used for title.
        if (sText != null) {
            if (bSkipDeclaration)
                outputJava.println("setTitle(\"" + sText + "\");");
            else
                outputJava.println(sName + ".setTitle(\"" + sText + "\");");
        }
    }


    public void ButtonCodeGenerator(UIFactoryCodeGenerator outputJava, String sName) throws UIFactoryException {
        ControlCodeGenerator(outputJava, sName);
        String sBitmapName = (String) getPropertyValue("BitmapName", 0);
        if ((sBitmapName != null) && (sBitmapName.trim().length() != 0)) {
            outputJava.println("Icon icon" + sName + " = new ImageIcon(\"" + sBitmapName + "\");");
            outputJava.print(sName + ".setIcon(icon" + sName + ");");
        }
    }

    public void CheckBoxCodeGenerator(UIFactoryCodeGenerator outputJava, String sName) throws UIFactoryException {
        ControlCodeGenerator(outputJava, sName);
    }

    public void ComboBoxCodeGenerator(UIFactoryCodeGenerator outputJava, String sName) throws UIFactoryException {
        // Combobox might be filled by model or by initial values.
        String sModelObject = (String) getPropertyValue("Model", 0);
        if ((sModelObject != null) && (sModelObject.trim().length() != 0)) {
            outputJava.print(sName + ".");
            outputJava.println("setModel(" + sModelObject + ");");  // no quotes around object name.
        } else {  // Add values for each initial value.
            int iCount = getPropertyValueCount("initValues");
            for (int i = 0; i < iCount; i++) {
                String sValue = (String) getPropertyValue("initValues", i);
                outputJava.print(sName + ".");
                outputJava.println("addItem(new String(\"" + sValue + "\"));");
            }
        }
        SwingCodeGenerator(outputJava, sName);
    }

    public void TextFieldCodeGenerator(UIFactoryCodeGenerator outputJava, String sName) throws UIFactoryException {
        ControlCodeGenerator(outputJava, sName);
        Integer iColumns = getIntegerPropertyValue("Columns", 0);
        if (iColumns != null)
            outputJava.println(sName + ".setColumns(" + iColumns.toString() + ");");
    }

    public void ListCodeGenerator(UIFactoryCodeGenerator outputJava, String sName) throws UIFactoryException {
        String sModelObject = (String) getPropertyValue("Model", 0);
        if ((sModelObject != null) && (sModelObject.trim().length() != 0)) {
            outputJava.print(sName + ".");
            outputJava.println("setModel(" + sModelObject + ");");  // no quotes around object name.
        } else {  // Add values for each initial value.
            int iCount = getPropertyValueCount("initValues");
            outputJava.print("Object[] " + sName + "ListData = {");
            for (int i = 0; i < iCount; i++) {
                String sValue = (String) getPropertyValue("initValues", i);
                if (i == iCount - 1) {
                    outputJava.println("\"" + sValue + "\"};");
                } else
                    outputJava.print("\"" + sValue + "\", ");
            }
            outputJava.print(sName + ".");
            outputJava.println("setListData(" + sName + "ListData );");
        }
        SwingCodeGenerator(outputJava, sName);
    }

    public void RadioButtonCodeGenerator(UIFactoryCodeGenerator outputJava, String sName) throws UIFactoryException {
        ControlCodeGenerator(outputJava, sName);
    }

    public void ScrollBarCodeGenerator(UIFactoryCodeGenerator outputJava, String sName) throws UIFactoryException {
        Integer iMax = getIntegerPropertyValue("Maximum", 0);
        Integer iMin = getIntegerPropertyValue("Minimum", 0);
        Integer iUnitInc = getIntegerPropertyValue("UnitIncrement", 0);
        String sOrientation = (String) getPropertyValue("Orientation", 0);
        SwingCodeGenerator(outputJava, sName);
        if (iMax != null)
            outputJava.println(sName + ".setMaximum(" + iMax.toString() + ");");
        if (iMin != null)
            outputJava.println(sName + ".setMinimum(" + iMin.toString() + ");");
        if (iUnitInc != null)
            outputJava.println(sName + ".setUnitIncrement(" + iUnitInc.toString() + ");");
        if (sOrientation != null)
            outputJava.println(sName + ".setOrientation(" + sOrientation + ");");
    }

    public void ScrollPaneCodeGenerator(UIFactoryCodeGenerator outputJava, String sName) throws UIFactoryException {
        ControlCodeGenerator(outputJava, sName);
    }

    public void SliderCodeGenerator(UIFactoryCodeGenerator outputJava, String sName) throws UIFactoryException {
        Integer iMax = getIntegerPropertyValue("Maximum", 0);
        Integer iMin = getIntegerPropertyValue("Minimum", 0);
        SwingCodeGenerator(outputJava, sName);
        if (iMax != null)
            outputJava.println(sName + ".setMaximum(" + iMax.toString() + ");");
        if (iMin != null)
            outputJava.println(sName + ".setMinimum(" + iMin.toString() + ");");
    }

    public void TextAreaCodeGenerator(UIFactoryCodeGenerator outputJava, String sName) throws UIFactoryException {
        ControlCodeGenerator(outputJava, sName);
        Integer iRows = getIntegerPropertyValue("Rows", 0);
        Integer iColumns = getIntegerPropertyValue("Columns", 0);
        if (iRows != null)
            outputJava.println(sName + ".setRows(" + iRows.toString() + ");");
        if (iColumns != null)
            outputJava.println(sName + ".setColumns(" + iColumns.toString() + ");");
        UIFactoryBlueprint theParentObject = (UIFactoryBlueprint) objParentMeta;
        if (theParentObject != null) {
            String sContainerType = (String) theParentObject.getPropertyValue("ContainerType", 0);
            MetaContainerType theContainerType = new MetaContainerType();
            if ((sContainerType != null) && (sContainerType.trim().length() != 0)) {
                theContainerType.setContainerType(sContainerType);
                if (theContainerType.getContainerType() == MetaContainerType.CONTAINER_SCROLLPANE) {
                    outputJava.println(sName + ".thePane = (JScrollPane) " + theParentObject.sIdentifier + ";");
                    outputJava.println(sName + ".fixupSize();");
                }
            }
        }
    }

    public void EntryCodeGenerator(UIFactoryCodeGenerator outputJava, String sName) throws UIFactoryException {
        ControlCodeGenerator(outputJava, sName);
//      String sFieldList = (String)getPropertyValue("FieldList", 0);
        String sPromptFieldList = (String) getPropertyValue("PromptFieldList", 0);
        String sTableName = (String) getPropertyValue("TableName", 0);
        String sTableEntryData = (String) getPropertyValue("TableEntryData", 0);
        String sTableIndex = (String) getPropertyValue("TableIndex", 0);
        String sEntryType = (String) getPropertyValue("EntryType", 0);
        String sSelectionEntryTo = (String) getPropertyValue("SelectionEntryTo", 0);
        String sSelectionEntryFrom = (String) getPropertyValue("SelectionEntryFrom", 0);
        String sEntryValue = (String) getPropertyValue("EntryValue", 0);
        if (sEntryType != null)
            outputJava.println(sName + ".setEntryType(" + sEntryType + ");");
        if (sTableEntryData != null) {
            outputJava.println("TableEntryData tblData_" + sName + " = new TableEntryData(" + sTableEntryData + ");");
            if (sTableName != null)
                outputJava.println(sName + ".setModel(" + sTableName + ".createTableModel( 2, tblData_" + sName + "));");
        }
        if (sEntryValue != null)
            outputJava.println(sName + ".setEntryValue(" + sEntryValue + ");");
        if (sSelectionEntryTo != null)
            outputJava.println(sName + ".setSelectionEntryTo(" + sSelectionEntryTo + ");");
        if (sSelectionEntryFrom != null)
            outputJava.println(sName + ".setSelectionEntryFrom(" + sSelectionEntryFrom + ");");
        if (sPromptFieldList != null)
            outputJava.println(sName + ".initPrompt(" + sTableName + ", " + sTableIndex + ", " + sPromptFieldList + ");");
    }

    public void TableCodeGenerator(UIFactoryCodeGenerator outputJava, String sName) throws UIFactoryException {
        String sFieldList = (String) getPropertyValue("FieldList", 0);
        outputJava.println(sName + ".setFieldList(" + sFieldList + ");");
        outputJava.println(sName + ".open();");
    }

    // Doesn't handle multiline condition.
    public Object getDefaultPropertyValue(String sPropertyName, ArrayList theGenerateProperties) {
        int iSize = theGenerateProperties.size();
        for (int i = 0; i < iSize; i = i + 5) {
            String sCompareName = (String) theGenerateProperties.get(i);
            if (sCompareName.equals(sPropertyName))
                return theGenerateProperties.get(i + 1);
        }
        return null;
    }

    public String getSetPropertyMethod(String sPropertyName, ArrayList theGenerateProperties) {
        int iSize = theGenerateProperties.size();
        for (int i = 0; i < iSize; i = i + 5) {
            String sCompareName = (String) theGenerateProperties.get(i);
            if (sCompareName.equals(sPropertyName))
                return (String) theGenerateProperties.get(i + 2);
        }
        return null;
    }

    public String getPropertyType(String sPropertyName, ArrayList theGenerateProperties) {
        int iSize = theGenerateProperties.size();
        for (int i = 0; i < iSize; i = i + 5) {
            String sCompareName = (String) theGenerateProperties.get(i);
            if (sCompareName.equals(sPropertyName))
                return (String) theGenerateProperties.get(i + 3);
        }
        return null;
    }

    public String getPropertyClass(String sPropertyName, ArrayList theGenerateProperties) {
        int iSize = theGenerateProperties.size();
        for (int i = 0; i < iSize; i = i + 5) {
            String sCompareName = (String) theGenerateProperties.get(i);
            if (sCompareName.equals(sPropertyName))
                return (String) theGenerateProperties.get(i + 4);
        }
        return null;
    }


    public void generateProperties(UIFactoryCodeGenerator outputJava, String sName) throws Throwable {
        boolean bSkipDeclaration = false;
        Boolean bTest = getBooleanPropertyValue("NoDeclaration", 0);
        if (bTest != null)
            bSkipDeclaration = bTest.booleanValue();

        // For all properties with the generate boolean set.
        UIFactoryBlueprintProperty theProperty = theFirstProperty;
        // Get class name.
        String sFullClassName = theClassName; //cClassObject.getName();
        String sClassName = sFullClassName.substring(sFullClassName.lastIndexOf('.') + 1);
        // If the class is an intentional subclass (has methods defined in the metaclass)
        // then use the subclass.  Otherwise use its super.
        ArrayList theDefaultValues =
                outputJava.metadataProvider.getDefaultPropertyValues(sClassName);

        while (theProperty != null) {
            // Find the default property value for the auto generating properties.
            String sCompareName = theProperty.sPropertyName;
            // Only the zeroth value is tested.  Auto generate does not work for arrays.
            Object objValue = theProperty.getValue(0);
            Object objDefaultValue = getDefaultPropertyValue(sCompareName, theDefaultValues);
            String sSetName = getSetPropertyMethod(sCompareName, theDefaultValues);
            String sPropTypeName = getPropertyType(sCompareName, theDefaultValues);
            String sPropClassName = getPropertyClass(sCompareName, theDefaultValues);
            // If sSetName is set, then the generate element is set in the metadata.
            // Check the property value against the default for the class.
            // Add the property if the value is not the same as the default.
            if (sSetName != null) {
                if ((sPropTypeName.equalsIgnoreCase("string")) ||
                        (sPropClassName.equalsIgnoreCase("string"))) {
                    if (objValue.equals(objDefaultValue) == false) {
                        if (bSkipDeclaration)
                            outputJava.println(sSetName + "(\"" + objValue.toString() + "\");");
                        else
                            outputJava.println(sName + "." + sSetName + "(\"" + objValue.toString() + "\");");
                    }
                } else {
                    if (objValue.equals(objDefaultValue) == false) {
                        if (bSkipDeclaration)
                            outputJava.println(sSetName + "(" + objValue.toString() + ");");
                        else
                            outputJava.println(sName + "." + sSetName + "(" + objValue.toString() + ");");
                    }
                }
            }
            theProperty = theProperty.theNextProperty;
        }
    }

    public String getLayoutDeclaration() {
        String sLayoutDeclaration = "null";
        String sLayoutName = (String) getPropertyValue("LayoutName", 0);
        if ((sLayoutName != null) && (sLayoutName.length() != 0)) {
            sLayoutDeclaration = "new " + sLayoutName + "(";
            int iLayoutParamCount = getPropertyValueCount("LayoutParameter");
            for (int i = 0; i < iLayoutParamCount; i++)
                sLayoutDeclaration = sLayoutDeclaration + getPropertyValue("LayoutParameter", i);
        }
        return sLayoutDeclaration;
    }

    public void generateCode(UIFactoryCodeGenerator outputJava, String sName) throws Throwable {
        // Declaration can go either as member data or in init function/constructor.
        boolean bSkipDeclaration = false;
        boolean bFrame = false;
        boolean bAbalet = false;
        boolean bPanel = false;
        boolean bGroup = false;
        boolean bAttached = false;
        boolean bTabbedPane = false;
        boolean bTabPage = false;
        boolean bScrollPane = false;
        boolean bMenu        = false;
        boolean bMenuItem   = false;


        UIFactoryBlueprint theParentObject = null;

        if (objParentMeta instanceof UIFactoryBlueprint)
            theParentObject = (UIFactoryBlueprint) objParentMeta;
        else
            theParentObject = null;

        Boolean bTest = getBooleanPropertyValue("NoDeclaration", 0);
        if (bTest != null)
            bSkipDeclaration = bTest.booleanValue();

        if (!bSkipDeclaration)
            outputJava.generateDeclaration(this, sName, sPreConstructionCode, bDeclareAsMember);

        bTest = getBooleanPropertyValue("Attached", 0);
        if (bTest != null)
            bAttached = bTest.booleanValue();

        bTest = getBooleanPropertyValue("AnchorLeft", 0);
        boolean bAnchorLeft = false;
        if (bTest != null)
            bAnchorLeft = bTest.booleanValue();

        bTest = getBooleanPropertyValue("AnchorRight", 0);
        boolean bAnchorRight = false;
        if (bTest != null)
            bAnchorRight = bTest.booleanValue();

        bTest = getBooleanPropertyValue("AnchorTop", 0);
        boolean bAnchorTop = false;
        if (bTest != null)
            bAnchorTop = bTest.booleanValue();

        bTest = getBooleanPropertyValue("AnchorBottom", 0);
        boolean bAnchorBottom = false;
        if (bTest != null)
            bAnchorBottom = bTest.booleanValue();

        String sContainerType = (String) getPropertyValue("ContainerType", 0);
        MetaContainerType theContainerType = new MetaContainerType();
        if ((sContainerType != null) && (sContainerType.trim().length() != 0))
            theContainerType.setContainerType(sContainerType);

        if (theContainerType.getContainerType() == MetaContainerType.CONTAINER_FRAME)
            bFrame = true;
        if (theContainerType.getContainerType() == MetaContainerType.CONTAINER_ABALET) {
            bAbalet = true;
            bFrame = true;
        }
        if (theContainerType.getContainerType() == MetaContainerType.CONTAINER_PANEL)
            bPanel = true;
        if (theContainerType.getContainerType() == MetaContainerType.CONTAINER_BUTTONGROUP)
            bGroup = true;
        if (theContainerType.getContainerType() == MetaContainerType.CONTAINER_SCROLLPANE)
            bScrollPane = true;
        if (theContainerType.getContainerType() == MetaContainerType.CONTAINER_TABBEDPANE)
            bTabbedPane = true;
        if (theContainerType.getContainerType() == MetaContainerType.CONTAINER_TABPAGE)
            bTabPage = true;
        if (theContainerType.getContainerType() == MetaContainerType.CONTAINER_JMENUBAR)
            bMenu = true;
        if (theContainerType.getContainerType() == MetaContainerType.CONTAINER_JMENU)
            bMenuItem = true;

        String sLayoutDeclaration = getLayoutDeclaration();

        if (bSkipDeclaration && !bAbalet) { // for main panel
            setPropertyValue("PanelName", 0, sName);
            setPropertyValue("ContentPane", 0, "getContentPane()");
            outputJava.println("//  Set the layout.");
            outputJava.print("AnchoringLayoutManager layoutManager" + sName + " = new AnchoringLayoutManager();");
            outputJava.println("getContentPane().setLayout( layoutManager" + sName + " );");
        } else if (bTabbedPane) {
            setPropertyValue("PanelName", 0, sName);
            setPropertyValue("ContentPane", 0, sName);
            outputJava.generateBasicTabbedPaneInitialization(sName);
        } else if (bScrollPane) {
            setPropertyValue("PanelName", 0, sName);
            setPropertyValue("ContentPane", 0, sName);
        } else if (bFrame) { // for frames and abalets.
            setPropertyValue("PanelName", 0, sName);
            setPropertyValue("ContentPane", 0, "contentPane" + sName);
            outputJava.generateBasicContainerInitialization(sName, sLayoutDeclaration);
        } else if (bPanel || bTabPage) { // for panels.
            setPropertyValue("PanelName", 0, sName);
            setPropertyValue("ContentPane", 0, sName);
            outputJava.generateBasicPanelInitialization(sName, sLayoutDeclaration);
        } else if (bMenu) { // Menu
            setPropertyValue("PanelName", 0, sName);
            setPropertyValue("ContentPane", 0, sName);
        } else if (bMenuItem) { // MenuItem
            setPropertyValue("PanelName", 0, sName);
            setPropertyValue("ContentPane", 0, sName);

        } else if (bGroup) {
            setPropertyValue("PanelName", 0, sName);
            setPropertyValue("ContentPane", 0, sName);
            outputJava.generateBasicGroupInitialization(sName, sLayoutDeclaration);
        }

        String sSettingName = (String) getPropertyValue("SettingName", 0);
        Boolean pbSelected = getBooleanPropertyValue("Selected", 0);
        if (sSettingName != null)
            outputJava.println(sName + ".setSettingName(" + sSettingName + ");");
        if (pbSelected != null)
            outputJava.println(sName + ".setSelected(" + pbSelected.toString() + ");");
        String sCustomCodeGenerator = outputJava.getCustomCodeGenerator(theClassName);
        // Custom code generator takes "outputJava", sName.
        if (sCustomCodeGenerator != null) {
            sCustomCodeGenerator = sCustomCodeGenerator.trim();
            if (sCustomCodeGenerator.length() > 0) {
                // Invoke the proper custom code generator for this class.
                Class[] cList = new Class[2];
                cList[0] = outputJava.getClass();
                cList[1] = sName.getClass();
                Object[] objArgList = new Object[2];
                objArgList[0] = outputJava;
                objArgList[1] = sName;
                Method m = getClass().getMethod(sCustomCodeGenerator, cList);
                if (m != null) {
                    m.invoke(this, objArgList);
                }
            }
        }

        // Attach to focus parent?
        if (bAttached && (theParentObject != null)) {
//          boolean bParentIsScrollPane = false;
//          boolean bParentIsTabbedPane = false;
            boolean bParentIsGroup = false;

            String sParentContainerType = (String) theParentObject.getPropertyValue("ContainerType", 0);
            MetaContainerType theParentContainerType = new MetaContainerType();
            if ((sParentContainerType != null) && (sParentContainerType.trim().length() != 0))
                theParentContainerType.setContainerType(sParentContainerType);
            if (theParentContainerType.getContainerType() == MetaContainerType.CONTAINER_BUTTONGROUP)
                bParentIsGroup = true;

            // This breaks if groups are nested in groups.
            if (bParentIsGroup) {
                // Add control to group.
                outputJava.generateBasicControlToGroupAttachment(sName, sContainerName);
                // Now change it so we add to grandparent instead of parent.
                if (theParentObject.objParentMeta == null)
                    throw new UIFactoryException("Parent of group is null.  Groups cannot be top level!");
                theParentObject = (UIFactoryBlueprint) theParentObject.objParentMeta;
                sParentContainerType = (String) theParentObject.getPropertyValue("ContainerType", 0);
                theParentContainerType = new MetaContainerType();
                if ((sParentContainerType != null) && (sParentContainerType.trim().length() != 0))
                    theParentContainerType.setContainerType(sParentContainerType);
            }
            String sPanelName = (String) theParentObject.getPropertyValue("PanelName", 0);
            String sContentPane = (String) theParentObject.getPropertyValue("ContentPane", 0);
            // Attach to parent (or grandparent for groups)
            if (theParentContainerType.getContainerType() == MetaContainerType.CONTAINER_SCROLLPANE)
                outputJava.generateBasicControlToScrollPaneAttachment(sName, sPanelName);
            else if (theParentContainerType.getContainerType() == MetaContainerType.CONTAINER_TABBEDPANE) {
                Integer iTabs = (Integer) theParentObject.getPropertyValue("TabsGenerated", 0);
                if (iTabs == null)
                    iTabs = new Integer(0);
                String sTabTitle = (String) theParentObject.getPropertyValue("TabTitle", iTabs.intValue());
                if (sTabTitle == null)
                    sTabTitle = sName;
                outputJava.generateBasicControlToTabbedPaneAttachment(sName, sPanelName, sTabTitle);
                iTabs = new Integer(iTabs.intValue() + 1);
                theParentObject.setPropertyValue("TabsGenerated", 0, iTabs);
            }
            else if (theParentContainerType.getContainerType() == MetaContainerType.CONTAINER_JMENUBAR ||
                    theParentContainerType.getContainerType() == MetaContainerType.CONTAINER_JMENU) {
                outputJava.generateBasicControlToMenuAttachment(sName, sPanelName);
            }
            else
                outputJava.generateBasicControlToContentPaneAttachment(sName, sContentPane, sPanelName,
                        bAnchorLeft, bAnchorRight, bAnchorTop, bAnchorBottom);

        }

        // Set properties for object.
        generateProperties(outputJava, sName);

        if (sCustomCode != null)
            outputJava.generateAdditionalCustomCode(sCustomCode);

        if (theDiscriminators != null) {
            int iSize = theDiscriminators.size();
            for (int iListener = 0; iListener < iSize; iListener++) {
                ListenerDiscriminator theDiscriminator = (ListenerDiscriminator) theDiscriminators.get(iListener);
                MetaClassDetail theListener = theDiscriminator.theClass;
                String sListenerName = theListener.sClassName;
                int bMatch = sListenerName.lastIndexOf("Adapter");
                String sAttachMethodName = "";
                if (bMatch != -1) {
                    sAttachMethodName = "add" + sListenerName.substring(0, bMatch) + "Listener";
                } else
                    sAttachMethodName = "add" + sListenerName;
                if (bSkipDeclaration)
                    outputJava.println(sAttachMethodName + "( new " + sName + "$$$" + sListenerName + "() );");
                else
                    outputJava.println(sName + "." + sAttachMethodName + "( new " + sName + "$$$" + sListenerName + "() );");
            }
        }

        if (bAbalet)
            outputJava.generateAbaletLinkage();

        if (bTabbedPane)
            outputJava.generateTabbedPaneListener();  // This is only done once.

    }

    public UIFactoryBlueprint(MetaObject theObject) {
        sIdentifier = theObject.getName();
        theClassName = theObject.theClass.getMetadata().sClassName;
        this.theObject = theObject;
    }

    public String getClassName() {
        return theClassName;
    }
}
