/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.lib.ui.renderer.common;

/**
 * Title:        UIFactoryBroadcaster
 * Description:  This simple class places objects in a simple tree structure.
 *              This is used to invoke a method to generate code which
 *              (besides assembling the structure) is all this class does.
 * Copyright:    Copyright (c) 2001
 * Company:      Abacus Software
 * @author       Michael Gouker (Cagey Logic)
 * @version 0.1
 */

import java.lang.reflect.InvocationTargetException;

/**
 * UIFactoryBroadcaster:  This simple class maintains a tree structure where each level
 *              is just a simple linked list of objects.  There is no ordering
 *              or traversal except first-last.  Its sole purpose is to maintain
 *              the objects in a structure (while storing the nesting of ui
 *              containership) so that code can later be generated. Each
 *              "UIFactoryBroadcaster" is a node in the UIFactoryBroadcaster nested structure.
 */

public class UIFactoryBroadcaster {

    /**
     * oTheObject - This is a pointer back to the object that the node represents.
     */
    protected UIFactoryBlueprint oTheObject;

    /**
     * oFirstChild - If the node has children, oFirstChild is the first of the next
     * level.
     */
    protected UIFactoryBroadcaster oFirstChild;

    /**
     * oLastChild - If the node has children, oLastChild is the last of the next
     * level.
     */
    protected UIFactoryBroadcaster oLastChild;

    /**
     * oNextSibling - If this object has other objects that have been declared at
     * this level that follow its declaration, oNextSibling is the next object.
     */
    protected UIFactoryBroadcaster oNextSibling;

    /**
     * sObjectIdentifer - a String - This is the name of the object to place in the
     * source code.  Java hashes names into unfriendly looking identifiers.
     */
    protected String sObjectIdentifier;

    /**
     * xmlListeners - the elements that describe the listeners of the objects.
     * These are used to generate event listener classes for the object.
     */

    protected electric.xml.Elements theListenerElements1;
    protected electric.xml.Elements theListenerElements2;


    /**
     * UIFactoryBroadcaster - constructor for broadcast node.
     *
     * @param   oObjectToAdd - an object.  The object that is linked to this node in
     * the broadcast tree.
     * @param   sIdentifier - a String.  The name of the object that will appear in
     * the code that is generated.
     */
    public UIFactoryBroadcaster(UIFactoryBlueprint oObjectToAdd, String sIdentifier) {
        oTheObject = oObjectToAdd;
        sObjectIdentifier = sIdentifier;
    }

    /**
     * findDeclaredObject  - Looks for an object by name in the broadcast list and returns it.
     *
     *  @param sObjName - the name of the object to search for.
     *
     */

    public Object findDeclaredObject(String sObjName) {
        UIFactoryBroadcaster oNext = oFirstChild;
        while (oNext != null) {
            // See if there's a match
            if (sObjName.equals(sObjectIdentifier))
                return oTheObject;
            // Descend tree
            Object oTest = oNext.findDeclaredObject(sObjName);
            // Yes? Return then.
            if (oTest != null)
                return oTest;
            // Keep looking on next.
            else
                oNext = oNext.oNextSibling;
        }
        return null;
    }


    /**
     * addChildObject - Adds a new broadcaster node at this level in the structure.
     *
     * @param oNewNode - UIFactoryBroadcaster. The node to be added at this level.
     */
    public void addChildObject(UIFactoryBroadcaster oNewNode) {
        if (oFirstChild == null) {
            oFirstChild = oNewNode;
            oLastChild = oNewNode;
        } else {
            oLastChild.oNextSibling = oNewNode;
            oLastChild = oNewNode;
        }
    }

    /**
     * generateCode - Invoke the generateCode method for all objects in the
     * broadcast tree.
     *
     * @param codeOutput - UIFactoryCodeGenerator - The object that generates the source.
     *        It is passed on as a parameter to the generateCode method along with
     *        the name to use to specify the object.
     */

    public void generateCode(UIFactoryCodeGenerator codeOutput)
            throws InvocationTargetException, IllegalAccessException, UIFactoryException {

        // Generate classes for event listeners.
        codeOutput.generateListenerClasses(sObjectIdentifier, oTheObject);

        try {
            oTheObject.generateCode(codeOutput, sObjectIdentifier);
        } catch (java.lang.Throwable e1) {
            throw new UIFactoryException("Object " + sObjectIdentifier + "doesn't know how to generate its code!");
        }

        UIFactoryBroadcaster child = oFirstChild;
        while (child != null) {
            child.generateCode(codeOutput);
            child = child.oNextSibling;
        }
    }
}
