/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

package ch.abacus.lib.ui.renderer.common;

import java.io.LineNumberReader;
import java.io.StringReader;

/**
 * Title:        uifactory
 * Description:
 * Copyright:    Copyright (c) 2001
 * Company:      Abacus Research
 * @author Michael Gouker (Cagey Logic)
 * @version 1.0
 */

class UIFactoryClassJumpPoint {
    String sClassName;
    int iBeginLineNumber;
    int iEndLineNumber;
    UIFactoryClassJumpPoint theNextClassJumpPoint;
    UIFactoryClassJumpPoint theFirstClassChild;
    UIFactoryClassJumpPoint theLastClassChild;
    UIFactoryClassJumpPoint theParent;
    UIFactoryMemberDataJumpPoint theFirstMemberDataJumpPoint;
    UIFactoryMemberDataJumpPoint theLastMemberDataJumpPoint;
    UIFactoryPropertyJumpPoint theFirstPropertyJumpPoint;
    UIFactoryPropertyJumpPoint theLastPropertyJumpPoint;
    UIFactoryMethodJumpPoint theFirstMethodJumpPoint;
    UIFactoryMethodJumpPoint theLastMethodJumpPoint;
    // Listeners live in classes - generally the main class.
    UIFactoryListenerJumpPoint theFirstListenerJumpPoint;
    UIFactoryListenerJumpPoint theLastListenerJumpPoint;

    UIFactoryListenerJumpPoint getListenerJump(int iLineNumber) {
        UIFactoryListenerJumpPoint theListenerJump = theFirstListenerJumpPoint;
        while (theListenerJump != null) {
            if ((iLineNumber > theListenerJump.iBeginLineNumber) &&
                    (iLineNumber < theListenerJump.iEndLineNumber))
                return theListenerJump;
            theListenerJump = theListenerJump.theNextListenerJumpPoint;
        }
        return null;
    }

    String getListenerName(int iLineNumber) {
        UIFactoryListenerJumpPoint theListener = getListenerJump(iLineNumber);
        if (theListener != null)
            return theListener.sListenerName;
        else
            return null;
    }

    String getMethodName(int iLineNumber) {
        UIFactoryMethodJumpPoint theMethodJump = theFirstMethodJumpPoint;
        while (theMethodJump != null) {
            if ((iLineNumber > theMethodJump.iBeginLineNumber) &&
                    (iLineNumber < theMethodJump.iEndLineNumber))
                return theMethodJump.sMethodName;
            theMethodJump = theMethodJump.theNextMethodJumpPoint;
        }
        // May be the method of one of the listeners.
        UIFactoryListenerJumpPoint theListenerJump = getListenerJump(iLineNumber);
        if (theListenerJump == null)
            return null;
        theMethodJump = theListenerJump.theFirstMethodJumpPoint;
        while (theMethodJump != null) {
            if ((iLineNumber > theMethodJump.iBeginLineNumber) &&
                    (iLineNumber < theMethodJump.iEndLineNumber))
                return theMethodJump.sMethodName;
            theMethodJump = theMethodJump.theNextMethodJumpPoint;
        }
        return null;
    }

    String getMemberDataName(int iLineNumber) {
        UIFactoryMemberDataJumpPoint theMemberDataJump = theFirstMemberDataJumpPoint;
        while (theMemberDataJump != null) {
            if ((iLineNumber > theMemberDataJump.iBeginLineNumber) &&
                    (iLineNumber < theMemberDataJump.iEndLineNumber))
                return theMemberDataJump.sMemberDataName;
            theMemberDataJump = theMemberDataJump.theNextMemberDataJumpPoint;
        }
        return null;
    }

    String getPropertyName(int iLineNumber) {
        UIFactoryPropertyJumpPoint thePropertyJump = theFirstPropertyJumpPoint;
        while (thePropertyJump != null) {
            if ((iLineNumber > thePropertyJump.iBeginLineNumber) &&
                    (iLineNumber < thePropertyJump.iEndLineNumber))
                return thePropertyJump.sPropertyName;
            thePropertyJump = thePropertyJump.theNextPropertyJumpPoint;
        }
        return null;
    }
}

class UIFactoryListenerJumpPoint {
    String sListenerName;
    int iBeginLineNumber;
    int iEndLineNumber;
    UIFactoryListenerJumpPoint theNextListenerJumpPoint;
    UIFactoryMethodJumpPoint theFirstMethodJumpPoint;
    UIFactoryMethodJumpPoint theLastMethodJumpPoint;
}


class UIFactoryMemberDataJumpPoint {
    String sMemberDataName;
    int iBeginLineNumber;
    int iEndLineNumber;
    UIFactoryMemberDataJumpPoint theNextMemberDataJumpPoint;
}

class UIFactoryPropertyJumpPoint {
    String sPropertyName;
    int iBeginLineNumber;
    int iEndLineNumber;
    UIFactoryPropertyJumpPoint theNextPropertyJumpPoint;
}

class UIFactoryMethodJumpPoint {
    String sMethodName;
    int iBeginLineNumber;
    int iEndLineNumber;
    UIFactoryMethodJumpPoint theNextMethodJumpPoint;
}

public class UIFactoryCodeJumper {

    UIFactoryClassJumpPoint theFirstClassJumpPoint;
    UIFactoryClassJumpPoint theLastClassJumpPoint;

    public static final int JUMPPOINT_NONE = 0;
    public static final int JUMPPOINT_METHOD = 1;
    public static final int JUMPPOINT_MEMBERDATA = 2;
    public static final int JUMPPOINT_PROPERTY = 3;

    public static final int JUMPCLASS_NONE = 0;
    public static final int JUMPCLASS_CLASS = 1;
    public static final int JUMPCLASS_LISTENER = 2;

    public UIFactoryCodeJumper(String sSourceCode) {
        StringReader theBufferAccess = new StringReader(sSourceCode);
        LineNumberReader theLineCounter = new LineNumberReader(theBufferAccess);
        try {
            String sTheLine = theLineCounter.readLine();
            UIFactoryClassJumpPoint currentClassJump = null;
            while (sTheLine != null) {
                // Scan for tag.
                int iMatch = sTheLine.indexOf("/***__@");
                if (iMatch != -1) {
                    int iMatchIdentifier = sTheLine.indexOf("[", iMatch);
                    if (iMatchIdentifier != -1) {
                        int iMatchEndIdentifier = sTheLine.indexOf("]", iMatchIdentifier);
                        if (iMatchEndIdentifier != -1) {
                            String sIdentifier = sTheLine.substring(iMatchIdentifier + 1,
                                    iMatchEndIdentifier);
                            int iMatchEndTag = sTheLine.indexOf(":", iMatch);
                            if (iMatchEndTag != -1) {
                                String sTag = "";
                                boolean bEnd = false;
                                if (sTheLine.charAt(iMatch + 7) == '@') {  // end of a tag.
                                    sTag = sTheLine.substring(iMatch + 8, iMatchEndTag);
                                    bEnd = true;
                                } else
                                    sTag = sTheLine.substring(iMatch + 7, iMatchEndTag);
                                if (sTag.equalsIgnoreCase("class")) {
                                    if (bEnd) {
                                        currentClassJump.iEndLineNumber = theLineCounter.getLineNumber();
                                        // Return to the parent class.
                                        currentClassJump = currentClassJump.theParent;
                                    } else {
                                        UIFactoryClassJumpPoint theNewClass = new UIFactoryClassJumpPoint();
                                        theNewClass.iBeginLineNumber = theLineCounter.getLineNumber();
                                        theNewClass.sClassName = sIdentifier;
                                        theNewClass.theParent = currentClassJump;
                                        // First for the UIFactoryCodeJumper.
                                        if (theFirstClassJumpPoint == null)
                                            theFirstClassJumpPoint = theNewClass;
                                        else
                                            theLastClassJumpPoint.theNextClassJumpPoint = theNewClass;
                                        theLastClassJumpPoint = theNewClass;
                                        // Now for the Inside levels.
                                        if (currentClassJump != null) {
                                            if (currentClassJump.theFirstClassChild == null)
                                                currentClassJump.theFirstClassChild = theNewClass;
                                            else
                                                currentClassJump.theLastClassChild.theNextClassJumpPoint = theNewClass;
                                            currentClassJump.theLastClassChild = theNewClass;
                                        }
                                        // Now begin processing in the child class.
                                        currentClassJump = theNewClass;
                                    }
                                }
                                if (sTag.equalsIgnoreCase("listener")) {
                                    if (bEnd) {
                                        currentClassJump.theLastListenerJumpPoint.iEndLineNumber = theLineCounter.getLineNumber();
                                    } else { // Allocate the new listener and attach it to the class.
                                        UIFactoryListenerJumpPoint theNewListener = new UIFactoryListenerJumpPoint();
                                        theNewListener.iBeginLineNumber = theLineCounter.getLineNumber();
                                        theNewListener.sListenerName = sIdentifier;
                                        if (currentClassJump.theFirstListenerJumpPoint == null)
                                            currentClassJump.theFirstListenerJumpPoint = theNewListener;
                                        else
                                            currentClassJump.theLastListenerJumpPoint.theNextListenerJumpPoint = theNewListener;
                                        currentClassJump.theLastListenerJumpPoint = theNewListener;
                                    }
                                }
                                if (sTag.equalsIgnoreCase("member data")) {
                                    if (bEnd) {
                                        currentClassJump.theLastMemberDataJumpPoint.iEndLineNumber = theLineCounter.getLineNumber();
                                    } else { // Allocate the new listener and attach it to the class.
                                        UIFactoryMemberDataJumpPoint theNewMemberData = new UIFactoryMemberDataJumpPoint();
                                        theNewMemberData.iBeginLineNumber = theLineCounter.getLineNumber();
                                        theNewMemberData.sMemberDataName = sIdentifier;
                                        if (currentClassJump.theFirstMemberDataJumpPoint == null)
                                            currentClassJump.theFirstMemberDataJumpPoint = theNewMemberData;
                                        else
                                            currentClassJump.theLastMemberDataJumpPoint.theNextMemberDataJumpPoint = theNewMemberData;
                                        currentClassJump.theLastMemberDataJumpPoint = theNewMemberData;
                                    }
                                }
                                if (sTag.equalsIgnoreCase("property")) {
                                    if (bEnd) {
                                        currentClassJump.theLastPropertyJumpPoint.iEndLineNumber = theLineCounter.getLineNumber();
                                    } else { // Allocate the new Property and attach it to the class.
                                        UIFactoryPropertyJumpPoint theNewProperty = new UIFactoryPropertyJumpPoint();
                                        theNewProperty.iBeginLineNumber = theLineCounter.getLineNumber();
                                        theNewProperty.sPropertyName = sIdentifier;
                                        if (currentClassJump.theFirstPropertyJumpPoint == null)
                                            currentClassJump.theFirstPropertyJumpPoint = theNewProperty;
                                        else
                                            currentClassJump.theLastPropertyJumpPoint.theNextPropertyJumpPoint = theNewProperty;
                                        currentClassJump.theLastPropertyJumpPoint = theNewProperty;
                                    }
                                }
                                if (sTag.equalsIgnoreCase("method")) {
                                    if (bEnd) {
                                        currentClassJump.theLastMethodJumpPoint.iEndLineNumber = theLineCounter.getLineNumber();
                                    } else { // Allocate the new Method and attach it to the class.
                                        UIFactoryMethodJumpPoint theNewMethod = new UIFactoryMethodJumpPoint();
                                        theNewMethod.iBeginLineNumber = theLineCounter.getLineNumber();
                                        theNewMethod.sMethodName = sIdentifier;
                                        if (currentClassJump.theFirstMethodJumpPoint == null)
                                            currentClassJump.theFirstMethodJumpPoint = theNewMethod;
                                        else
                                            currentClassJump.theLastMethodJumpPoint.theNextMethodJumpPoint = theNewMethod;
                                        currentClassJump.theLastMethodJumpPoint = theNewMethod;
                                    }
                                }
                            }
                        }
                    }
                }
                sTheLine = theLineCounter.readLine();
            }
        } catch (java.io.IOException e0) {
            System.out.println("Source file corrupt!");
        }
    }

    private UIFactoryClassJumpPoint getClassJumpNested(int iLineNumber, UIFactoryClassJumpPoint theJumpPoint) {
        while (theJumpPoint != null) {
            if ((iLineNumber > theJumpPoint.iBeginLineNumber) &&
                    (iLineNumber < theJumpPoint.iEndLineNumber)) {
                // Definitely in this class, but may be in a subclass.  Check.
                UIFactoryClassJumpPoint theSubClasses = getClassJumpNested(iLineNumber, theJumpPoint.theFirstClassChild);
                if (theSubClasses != null)
                    return theSubClasses;
                else
                    return theJumpPoint;
            }
            theJumpPoint = theJumpPoint.theNextClassJumpPoint;
        }
        return theJumpPoint;
    }

    public UIFactoryClassJumpPoint getClassJump(int iLineNumber) {
        UIFactoryClassJumpPoint theJumpPoint = theFirstClassJumpPoint;
        return getClassJumpNested(iLineNumber, theJumpPoint);
    }

    public String getMethodName(int iLineNumber) {
        // Search to find class.
        UIFactoryClassJumpPoint theClassJump = getClassJump(iLineNumber);
        // Search methods of class and methods of nested listeners for match.
        if (theClassJump != null)
            return theClassJump.getMethodName(iLineNumber);
        else
            return null;
    }

    public String getPropertyName(int iLineNumber) {
        // Search to find class.
        UIFactoryClassJumpPoint theClassJump = getClassJump(iLineNumber);
        // Search property for match.
        if (theClassJump != null)
            return theClassJump.getPropertyName(iLineNumber);
        else
            return null;
    }

    public String getMemberDataName(int iLineNumber) {
        // Search to find class.
        UIFactoryClassJumpPoint theClassJump = getClassJump(iLineNumber);
        // Search member data for match.
        if (theClassJump != null)
            return theClassJump.getMemberDataName(iLineNumber);
        else
            return null;
    }

    public String getClassName(int iLineNumber) {
        UIFactoryClassJumpPoint theClassJump = getClassJump(iLineNumber);
        // Find the class name.
        if (theClassJump != null)
            return theClassJump.sClassName;
        else
            return null;
    }

    public String getListenerName(int iLineNumber) {
        UIFactoryClassJumpPoint theClassJump = getClassJump(iLineNumber);
        if (theClassJump != null) {
            UIFactoryListenerJumpPoint theListenerJump = theClassJump.getListenerJump(iLineNumber);
            if (theListenerJump != null) // Find the listener name.
                return theListenerJump.sListenerName;
        }
        return null;
    }

    public int getJumpPointType(int iLineNumber) {
        // Search through classes to find the class.
        UIFactoryClassJumpPoint theClassJump = getClassJump(iLineNumber);
        if (theClassJump != null) {
            // Search through each list (method, data, property order)
            String theMethodName = theClassJump.getMethodName(iLineNumber);
            if (theMethodName != null)
                return UIFactoryCodeJumper.JUMPPOINT_METHOD;
            String theMemberDataName = theClassJump.getMemberDataName(iLineNumber);
            if (theMemberDataName != null)
                return UIFactoryCodeJumper.JUMPPOINT_MEMBERDATA;
            String thePropertyName = theClassJump.getPropertyName(iLineNumber);
            if (thePropertyName != null)
                return UIFactoryCodeJumper.JUMPPOINT_PROPERTY;
        }
        return UIFactoryCodeJumper.JUMPPOINT_NONE;
    }

    public int getJumpPointClass(int iLineNumber) {

        // Search through classes to find the class.
        UIFactoryClassJumpPoint theClassJump = getClassJump(iLineNumber);

        // Check if it's in a listener first.
        UIFactoryListenerJumpPoint theListenerJump = theClassJump.getListenerJump(iLineNumber);
        if (theListenerJump != null) {
            String theMethodName = theClassJump.getMethodName(iLineNumber);
            if (theMethodName != null)
                return UIFactoryCodeJumper.JUMPCLASS_LISTENER;
        }

        if (theClassJump != null) {
            // Search through each list (method, data, property order)
            String theMethodName = theClassJump.getMethodName(iLineNumber);
            if (theMethodName != null)
                return UIFactoryCodeJumper.JUMPCLASS_CLASS;
            String theMemberDataName = theClassJump.getMemberDataName(iLineNumber);
            if (theMemberDataName != null)
                return UIFactoryCodeJumper.JUMPCLASS_CLASS;
            String thePropertyName = theClassJump.getPropertyName(iLineNumber);
            if (thePropertyName != null)
                return UIFactoryCodeJumper.JUMPCLASS_CLASS;
        }
        return UIFactoryCodeJumper.JUMPCLASS_NONE;
    }

}
