/*
 * 2004  Abacus Research AG , St. Gallen , Switzerland . All rights reserved.
 * Terms of Use under The GNU GENERAL PUBLIC LICENSE Version 2
 *
 * THIS SOFTWARE IS PROVIDED BY ABACUS RESEARCH AG ``AS IS'' AND ANY EXPRESS 
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR 
 * NON-INFRINGEMENT, ARE DISCLAIMED. IN NO EVENT SHALL ABACUS RESEARCH AG BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* $Header: /cvs/v2006_cdnew/aba/java/src/ch/abacus/lib/ui/renderer/programmableCompiler/Exec.java,v 1.6 2004/08/31 16:25:50 gouker Exp $
 * $Log: Exec.java,v $
 * Revision 1.6  2004/08/31 16:25:50  gouker
 * Changes for optimization.  Droplet code generator changes.
 * This code still has some debugging that must be removed.
 *
 * Revision 1.5  2003/12/12 16:26:56  gouker
 * Moved source code from design cockpit to renderer
 * Merged in all 11/30 fixes for resource losses
 * Added metadata dispenser
 *
 * Revision 1.1  2003/07/02 18:53:18  gouker
 * Design Cockpit - New check in after refactoring.  In progress still so don't even try to build yet.
 *
 * Thanks!
 *
 * Revision 1.1  2003/03/24 14:59:57  gouker
 * First time checking in design cockpit.  This time it allows renderer.
 *
 * Revision 1.2  1998/11/06 10:37:03  graham
 * Added support for multiple packages and inner classes.
 *
 * Revision 1.1  1998/01/07 10:00:00  graham
 * Initial revision
 *
 */

package ch.abacus.lib.ui.renderer.programmableCompiler;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.InputStreamReader;

// This appears in Core Web Programming from
// Prentice Hall Publishers, and may be freely used
// or adapted. 1997 Marty Hall, hall@apl.jhu.edu.

/** A class that eases the pain of running external
 *  processes from applications.
 *  Lets you run a program three ways:
 *  <OL>
 *     <LI><B>exec</B>: Execute the command, returning
 *         immediately even if the command is still
 *         running. This would be appropriate
 *         for printing a file.
 *     <LI><B>execWait</B>: Execute the command, but
 *         don't return until the command finishes.
 *         This would be appropriate for
 *         sequential commands where the first depends
 *         on the second having finished (e.g.
 *         <CODE>javac</CODE> followed by
 *         <CODE>java</CODE>).
 *     <LI><B>execPrint</B>: Execute the command and
 *         print the output. This would be appropriate
 *         for the UNIX command <CODE>ls</CODE>.
 *  </OL>
 *  Note that the PATH is not taken into account,
 *  so  you must specify the <B>full</B> pathname to
 *  the command, and shell builtin commands
 *  will not work. For instance, on Unix the above
 *  three examples might look like:
 *  <OL>
 *    <LI><PRE>Exec.exec("/usr/ucb/lpr Some-File");</PRE>
 *    <LI><PRE>
 *        Exec.execWait("/usr/local/bin/javac Foo.java");
 *        Exec.execWait("/usr/local/bin/java Foo");
 *        </PRE>
 *    <LI><PRE>Exec.execPrint("/usr/bin/ls -al");</PRE>
 *  </OL>
 *
 * The <A HREF="../../compiler/Exec.java">source code</A> is available.
 *
 * @author Marty Hall
 *  (<A HREF="mailto:hall@apl.jhu.edu">
 *   hall@apl.jhu.edu</A>)
 * @version 1.0 1997
 */

public class Exec {
    //----------------------------------------------------

    private static boolean verbose = true;

    /** Determines if the Exec class should print which
     *  commands are being executed, and print error
     *  messages if a problem is found. Default is true.
     *
     * @param verboseFlag true: print messages.
     *        false: don't.
     */

    public static void setVerbose(boolean verboseFlag) {
        verbose = verboseFlag;
    }

    /** Starts a process to execute the command. Returns
     *  immediately, even if the new process is still
     *  running.
     *
     * @param command The <B>full</B> pathname of the
     *        command to be executed. No shell builtins
     *        (e.g. "cd") or shell meta-chars (e.g. "&gt;")
     *        allowed.
     * @return false if a problem is known to occur, but
     *         since this returns immediately, problems
     *         aren't usually found in time.
     *         Returns true otherwise.
     */

    public static boolean exec(String command) {
        return (exec(command, false, false));
    }

    /** Tests whether Exec will print status messages.
     *
     * @return true if Exec will print status messages
     */

    public static boolean getVerbose() {
        return (verbose);
    }

    /** Starts a process to execute the command. Waits
     *  for the process to finish before returning.
     *
     * @param command The <B>full</B> pathname of the
     *        command to be executed. No shell builtins
     *        or shell meta-chars allowed.
     * @return false if a problem is known to occur,
     *         either due to an exception or from the
     *         subprocess returning a non-zero value.
     *         Returns true otherwise.
     */

    public static boolean execWait(String command) {
        return (exec(command, false, true));
    }

    /** Starts a process to execute the command. Prints
     *  all output the command gives.
     *
     * @param command The <B>full</B> pathname of the
     *        command to be executed. No shell builtins
     *        or shell meta-chars allowed.
     * @return false if a problem is known to occur,
     *         either due to an exception or from the
     *         subprocess returning a non-zero value.
     *         Returns true otherwise.
     */

    public static boolean execPrint(String command) {
        return (exec(command, true, false));
    }

    //----------------------------------------------------
    // This creates a Process object via
    // Runtime.getRuntime.exec(). Depending on the
    // flags, it may call waitFor on the process
    // to avoid continuing until the process terminates,
    // or open an input stream from the process to read
    // the results.

    private static boolean exec(String command,
                                boolean printResults,
                                boolean wait) {
        if (verbose) {
            printSeparator();
            System.out.println("Executing '" + command + "'.");
        }
        try {
            // Start running command, returning immediately.
            Process p = Runtime.getRuntime().exec(command);

            // Print the output. Since we read until
            // there is no more input, this causes us
            // to wait until the process is completed
            if (printResults) {
                BufferedInputStream buffer =
                        new BufferedInputStream(p.getInputStream());
                BufferedReader commandResult =
                        new BufferedReader(new InputStreamReader(buffer));
                String s = null;
                try {
                    while ((s = commandResult.readLine()) != null)
                        System.out.println("Output: " + s);
                    commandResult.close();
                    if (p.exitValue() != 0) {
                        if (verbose)
                            printError(command +
                                    " -- p.exitValue() != 0");
                        return (false);
                    }
                    // Ignore read errors; they mean process is done
                } catch (Exception e) {
                }

                // If you don't print the results, then you
                // need to call waitFor to stop until the process
                // is completed
            } else if (wait) {
                try {
                    System.out.println(" ");
                    int returnVal = p.waitFor();
                    if (returnVal != 0) {
                        if (verbose)
                            printError(command);
                        return (false);
                    }
                } catch (Exception e) {
                    if (verbose)
                        printError(command, e);
                    return (false);
                }
            }
        } catch (Exception e) {
            if (verbose)
                printError(command, e);
            return (false);
        }
        return (true);
    }

    //----------------------------------------------------

    private static void printError(String command,
                                   Exception e) {
        System.out.println("Error doing exec(" +
                command + "): " + e.getMessage());
        System.out.println("Did you specify the full " +
                "pathname?");
    }

    private static void printError(String command) {
        System.out.println("Error executing '" +
                command + "'.");
    }

    //----------------------------------------------------

    private static void printSeparator() {
        System.out.println
                ("==============================================");
    }

    //----------------------------------------------------
}
