/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.model;

import java.util.*;

/**
 * Abstract base class that stores property values.
 *
 * @author Karl Tauber
 */
public abstract class FormObject
{
	/**
	 * A property value that explicitly sets a property to <code>null</code>.
	 * <p>
	 * Example:
	 * <pre>
	 *   FormComponent label = new FormComponent("javax.swing.JLabel");
	 *   label.setProperty("text", <b>FormComponent.NULL_VALUE</b>);
	 * </pre>
	 * The same in Swing:
	 * <pre>
	 *   JLabel label = new JLabel();
	 *   label.setText(<b>null</b>);
	 * </pre>
	 */
	public static final Object NULL_VALUE = new Object() {
		@Override
		public String toString() {
			return "(null)";
		}
	};

	private static final EmptyIterator<Map.Entry<String, Object>> EMPTY_ITERATOR
		= new EmptyIterator<Map.Entry<String, Object>>();

	private LinkedHashMap<String, Object> properties;
	private int referenceCount;

	FormObject() {
	}

	FormObject( FormObject obj, int dummy ) {
		// dummy parameter:
		//   Workaround for a Java 1.5 bug where the wrong constructor
		//   is invoked if the parameter value is null.

		// Do not clone property values because JFormDesigner handles all values
		// as immutable (never modifies a value; always creates a new instance).
		if( obj.properties != null )
			properties = new LinkedHashMap<String, Object>( obj.properties );
		referenceCount = obj.referenceCount;
	}

	/**
	 * Returns the value of a property (or <code>null</code> if inexistent).
	 */
	public Object getProperty( String name ) {
		if( properties == null )
			return null;
		return properties.get( name );
	}

	/**
	 * Returns the value of a property (or <code>def</code> if inexistent).
	 */
	public Object getProperty( String name, Object def ) {
		Object value = getProperty( name );
		return (value != null) ? value : def;
	}

	/**
	 * Sets the value of a property.
	 */
	public void setProperty( String name, Object value ) {
		setProperty( name, -1, value );
	}

	/**
	 * Sets the value of a property.
	 * If the property is not yet set, then it is inserted at the given index.
	 * If index is -1, then the property is appended to the end of the properties list.
	 *
	 * @since 3.1
	 */
	public void setProperty( String name, int index, Object value ) {
		if( properties == null )
			properties = new LinkedHashMap<String, Object>( 8 );

		int removedIndex = -1;
		Object oldValue;
		if( value != null ) {
			if( index < 0 || index >= properties.size() || properties.containsKey( name ) ) {
				// replace property or append to the end of the list
				oldValue = properties.put( name, value );
			} else {
				// insert property at index
				properties = insertAt( properties, name, index, value );
				oldValue = null;
			}
		} else {
			// determine index of property (for correct undo/redo)
			removedIndex = getIndexOf( properties, name );

			// remove property
			oldValue = properties.remove( name );
		}

		if( oldValue instanceof FormReference )
			referenceCount--;
		if( value instanceof FormReference )
			referenceCount++;

		firePropertyChanged( name, removedIndex, oldValue, value );
	}

	/**
	 * Creates a copy of the given map and inserts the key/value at the given index.
	 */
	private static LinkedHashMap<String, Object> insertAt( Map<String, Object> map, String key, int index, Object value ) {
		LinkedHashMap<String, Object> newMap = new LinkedHashMap<String, Object>( Math.max( map.size() + 1, 16 ) );

		Iterator<Map.Entry<String, Object>> it = map.entrySet().iterator();
		for( int i = 0; it.hasNext(); i++ ) {
			if( i == index )
				newMap.put( key, value );

			Map.Entry<String, Object> entry = it.next();
			newMap.put( entry.getKey(), entry.getValue() );
		}
		return newMap;
	}

	/**
	 * Returns the index of the key in the given map or -1 if not found.
	 */
	private static int getIndexOf( Map<String, Object> map, String key ) {
		Iterator<String> it = map.keySet().iterator();
		for( int i = 0; it.hasNext(); i++ ) {
			if( key.equals( it.next() ) ) {
				return i;
			}
		}
		return -1;
	}

	/**
	 * Sets the value of a property.
	 * Removes the property if the value equals to <code>def</code>.
	 */
	public void setProperty( String name, Object value, Object def ) {
		setProperty( name, !def.equals( value ) ? value : null );
	}

	/**
	 * Convenience method to get a String property value.
	 * Returns <code>null</code> if the property is not a String or does not exist.
	 */
	public String getPropertyString( String name ) {
		return getPropertyString( name, null );
	}

	/**
	 * Convenience method to get a String property value.
	 * Returns <code>def</code> if the property is not a String or does not exist.
	 */
	public String getPropertyString( String name, String def ) {
		Object value = getProperty( name );
		return (value instanceof String) ? (String) value : def;
	}

	/**
	 * Convenience method to set an String property value.
	 */
	public void setPropertyString( String name, String value ) {
		setProperty( name, value );
	}

	/**
	 * Convenience method to get an integer property value.
	 * Returns <code>0</code> if the property is not an integer or does not exist.
	 */
	public int getPropertyInt( String name ) {
		return getPropertyInt( name, 0 );
	}

	/**
	 * Convenience method to get an integer property value.
	 * Returns <code>def</code> if the property is not an integer or does not exist.
	 */
	public int getPropertyInt( String name, int def ) {
		Object value = getProperty( name );
		return (value instanceof Integer) ? ((Integer)value).intValue() : def;
	}

	/**
	 * Convenience method to set an integer property value.
	 */
	public void setPropertyInt( String name, int value ) {
		setProperty( name, new Integer( value ) );
	}

	/**
	 * Convenience method to set an integer property value.
	 * Removes the property if the value equals to <code>def</code>.
	 */
	public void setPropertyInt( String name, int value, int def ) {
		setProperty( name, (value != def) ? new Integer( value ) : null );
	}

	/**
	 * Convenience method to get a double property value.
	 * Returns <code>0</code> if the property is not an double or does not exist.
	 */
	public double getPropertyDouble( String name ) {
		return getPropertyDouble( name, 0 );
	}

	/**
	 * Convenience method to get a double property value.
	 * Returns <code>def</code> if the property is not an double or does not exist.
	 */
	public double getPropertyDouble( String name, double def ) {
		Object value = getProperty( name );
		return (value instanceof Double) ? ((Double)value).doubleValue() : def;
	}

	/**
	 * Convenience method to set a double property value.
	 */
	public void setPropertyDouble( String name, double value ) {
		setProperty( name, new Double( value ) );
	}

	/**
	 * Convenience method to set a double property value.
	 * Removes the property if the value equals to <code>def</code>.
	 */
	public void setPropertyDouble( String name, double value, double def ) {
		setProperty( name, (value != def) ? new Double( value ) : null );
	}

	/**
	 * Convenience method to get a boolean property value.
	 * Returns <code>0</code> if the property is not an boolean or does not exist.
	 */
	public boolean getPropertyBoolean( String name ) {
		return getPropertyBoolean( name, false );
	}

	/**
	 * Convenience method to get a boolean property value.
	 * Returns <code>def</code> if the property is not an boolean or does not exist.
	 */
	public boolean getPropertyBoolean( String name, boolean def ) {
		Object value = getProperty( name );
		return (value instanceof Boolean) ? ((Boolean)value).booleanValue() : def;
	}

	/**
	 * Convenience method to set a boolean property value.
	 */
	public void setPropertyBoolean( String name, boolean value ) {
		setProperty( name, Boolean.valueOf( value ) );
	}

	/**
	 * Convenience method to set a boolean property value.
	 * Removes the property if the value equals to <code>def</code>.
	 */
	public void setPropertyBoolean( String name, boolean value, boolean def ) {
		setProperty( name, (value != def) ? Boolean.valueOf( value ) : null );
	}

	/**
	 * Returns an iterator over the properties in this object.
	 * <code>Iterator.next()</code> returns an <code>java.util.Map.Entry</code>
	 * where the key is the property name and the value is the property value.
	 * <p>
	 * <strong>Note:</strong> This method is not available in the Java 1.4 version of this library.
	 *
	 * @since 5.0
	 */
	public Iterable<Map.Entry<String, Object>> properties() {
		return (properties != null && properties.size() > 0)
			? new MyIterator<Map.Entry<String, Object>>( properties.entrySet().iterator() )
			: EMPTY_ITERATOR;
	}

	/**
	 * Returns an iterator over the properties in this object.
	 * <code>Iterator.next()</code> returns an <code>java.util.Map.Entry</code>
	 * where the key is the property name and the value is the property value.
	 */
	public Iterator<Map.Entry<String, Object>> getProperties() {
		return properties().iterator();
	}

	/**
	 * Returns an iterator over the property names in this object.
	 * <code>Iterator.next()</code> returns a <code>String</code>.
	 * <p>
	 * <strong>Note:</strong> This method is not available in the Java 1.4 version of this library.
	 *
	 * @since 5.0
	 */
	public Iterable<String> propertyNames() {
		return (properties != null && properties.size() > 0)
			? new MyIterator<String>( properties.keySet().iterator() )
			: new EmptyIterator<String>();
	}

	/**
	 * Returns an iterator over the property names in this object.
	 * <code>Iterator.next()</code> returns a <code>String</code>.
	 */
	public Iterator<String> getPropertyNames() {
		return propertyNames().iterator();
	}

	/**
	 * Returns the number of properties in this object.
	 */
	public int getPropertyCount() {
		return (properties != null) ? properties.size() : 0;
	}

	abstract void firePropertyChanged( String name, int index, Object oldValue, Object newValue );

	boolean isEqual( FormObject formObject ) {
		if( formObject == this )
			return true;
		return safeEquals( properties, formObject.properties );
	}

	/**
	 * For internal use only.
	 */
	public int getReferenceCount() {
		return referenceCount;
	}

	void updateReferences( String oldName, String newName ) {
		if( properties == null || referenceCount == 0 )
			return;

		int refCount = referenceCount;
		for( Map.Entry<String, Object> entry : properties() ) {
			Object value = entry.getValue();

			if( value instanceof FormReference ) {
				if( oldName.equals( ((FormReference)value).getName() ) )
					setProperty( entry.getKey(), (newName != null)
									? new FormReference( newName ) : null );

				if( --refCount <= 0 )
					break;
			}
		}
	}

	/**
	 * Returns a string representation of the object.
	 */
	@Override
	public String toString() {
		return (properties != null) ? ("properties=" + properties.toString()) : "";
	}


	/**
	 * Returns the unqualified name of a class
	 * (e.g. JTextField for javax.swing.JTextField).
	 */
	static String unqualifiedClassName( Class<?> cls ) {
		if( cls == null )
			return "null";
		String className = cls.getName();
		return className.substring( className.lastIndexOf( '.' ) + 1 );
	}

	static boolean safeEquals( Object o1, Object o2 ) {
		if( o1 == o2 )
			return true;
		if( o1 == null || o2 == null )
			return false;
		return o1.equals( o2 );
	}

	//---- inner class MyIterator ---------------------------------------------

	static class MyIterator<E>
		implements Iterator<E>, Iterable<E>
	{
		private final Iterator<E> it;

		public MyIterator( Iterator<E> it ) {
			this.it = it;
		}

		public boolean hasNext() {
			return it.hasNext();
		}

		public E next() {
			return it.next();
		}

		public void remove() {
			throw new UnsupportedOperationException( "remove not allowed" );
		}

		public Iterator<E> iterator() {
			return this;
		}
	}

	//---- inner class EmptyIterator ------------------------------------------

	static class EmptyIterator<E>
		implements Iterator<E>, Iterable<E>
	{
		public boolean hasNext() {
			return false;
		}

		public E next() {
			throw new NoSuchElementException();
		}

		public void remove() {
			throw new IllegalStateException();
		}

		public Iterator<E> iterator() {
			return this;
		}
	}
}
