/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.model;

import java.awt.*;
import java.beans.Beans;
import java.net.URL;
import javax.swing.*;

/**
 * Used to specify a <code>javax.swing.Icon</code>.
 * The icon is either loaded from the classpath using
 * <code>new ImageIcon(classLoader.getResource(name))</code>,
 * from the file system using <code>new ImageIcon(name)</code>
 * or from the current look and feel using <code>UIManager.getIcon(name)</code>.
 *
 * @author Karl Tauber
 */
public class SwingIcon
	implements Icon, SwingResource
{
	public static final int CLASSPATH = 0;
	public static final int FILE = 1;
	/** @since 3.0 */
	public static final int SWING = 2;

	private final int type;
	private final String name;

	/**
	 * Constructs a icon.
	 *
	 * @param type The icon type ({@link #CLASSPATH}, {@link #FILE} or {@link #SWING}).
	 * @param name The icon name.
	 * 		A resource name (see <code>java.lang.ClassLoader.getResource(String)</code>)
	 * 		if type is {@link #CLASSPATH}, a file name (see <code>javax.swing.ImageIcon(String)</code>)
	 * 		if type is {@link #FILE} or a key (see <code>javax.swing.UIManager.getIcon(String)</code>)
	 * 		if type is {@link #SWING}.
	 */
	public SwingIcon( int type, String name ) {
		this.type = type;
		this.name = name;
	}

	/**
	 * Returns the icon type ({@link #CLASSPATH}, {@link #FILE} or {@link #SWING}).
	 */
	public int getType() {
		return type;
	}

	/**
	 * Returns the icon name.
	 * A resource name if type is {@link #CLASSPATH},
	 * a file name if type is {@link #FILE} or
	 * a key if type is {@link #SWING}.
	 */
	public String getName() {
		return name;
	}

	/**
	 * For internal use only.
	 */
	public Icon createIcon( ClassLoader loader ) {
		if( type == CLASSPATH ) {
			String resName = name.startsWith( "/" ) ? name.substring( 1 ) : name;
			URL url = loader.getResource( resName );
			if( url != null )
				return loadIcon( url );
		} else if( type == FILE )
			return loadIcon( name );
		else if( type == SWING )
			return UIManager.getIcon( name );

		return null;
	}

	private static ImageIcon loadIcon( URL url ) {
		if( Beans.isDesignTime() ) {
			Image image = Toolkit.getDefaultToolkit().createImage( url );
			return (image != null) ? new ImageIcon( image, url.toExternalForm() )
								   : new ImageIcon();
		} else
			return new ImageIcon( url );
	}

	private static ImageIcon loadIcon( String filename ) {
		if( Beans.isDesignTime() ) {
			Image image = Toolkit.getDefaultToolkit().createImage( filename );
			return (image != null) ? new ImageIcon( image, filename )
								   : new ImageIcon();
		} else
			return new ImageIcon( filename );
	}

	public int getIconWidth() {
		return 1;
	}

	public int getIconHeight() {
		return 1;
	}

	public void paintIcon( Component c, Graphics g, int x, int y ) {
	}

	/**
	 * Determines whether another object is equal to this object.
	 */
	@Override
	public boolean equals( Object obj ) {
		if( !(obj instanceof SwingIcon) )
			return false;
		return type == ((SwingIcon)obj).type &&
			   name.equals( ((SwingIcon)obj).name );
	}

	/**
	 * Returns the hash code for this object.
	 */
	@Override
	public int hashCode() {
		return type + name.hashCode();
	}

	/**
	 * Returns a string representation of the object.
	 */
	@Override
	public String toString() {
		return FormObject.unqualifiedClassName( getClass() )
			+ "[type=" + type + ",name=" + name + "]";
	}
}
