/*
 * Copyright (C) 2008-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import java.lang.reflect.Method;
import java.util.Collections;
import java.util.List;
import javax.swing.*;
import org.jdesktop.beansbinding.*;
import org.jdesktop.swingbinding.*;
import com.jformdesigner.model.*;
import static com.jformdesigner.runtime.BeansBindingConstants.*;

/**
 * A bindings creator for Beans Binding (JSR 295).
 *
 * @author Karl Tauber
 * @since 5.0
 */
@SuppressWarnings({ "unchecked", "rawtypes" })
public class BeansBindingCreator
	implements BindingCreator<BindingGroup, Binding>
{
	private FormCreator creator;

	public void setFormCreator( FormCreator creator ) {
		this.creator = creator;
	}

	public BindingGroup createBindingGroup( FormBindingGroup formBindingGroup )
		throws Exception
	{
		BindingGroup group = new BindingGroup();
		FormBinding[] formBindings = formBindingGroup.getBindings();
		for( int i = 0; i < formBindings.length; i++ )
			group.addBinding( createBinding( formBindings[i] ) );
		return group;
	}

	public Binding createBinding( FormBinding formBinding )
		throws Exception
	{
		AutoBinding.UpdateStrategy strategy = (AutoBinding.UpdateStrategy)
			formBinding.getProperty( PROP_UPDATE_STRATEGY, AutoBinding.UpdateStrategy.READ_WRITE );
		String source = formBinding.getSource();
		String target = formBinding.getTarget();
		String sourcePath = formBinding.getSourcePath();
		String targetPath = formBinding.getTargetPath();
		String name = formBinding.getPropertyString( PROP_NAME );

		if( source == null )
			throw new IllegalArgumentException( "Binding source is null." );
		if( target == null )
			throw new IllegalArgumentException( "Binding target is null." );
		if( targetPath == null )
			throw new IllegalArgumentException( "Binding target path is null." );

		Object sourceObject = getBean( source );
		Object targetObject = getBean( target );
		Property sourceProperty = (sourcePath != null)
			? createProperty( sourcePath, formBinding.getPropertyString( PROP_SOURCE_PATH_TYPE ) )
			: ObjectProperty.create();

		Binding binding = createSwingBinding( formBinding, strategy, sourceObject,
			sourcePath, sourceProperty, targetObject, targetPath, name );

		if( binding == null ) {
			Property targetProperty = createProperty( targetPath, formBinding.getPropertyString( PROP_TARGET_PATH_TYPE ) );

			binding = Bindings.createAutoBinding( strategy,
				sourceObject, sourceProperty, targetObject, targetProperty, name );
		}

		initBindingProperties( formBinding, binding );

		if( formBinding.getPropertyBoolean( PROP_BIND_IMMEDIATELY ) )
			bind( binding );

		return binding;
	}

	private Binding createSwingBinding( FormBinding formBinding,
			AutoBinding.UpdateStrategy strategy,
			Object sourceObject, String sourcePath, Property sourceProperty,
			Object targetObject, String targetPath, String name )
		throws Exception
	{
		if( "elements".equals( targetPath ) &&
			(sourceObject instanceof List || sourcePath != null) )
		{
			if( targetObject instanceof JComboBox ) {
				return SwingBindings.createJComboBoxBinding( strategy,
					sourceObject, sourceProperty, (JComboBox) targetObject, name );

			} else if( targetObject instanceof JList ) {
				JListBinding binding = SwingBindings.createJListBinding( strategy,
					sourceObject, sourceProperty, (JList) targetObject, name );
				String detailPath = formBinding.getPropertyString( PROP_DETAIL_PATH );
				if( detailPath != null )
					binding.setDetailBinding( createProperty( detailPath, formBinding.getPropertyString( PROP_DETAIL_PATH_TYPE ) ) );
				return binding;

			} else if( targetObject instanceof JTable ) {
				JTableBinding binding = SwingBindings.createJTableBinding( strategy,
						sourceObject, sourceProperty, (JTable) targetObject, name );
				if( !formBinding.getPropertyBoolean( PROP_EDITABLE, true ) )
					binding.setEditable( false );

				// ensure that JTableBinding sets a table model (and columns)
				binding.setSourceNullValue( Collections.EMPTY_LIST );
				binding.setSourceUnreadableValue( Collections.EMPTY_LIST );

				FormBinding[] columns = (FormBinding[]) formBinding.getProperty( PROP_SUB_BINDINGS );
				if( columns != null ) {
					for( int i = 0; i < columns.length; i++ ) {
						FormBinding column = columns[i];
						JTableBinding.ColumnBinding columnBinding = binding.addColumnBinding(
								createProperty( column.getSourcePath(), column.getPropertyString( PROP_SOURCE_PATH_TYPE ) ) );
						String columnName = getPropertyStringI18n( column, PROP_COLUMN_NAME, null );
						if( columnName != null )
							columnBinding.setColumnName( columnName );
						String columnClassName = column.getPropertyString( PROP_COLUMN_CLASS );
						if( columnClassName != null ) {
							Class<?> columnClass;
							if( "boolean".equals( columnClassName ) )
								columnClass = boolean.class;
							else if( "int".equals( columnClassName ) )
								columnClass = int.class;
							else if( "short".equals( columnClassName ) )
								columnClass = short.class;
							else if( "byte".equals( columnClassName ) )
								columnClass = byte.class;
							else if( "long".equals( columnClassName ) )
								columnClass = long.class;
							else if( "float".equals( columnClassName ) )
								columnClass = float.class;
							else if( "double".equals( columnClassName ) )
								columnClass = double.class;
							else if( "char".equals( columnClassName ) )
								columnClass = char.class;
							else
								columnClass = getLoader().loadClass( columnClassName );
							columnBinding.setColumnClass( columnClass );
						}
						if( !column.getPropertyBoolean( PROP_EDITABLE, true ) )
							columnBinding.setEditable( false );

						initBindingProperties( column, columnBinding );
					}
				}
				return binding;
			}
		}
		return null;
	}

	private Property createProperty( String path, String pathType )
		throws Exception
	{
		if( pathType != null ) {
			if( pathType.equals( ELProperty.class.getName() ) )
				return ELProperty.create( path );
			else if( pathType.equals( BeanProperty.class.getName() ) )
				return BeanProperty.create( path );
			else {
				Class<?> cls = getLoader().loadClass( pathType );
				Method createMethod = cls.getMethod( "create", String.class );
				return (Property) createMethod.invoke( null, path );
			}
		} else
			return isBeanPropertyPath( path ) ? BeanProperty.create( path ) : ELProperty.create( path );
	}

	public static boolean isBeanPropertyPath( String path ) {
		boolean first = true;
		int length = path.length();
		for( int i = 0; i < length; i++ ) {
			char ch = path.charAt( i );
			if( first ) {
				if( !Character.isJavaIdentifierStart( ch ) )
					return false;
				first = false;
			} else {
				if( ch == '.' )
					first = true;
				else if( !Character.isJavaIdentifierPart( ch ) )
					return false;
			}
		}
		if( first )
			return false;
		return true;
	}

	private void initBindingProperties( FormBinding formBinding, Binding binding )
		throws Exception
	{
		Object converter = formBinding.getProperty( PROP_CONVERTER );
		Object validator = formBinding.getProperty( PROP_VALIDATOR );
		Object sourceNullValue = getPropertyI18n( formBinding, PROP_SOURCE_NULL_VALUE );
		Object targetNullValue = getPropertyI18n( formBinding, PROP_TARGET_NULL_VALUE );
		Object sourceUnreadableValue = getPropertyI18n( formBinding, PROP_SOURCE_UNREADABLE_VALUE );

		if( sourceNullValue != null )
			binding.setSourceNullValue( nullValue( sourceNullValue ) );
		if( sourceUnreadableValue != null )
			binding.setSourceUnreadableValue( nullValue( sourceUnreadableValue ) );
		if( targetNullValue != null )
			binding.setTargetNullValue( nullValue( targetNullValue ) );

		if( converter != null ) {
			Object converterObject = getBean( ((FormReference)converter).getName() );
			binding.setConverter( (Converter) converterObject );
		}
		if( validator != null ) {
			Object validatorObject = getBean( ((FormReference)validator).getName() );
			binding.setValidator( (Validator) validatorObject );
		}
	}

	private Object getPropertyI18n( FormObject formObject, String name ) {
		Object value = formObject.getProperty( name );
		return (value instanceof FormMessage) ? getString( (FormMessage) value ) : value;
	}

	private Object nullValue( Object value ) {
		return (value != FormObject.NULL_VALUE) ? value : null;
	}

	public void bindGroup( BindingGroup bindingGroup ) {
		bindingGroup.bind();
	}

	public void unbindGroup( BindingGroup bindingGroup ) {
		bindingGroup.unbind();
	}

	public void bind( Binding binding ) {
		if( !binding.isBound() )
			binding.bind();
	}

	public void unbind( Binding binding ) {
		if( binding.isBound() )
			binding.unbind();
	}

	//---- FormCreator access methods -----------------------------------------

	protected ClassLoader getLoader() {
		return creator.getLoader();
	}

	protected Object getBean( String name ) throws Exception {
		return creator.getBean( name, true );
	}

	protected String getPropertyStringI18n( FormObject formObject, String name, String def ) {
		return creator.getPropertyStringI18n( formObject, name, def );
	}

	protected String getString( FormMessage message ) {
		return creator.getString( message );
	}
}
