/*
 * Copyright (C) 2003-2011 Karl Tauber <karl at jformdesigner dot com>
 * All Rights Reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *  o Neither the name of JFormDesigner or Karl Tauber nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.jformdesigner.runtime;

import java.awt.*;
import java.lang.reflect.Array;
import java.util.Date;
import java.util.Vector;
import javax.swing.*;
import com.jformdesigner.model.SwingTableModel;

/**
 * @author Karl Tauber
 */
public class ObjectCloner
{
	public static Object cloneObject( Object o ) {
		if( o == null )
			return null;

		Class<?> cls = o.getClass();
		if( isImmutable( cls ) )
			return o;

		if( cls.isArray() ) {
			Class<?> componentType = cls.getComponentType();
			int arrayLength = Array.getLength( o );
			Object newArray = Array.newInstance( componentType, arrayLength );
			if( isImmutable( componentType ) || componentType.isPrimitive() )
				System.arraycopy( o, 0, newArray, 0, arrayLength );
			else {
				for( int i = 0; i < arrayLength; i++ )
					Array.set( newArray, i, cloneObject( Array.get( o, i ) ) );
			}
			return newArray;
		}

		if( cls == Dimension.class )
			return new Dimension( (Dimension) o );
		if( cls == Insets.class )
			return new Insets( ((Insets)o).top, ((Insets)o).left, ((Insets)o).bottom, ((Insets)o).right );
		if( cls == Point.class )
			return new Point( (Point) o );
		if( cls == Rectangle.class )
			return new Rectangle( (Rectangle) o );

		if( cls == DefaultComboBoxModel.class ) {
			DefaultComboBoxModel m = (DefaultComboBoxModel) o;
			int size = m.getSize();
			Vector<Object> v = new Vector<Object>( size );
			for( int i = 0; i < size; i++ )
				v.add( m.getElementAt( i ) );
			return new DefaultComboBoxModel( v );
		}
		if( cls == SwingTableModel.class ) {
			SwingTableModel m = (SwingTableModel) o;
			return new SwingTableModel( m );
		}

		if( cls == SpinnerNumberModel.class ) {
			SpinnerNumberModel m = (SpinnerNumberModel) o;
			return new SpinnerNumberModel( m.getNumber(), m.getMinimum(), m.getMaximum(), m.getStepSize() );
		}
		if( cls == SpinnerDateModel.class ) {
			SpinnerDateModel m = (SpinnerDateModel) o;
			Date value = m.getDate();
			Comparable<?> start = m.getStart();
			Comparable<?> end = m.getEnd();
			long now = (System.currentTimeMillis() / 60000) * 60000;
			if( value.getTime() == Long.MIN_VALUE )
				value = new Date( now );
			if( start instanceof Date && ((Date)start).getTime() == Long.MIN_VALUE )
				start = new Date( now );
			if( end instanceof Date && ((Date)end).getTime() == Long.MIN_VALUE )
				end = new Date( now );
			return new SpinnerDateModel( value, start, end, m.getCalendarField() );
		}
		if( cls == SpinnerListModel.class ) {
			SpinnerListModel m = (SpinnerListModel) o;
			SpinnerListModel m2 = new SpinnerListModel( m.getList() );
			m2.setValue( m.getValue() );
			return m2;
		}

		return o;
	}

	public static boolean isImmutable( Class<?> cls ) {
		return cls == Boolean.class || // java.lang
			   cls == Byte.class ||
			   cls == Character.class ||
			   cls == Double.class ||
			   cls == Float.class ||
			   cls == Integer.class ||
			   cls == Long.class ||
			   cls == Short.class ||
			   cls == String.class ||
			   cls == Void.class ||
			   cls == Color.class || // java.awt
			   cls == Font.class;
	}
}
