/* DataContainer: This class implements an OS/2 style container.  It */
/*    requires the ContainerItem class.                              */

package extend.awt;

import java.awt.*;
import java.util.BitSet;

public class DataContainer extends Panel implements ItemProcessor {

   public DataContainer (int newView) {

      root = null;

      setLayout (null);
      setBackground (Color.lightGray);

      hScrollLeft = new Scrollbar (Scrollbar.HORIZONTAL);
      hScrollRight = new Scrollbar (Scrollbar.HORIZONTAL);
      vScroll = new Scrollbar (Scrollbar.VERTICAL);

      iconView = new IconCanvas (this);
      listView = new ListCanvas (this);
      treeView = new TreeCanvas (this);
      detailsLeftView = new DetailsCanvas (this, hScrollLeft, true);
      detailsRightView = new DetailsCanvas (this, hScrollRight, false);
      splitBar = new Canvas ();
      splitBar.setBackground (Color.darkGray);

      add (splitBar);
      add (hScrollLeft);
      add (hScrollRight);
      add (vScroll);
      add (iconView);
      add (listView);
      add (treeView);
      add (detailsLeftView);
      add (detailsRightView);

      title = null;
      titles = null;
      attributes = null;
      first = 0;
      firstHotSpot = null;
      clicked = null;
      selectionType = Single_Selection;
      splitBarColumn = No_Split_Bar;

      setView (newView);
   }

   public void setView (int newView) {

      switch (newView) {

         case Icon_View:
            setIconView ();
            break;

         case List_View:
            setListView ();
            break;

         case Tree_View:
            setTreeView ();
            break;

         case Details_View:
            setDetailsView ();
            break;

         default:
            throw new IllegalArgumentException ("Invalid view selection");
      }

      view = newView;

      layout ();
   }

   private void hideAll () {

      iconView.hide ();
      listView.hide ();
      treeView.hide ();
      detailsLeftView.hide ();
      detailsRightView.hide ();
      splitBar.hide ();
      hScrollLeft.hide ();
      hScrollRight.hide ();
   }

   private void setIconView () {

      hideAll ();
      iconView.show ();
      currentCanvas = iconView;
   }

   private void setListView () {

      hideAll ();
      listView.show ();
      currentCanvas = listView;
   }

   private void setTreeView () {

      hideAll ();
      treeView.show ();
      currentCanvas = treeView;
   }

   private void setDetailsView () {

      hideAll ();
      detailsLeftView.show ();
      hScrollLeft.show ();
      currentCanvas = detailsLeftView;

      if (splitBarColumn == No_Split_Bar)

         detailsLeftView.setColumns (0, DetailsCanvas.All_Columns);

      else {

         detailsRightView.show ();
         hScrollRight.show ();
         detailsLeftView.setColumns (0, splitBarColumn);
         detailsRightView.setColumns (splitBarColumn + 1, DetailsCanvas.All_Columns);
      }
   }

   public void setRoot (ContainerItem baseItem) {

      root = baseItem;
      ContainerItem [] sel;

      if (root != null) {

         root = root.getFirst ();
         first = 0;
         clicked = null;

         sel = getSelected ();

         if (selectionType == Forced_Single_Selection && sel == null)
            root.setSelected (true);

         if (sel != null)
            if ((selectionType == Forced_Single_Selection ||
                 selectionType == Single_Selection) && sel.length > 1) {

               newSelectionState = false;
               root.traverse (this, Setting_Selection);
               sel [0].setSelected (true);
            }
      }

      repaint ();
   }

   public void setTitle (String text) {

      title = text;
      repaint ();
   }

   public void setColumnData (Object title, int justify, int vertLine) {

      if ((justify != Left_Justify && justify != Center_Justify &&
           justify != Right_Justify) ||
          (vertLine != Vertical_Line && vertLine != No_Vertical_Line))
         throw new IllegalArgumentException ("Invalid column attribute");

      Object [] newOne = new Object [(titles == null) ? 1 : titles.length + 1];
      int [] newAttrs = new int [newOne.length];

      if (titles != null) {

         System.arraycopy (titles, 0, newOne, 0, titles.length);
         System.arraycopy (attributes, 0, newAttrs, 0, titles.length);
      }

      titles = newOne;
      attributes = newAttrs;

      titles [titles.length - 1] = title;
      attributes [titles.length - 1] = justify + vertLine;
   }

   public void setSelectionType (int newType) {

      if (newType != Single_Selection && newType != Forced_Single_Selection &&
          newType != Multiple_Selection)
         throw new IllegalArgumentException ("Invalid view selection");

      selectionType = newType;

      if (newType == Forced_Single_Selection && getSelected () == null && root != null)
         root.setSelected (true);
   }

   public void setSplitBar (int afterColumn, int windowPos) {

      if (titles == null)

         throw new IllegalArgumentException ("No columns in container");

      else {

         if (afterColumn < No_Split_Bar || afterColumn >= titles.length)
            throw new IllegalArgumentException ("Column out of range");

         if (windowPos < 10 || windowPos > 90)
            throw new IllegalArgumentException ("Split bar position out of range");
      }

      splitBarColumn = afterColumn;
      splitBarPosition = windowPos;

      setView (view);
   }

   public void refresh () {

      if (selectionType == Forced_Single_Selection &&
          getSelected () == null && root != null)
         root.setSelected (true);

      repaint ();
   }

   public void layout () {

      Dimension d = size ();
      Font font = getFont ();
      FontMetrics fm = (font == null) ? null : getFontMetrics (font);
      top = (title == null || fm == null) ? 0 : fm.getHeight () + 6;
      int extra = (view == Details_View && fm != null && titles != null) ?
                  Utilities.lineHeight (fm, titles, this) : 0;
      splitBarSpot = null;

      vScroll.reshape (d.width - 22, top + extra + 2, 20,
                       d.height - top - extra - 2 -
                       ((view == Details_View) ? 20 : 0));

      if (view == Details_View) {

         int leftWidth = d.width - 22;

         if (splitBarColumn != No_Split_Bar) {

            leftWidth = splitBarPosition * (leftWidth) / 100;

            detailsRightView.reshape (leftWidth + 7, top,
                                      d.width - leftWidth - 29,
                                      d.height - top - 20);
            hScrollRight.reshape (leftWidth + 7, d.height - 20,
                                  d.width - leftWidth - 29, 20);
            splitBarSpot = new HotSpot (new Rectangle (leftWidth, top, 7,
                                                       d.width - top),
                                        null, Split_Bar);
         }

         detailsLeftView.reshape (0, top, leftWidth, d.height - top - 20);
         hScrollLeft.reshape (0, d.height - 20, leftWidth, 20);

      } else
         currentCanvas.reshape (0, top, d.width - 20, d.height - top);
   }

   public void paint (Graphics g) {

      FontMetrics fm = getFontMetrics (getFont ());
      Dimension d = size ();
      int top = 0;

      if (title != null) {

         g.drawString (title, (d.width - fm.stringWidth (title)) / 2,
                       fm.getHeight () - 1);
         top = fm.getHeight () + 6;
      }

      if (view == Details_View)
         drawSplitBar (g, d, top);

      currentCanvas.repaint ();

      if (view == Details_View && splitBarColumn != No_Split_Bar)
         detailsRightView.repaint ();
   }

   private void drawSplitBar (Graphics g, Dimension d, int top) {

      int x = splitBarPosition * (d.width - 22) / 100;

      g.setColor (Color.white);
      g.drawLine (x, top + 1, x, d.height - 1);

      g.setColor (Color.darkGray);
      g.drawLine (x + 5, top + 1, x + 5, d.height - 1);
   }

   protected void drawDataArea (Graphics g, int top, Dimension d) {

      g.setColor (Color.darkGray);
      g.drawLine (0, top, d.width - 1, top);
      g.drawLine (0, top + 1, d.width - 2, top + 1);
      g.drawLine (0, top + 2, 0, d.height - 1);
      g.drawLine (1, top + 2, 1, d.height - 2);

      g.setColor (Color.white);
      g.drawLine (d.width - 1, top + 1, d.width - 1, d.height - 1);
      g.drawLine (d.width - 2, top + 2, d.width - 2, d.height - 1);
      g.drawLine (1, d.height - 1, d.width - 3, d.height - 1);
      g.drawLine (2, d.height - 2, d.width - 3, d.height - 2);

      g.setColor (Color.black);
   }

   private int lineCount () {

      int result = (root == null) ? 0 : root.itemCount ();

      if (root != null) {

         if (view == Tree_View) 
            result += childCount (root.getFirst ());

         if (view == Icon_View)
            result = flowCount (result);
      }

      return result;
   }

   private int flowCount (int fullCount) {

      FontMetrics fm = getFontMetrics (getFont ());
      ContainerItem item = null;
      int left = 4;
      Dimension box = null;

      if (root != null) {

         item = root.getFirst ();
         fullCount = 1;

      } else
         fullCount = 0;

      while (item != null) {

         box = iconView.iconSpan (item.getImage (),
                                  iconView.splitLine (item.textFor ()), fm);

         if (left + box.width > size ().width) {

            left = 4 + box.width;
            fullCount++;

         } else
            left += box.width + 8;

         item = item.getNext ();
      }

      return fullCount;
   }

   private int childCount (ContainerItem item) {

      int result = 0;

      while (item != null) {

         if (item.isExpanded ()) {

            ContainerItem child = item.getFirstChild ();

            result += child.itemCount () + childCount (child);
         }

         item = item.getNext ();
      }

      return result;
   }

   protected void setLines (int lines) {

      if (lineCount () <= lines)
         vScroll.disable ();
      else {
         vScroll.enable ();
         vScroll.setPageIncrement (lines);
         vScroll.setValues (vScroll.getValue (), lines, 0,
                            lineCount () - lines);
      }
   }

   protected void addHotSpot (int x, int y, int w, int h, ContainerItem item,
                              int type, Component source) {

      Point p = (source == null) ? new Point (0, 0) : source.location ();

      x += p.x;
      y += p.y;

      HotSpot hs = new HotSpot (new Rectangle (x, y, w, h), item, type);

      if (firstHotSpot == null)
         firstHotSpot = hs;
      else
         firstHotSpot.add (hs);
   }

   protected void clearHotSpotList () {

      firstHotSpot = splitBarSpot;

      if (firstHotSpot != null)
         firstHotSpot.chop ();
   }

   public ContainerItem getRoot () {return root;}

   protected Object [] getTitles () {return titles;}

   protected int [] getAttributes () {return attributes;}

   protected int getFirst () {return first;}

   public ContainerItem [] getSelected () {

      selected = null;

      if (root != null)
         root.traverse (this, Finding_Selected);

      return selected;
   }

   public void setAllSelection (boolean newState) {

      newSelectionState = newState;

      if (root != null)
         root.traverse (this, Setting_Selection);

      repaint ();
   }

   public boolean processItem (ContainerItem item, int tag) {

      switch (tag) {

         case Finding_Selected:
            if (item.isSelected ()) {

               ContainerItem [] newOne = new ContainerItem [((selected == null) ? 1 : selected.length + 1)];

               if (selected != null)
                  System.arraycopy (selected, 0, newOne, 0, selected.length);

               selected = newOne;
               selected [selected.length - 1] = item;
            }

            break;

         case Setting_Selection:
            item.setSelected (newSelectionState);
      }

      return true;
   }

   public boolean handleEvent (Event event) {

      if (event.id == Event.SCROLL_ABSOLUTE  ||
          event.id == Event.SCROLL_LINE_UP   ||
          event.id == Event.SCROLL_LINE_DOWN ||
          event.id == Event.SCROLL_PAGE_UP   ||
          event.id == Event.SCROLL_PAGE_DOWN) {

         boolean repaintIsNeeded = false;

         if (first != vScroll.getValue ()) {

            first = vScroll.getValue ();
            repaintIsNeeded = true;
         }

         if (event.target == hScrollLeft) {

            detailsLeftView.horizontalScroll ();
            repaintIsNeeded = true;
         }

         if (event.target == hScrollRight) {

            detailsRightView.horizontalScroll ();
            repaintIsNeeded = true;
         }

         if (repaintIsNeeded)
            repaint ();

      } else if (event.id == Event.WINDOW_MOVED) {

         boolean result = super.handleEvent (event);

         return result;

      } else
         return super.handleEvent (event);

      return true;
   }

   public boolean mouseDown (Event event, int x, int y) {

      if (firstHotSpot != null) {

         clicked = firstHotSpot.hit (x, y);

         if (clicked != null)
            if (clicked.getType () == Split_Bar) {

               int bx = splitBarPosition * (size ().width - 22) / 100;

               splitBar.show ();
               splitBar.reshape (bx, top, 7, size ().height - top);
            }
      }

      clickCount = event.clickCount;

      return super.mouseDown (event, x, y);
   }

   public boolean mouseDrag (Event event, int x, int y) {

      if (clicked != null)
         if (clicked.getType () == Split_Bar) {

            int nx = x * 100 / size ().width;

            if (nx >= 10 && nx <= 90)
               splitBar.move (x, splitBar.location ().y);
         }

      return super.mouseDrag (event, x, y);
   }

   public boolean mouseUp (Event event, int x, int y) {

      HotSpot test = null;
      ContainerItem [] list = null;
      ContainerItem work = null;
      ContainerItem ref = null;
      boolean redrawIsNeeded = false;

      if (firstHotSpot != null)
         test = firstHotSpot.hit (x, y);

      if (test == clicked && clicked != null) {

         if (clicked.getType () == Tree_Button) {

            work = (ContainerItem) (clicked.getReference ());
            work.setExpanded (!work.isExpanded ());
            redrawIsNeeded = true;
         }

         if (clicked.getType () == Selectable) {

            ref = (ContainerItem) (clicked.getReference ());

            if (selectionType == Single_Selection ||
                selectionType == Forced_Single_Selection) {

               list = getSelected ();

               if (list != null)
                  work = list [0];

               if (work != null && clickCount == 1 &&
                   (selectionType == Single_Selection || work != ref)) {

                  work.setSelected (false);
                  deliverEvent (new Event (this, Event.LIST_DESELECT, work));
                  redrawIsNeeded = true;
               }
            }

            if (work != ref || clickCount > 1) {

               if (!ref.isSelected () || clickCount == 1)
                  ref.setSelected (!ref.isSelected ());

               if (ref.isSelected ())
                  deliverEvent (new Event (this,
                                           (clickCount == 1) ?
                                            Event.LIST_SELECT :
                                            Event.ACTION_EVENT, ref));
               else
                  deliverEvent (new Event (this, Event.LIST_DESELECT, ref));

               redrawIsNeeded = true;
            }
         }
      }

      if (clicked != null)
         if (clicked.getType () == Split_Bar) {

            int nx = x * 100 / size ().width;

            splitBar.hide ();

            if (nx >= 10 && nx <= 90) {
               splitBarPosition = nx;
               layout ();
            }
         }

      if (redrawIsNeeded)
         repaint ();

      clicked = null;

      return super.mouseUp (event, x, y);
   }

   protected ContainerItem root;
   private Scrollbar hScrollLeft;
   private Scrollbar hScrollRight;
   private Scrollbar vScroll;
   private Canvas currentCanvas;
   private IconCanvas iconView;
   private ListCanvas listView;
   private TreeCanvas treeView;
   private DetailsCanvas detailsLeftView;
   private DetailsCanvas detailsRightView;
   private Canvas splitBar;
   private Object [] titles;
   private int [] attributes;
   private String title;
   private int view;
   private int first;
   private int top;
   private HotSpot splitBarSpot;
   private HotSpot firstHotSpot;
   private HotSpot clicked;
   private int clickCount;
   private int selectionType;
   private ContainerItem [] selected;
   private boolean newSelectionState;
   private int splitBarColumn;
   private int splitBarPosition;

   protected static final int Tree_Button = 1;
   protected static final int Selectable = 2;
   private static final int Split_Bar = 3;
   private static final int Finding_Selected = 1;
   private static final int Setting_Selection = 2;

   public static final int Details_View = 1;
   public static final int List_View = 2;
   public static final int Tree_View = 4;
   public static final int Icon_View = 8;
   public static final int Left_Justify = 0;
   public static final int Center_Justify = 1;
   public static final int Right_Justify = 2;
   public static final int No_Vertical_Line = 0;
   public static final int Vertical_Line = 4;
   public static final int Single_Selection = 0;
   public static final int Forced_Single_Selection = 1;
   public static final int Multiple_Selection = 2;
   public static final int No_Split_Bar = -1;
}
