/* NBSpiralLook: This class implements the sizing and painting methods for */
/*    the Simple notebook style.                                           */

package extend.awt;

import java.awt.*;

public class NBSpiralLook implements NBRenderer {

   protected NBSpiralLook (Notebook nb) {

      parent = nb;
      first = 0;
      activeLeft = false;
      activeRight = false;
      paging = null;
   }

   public Insets insets (FontMetrics fm, Dimension d) {

      int orientation = parent.getOrientation ();
      int top = (orientation == Notebook.Right) ? 4 : 22;
      int left = (orientation == Notebook.Bottom) ? 4 : 22;
      int right = 21;
      int bottom = 21;

      switch (orientation) {

         case Notebook.Right:
            right = maxWidth (fm);
            break;

         case Notebook.Bottom:
            bottom = Utilities.lineHeight (fm, parent.getTabs (), parent) + 27;
            break;
      }

      space = new Insets (top, left, bottom, right);

      return (new Insets (top, left, bottom + fm.getHeight (), right));
   }

   private int maxWidth (FontMetrics fm) {

      Object [] tabs = parent.getTabs ();
      int max = 0;

      if (tabs != null) 
         for (int loop = 0; loop < tabs.length; loop++)
            max = Math.max (max, Utilities.itemWidth (fm, tabs [loop], parent) +
                                 fm.stringWidth ("  "));

      return (max + 20);
   }

   public void paintTheLook (Graphics g, FontMetrics fm, Dimension d) {

      int orientation = parent.getOrientation ();
      int x1 = (orientation == Notebook.Right) ? 8 : 2;
      int y1 = (orientation == Notebook.Right) ? 2 : 8;
      int x2 = d.width - space.right;
      int y2 = d.height - space.bottom;
      int w  = fm.getHeight ();

      g.setColor (Color.darkGray);

      g.drawLine (x1, y1, x2, y1);
      g.drawLine (x2, y1, x2, y2);
      g.drawLine (x2, y2, x1, y2);
      g.drawLine (x1, y2, x1, y1);

      for (int loop = 1; loop < 7; loop++) {

         int c1 = loop * 3;
         int c2 = (loop - 1) * 3;

         g.drawLine (x1 + c1, y2 + c2, x1 + c1, y2 + c1);
         g.drawLine (x1 + c1, y2 + c1, x2 + c1, y2 + c1);
         g.drawLine (x2 + c1, y2 + c1, x2 + c1, y1 + c1);
         g.drawLine (x2 + c1, y1 + c1, x2 + c2, y1 + c1);
      }

      g.drawLine (x1 + 18, y2 + 19, x2 + 19, y2 + 19);
      g.drawLine (x2 + 19, y2 + 19, x2 + 19, y1 + 18);

      if (activeLeft)
         drawLeftArrow (g, fm, new Point (x2 - (w * 2), y2 - (w * 2) - 2),
                        Notebook.Page_Backward);
      else
         drawArrowFrame (g, x2 - (w * 2), y2 - w, w);

      if (activeRight)
         drawRightArrow (g, fm, new Point (x2 - w, y2 - (w * 2) - 2),
                         Notebook.Page_Forward);
      else
         drawArrowFrame (g, x2 - w, y2 - w, w);

      if (paging != null)
         g.drawString (paging, x2 - (w * 2) - fm.stringWidth (paging + " "),
                       y2 - 6);

      switch (orientation) {

         case Notebook.Right:
            drawVerticalSpiral (g, y1, y2);
            drawTabs (g, fm, new Point (x2, y1 + 18));
            break;

         case Notebook.Bottom:
            drawHorizontalSpiral (g, x1, x2);
            drawTabs (g, fm, new Point (x1 + 18, y2));
            break;
      }
   }

   private void drawVerticalSpiral (Graphics g, int low, int high) {

      int height = (high - low) + 1;
      int steps = height / 16;
      int y = ((height % 16) / 2) + 2;

      for (int loop = 0; loop < steps; loop++) {

         drawWestSpiral (g, y);
         y += 16;
      }
   }

   private void drawWestSpiral (Graphics g, int y) {

      g.setColor (Color.white);

      g.drawLine (1, y + 4, 13, y + 4);
      g.drawLine (1, y + 8, 13, y + 8);
      g.drawLine (0, y + 5, 0, y + 5);
      g.drawLine (0, y + 9, 0, y + 9);

      g.setColor (Color.lightGray);

      g.drawLine (1, y + 5, 14, y + 5);
      g.drawLine (1, y + 9, 14, y + 9);
      g.drawLine (14, y + 4, 14, y + 4);
      g.drawLine (14, y + 8, 14, y + 8);

      g.setColor (Color.darkGray);

      g.drawLine (10, y + 3, 15, y + 3);
      g.drawLine (1, y + 6, 16, y + 6);
      g.drawLine (8, y + 7, 16, y + 7);
      g.drawLine (1, y + 11, 16, y + 11);
      g.drawLine (10, y + 12, 15, y + 12);
      g.drawLine (16, y + 4, 16, y + 11);
      g.drawLine (15, y + 4, 15, y + 11);
   }

   private void drawHorizontalSpiral (Graphics g, int low, int high) {

      int width = (high - low) + 1;
      int steps = width / 16;
      int x = ((width % 16) / 2) + 2;

      for (int loop = 0; loop < steps; loop++) {

         drawNorthSpiral (g, x);
         x += 16;
      }
   }

   private void drawNorthSpiral (Graphics g, int x) {

      g.setColor (Color.white);

      g.drawLine (x + 4, 1, x + 4, 13);
      g.drawLine (x + 8, 1, x + 8, 13);
      g.drawLine (x + 5, 0, x + 5, 0);
      g.drawLine (x + 9, 0, x + 9, 0);

      g.setColor (Color.lightGray);

      g.drawLine (x + 5, 1, x + 5, 14);
      g.drawLine (x + 9, 1, x + 9, 14);
      g.drawLine (x + 4, 14, x + 4, 14);
      g.drawLine (x + 8, 14, x + 8, 14);

      g.setColor (Color.darkGray);

      g.drawLine (x + 3, 10, x + 3, 15);
      g.drawLine (x + 6, 1, x + 6, 16);
      g.drawLine (x + 7, 8, x + 7, 16);
      g.drawLine (x + 11, 1, x + 11, 16);
      g.drawLine (x + 12, 10, x + 12, 15);
      g.drawLine (x + 4, 16, x + 11, 16);
      g.drawLine (x + 4, 15, x + 11, 15);
   }
  
   private void drawTabs (Graphics g, FontMetrics fm, Point location) {

      Object [] tabs = parent.getTabs ();
      int orientation = parent.getOrientation ();
      Color [] bgColors = parent.getBgColors ();
      Color [] fgColors = parent.getFgColors ();
      int lHeight = Utilities.lineHeight (fm, tabs, parent);
      Point p = new Point (location.x, location.y);
      int item = first;
      int max = (orientation == Notebook.Right) ?
                parent.size ().height - space.bottom :
                parent.size ().width - space.right;
      boolean atEnd = false;

      if (first > 0)
         drawBackArrow (g, fm, orientation, p);

      for (int loop = 0; loop < tabs.length && !atEnd; loop++) {

         switch (orientation) {

            case Notebook.Right:
               drawEastTab (g, fm, tabs [item], bgColors [item],
                            fgColors [item], p, item, lHeight);

               if (item < (tabs.length - 1))
                  if (lHeight + 2 +
                      ((item == tabs.length - 2) ?
                        0 : fm.getHeight ()) > max - p.y) {

                      p.y = parent.size ().height - fm.getHeight () - 1;
                      drawFwdArrow (g, fm, orientation, p);
                      atEnd = true;
                  }

               break;
  
            case Notebook.Bottom:
               drawSouthTab (g, fm, tabs [item], bgColors [item],
                             fgColors [item], p, item, lHeight);

               if (item < (tabs.length - 1))
                  if (Utilities.itemWidth (fm, tabs [item + 1], parent) +
                      fm.stringWidth ("  ") + 2 > max - p.x) {

                      p.x = parent.size ().width - fm.getHeight () - 1;
                      drawFwdArrow (g, fm, orientation, p);
                      atEnd = true;
                  }

               break;
         }

         item++;

         if (!atEnd)
            atEnd = (item > tabs.length - 1);
      }
   }

   private void drawEastTab (Graphics g, FontMetrics fm, Object tab,
                             Color bgc, Color fgc, Point p, int index,
                             int lHeight) {

      int currentTab = parent.getCurrentTab ();
      int x = (index < currentTab) ? 18 : (index - currentTab) * 3;
      int dx = space.right - 20;
      int delta = (lHeight - fm.getHeight ()) / 2;

      if (x > 18) x = 18;
      if (x > 0)  x++;

      g.setColor (Color.lightGray);

      g.fillRect (p.x + x, p.y, dx, lHeight);

      g.setColor (Color.darkGray);

      g.drawLine (p.x + x, p.y, p.x + x + dx, p.y);
      g.drawLine (p.x + x + dx, p.y, p.x + x + dx, p.y + lHeight);
      g.drawLine (p.x + x + dx, p.y + lHeight, p.x + x, p.y + lHeight);

      g.setColor (bgc);

      g.fillRect (p.x + x + 2, p.y + 2, dx - 3, lHeight - 3);

      g.setColor (fgc);

      if (tab instanceof Image)
         g.drawImage ((Image) tab, p.x + x + 6, p.y + 4, parent);
      else
         g.drawString (tab.toString (), p.x + x + 6, p.y + lHeight - delta - 3);

      parent.addHotSpot (p.x + x + 2, p.y + 2, dx - 4, lHeight,
                         Notebook.Tab_Click, new Integer (index));

      p.y += (lHeight + 2);
   }

   private void drawSouthTab (Graphics g, FontMetrics fm, Object tab,
                              Color bgc, Color fgc, Point p, int index,
                              int lHeight) {

      int currentTab = parent.getCurrentTab ();
      int y = (index < currentTab) ? 18 : (index - currentTab) * 3;
      int dx = Utilities.itemWidth (fm, tab, parent) + fm.stringWidth ("  ") + 2;
      int delta = (lHeight - fm.getHeight ()) / 2;
      int tx = p.x + ((dx - Utilities.itemWidth (fm, tab, parent)) / 2) + 3;

      if (y > 18) y = 18;
      if (y > 0) y++;

      g.setColor (Color.lightGray);

      g.fillRect (p.x, p.y + y, dx, lHeight);

      g.setColor (Color.darkGray);

      g.drawLine (p.x, p.y + y, p.x, p.y + y + lHeight);
      g.drawLine (p.x + dx, p.y + y, p.x + dx, p.y + y + lHeight);
      g.drawLine (p.x + dx, p.y + y + lHeight, p.x, p.y + y + lHeight);

      g.setColor (bgc);

      g.fillRect (p.x + 2, p.y + y + 2, dx - 3, lHeight - 3);

      g.setColor (fgc);

      if (tab instanceof Image)
         g.drawImage ((Image) tab, tx, p.y + y + 4, parent);
      else
         g.drawString (tab.toString (), tx, p.y + y + lHeight - delta - 5);

      parent.addHotSpot (p.x + 2, p.y + y + 2, dx - 3, lHeight - 3,
                         Notebook.Tab_Click, new Integer (index));

      p.x += (dx + 2);
   }

   private void drawBackArrow (Graphics g, FontMetrics fm, int side, Point p) {

      if (side == Notebook.Right)
         drawUpArrow (g, fm, p);
      else
         drawLeftArrow (g, fm, p, Notebook.Up_Click);
   }

   private void drawUpArrow (Graphics g, FontMetrics fm, Point p) {

      int width = fm.getHeight ();
      int x = p.x + 18;
      int y = p.y;

      drawArrowFrame (g, x, y, width);
      drawVerticalArrow (g, x, y, width, 1);

      parent.addHotSpot (x, y, width, width, Notebook.Up_Click, null);

      p.y = y + width + 2;
   }

   private void drawLeftArrow (Graphics g, FontMetrics fm, Point p, int clickType) {

      int width = fm.getHeight ();
      int x = p.x;
      int y = p.y + 18;

      drawArrowFrame (g, x, y, width);
      drawHorizontalArrow (g, x, y, width, 1);

      parent.addHotSpot (x, y, width, width, clickType, null);

      p.x = x + width + 2;
   }

   private void drawFwdArrow (Graphics g, FontMetrics fm, int side, Point p) {

      if (side == Notebook.Right)
         drawDownArrow (g, fm, p);
      else
         drawRightArrow (g, fm, p, Notebook.Down_Click);
   }

   private void drawRightArrow (Graphics g, FontMetrics fm, Point p, int clickType) {

      int width = fm.getHeight ();
      int x = p.x;
      int y = p.y + 18;

      drawArrowFrame (g, x, y, width);
      drawHorizontalArrow (g, x + width, y, width, -1);

      parent.addHotSpot (x, y, width, width, clickType, null);

      p.x = x + width + 2;
   }

   private void drawDownArrow (Graphics g, FontMetrics fm, Point p) {

      int width = fm.getHeight ();
      int x = p.x + 18;
      int y = p.y;

      drawArrowFrame (g, x, y, width);
      drawVerticalArrow (g, x, y + width - 1, width, -1);

      parent.addHotSpot (x, y, width, width, Notebook.Down_Click, null);
   }

   private void drawArrowFrame (Graphics g, int x, int y, int width) {

      g.setColor (Color.darkGray);

      g.drawLine (x, y, x + width - 1, y);
      g.drawLine (x + width - 1, y, x + width - 1, y + width - 1);
      g.drawLine (x + width - 1, y + width - 1, x, y + width - 1);
      g.drawLine (x, y + width - 1, x, y);
   }

   private void drawVerticalArrow (Graphics g, int x, int y, int width, int dir) {

      int [] px = new int [7];
      int [] py = new int [7];

      px [0] = x + (width / 2);   py [0] = y + (4 * dir);
      px [1] = x + width - 4;     py [1] = y + (width / 2 * dir);
      px [2] = px [0] + 2;        py [2] = py [1];
      px [3] = px [2];            py [3] = y + ((width - 4) * dir);
      px [4] = px [0] - 2;        py [4] = py [3];
      px [5] = px [4];            py [5] = py [2];
      px [6] = x + 3;             py [6] = py [1];

      g.setColor (Color.black);
      g.fillPolygon (px, py, 7);
   }
  
   private void drawHorizontalArrow (Graphics g, int x, int y, int width, int dir) {

      int [] px = new int [7];
      int [] py = new int [7];

      px [0] = x + (4 * dir);            py [0] = y + (width / 2);
      px [1] = x + (width / 2 * dir);    py [1] = y + width - 4;
      px [2] = px [1];                   py [2] = py [0] + 2;
      px [3] = x + ((width - 4) * dir);  py [3] = py [2];
      px [4] = px [3];                   py [4] = py [0] - 2;
      px [5] = px [2];                   py [5] = py [4];
      px [6] = px [1];                   py [6] = y + 3;

      g.setColor (Color.black);
      g.fillPolygon (px, py, 7);
   }
  
   public void scroll (int amount) {

      first += amount;
   }

   public void setButtonState (int which, boolean newState) {

      switch (which) {

         case Left_Button:
            activeLeft = newState;
            break;

         case Right_Button:
            activeRight = newState;
            break;
      }
   }

   public void setPaging (String text) {paging = text;}

   private Notebook parent;
   private Insets space;
   private int index;
   private int first;
   private boolean activeLeft;
   private boolean activeRight;
   private String paging;
}
