/* Notebook: This class provides for a simple notebook control.  It may be */
/*    used anywhere a Panel can be used.                                   */

package extend.awt;

import java.awt.*;
import java.awt.image.*;

public class Notebook extends Panel {

   public Notebook (int pos, int style) {

      setSideAndLook (pos, style);

      setLayout (layout = new CardLayout ());
      setBackground (Color.lightGray);

      currentTab = 0;

      setOrientationAndStyle (pos, style);

      hsList = null;
   }

   public void setOrientation (int pos) {

      setOrientationAndStyle (pos, look);
   }

   public int getOrientation () {return side;}

   public void setStyle (int style) {

      setOrientationAndStyle (side, style);
   }

   public int getStyle () {return look;}

   public void setOrientationAndStyle (int pos, int style) {

      setSideAndLook (pos, style);

      switch (style) {

         case Simple:   renderer = new NBSimpleLook   (this); break;
         case Spiral:   renderer = new NBSpiralLook   (this); break;
//       case Extended: renderer = new NBExtendedLook (this); break;
      }

      currentPage = 1;

      buildStatusBar ();

      if (tabs != null)
         rearrainge ();
   }

   private void setSideAndLook (int pos, int style) {

      switch (style) {

         case Simple:
            if (pos != Top && pos != Left && pos != Right && pos != Bottom)
               throw new IllegalArgumentException ("Invalid position for Simple style");
            break;

         case Spiral:
            if (pos != Right && pos != Bottom)
               throw new IllegalArgumentException ("Invalid position for Spiral style");
            break;

//       case Extended:
//          if (pos != Top)
//             throw new IllegalArgumentException ("Invalid position for Extended style");
//          break;

         default:
            throw new IllegalArgumentException ("Invalid notebook style");
      }

      side = pos;
      look = style;
   }

   private void rearrainge () {

      Component [] list = getComponents ();

      layout ();

      if (list != null) {

         for (int loop = 0; loop < list.length; loop++) {

            list [loop].layout ();
            list [loop].repaint ();
         }
      }

      repaint ();
   }

   public void addPage (Object label, Component component, PageVerifier verifier) {

      addPage (label, Color.lightGray, Color.black, component, verifier);
   }

   public void addPage (Object label, Color bgColor, Component component, PageVerifier verifier) {

      addPage (label, bgColor, Color.black, component, verifier);
   }

   public void addPage (Object label, Color bgColor, Color fgColor,
                        Component component, PageVerifier verifier) {

      int place = pageLocation (label);

      if (place < 0) {

         Object [] newOne = new Object [(tabs == null) ? 1 : tabs.length + 1];
         Color [] newBgClrs = new Color [newOne.length];
         Color [] newFgClrs = new Color [newOne.length];
         int [] newCounts = new int [newOne.length];

         if (tabs != null) {

            System.arraycopy (tabs, 0, newOne, 0, tabs.length);
            System.arraycopy (bgColors, 0, newBgClrs, 0, bgColors.length);
            System.arraycopy (fgColors, 0, newFgClrs, 0, fgColors.length);
            System.arraycopy (counts, 0, newCounts, 0, counts.length);
         }

         tabs = newOne;
         bgColors = newBgClrs;
         fgColors = newFgClrs;
         counts = newCounts;

         place = tabs.length - 1;
         tabs [place] = label;
         bgColors [place] = bgColor;
         fgColors [place] = fgColor;
         counts [place] = 0;
      }

      counts [place]++;

      PageVerifier newList [] = new PageVerifier [((place == 0 && counts [0] == 1) ?
                                                   1 : verifierList.length + 1)];

      if (newList.length > 1)
         System.arraycopy (verifierList, 0, newList, 0, verifierList.length);

      verifierList = newList;

      verifierList [verifierList.length - 1] = verifier;

      add (label.toString () + Separator + counts [place], component);
      showPage ();
   }

   private int pageLocation (Object tab) {

      int result = -1;

      if (tabs != null)
         for (int loop = 0; loop < tabs.length; loop++)
            if (tabs [loop].toString ().equals (tab.toString ()))
               result = loop;

      return result;
   }

   private PageVerifier pageVerifier () {

      int result = 0;

      for (int loop = 0; loop < currentTab; loop++)
         result += counts [loop];

      return verifierList [result + currentPage - 1];
   }

   private boolean okToFlip () {

      PageVerifier verifier = pageVerifier ();
      boolean result = true;

      if (verifier != null)
         result = verifier.okToChangePage ();

      return result;
   }

   public Insets insets () {

      FontMetrics fm = getFontMetrics (getFont ());
      Dimension d = size ();

      return renderer.insets (fm, d);
   }

   public void paint (Graphics g) {

      FontMetrics fm = getFontMetrics (getFont ());
      Dimension d = size ();

      hsList = null;

      renderer.paintTheLook (g, fm, d);
   }

   public boolean imageUpdate (Image image, int flags, int x, int y, int w,
                               int h) {

      boolean result = super.imageUpdate (image, flags, x, y, w, h);

      if ((flags & (ALLBITS | ABORT)) != 0)
         rearrainge ();

      return result;
   }

   protected void addHotSpot (int x, int y, int w, int h, int type, Object ref) {

      HotSpot hs = new HotSpot (new Rectangle (x, y, w, h), ref, type);

      if (hsList == null)
         hsList = hs;
      else
         hsList.add (hs);
   }

   protected Object [] getTabs () {return tabs;}

   protected Color [] getBgColors () {return bgColors;}

   protected Color [] getFgColors () {return fgColors;}

   protected int getCurrentTab () {return currentTab;}

   private void buildStatusBar () {

      if (counts != null && look == Spiral) {

         boolean newLeftState = false;
         boolean newRightState = false;

         if (counts [currentTab] > 1)
            renderer.setPaging ("Page " + currentPage + " of " +
                                 counts [currentTab]);
         else
            renderer.setPaging (null);

         if (counts [currentTab] > 1 || currentTab < tabs.length - 1)
            newRightState = true;

         if (currentPage > 1 || currentTab > 0)
            newLeftState = true;

         renderer.setButtonState (NBRenderer.Left_Button, newLeftState);
         renderer.setButtonState (NBRenderer.Right_Button, newRightState);
      }
   }
   
   private void showPage () {

      buildStatusBar ();

      if (tabs != null) {

         layout.show (this, pageShowing ());

         if (!lastTop.equals ("") && !lastTop.equals (pageShowing ()))
            deliverEvent (new Event (this, Event.ACTION_EVENT, pageShowing ()));
         
         lastTop = pageShowing ();
      }
   }

   public String pageShowing () {

      return (tabs [currentTab].toString () + Separator + currentPage);
   }

   public void verifyPage () {okToFlip ();}

   public void turnToPage (Object tab, int page) {

      int newTab = pageLocation (tab);

      if (newTab >= 0)
         if (page < 1 || page > counts [newTab])
            newTab = -1;

      if (newTab < 0)
         throw new IllegalArgumentException ("Invalid notebook page requested.");

      if (okToFlip ()) {

         currentTab = newTab;
         currentPage = page;

         showPage ();

         if (getFont () != null)
            rearrainge ();
      }
   }

   public boolean mouseDown (Event event, int x, int y) {

      if (hsList != null)
         clicked = hsList.hit (x, y);

      return super.mouseDown (event, x, y);
   }

   public boolean mouseUp (Event event, int x, int y) {

      HotSpot test = null;

      if (hsList != null)
         test = hsList.hit (x, y);

      if (test == clicked && clicked != null)

         switch (clicked.getType ()) {

            case Tab_Click:
               if (okToFlip ()) {

                  currentTab = ((Integer) (test.getReference ())).intValue ();
                  currentPage = 1;

                  showPage ();
                  rearrainge ();
               }
               break;

            case Up_Click:
               renderer.scroll (-1);
               repaint ();
               break;

            case Down_Click:
               renderer.scroll (1);
               repaint ();
               break;

            case Page_Backward:
               if (okToFlip ()) {
     
                  if (currentPage == 1) {
    
                     if (currentTab > 0) {

                        currentTab--;
                        currentPage = counts [currentTab];
                     }
    
                  } else
                     currentPage--;

                  showPage ();
                  rearrainge ();
               }
               break;

            case Page_Forward:
               if (okToFlip ()) {
     
                  if (currentPage == counts [currentTab]) {

                     if (currentTab < tabs.length - 1) {

                        currentTab++;
                        currentPage = 1;
                     }

                  } else
                     currentPage++;

                  showPage ();
                  rearrainge ();
               }
               break;
         }

      clicked = null;

      return super.mouseUp (event, x, y);
   }

   private Object [] tabs;
   private Color [] bgColors;
   private Color [] fgColors;
   private int [] counts;
   private PageVerifier [] verifierList;
   private NBRenderer renderer;
   private int currentTab;
   private int currentPage;
   private CardLayout layout;
   private int side;
   private int look;
   private String lastTop = "";
   private HotSpot hsList;
   private HotSpot clicked;

   protected static final int Tab_Click = 0;
   protected static final int Up_Click = 1;
   protected static final int Down_Click = 2;
   protected static final int Page_Backward = 3;
   protected static final int Page_Forward = 4;

   public static final int Top = 0;
   public static final int Left = 1;
   public static final int Right = 2;
   public static final int Bottom = 3;

   public static final int Simple = 0;
   public static final int Spiral = 1;
// public static final int Extended = 2;

   public static final String Separator = ".";
}
